/*
 *	recite - english text speech synthesizer
 *	Copyright (C) 1993 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 1, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: tables to map english to phonemes
 *
 *
 *	English to Phoneme rules.
 *
 *	Derived from:
 *
 *	     AUTOMATIC TRANSLATION OF ENGLISH TEXT TO PHONETICS
 *		    BY MEANS OF LETTER-TO-SOUND RULES
 *
 *			NRL Report 7948
 *
 *		      January 21st, 1976
 *	    Naval Research Laboratory, Washington, D.C.
 *
 *
 *	Published by the National Technical Information Service as
 *	document "AD/A021 929".
 *
 *
 *
 *	The Phoneme codes:
 *
 *	Vowels
 *		AA	fAther
 *		AE	fAt
 *		AH	bUt
 *		AO	lAWn
 *		AW	hOW
 *		AX	About
 *		(AXR)	bAR		peephole
 *		AY	hIde
 *		EH	gEt
 *		ER	mURdER
 *		(EXR)	bARE		peephole
 *		EY	gAte
 *		IH	bIt
 *		IX	Imp		PM
 *		(IXR)	bEER		peephole
 *		IY	bEEt
 *		OW	lOne
 *		(OXR)	bOAR		peephole
 *		OY	tOY
 *		UH	fUll
 *		UW	fOOl
 *		(UXR)	pOOR		peephole
 *		YU	YOU
 *
 *	Sonorants:
 *		EL	as in?		PM
 *		h	How
 *		HX	as in?		PM
 *		l	Laugh
 *		LX	as in?		PM
 *		r	Rate
 *		RX	as in?		PM
 *		w	Wear
 *		WH	WHere
 *		y	Young
 *
 *	Nasals:
 *		EM	as in?		PM
 *		EN	as in?		PM
 *		m	suM
 *		n	suN
 *		NG	suNG
 *
 *	Fricatives:
 *		DH	eiTHer
 *		f	Fault
 *		s	Sue
 *		SH	leaSH
 *		TH	eTHer
 *		v	Vault
 *		z	Zoo
 *		ZH	leiSure
 *
 *	Plosives:
 *		b	Back
 *		d	Dime
 *		DX	as in?		PM
 *		g	Goat
 *		GP	as in?		PM
 *		k	Coat
 *		KP	as in?		PM
 *		p	Pack
 *		t	Time
 *		TQ	as in?		PM
 *
 *	Afficates:
 *		CH	CHar
 *		j	Jar
 *
 *	Pseudo Vowels:
 *		(AXP)	as in?		PM
 *
 *
 *
 *	Rules are made up of four parts:
 *
 *		The left context.
 *		The text to match.
 *		The right context.
 *		The phonemes to substitute for the matched text.
 *
 *	Procedure:
 *
 *		Seperate each block of letters (apostrophes included)
 *		and add a space on each side.  For each unmatched
 *		letter in the word, look through the rules where the
 *		text to match starts with the letter in the word.  If
 *		the text to match is found and the right and left
 *		context patterns also match, output the phonemes for
 *		that rule and skip to the next unmatched letter.
 *
 *
 *	Special Context Symbols:
 *
 *		#	One or more vowels
 *		:	Zero or more consonants
 *		^	One consonant.
 *		.	One of B, D, V, G, J, L, M, N, R, W or Z (voiced
 *			consonants)
 *		%	One of ER, E, ES, ED, ING, ELY (a suffix)
 *			(Found in right context only)
 *		+	One of E, I or Y (a "front" vowel)
 *		@	one of T S R D L Z N J TH CH SH (left only)
 *		&	one of S C G Z X J CH SH (left only)
 *
 */

#include <rules.h>

/* Context definitions */
static char Anything[] = "";    /* No context requirement */
static char Nothing[] = " ";    /* Context is beginning or end of word */

/* Phoneme definitions */
static char Pause[] = " ";      /* Short silence */
static char Silent[] = "";      /* No phonemes */

#define LEFT_PART	0
#define MATCH_PART	1
#define RIGHT_PART	2
#define OUT_PART	3


/*0 = Punctuation */

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t punct_rules[] =
{
        {Anything,      "0",            Anything,       " zIY4rOW "      },
        {Nothing,       "1ST",          Anything,       "fER4st"         },
        {Nothing,       "10TH",         Anything,       "tEH4nTH"        },
        {Nothing,       "10 ",          Anything,       " tEH4n "        },
        {Anything,      "1",            Anything,       " wAH4n "        },
        {Nothing,       "2ND",          Anything,       "sEH4kAEnd"      },
        {Anything,      "2",            Anything,       " tUW4 "         },
        {Nothing,       "3RD",          Anything,       "THER4d"         },
        {Anything,      "3",            Anything,       " THrIY4 "       },
        {Anything,      "4",            Anything,       " fAX4r "        },
        {Nothing,       "5TH",          Anything,       "fIH4fTH"        },
        {Anything,      "5",            Anything,       " fAY4v "        },
        {Anything,      "6",            Anything,       " sIH4ks "       },
        {Anything,      "7",            Anything,       " sEH4vAEn "     },
        {Nothing,       "8TH",          Anything,       "EY4TH"          },
        {Anything,      "8",            Anything,       " EY4t "         },
        {Anything,      "9",            Anything,       " nAY4n "        },
	{Anything,	" ",            "'",            Silent		},
	{Anything,	" ",            Anything,       Pause		},
	{Anything,	"-",            Anything,       Silent		},
        {Anything,      "...",          Anything,       " THrUW "       },
        {Anything,      ".",            "?",            " pOY2nt "       },
        {Anything,      ".",            Nothing,        "."             },
        {Anything,      ".",            Anything,       Pause           },
        {Anything,      "!",            Anything,       "."             },
        {Anything,      "\"",           Nothing,        "-AH5nkwOWt- "   },
        {Anything,      "\"",           Anything,       "kwOW4t-"        },
        {Anything,      "##",           Anything,       "#"             },
        {Anything,      "#",            Anything,       " nAH4mbER "     },
        {"C",           "'S",           Anything,       "s"             },
        {"G",           "'S",           Anything,       "z"             },
        {"&",           "'S",           Anything,       "IHz"           },
	{".",           "'S",           Anything,       "z"     },
	{"#:&E",        "'S",           Anything,       "IHz"           },
	{"#:.E",        "'S",           Anything,       "z"     },
	{"#:^E",        "'S",           Anything,       "s"             },
	{"#",           "'S",           Anything,       "z"     },
        {Anything,      "'S",           Anything,       "s"             },
        {Anything,      "'T",           Anything,       "t"             },
        {Anything,      "'LL",          Anything,       "l"             },
        {Anything,      "'D",           Anything,       "d"             },
        {Anything,      "'M",           Anything,       "m"             },
	{Anything,      "$",            Anything,       " dAA2lER "      },
	{Anything,      "%",            Anything,       " pERsEH2nt "    },
	{Anything,      "&",            Anything,       " AEnd "        },
	{Anything,	"'",            Anything,       Silent  },
        {Anything,      "*",            Anything,       " AE4stERIHsk "  },
        {Anything,      "+",            Anything,       " plAH4s "       },
	{Anything,	",",            Anything,       Pause   },
	{Nothing,       "-",            Nothing,        "-"             },
	{Anything,      "-",            Anything,       Silent          },
        {Anything,      "/",            Anything,       " slAE4SH "      },
        {Anything,      ":",            Anything,       ". "            },
        {Anything,      ";",            Anything,       "."             },
        {Anything,      "<",            Anything,       " lEH4s DHAEn "  },
        {Anything,      "=",            Anything,       " IY4kwAElz "    },
        {Anything,      ">",            Anything,       " grEY4tER DHAEn " },
        {Anything,      "?",            Anything,       "."             },
        {Anything,      "@",            Anything,       " AE6t "         },
        {Anything,      "(",            Anything,       ","             },
        {Anything,      ")",            Anything,       ","             },
        {Anything,      "^",            Anything,       " kAE4rIHt "     },
        {Anything,      "~",            Anything,       "nAA4t "         },
        {Anything,      "\\",            Anything,       Pause           },
        {Anything,      "[",            Anything,       Pause           },
        {Anything,      "]",            Anything,       Pause           },
        {Anything,      "{",            Anything,       Pause           },
        {Anything,      "}",            Anything,       Pause           },
        {Anything,      "|",            Anything,       "AX4r "          },
        {Anything,      "_",            Anything,       Pause           },
        {Anything,      "`",            Anything,       Pause           },
        {Anything,      "",             Anything,       Pause           },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t A_rules[] =
{
	{Nothing,       "A. ",          Anything,       "EH4y. "         },
	{Anything,	"A",            Nothing,        "AH"    },
	{Nothing,	"ARE",          Nothing,        "AAr"   }, 
        {Nothing,       "AREN'T",       Nothing,        "AA2rIHnt"       },
        {Nothing,       "AND",          Nothing,        "AEnd"          },
        {Nothing,       "AS",           Nothing,        "AEz"           },
        {Nothing,       "AT",           Nothing,        "AEt"           },
        {Nothing,       "AN",           Nothing,        "AEn"           },
        {Nothing,       "AVE.",         Nothing,        "AE2vIHnUW"      },
	{Nothing,	"AR",           "O",            "AXr"   },
	{Anything,	"AR",           "#",            "EH4r"   },
	{" ^",          "AS",           "#",            "EY4s"   },
	{Anything,	"A",            "WA",           "AX"    },
	{Anything,	"AW",           Anything,       "AO5"    },
	{" :",          "ANY",          Anything,       "EH4nIY" },
	{Anything,	"A",            "^+#",          "EY5"    },
	{"#:",          "ALLY",         Anything,       "AElIY"		},
	{Nothing,	"AL",           "#",            "AEl"		},
	{Anything,	"AGAIN",        Anything,       "AXgEH4n" },
        {Anything,      "ABOUT",        Anything,       "AXbAW1t"        },
        {Anything,      "APPLE",        Anything,       "AE3pAEl"        },
	{"#:",          "AG",           "E",            "IHj"   },
	{Anything,	"A",            "^%",           "EY2"    },
	{Anything,      "AL",           "F ",           "AE2"            },
	{Anything,	"A",            "^+:#",         "AE1"    },
	{" :",          "A",            "^+ ",          "EY4"    },
	{Nothing,	"ARR",          Anything,       "AXr"   },
	{Anything,	"ARR",          Anything,       "AE4r"   },
	{" :",          "AR",           Nothing,        "AA5r"   },
	{Anything,	"AR",           Nothing,        "ER"    },
	{Anything,	"AR",           Anything,       "AA1r"   },
	{Anything,	"AIR",          Anything,       "EH4r"   },
	{Anything,	"AI",           Anything,       "EY4"    },
	{Anything,	"AY",           Anything,       "EY5"    },
	{Anything,	"AU",           Anything,       "AO4"    },
	{"#:",          "AL",           Nothing,        "AEl"   },
	{"#:",          "ALS",          Nothing,        "AElz"  },
	{Anything,	"ALK",          Anything,       "AO4k"   },
	{Anything,	"AL",           "^",            "AOl"   },
	{" :",          "ABLE",         Anything,       "EY4bAEl" },
	{Anything,	"ABLE",         Anything,       "AXbAEl" },
	{Anything,      "A",            "VO",           "EY4"            },
	{Anything,	"ANG",          "+",            "EY4nj"  },
        {Nothing,       "AMIGA",        Anything,       "AHmIY5gAH"      },
        {Anything,      "ATARI",        Anything,       "AHtAA4rIY"      },
        {Anything,      "ATOMIC",       Anything,       "AHtAA4mIHk"     },
        {Anything,      "A",            "TOM",          "AE2"            },
        {Anything,      "A",            "TTI",          "AE"            },
        {Nothing,       "AT",           Nothing,        "AEt"           },
        {Nothing,       "A",            "T",            "AH"            },
        {Anything,      "A",            "A",            Silent          },
	{Anything,	"A",            Anything,       "AE"    },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t B_rules[] =
{
	{Nothing,       "B",            Nothing,        "bIY4"         },
        {Anything,      "BECAUSE",      Anything,       "bIHkAH1z"       },
	{Nothing,	"BE",           "^#",           "bIH"   },
	{Anything,	"BEING",        Anything,       "bIY4IHNG" },
	{Nothing,	"BOTH",         Nothing,        "bOW4TH" },
        {Nothing,       "BY",           Nothing,        "bAY"           },
        {Nothing,       "BUT",          Nothing,        "bAHt"          },
        {Nothing,       "BEEN",         Nothing,        "bIHn"          },
	{Nothing,	"BUS",          "#",            "bIH4z"  },
        {Anything,      "BREAK",        Anything,       "brEY4k"         },
	{Anything,	"BUIL",         Anything,       "bIH4l"  },
        {"B",           "B",            Anything,       Silent          },
	{Anything,	"B",            Anything,       "b"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t C_rules[] =
{
        {Nothing,       "C",            Nothing,        "sIY4"         },
        {Nothing,       "CONSISTENT",   Anything,       "kAEnsIH5stIHnt" },
        {Anything,      "COMMODORE",    Anything,       "kAA4mAXdAXr"  },
        {Anything,      "CERTAIN",      Anything,       "sER5tIHn"       },
        {Anything,      "CONTOUR",      Anything,       "kAA5ntUH1r"      },
        {Nothing,       "COS",          Nothing,        "kOW2sAY1n"       },
	{Nothing,	"CH",           "^",            "k"     },
	{"^E",          "CH",           Anything,       "k"     },
	{Anything,      "CHA",          "R#",           "kEH5"           },
	{Anything,	"CH",           Anything,       "CH"    },
	{" S",          "CI",           "#",            "sAY4"   },
	{Anything,	"CI",           "A",            "SH"    },
	{Anything,	"CI",           "O",            "SH"    },
	{Anything,	"CI",           "EN",           "SH"    },
	{Anything,      "CITY",         Anything,       "sIHtIY"        },
	{Anything,	"C",            "+",            "s"     },
	{Anything,	"CK",           Anything,       "k"     },
	{Anything,	"COM",          "%",            "kAHm"  },
        {Anything,      "CUIT",         Anything,       "kIHt"          },
        {Anything,      "CREA",         "^+",           "krIYEY4"        },
        {Anything,      "CC",           "+",            "ks"            },
        {Anything,      "CC",           Anything,       "k"             },
	{Anything,	"C",            Anything,       "k"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t D_rules[] =
{
	{Nothing,       "DR.",          Nothing,        "dAA4ktER"       },
        {Nothing,       "D",            Nothing,        " dIY4 "         },
	{"#:",          "DED",          Nothing,        "dIHd"  },
	{".E",          "D",            Nothing,        "d"     },
	{"#:^E",        "D",            Nothing,        "t"     },
	{Nothing,	"DE",           "^#",           "dIH1"   },
	{Nothing,	"DO",           Nothing,        "dUW"   },
	{Nothing,	"DOES",         Anything,       "dAHz"  },
	{Anything,      "DONE",         Nothing,        "dAH5n"          },
	{Anything,	"DOING",        Anything,       "dUW4IHNG" },
	{Anything,	"DOW",          Anything,       "dAW2"   },
	{"#",		"DU",           "A",            "jUW"   },
	{Anything,      "DUC",          "+",            "dUW4s"          },
        {"#",           "DU",           "^#",           "jAX"           },
        {"D",           "D",            Anything,       Silent          },
        {Anything,      "DOLLAR",       Anything,       "dAA4lER"        },
	{Anything,	"D",            Anything,       "d"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t E_rules[] =
{
	{Nothing,       "E",            Nothing,        "IY4"          },
	{"#:",          "E",            Nothing,        Silent  },
	{"':^",         "E",            Nothing,        Silent  },
	{" :",          "E",            Nothing,        "IY"    },
	{"#",           "ED",           Nothing,        "d"     },
	{"#:",          "E",            "D ",           Silent  },
	{Anything,	"EV",           "ER",           "EH4v"   },
        {"#:",          "ERED",         Nothing,        "ERd"           },
        {"#:",          "ERING",        Anything,       "ERIHNG"        },
        {"#:",          "EN",           Nothing,        "EHn"           },
        {"#:",          "ENED",         Nothing,        "EHnd"          },
        {"#:",          "ENESS",        Nothing,        "nEHs"          },
        {Anything,      "EDGE",         Anything,       "EHj"           },
        {Anything,      "ENG",          "ISH",          "IY3NGg"         },
	{Anything,	"E",            "^%",           "IY4"    },
	{Anything,	"ERI",          "#",            "IY4rIY" },
	{Anything,	"ERI",          Anything,       "EH4rIH" },
	{"#:",          "ER",           "#",            "ER1"    },
	{Anything,      "ERROR",        Anything,       "EH4rAXr"      },
	{Anything,      "ERAS",         "E",            "IHrEY5s"        },
	{Anything,	"ER",           "#",            "EH1r"   },
	{"#:",          "ER",           Nothing,        "ER"            },
	{"#:",          "ERS",          Nothing,        "ERz"           },
	{Anything,	"ER",           Anything,       "ER"    },
	{Nothing,	"EVEN",         Anything,       "IYvEHn" },
	{"#:",          "E",            "W",            Silent  },
	{"@",           "EW",           Anything,       "UW1"    },
	{Anything,	"EW",           Anything,       "yUW1"   },
	{Anything,	"E",            "O",            "IY"    },
	{"#:&",         "ES",           Nothing,        "IHz"   },
	{"#:",          "E",            "S ",           Silent  },
	{"#:",          "ELY",          Nothing,        "lIY"   },
	{"#:",          "EMENT",        Anything,       "mEHnt" },
	{Anything,	"EFUL",         Anything,       "fUHl"  },
	{Anything,	"EE",           Anything,       "IY4"    },
	{Anything,	"EARN",         Anything,       "ER5n"   },
	{Nothing,	"EAR",          "^",            "ER5"    },
	{Anything,	"EAD",          Anything,       "EH1d"   },
	{"#:",          "EA",           Nothing,        "IYAH"  },
	{Anything,	"EA",           "SU",           "EH5"    },
	{Anything,	"EA",           Anything,       "IY5"    },
	{Anything,	"EIGH",         Anything,       "EY4"    },
	{Anything,	"EI",           Anything,       "IY4"    },
	{Nothing,	"EYE",          Anything,       "AY4"    },
	{Anything,	"EY",           Anything,       "IY1"    },
	{Anything,	"EU",           Anything,       "yUW5"   },
	{Anything,      "EQUAL",        Anything,       "IY5kwAEl"       },
	{Anything,	"E",            Anything,       "EH1"    },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t F_rules[] =
{
	{Nothing,       "F",            Nothing,        "EH4f"         },
	{Nothing,       "FOR",          Nothing,        "fAXr"        },
	{Nothing,       "FROM",         Nothing,        "frAHm"         },
	{Nothing,       "FT.",          Nothing,        "fIY2t"          },
	{Anything,	"FUL",          Anything,       "fUHl"  },
	{Anything,      "FRIEND",       Anything,       "frEH5nd"        },
	{Anything,      "FE",           "MALE",         "fIY4"           },
	{Anything,      "FATHER",       Anything,       "fAA4DHER"       },
	{Anything,      "F",            "F",            Silent          },
	{Anything,	"F",            Anything,       "f"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t G_rules[] =
{
	{Nothing,       "G",            Nothing,        " jIY4 "         },
	{Anything,	"GIV",          Anything,       "gIH5v"  },
	{Nothing,	"G",            "I^",           "g"     },
	{Anything,	"GE",           "T",            "gEH5"   },
	{"SU",          "GGES",         Anything,       "gjEH4s" },
	{Anything,	"GG",           Anything,       "g"     },
	{" B#",         "G",            Anything,       "g"     },
	{Anything,	"G",            "+",            "j"     },
	{Anything,	"GREAT",        Anything,       "grEY4t" },
	{Anything,      "GON",          "E",            "gAA3n"          },
	{"#",           "GH",           Anything,       Silent  },
	{Nothing,       "GN",           Anything,       "n"             },	
	{Anything,	"G",            Anything,       "g"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t H_rules[] =
{
	{Nothing,       "H",            Nothing,        "EY4CH"        },
	{Nothing,	"HAV",          Anything,       "hAEv"  },
	{Nothing,       "HAS",          Nothing,        "hAEz"  },
	{Nothing,       "HAD",          Nothing,        "hAEd"  },
	{Nothing,	"HERE",         Anything,       "hIYr"  },
	{Nothing,	"HOUR",         Anything,       "AW5ER"  },
	{Anything,	"HOW",          Anything,       "hAW"   },
	{Anything,	"H",            "#",            "h"     },
	{Anything,	"H",            Anything,       Silent  },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t I_rules[] =
{
	{Nothing,       "IN",           Nothing,        "IHn"           },
	{Nothing,       "IBM",          Nothing,        "AY5 bIY EH5m"    },
	{Anything,      "ISLA",         "ND",           "AY3lIX"         },
	{Nothing,	"IN",           Anything,       "IH4n"   },
	{"#:",          "I",            "NG",           "IH"            },
	{Nothing,       "IS",           Nothing,        "IHz"           },
	{Nothing,       "IF",           Nothing,        "IHf"           },
	{Nothing,       "INTO",         Nothing,        "IH2ntUW"        },
	{Nothing,       "I",            Nothing,        "AY4"          },
	{Anything,      "I",            Nothing,        "AY"            },
	{Anything,	"IN",           "D",            "AY5n"   },
	{"SEM",         "I",            Anything,       "IY"            },
	{" ANT",        "I",            Anything,       "AY"            },
	{Anything,	"IER",          Anything,       "IY1ER"  },
	{"#:R",         "IED",          Anything,       "IYd"   },
	{Anything,	"IED",          Nothing,        "AY5d"   },
	{Anything,	"IEN",          Anything,       "IYEHn" },
	{Anything,	"IE",           "T",            "AY4EH"  },
        {Anything,      "I'",           Anything,       "AY5"            },
        {" :",          "I",            "^%",           "AY5"            },
	{" :",          "I",            "%",            "AY5"    },
	{Anything,	"I",            "%",            "IY"    },
	{Anything,	"IE",           Anything,       "IY4"    },
	{Nothing,       "IDEA",         Anything,       "AYdIY5AH"       },
	{Anything,	"I",            "^+:#",         "IH"    },
	{"#:",          "I",            "^AL",          "IH"            },
	{Anything,	"IR",           "#",            "AY1r"   },
	{Anything,	"IZ",           "%",            "AY1z"   },
	{Anything,	"IS",           "%",            "AY4z"   },
	{Anything,	"I",            "D%",           "AY4"    },
	{"#:",          "ITY",          Nothing,        "IHtIY"         },
	{"I^",          "I",            "^#",           "IH1"            },
	{"+^",          "I",            "^+",           "AY1"    },
	{Anything,	"I",            "T%",           "AY2"    },
	{"#:^",         "I",            "^%",           "AY2"            },
	{"#:^",         "I",            "^+",           "IH1"    },
	{Anything,	"I",            "^+",           "AY1"    },
	{Anything,	"IR",           Anything,       "ER1"    },
	{Anything,	"IGH",          Anything,       "AY4"    },
	{Anything,	"ILD",          Anything,       "AY5ld"  },
	{Nothing,       "IGN",          Anything,       "IHgn"          },
	{Anything,	"IGN",          Nothing,        "AY4n"   },
	{Anything,	"IGN",          "^",            "AY4n"   },
	{Anything,	"IGN",          "%",            "AY4n"   },
	{"#:",          "IC",           Nothing,        "IHk"           },
	{Anything,      "ICRO",         Anything,       "AY5krOW"        },
	{Anything,	"IQUE",         Anything,       "IY4k"   },
	{Anything,	"I",            Anything,       "IH1"    },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t J_rules[] =
{
	{Nothing,       "J",            Nothing,        "jEY4"         },
	{Anything,      "JOSE",         "PH",           "jOW5sIH"        },
	{"J",           "J",            Anything,       Silent          },
	{Anything,	"J",            Anything,       "j"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t K_rules[] =
{
	{Nothing,       "K",            Nothing,        "kEY4"         },
	{Anything,      "KNOW",         "L",            "nAA4"           },
	{Nothing,	"K",            "N",            Silent  },
	{"K",           "K",            Anything,       Silent          },
	{Anything,	"K",            Anything,       "k"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t L_rules[] =
{
	{Nothing,       "L",            Nothing,        "EH4l"         },
	{Anything,      "LIV",          "ELY",          "lAY3v"          },
	{Nothing,       "LIV",          "%",            "lIH4v"          },
	{Nothing,       "LIV",          "ING",          "lIH3v"          },
	{Nothing,       "LISA",         Nothing,        "lIY5sAH"        },
	{Anything,      "LEVEL",        Anything,       "lEH4vAEl"       },
	{Anything,      "LISTEN",       Anything,       "lIH4sIHn"       },
	{Nothing,       "LB.",          Nothing,        "pAW3ndz"        },
	{Nothing,       "LN",           Nothing,        "lAO2g"          },
	{Anything,	"LO",           "C#",           "lOW1"   },
	{"L",           "L",            Anything,       Silent  },
	{"#:^",         "L",            "%",            "AEl"   },
	{Anything,	"LEAD",         Anything,       "lIYd"  },
	{Nothing,       "LAUGH",        Anything,       "lAE4f"          },
	{Anything,	"L",            Anything,       "l"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t M_rules[] =
{
	{Anything,      "MACINTO",      "SH",           "mAE5kIHntAA1" },
	{Anything,      "MACINTAL",     "K",            "mAE5kIHntAO1" },
	{Anything,      "MAJ",          "OR ",          "mEY4j"        },
	{Nothing,       "MR.",          Nothing,        "mIH2stER"     },
	{Nothing,       "MS.",          Anything,       "mIH5z"        },
	{Nothing,       "MRS.",         Nothing,        "mIH2sIXz"     },
	{Nothing,       "MI.",          Nothing,        "mAY2lz"       },
	{Nothing,       "M",            Nothing,        "EH3m"         },
	{Anything,	"MOV",          Anything,       "mUW4v"        },
	{Anything,      "MACHIN",       Anything,       "mAHSHIY5n"    },
	{"M",           "M",            Anything,       Silent         },
	{Anything,	"M",            Anything,       "m"            },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t N_rules[] =
{
	{Nothing,       "N",            Nothing,        "EH4n"         },
	{"E",           "NG",           "+",            "nj"    },
	{Anything,	"NG",           "R",            "NGg"   },
	{Anything,	"NG",           "#",            "NGg"   },
	{Anything,	"NGL",          "%",            "NGgAEl" },
	{Anything,	"NG",           Anything,       "NG"    },
	{Anything,	"NK",           Anything,       "NGk"   },
	{Nothing,	"NOW",          Nothing,        "nAW4"   },
	{"N",           "N",            Anything,       Silent          },
	{Anything,      "NON",          "E",            "nAH4n"          },
	{Anything,      "NEU",          Anything,       "nUW2"           },
	{Anything,	"N",            Anything,       "n"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t O_rules[] =
{
	{Nothing,       "O",            Nothing,        "OW4"          },
	{Anything,      "OF",           Nothing,        "AHv"           },
	{Nothing,       "ON",           Nothing,        "AAn"           },
	{Nothing,       "OH",           Nothing,        "OW5"            },
	{Anything,	"OROUGH",       Anything,       "ER4OW"  },
	{"#:",          "OR",           Nothing,        "ER"    },
	{Nothing,       "OZ.",          Nothing,        "AW2nsIHz"       },
	{Nothing,       "OR",           Nothing,        "AXr"         },
	{Anything,      "OR",           Nothing,        "ER"            },
	{"#:",          "ORS",          Nothing,        "ERz"   },
	{Anything,	"ORE",          "^ ",           "OWr" },
	{Anything,	"OR",           Anything,       "AXr"   },
	{"H",		"ONEY",         Anything,       "AHnIY" },
	{"M",		"ONEY",         Anything,       "AHnIY" },
	{Anything,      "OR",           Anything,       "OWr"         },
	{Nothing,	"ONE",          Anything,       "wAH5n"  },
	{"#",           "ONE",          Nothing,        "wAH1n"          },
	{Anything,	"OW",           Anything,       "OW1"    },
	{Nothing,	"OVER",         Anything,       "OW5vER" },
	{"R",           "O",            "L",            "AX2"            },
	{"PR",          "O",            "V%",           "UW4"            },
	{"PR",          "O",            "VING",         "UW4"            },
	{"PR",          "O",            "V",            "AX"            },
	{Anything,	"OV",           Anything,       "AH4v"   },
	{Anything,      "OL",           "K",            "OW3"            },
	{Anything,      "OL",           "T",            "AX2l"           },
	{Anything,      "OL",           "D",            "AX2l"           },
	{Anything,	"O",            "^%",           "OW5"    },
	{Anything,	"O",            "^EN",          "OW4"    },
	{Anything,	"O",            "^I#",          "OW5"    },
	{Anything,	"OL",           "D",            "OW4l"   },
	{Anything,	"OUGHT",        Anything,       "AO5t"   },
	{Anything,	"OUGH",         Anything,       "AH5f"   },
	{Nothing,	"OU",           Anything,       "AW"    },
	{"H",           "OU",           "S#",           "AW4"    },
	{Anything,	"OUS",          Anything,       "AXs"   },
	{Anything,	"OUR",          Anything,       "AX1r"   },
	{Anything,	"OULD",         Anything,       "UH5d"   },
	{"^",           "OU",           "^L",           "AH5"    },
	{Anything,	"OUP",          Anything,       "UW5p"   },
	{Anything,	"OU",           Anything,       "AW1"    },
	{Anything,	"OY",           Anything,       "OY1"    },
	{Anything,	"OING",         Anything,       "OW4IHNG" },
	{Anything,	"OI",           Anything,       "OY5"    },
	{Anything,	"OOR",          Anything,       "UH5r" },
	{Anything,	"OOK",          Anything,       "UH5k"   },
	{"F",           "OOD",          Anything,       "UW5d"           },
	{"L",           "OOD",          Anything,       "AH5d"           },
	{"M",           "OOD",          Anything,       "UW5d"           },
	{Anything,	"OOD",          Anything,       "UH5d"   },
	{"F",           "OOT",          Anything,       "UH5t"           },
	{Anything,	"OO",           Anything,       "UW5"    },
	{Anything,      "O'",           Anything,       "AX"            },
	{Anything,	"O",            "E",            "OW1"    },
	{Anything,	"O",            Nothing,        "OW1"    },
	{Anything,	"OA",           Anything,       "OW4"    },
	{Nothing,	"ONLY",         Anything,       "OW4nlIY" },
	{Nothing,	"ONCE",         Anything,       "wAH4ns" },
	{Anything,	"ON'T",         Anything,       "OW4nt"  },
	{" C",          "O",            "N",            "AX"            },
	{"C",           "O",            "N",            "AA1"    },
	{Anything,	"O",            "NG",           "AO1"    },
	{" :^",         "O",            "N",            "AH1"    },
	{"I",           "ON",           Anything,       "AEn"   },
	{"#:",          "ON",           Nothing,        "AEn"   },
	{"#^",          "ON",           Anything,       "AEn"   },
	{"FR",          "O",            "ST",           "AO4"            },
	{"L",           "O",            "ST",           "AO4"            },
	{"C",           "O",            "ST",           "AO4"            },
	{Anything,      "O",            "ST%",          "OW4"            },
	{Anything,	"O",            "ST ",          "OW5"    },
	{Anything,	"OF",           "^",            "AO4f"   },
	{Anything,	"OTHER",        Anything,       "AH5DHER" },
	{"R",           "O",            "B",            "AA1"            },
	{"PR",          "O",            ":#",           "OW5"            },
	{Anything,      "OSS",          Nothing,        "AA5s"           },
	{"#:^",         "OM",           Anything,       "AHm"   },
	{Anything,	"O",            Anything,       "AA1"    },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t P_rules[] =
{
	{Nothing,       "P",            Nothing,        "pIY4"         },
	{Anything,	"PH",           Anything,       "f"     },
	{Anything,      "PEOPL",        Anything,       "pIY5pAEl"       },
	{Anything,      "PURPOSE",      Anything,       "pER5pAXs"       },
	{Anything,      "PIT",          "CH",           "pIH5"           },
	{Anything,	"POW",          Anything,       "pAW4"   },
	{Anything,	"PUT",          Nothing,        "pUHt"  },
	{Anything,      "P",            "P",            Silent          },
	{Nothing,       "P",            "N",            Silent          },
	{Nothing,       "P",            "S",            Silent          },
	{Nothing,       "PROF.",        Anything,       "prAXfEH4sER"    },
	{Anything,	"P",            Anything,       "p"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t Q_rules[] =
{
	{Nothing,       "Q",            Nothing,        "kyUW4"        },
	{"S",           "QUAR",         Anything,       "kwEH4r"         },
	{Anything,	"QUAR",         Anything,       "kwAX5r" },
	{Anything,      "QUA",          "L",            "kwAA3"          },
	{Anything,	"QU",           Anything,       "kw"    },
	{"Q",           "Q",            Anything,       Silent          },
	{Anything,	"Q",            Anything,       "k"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t R_rules[] =
{
	{Nothing,       "R",            Nothing,        "AA4r"         },
	{Anything,      "READY",        Anything,       "rEH5dIY"        },
	{Anything,      "READ",         Anything,       "rIY4d"          },
	{Nothing,	"RE",           "^#",           "rIY1"   },
	{Anything,      "R",            "R",            Silent          },
	{Anything,	"R",            Anything,       "r"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t S_rules[] =
{
	{Nothing,       "S",            Nothing,        "EH4s"         },
	{Nothing,       "SO",           Nothing,        "sOW"           },
	{Anything,	"SH",           Anything,       "SH"    },
	{"#",           "SION",         Anything,       "ZHAEn" },
	{Anything,	"SOME",         Anything,       "sAHm"  },
	{"#",           "SUR",          "#",            "ZHER"  },
	{Anything,	"SUR",          "#",            "SHER"  },
	{"#",           "SU",           "#",            "ZHUW"  },
	{"#",           "SSU",          "#",            "SHUW"  },
	{"#",           "SED",          Nothing,        "zd"    },
	{Anything,      "SIS",          Anything,       "sIH1s"          },
	{Nothing,       "ST.",          Nothing,        "strIY2t"        },
	{"#",           "S",            "#",            "z"     },
	{Anything,	"SAID",         Anything,       "sEH2d"  },
	{Anything,      "SAYS",         Anything,       "sEH2z"          },
	{Nothing,       "SIN",          Nothing,        "sAY2n"          },
	{"^",           "SION",         Anything,       "SHAEn" },
	{Anything,	"S",            "S",            Silent  },
	{".",           "S",            Nothing,        "z"     },
	{"#:.E",        "S",            Nothing,        "z"     },
	{"#:^##",       "S",            Nothing,        "z"     },
	{"#:^#",        "S",            Nothing,        "s"     },
	{"U",           "S",            Nothing,        "s"     },
	{" :#",         "S",            Nothing,        "z"     },
	{"##",          "S",            Nothing,        "z"             },
	{Nothing,	"SCH",          Anything,       "sk"    },
	{Anything,	"S",            "C+",           Silent  },
	{"#",           "SM",           Anything,       "zAEm"    },
	{"#",           "SN",           "'",            "zAEn"  },
	{Anything,      "STLE",         Anything,       "sAEl"          },
	{Anything,	"S",            Anything,       "s"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t T_rules[] =
{
	{Nothing,       "T",            Nothing,        "tIY4"         },
	{Nothing,       "THE",          " #",           "DHIY"          },
	{Nothing,	"THE",          Nothing,        "DHAX"  },
	{Anything,	"TO",           Nothing,        "tUW"   },
	{Nothing,	"THAT",         Anything,       "DHAEt" },
	{Nothing,	"THIS",         Nothing,        "DHIHs" },
	{Nothing,	"THEY",         Anything,       "DHEY1"  },
	{Nothing,	"THERE",        Anything,       "DHEH1r" },
	{Anything,	"THER",         Anything,       "DHER"  },
 	{Anything,	"THEIR",        Anything,       "DHEH2r" },
	{Nothing,	"THAN",         Nothing,        "DHAEn" },
	{Nothing,	"THEM",         Nothing,        "DHEH2m" },
	{Anything,	"THESE",        Nothing,        "DHIYz" },
	{Nothing,	"THEN",         Anything,       "DHEH1n" },
	{Anything,	"THROUGH",      Anything,       "THrUW4" },
	{Anything,	"THOSE",        Anything,       "DHAX1z" },
	{Anything,	"THOUGH",       Nothing,        "DHOW1"  },
	{Anything,      "TODAY",        Anything,       "tUWdEY5"        },
	{Anything,      "TOMO",         "RROW",         "tAEmAA5"        },
	{Anything,      "TO",           "TAL",          "tOW5"           },
	{Nothing,	"THUS",         Anything,       "DHAH4s" },
	{Anything,	"TH",           Anything,       "TH"    },
	{"#:",          "TED",          Nothing,        "tIXd"  },
	{"S",           "TI",           "#N",           "CH"    },
	{Anything,	"TI",           "O",            "SH"    },
	{Anything,	"TI",           "A",            "SH"    },
	{Anything,	"TIEN",         Anything,       "SHAEn" },
	{Anything,	"TUR",          "#",            "CHER"  },
	{Anything,	"TU",           "A",            "CHUW"  },
	{Nothing,	"TWO",          Anything,       "tUW5"   },
	{"&",           "T",            "EN",           Silent          },
	{"F",           "T",            "EN",           Silent          },
	{Anything,      "T",            "T",            Silent          },
	{Anything,	"T",            Anything,       "t"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t U_rules[] =
{
	{Nothing,       "U",            Nothing,        "yUW4"         },
	{Nothing,	"UN",           "I",            "yUWn"  },
	{Nothing,	"UN",           Anything,       "AH1n"   },
	{Nothing,	"UPON",         Anything,       "AXpAA3n" },
	{"@",           "UR",           "#",            "UH4r"   },
	{Anything,	"UR",           "#",            "yUH4r"  },
	{Anything,	"UR",           Anything,       "ER1"    },
	{Anything,	"U",            "^ ",           "AH"    },
	{Anything,	"U",            "^^",           "AH5"    },
	{Anything,	"UY",           Anything,       "AY5"    },
	{" G",          "U",            "#",            Silent  },
	{"G",           "U",            "%",            Silent  },
	{"G",           "U",            "#",            "w"     },
	{"#N",          "U",            Anything,       "yUW1"   },
	{"@",           "U",            Anything,       "UW1"    },
	{"U",           "U",            Anything,       Silent          },
 	{Nothing,       "USA",          Nothing,        "yUW5EHsEY2"      },
	{Anything,	"U",            Anything,       "yUW1"   },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t V_rules[] =
{
	{Nothing,       "V",            Nothing,        "vIY4"         },
	{Anything,	"VIEW",         Anything,       "vyUW5"  },
	{Anything,      "VOCABULAR",    Anything,       "vAXkAE4bYUlEHr" },
	{"V",           "V",            Anything,       Silent          },
	{Anything,	"V",            Anything,       "v"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t W_rules[] =
{
	{Nothing,       "W",            Nothing,        "dAH4bAElyUW"  },
	{Nothing,       "WIZARD",       Anything,       "wIH4zERd"       },
	{Nothing,       "WERE",         Anything,       "wER"           },
	{Anything,      "WA",           "SH",           "wAA2"           },
	{Anything,      "WA",           "ST",           "wEY2"           },
	{Anything,      "WA",           "S",            "wAH1"           },
	{Anything,	"WA",           "T",            "wAA1"   },
	{Anything,	"WHERE",        Anything,       "WHEH1r" },
	{Anything,	"WHAT",         Anything,       "WHAHt" },
	{Anything,	"WHOL",         Anything,       "hOW2l"  },
	{Anything,	"WHO",          Anything,       "hUW"   },
	{Anything,	"WH",           Anything,       "w"    },
	{Anything,      "WAR",          "#",            "wEH2r"          },
	{Anything,	"WAR",          Anything,       "wAO5r" },
	{Anything,	"WOR",          "^",            "wER"   },
	{Anything,	"WR",           Anything,       "r"     },
	{Anything,      "WOM",          "A",            "wUH5m"          },
	{Anything,      "WOM",          "E",            "wIH5m"          },
	{Anything,      "WEA",          "R",            "wEH"           },
	{Anything,      "WANT",         Anything,       "wAA5nt"         },
	{"ANS",         "WER",          Anything,       "ER"            },
	{"W",           "W",            Anything,       Silent          },
	{Anything,	"W",            Anything,       "w"     },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t X_rules[] =
{
	{"?",           "X",            "?",            " bAY "         },
	{"? ",          "X",            " ?",           "bAY"           },
	{Nothing,       "X",            Nothing,        "EH4ks"        },
	{Nothing,       "X",            Anything,       "z"             },
	{"X",           "X",            Anything,       Silent          },
	{Anything,      "XC",           "+",            "ks"            },
	{Anything,	"X",            Anything,       "ks"    },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t Y_rules[] =
{
	{Nothing,       "Y",            Nothing,        "wAY4"         },
	{Anything,	"YOUNG",        Anything,       "yAH2NG" },
	{Nothing,       "YOUR",         Anything,       "yAXr"  },
	{Nothing,	"YOU",          Anything,       "yUW"   },
	{Nothing,	"YES",          Anything,       "yEH2s"  },
	{Nothing,	"Y",            Anything,       "y"     },
	{"F",           "Y",            Anything,       "AY"            },
	{"PS",          "YCH",          Anything,       "AY1k"           },
	{"#:^",         "Y",            Nothing,        "IY"    },
	{"#:^",         "Y",            "I",            "IY"    },
	{" :",          "Y",            Nothing,        "AY"    },
	{" :",          "Y",            "#",            "AY"    },
	{" :",          "Y",            "^+:#",         "IH"    },
	{" :",          "Y",            "^#",           "AY"    },
	{"Y",           "Y",            Anything,       Silent          },
	{Anything,	"Y",            Anything,       "IH"    },
	{0, 0, 0, 0, },
};

/*
 *	LEFT_PART	MATCH_PART	RIGHT_PART	OUT_PART
 */
static rule_t Z_rules[] =
{
	{Nothing,       "Z",            Nothing,        "zIY4"         },
	{"Z",           "Z",            Anything,       Silent          },
	{Anything,	"Z",            Anything,       "z"     },
	{0, 0, 0, 0, },
};

rule_t *english_rules[] =
{
	punct_rules,
	A_rules, B_rules, C_rules, D_rules, E_rules, F_rules, G_rules,
	H_rules, I_rules, J_rules, K_rules, L_rules, M_rules, N_rules,
	O_rules, P_rules, Q_rules, R_rules, S_rules, T_rules, U_rules,
	V_rules, W_rules, X_rules, Y_rules, Z_rules
};
