#!/usr/bin/env python3
"""
Examples::

    $ python3 geet.py
    no command given

    $ python3 geet.py leet
    unknown command 'leet'

    $ python3 geet.py --help
    geet v1.7.2
    The l33t version control

    Usage: geet.py [SWITCHES] [SUBCOMMAND [SWITCHES]] args...
    Meta-switches:
        -h, --help                 Prints this help message and quits
        -v, --version              Prints the program's version and quits

    Subcommands:
        commit                     creates a new commit in the current branch; see
                                   'geet commit --help' for more info
        push                       pushes the current local branch to the remote
                                   one; see 'geet push --help' for more info

    $ python3 geet.py commit --help
    geet commit v1.7.2
    creates a new commit in the current branch

    Usage: geet commit [SWITCHES]
    Meta-switches:
        -h, --help                 Prints this help message and quits
        -v, --version              Prints the program's version and quits

    Switches:
        -a                         automatically add changed files
        -m VALUE:str               sets the commit message; required

    $ python3 geet.py commit -m "foo"
    committing...
"""

from __future__ import annotations

try:
    import colorama

    colorama.init()
except ImportError:
    pass

from plumbum import cli, colors


class Geet(cli.Application):
    SUBCOMMAND_HELPMSG = False
    DESCRIPTION = colors.yellow | """The l33t version control"""
    PROGNAME = colors.green
    VERSION = colors.blue | "1.7.2"
    COLOR_USAGE_TITLE = colors.bold | colors.magenta
    COLOR_USAGE = colors.magenta

    _group_names = ["Meta-switches", "Switches", "Sub-commands"]

    COLOR_GROUPS = dict(
        zip(_group_names, [colors.do_nothing, colors.skyblue1, colors.yellow])
    )

    COLOR_GROUP_TITLES = dict(
        zip(
            _group_names,
            [colors.bold, colors.bold | colors.skyblue1, colors.bold | colors.yellow],
        )
    )

    verbosity = cli.SwitchAttr(
        "--verbosity",
        cli.Set("low", "high", "some-very-long-name", "to-test-wrap-around"),
        help=colors.cyan
        | "sets the verbosity level of the geet tool. doesn't really do anything except for testing line-wrapping "
        "in help " * 3,
    )
    verbositie = cli.SwitchAttr(
        "--verbositie",
        cli.Set("low", "high", "some-very-long-name", "to-test-wrap-around"),
        help=colors.hotpink
        | "sets the verbosity level of the geet tool. doesn't really do anything except for testing line-wrapping "
        "in help " * 3,
    )


@Geet.subcommand(colors.red | "commit")
class GeetCommit(cli.Application):
    """creates a new commit in the current branch"""

    auto_add = cli.Flag("-a", help="automatically add changed files")
    message = cli.SwitchAttr("-m", str, mandatory=True, help="sets the commit message")

    def main(self):
        print("committing...")


GeetCommit.unbind_switches("-v", "--version")


@Geet.subcommand("push")
class GeetPush(cli.Application):
    """pushes the current local branch to the remote one"""

    tags = cli.Flag("--tags", help="whether to push tags (default is False)")

    def main(self, remote, branch="master"):
        print(f"pushing to {remote}/{branch}...")


if __name__ == "__main__":
    Geet.run()
