/*	Blinker

PIRL CVS ID: Blinker.java,v 1.8 2012/04/16 06:22:58 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import	javax.swing.Timer;
import	java.awt.Component;
import	java.awt.Color;
import	java.awt.event.*;

/**	An ActionListener for blinking a component to make it stand out
	from the crowd.
<P>
	Blinking occurs by setting the background color of the component to a
	specified color, and then returning to the original color, with a
	delay interval between background color switching. This on-off cycle
	is repeated for a specified number of times, or until instructed to
	stop. An optional delay after the new background color is first
	switched on until the blinking intervals start may be specified.
<P>
	@author		Bradford Castalia, UA/PIRL
	@version	1.8 
*/
public class Blinker
	extends Timer
	implements ActionListener
{
private static final String
	ID = "PIRL.Viewers.Blinker (1.8 2012/04/16 06:22:58)";

/**	Constant to specify an indefinate number of blinks;
	i.e&#46; forever until stopped.
*/
public static final int
	INDEFINITELY		=  Integer.MAX_VALUE;

/**	The default number of blink repeats: INDEFINITELY.
*/
public static final int
	REPEATS				= INDEFINITELY;

/**	The default initial delay before blinking starts: 0 milliseconds
	(immediately after start).
*/
public static final int
	INITIAL_DELAY		= 0;

/**	The default blink interval: 500 milliseconds (half a second).
*/
public static final int
	INTERVAL			= 500;

/**	The minimum blink interval: 100 milliseconds.
*/
public static final int
	MINIMUM_INTERVAL	= 100;
	
/**	The default background Color to use when the component is
	blinked "on": yellow.
*/
public static final Color
	COLOR				= Color.yellow;

private Component
	Blinkee;
private Color
	Colors[] = new Color[2];	// 0: off (component), 1: on (blink)
private int
	Blinks = 0,
	Max_Blinks;


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTORS	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_TIMER			= 1 << 2,
	DEBUG_ALL			= -1,

	DEBUG       		= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a component Blinker.
<P>
	@param	component	The Component to be blinked.
	@param	color	The Color to set the component background when
		it is blinked "on".
	@param	repeats	The maximum number of blinks. Use the <CODE>{@link
		#INDEFINITELY}</CODE> value to continue blinking until
		stopped.
	@param	initial_delay	The initial delay, after the component
		is turned on, before starting to blink. A value <= 0 means
		no initial delay.
	@param	interval	The interval between blinks (on to off, or
		off to on). A {@link #MINIMUM_INTERVAL} is enforced.
*/
public Blinker
	(
	Component	component,
	Color		color,
	int			repeats,
	int			initial_delay,
	int			interval
	)
{
super (((interval < MINIMUM_INTERVAL) ? MINIMUM_INTERVAL : interval), null);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> Blinker:\n"
		+"        component - " + component + '\n'
		+"            color - " + color + '\n'
		+"          repeats - " + repeats + '\n'
		+"    initial_delay - " + initial_delay + '\n'
		+"         interval - " + interval);
addActionListener (this);
if (initial_delay < 0)
	initial_delay = 0;
setInitialDelay (initial_delay);
setRepeats (true);
setRepeats (repeats);
Blinkee = component;
if (component != null)
	{
	Colors[0] = component.getBackground ();
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("       background - " + Colors[0]);
	}
Colors[1] = color;
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< Blinker");
}

/**	Constructs a Blinker for the component using all of the default
	Blinker parameters.
<P>
	@param	component	The Component to be blinked.
*/
public Blinker
	(
	Component	component
	)
{this (component, COLOR, REPEATS, INITIAL_DELAY, INTERVAL);}

/**	Constructs a disabled Blinker with default Blinker parameters.
*/
public Blinker ()
{this (null, COLOR, REPEATS, INITIAL_DELAY, INTERVAL);}

/*==============================================================================
	Timer methods overrides
*/
/**	Turns the component "on" (sets the new background color) and starts
	the blink timer. The blink count picks up from its previous value.
	Use restart to reset the blink count.
*/
public void start ()
{
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		(">>> Blinker.start");
if (Blinkee != null &&
	Blinks < Max_Blinks)
	{
	//	Turn the component "on".
	if ((Blinks % 2) == 0)
		Blinks++;
	if ((DEBUG & DEBUG_TIMER) != 0)
		System.out.println
			("    " + Blinks + " blinks");
	Blinkee.setBackground (Colors[1]);
	super.start ();
	}
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		("<<< Blinker.start: isRunning " + isRunning ());
return;
}

/**	Stops the blink timer and turns the component "off" (sets the
	original background color). The blink count is not reset.
*/
public void stop ()
{
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		(">>> Blinker.stop");
super.stop ();
//	Turn the component "off".
if ((Blinks % 2) != 0)
	Blinks--;
if (Blinkee != null)
	{
	if ((DEBUG & DEBUG_TIMER) != 0)
		System.out.println
			("    Restoring background");
	Blinkee.setBackground (Colors[0]);
	}
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		("<<< Blinker.stop");
}

/**	Restarts the Blinker as if no blinks had occured.
*/
public void restart ()
{
stop ();
Blinks = 0;
start ();
}

/**	Sets the maximum number of blinks.
<P>
	@param	repeats	The maximum number of times to blink the component.
		Use the <CODE>{@link #INDEFINITELY}</CODE> value to continue
		blinking until stopped.
	@return	The previous value.
*/
public int setRepeats
	(
	int		repeats
	)
{
int
	max = Max_Blinks;
if (repeats < 0)
	repeats = 0;
Max_Blinks = repeats;
return max;
}

/**	Gets the maximum number of blinks.
<P>
	@return	The maximum number of times to blink the component. For
		indefinate blinking this will be the <CODE>{@link
		#INDEFINITELY}</CODE> value.
*/
public int getRepeats ()
{return Max_Blinks;}

/*==============================================================================
	Extension methods
*/
/**	Sets the background color to use when blinking the component.
<P>
	@param	color	The background color when the component is "on".
	@return	The previous value.
*/
public Color setColor
	(
	Color	color
	)
{
Color
	previous_color = Colors[1];
Colors[1] = color;
return previous_color;
}

/**	Gets the background color that will be used when blinking the
	component.
<P>
	@return	The background blink Color.
*/
public Color getColor ()
{return Colors[1];}

/**	Sets the Component to be blinked.
<P>
	Any currently blinking component is stopped and the blink count is
	reset to zero. The new component will not yet be blinking.
<P>
	@param	component	The Component to be blinked.
	@return	This blinker.
*/
public Blinker setComponent (Component component)
{
stop ();
Blinks = 0;
if ((Blinkee = component) != null)
	Colors[0] = component.getBackground ();
return this;
}

/**	Gets the Component to be blinked.
<P>
	@return	The Component to be blinked.
*/
public Component getComponent ()
{return Blinkee;}

/**	Performs the blink event at the Timer interval.

	Switches the component from its "on" background color to its
	"off" background color, or vice versa.
*/
public void actionPerformed (ActionEvent event)
{
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		(">>> Blinker.actionPerformed");
if (Blinks >= Max_Blinks ||
	Blinkee == null)
	{
	if ((DEBUG & DEBUG_TIMER) != 0)
		System.out.println
			("    Stopping after " + Blinks + " blinks");
	Blinks = 0;
	stop ();
	}
else if (Blinkee != null)
	{
	Blinkee.setBackground (Colors[++Blinks % 2]);
	if ((DEBUG & DEBUG_TIMER) != 0)
		System.out.println
			("    Background set to " + Colors[Blinks % 2]);
	}
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		("<<< Blinker.actionPerformed");
}


}	//	class Blinker.
