/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/laplace/cdf' );
import Laplace = require( './../../../../../base/dists/laplace/ctor' );
import entropy = require( './../../../../../base/dists/laplace/entropy' );
import kurtosis = require( './../../../../../base/dists/laplace/kurtosis' );
import logcdf = require( './../../../../../base/dists/laplace/logcdf' );
import logpdf = require( './../../../../../base/dists/laplace/logpdf' );
import mean = require( './../../../../../base/dists/laplace/mean' );
import median = require( './../../../../../base/dists/laplace/median' );
import mgf = require( './../../../../../base/dists/laplace/mgf' );
import mode = require( './../../../../../base/dists/laplace/mode' );
import pdf = require( './../../../../../base/dists/laplace/pdf' );
import quantile = require( './../../../../../base/dists/laplace/quantile' );
import skewness = require( './../../../../../base/dists/laplace/skewness' );
import stdev = require( './../../../../../base/dists/laplace/stdev' );
import variance = require( './../../../../../base/dists/laplace/variance' );

/**
* Interface describing the `laplace` namespace.
*/
interface Namespace {
	/**
	* Laplace distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 10.0, 0.0, 3.0 );
	* // returns ~0.982
	*
	* y = ns.cdf( 0.0, 0.0, 3.0 );
	* // returns 0.5
	*
	* var myCDF = ns.cdf.factory( 2.0, 3.0 );
	* y = myCDF( 10.0 );
	* // returns ~0.965
	*
	* y = myCDF( 2.0 );
	* // returns 0.5
	*/
	cdf: typeof cdf;

	/**
	* Laplace distribution.
	*/
	Laplace: typeof Laplace;

	/**
	* Returns the differential entropy for a Laplace distribution with location `mu` and scale `b`.
	*
	* ## Notes
	*
	* -   If provided `b <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns entropy
	*
	* @example
	* var y = ns.entropy( 0.0, 1.0 );
	* // returns ~1.693
	*
	* @example
	* var y = ns.entropy( 5.0, 2.0 );
	* // returns ~2.386
	*
	* @example
	* var y = ns.entropy( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 0.0, 0.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis for a Laplace distribution with location `mu` and scale `b`.
	*
	* ## Notes
	*
	* -   If provided `b <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns kurtosis
	*
	* @example
	* var y = ns.kurtosis( 0.0, 1.0 );
	* // returns 3.0
	*
	* @example
	* var y = ns.kurtosis( 5.0, 2.0 );
	* // returns 3.0
	*
	* @example
	* var y = ns.kurtosis( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 0.0, 0.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Laplace distribution logarithm of cumulative distribution function (logCDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 10.0, 0.0, 3.0 );
	* // returns ~-0.018
	*
	* y = ns.logcdf( 0.0, 0.0, 3.0 );
	* // returns ~-0.693
	*
	* var mylogcdf = ns.logcdf.factory( 2.0, 3.0 );
	* y = mylogcdf( 10.0 );
	* // returns ~-0.036
	*
	* y = mylogcdf( 2.0 );
	* // returns ~-0.693
	*/
	logcdf: typeof logcdf;

	/**
	* Laplace distribution natural logarithm of probability density function (PDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 0.0, 1.0 );
	* // returns ~-2.688
	*
	* var mylogpdf = ns.logpdf.factory( 10.0, 2.0 );
	* y = mylogpdf( 10.0 );
	* // returns -1.386
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value for a Laplace distribution with location `mu` and scale `b`.
	*
	* ## Notes
	*
	* -   If provided `b <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns expected value
	*
	* @example
	* var y = ns.mean( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.mean( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.mean( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, 0.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median for a Laplace distribution with location `mu` and scale `b`.
	*
	* ## Notes
	*
	* -   If provided `b <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns median
	*
	* @example
	* var y = ns.median( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.median( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.median( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, 0.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Laplace distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.5, 0.0, 1.0 );
	* // returns ~1.333
	*
	* y = ns.mgf( 0.0, 0.0, 1.0 );
	* // returns 1.0
	*
	* y = ns.mgf( -1.0, 4.0, 0.2 );
	* // returns ~0.019
	*
	* var mymgf = ns.mgf.factory( 4.0, 2.0 );
	*
	* y = mymgf( 0.2 );
	* // returns ~2.649
	*
	* y = mymgf( 0.4 );
	* // returns ~13.758
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode for a Laplace distribution with location `mu` and scale `b`.
	*
	* ## Notes
	*
	* -   If provided `b <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns mode
	*
	* @example
	* var y = ns.mode( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.mode( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.mode( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, 0.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Laplace distribution probability density function (PDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.0, 1.0 );
	* // returns ~0.068
	*
	* var myPDF = ns.pdf.factory( 10.0, 2.0 );
	* y = myPDF( 10.0 );
	* // returns 0.25
	*/
	pdf: typeof pdf;

	/**
	* Laplace distribution quantile function.
	*
	* @param p - input value
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 0.0, 1.0 );
	* // returns ~0.916
	*
	* var myQuantile = ns.quantile.factory( 10.0, 2.0 );
	* y = myQuantile( 0.5 );
	* // returns 10.0
	*
	* y = myQuantile( 0.8 );
	* // returns ~11.833
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness for a Laplace distribution with location `mu` and scale `b`.
	*
	* ## Notes
	*
	* -   If provided `b <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns skewness
	*
	* @example
	* var y = ns.skewness( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.skewness( 5.0, 2.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.skewness( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 0.0, 0.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation for a Laplace distribution with location `mu` and scale `b`.
	*
	* ## Notes
	*
	* -   If provided `b <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns standard deviation
	*
	* @example
	* var y = ns.stdev( 0.0, 1.0 );
	* // returns ~1.414
	*
	* @example
	* var y = ns.stdev( 5.0, 2.0 );
	* // returns ~2.828
	*
	* @example
	* var y = ns.stdev( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, 0.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance for a Laplace distribution with location `mu` and scale `b`.
	*
	* ## Notes
	*
	* -   If provided `b <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param b - scale parameter
	* @returns variance
	*
	* @example
	* var y = ns.variance( 0.0, 1.0 );
	* // returns 2.0
	*
	* @example
	* var y = ns.variance( 5.0, 2.0 );
	* // returns 8.0
	*
	* @example
	* var y = ns.variance( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, 0.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Laplace distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
