# Copyright 2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Test whether or not a node is compatible with MAAS."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = [
    "main",
    ]

import logging
import os
import sys

from maastest import utils
from testtools.testresult.real import _details_to_str

# Most imports are done as needed in main() to improve responsiveness
# when maas-test is run interactively, especially when --help is passed.


class RETURN_CODES:
    """Return codes for this program.

    Not particularly useful in practice, but they help the test suite check
    for the right error diagnosis.
    """
    SUCCESS = 0
    TEST_FAILED = 1
    NOT_ROOT = 2
    NO_KVM_EXTENSIONS = 3
    UNEXPECTED_DHCP_SERVER = 4


class ProgramFailure(Exception):
    """A known kind of program failure.

    These are fatal, program-level failures that we know how to present to the
    user, including which error code to return.

    :ivar return_code: Return code, from `RETURN_CODES`, which the program
        should return to the shell.
    """
    def __init__(self, return_code, message):
        super(ProgramFailure, self).__init__(message)
        self.return_code = return_code


def check_for_root():
    """Verify that the program is being run as root.

    :raise ProgramFailure: if the program is not running as root.
    """
    if os.geteuid() != 0:
        raise ProgramFailure(
            RETURN_CODES.NOT_ROOT, "This test must be run as root.  Use sudo.")


def check_for_virtualisation_support():
    """Verify that the system supports efficient virtual machines.

    :raise ProgramFailure: if the CPU lacks virtualisation support, or support
        has been disabled in firmware.
    """
    if not utils.check_kvm_ok():
        raise ProgramFailure(
            RETURN_CODES.NO_KVM_EXTENSIONS,
            "Unable to continue without KVM extensions.")


def check_against_virtual_host():
    """Warn if the program is running in a virtual machine.

    For acceptable performance, maas-test should be run on a physical system.
    This function warns, but does not fail, if the system is virtual.
    """
    virt_type = utils.virtualization_type()
    if virt_type is not None:
        logging.info(
            "This machine is running on %s virtual hardware." % virt_type)
        logging.warning(
            "Running maas-test on this machine will result in significantly "
            "poorer performance than on physical hardware.")


def check_against_dhcp_servers(interface):
    """Check for unexpected DHCP servers on the testing network.

    :param interface: Network interface to the testing network.  This will
        send out a DHCP discovery request on that interface.
    :raise ProgramFailure: if any DHCP servers were detected.
    """
    # Importing locally for responsiveness.
    from maastest.detect_dhcp import probe_dhcp
    dhcp_servers = probe_dhcp(interface)
    if len(dhcp_servers) > 0:
        raise ProgramFailure(
            RETURN_CODES.UNEXPECTED_DHCP_SERVER,
            "DHCP server(s) detected on %s: %s. "
            "Pass an interface that is connected to the testing network. "
            "Ensure that the testing network connects only to the node and "
            "its BMC, and has no DHCP service running.  "
            "See the maas-test manual for details."
            % (interface, ', '.join(sorted(dhcp_servers))))


def has_maas_probe_dhcp(machine):
    """Does the given virtual machine have `maas-probe-dhcp` installed?"""
    return_code, _, _ = machine.run_command(['which', 'maas-probe-dhcp'])
    return (return_code == 0)


def check_against_dhcp_servers_from_vm(machine):
    """Check for unexpected DHCP servers, from within the virtual machine.

    This complements `check_against_dhcp_servers`, adding detection for two
    situations that it doesn't handle:
    - The interface has no IP address (the VM's interface gets a static one).
    - The DHCP server is running on the same interface we use.

    This second DHCP check uses the `maas-probe-dhcp` script that is installed
    on the VM as part of MAAS.

    :param machine: Virtual-machine fixture, with MAAS installed.
    :raise ProgramFailure: if any DHCP servers were detected.
    """
    if machine.direct_interface is None:
        # We need a direct interface to perform this check.
        return
    if not has_maas_probe_dhcp(machine):
        # Not all released MAAS versions have the maas-probe-dhcp tool.  If we
        # don't have it, skip this check.
        return

    # Importing locally for responsiveness.
    from maastest.kvmfixture import DIRECT_INTERFACE
    return_code, stdout, stderr = machine.run_command(
        ['sudo', 'maas-probe-dhcp', DIRECT_INTERFACE], check_call=False)
    if return_code == 0:
        # OK.  No DHCP servers detected.
        return

    # Importing locally for responsiveness.
    import re
    match = re.match("DHCP servers detected: (.*)$", stdout)
    if match is None:
        # Not the output we expect when maas-probe-dhcp detects DHCP servers,
        # so either something else went wrong, or the output has changed.
        raise Exception(
            "Call to maas-probe-dhcp failed in virtual machine: '%s'" % stderr)
    ips = match.group(1)
    if ips == machine.direct_ip:
        # A DHCP server was detected, but it's running on the virtual machine's
        # bridged IP address.  It's probably just the MAAS DHCP server running
        # in the virtual machine itself.
        return

    # The virtual machine detected a DHCP server that we missed before.
    raise ProgramFailure(
        RETURN_CODES.UNEXPECTED_DHCP_SERVER,
        "The virtual machine detected a DHCP server on the network: %s. "
        "Pass an interface that is connected to the testing network. "
        "Ensure that the testing network connects only to the node and "
        "its BMC, and has no DHCP service running.  "
        "See the maas-test manual for details."
        % ips)


def make_local_proxy_fixture():
    """Create a `LocalProxyFixture`.

    Defined here so that the import can be deferred for responsiveness, while
    still allowing tests to patch it.
    """
    from maastest.proxyfixture import LocalProxyFixture
    return LocalProxyFixture()


def set_up_proxy(args):
    """Obtain or set up an http/https proxy, as appropriate.

    The command-line arguments may disable this, or make it use an existing
    proxy, or require us to set up our own.  This function will do whichever
    fits, and set up the process environment as needed.

    :return: A tuple of (proxy URL, proxy fixture).  The fixture may be `None`.
        If it is not `None`, it will need to be cleaned up later.  The proxy
        URL may be the empty string if no proxy is to be used.
    """
    proxy_fixture = None
    proxy_is_set_up = False
    try:
        if args.http_proxy is not None:
            # The user has passed an external proxy; don't start polipo,
            # just use that proxy for everything.
            proxy_url = args.http_proxy
            logging.info("Using external proxy %s." % proxy_url)
        elif args.disable_cache:
            # The user has passed --disable-cache, so don't start polipo
            # and don't try to use an external proxy.
            proxy_url = ''
            logging.info("Caching disabled.")
        else:
            # The default case: start polipo and use that as our caching
            # proxy.
            proxy_fixture = make_local_proxy_fixture()
            proxy_fixture.setUp()
            proxy_is_set_up = True
            proxy_url = proxy_fixture.get_url()

        if proxy_url != '':
            os.putenv('http_proxy', proxy_url)
            os.putenv('https_proxy', proxy_url)
    except:
        if proxy_is_set_up:
            # We have a proxy fixture set up, but we're not going to return.
            # That means nobody else is responsible for cleaning it up.
            proxy_fixture.cleanUp()
        raise

    return proxy_url, proxy_fixture


def create_vm(args, proxy_url=None):
    """Create the virtual machine fixture, but do not set it up yet.

    :param args: Arguments object as returned by the arguments parser.
    :param proxy_url: Optional http/https proxy URL for the VM to use, as
        returned by `set_up_proxy`.
    :return: Virtual-machine fixture.  If you begin to set it up, clean it up
        as well once you're done.
    """
    # Importing locally for responsiveness.
    from maastest.kvmfixture import KVMFixture

    # TODO: series and architecture should be script parameters.
    architecture = utils.determine_vm_architecture()
    fixture = KVMFixture(
        series=args.maas_series, architecture=architecture,
        proxy_url=proxy_url, direct_interface=args.interface,
        archives=args.archive, kvm_timeout=args.kvm_timeout,
        simplestreams_filter=args.maas_simplestreams_filter)
    return fixture


def install_maas(args, machine, proxy_url):
    """Install MAAS in the virtual machine.

    :return: MAAS fixture.  It still needs to be set up, but it will already
        have MAAS installed in the virtual machine.  Clean up after use.
    """
    # Importing locally for responsiveness.
    from maastest.maasfixture import MAASFixture
    fixture = MAASFixture(
        machine, proxy_url=proxy_url, series=args.series,
        architecture=args.architecture,
        simplestreams_filter=args.simplestreams_filter)
    fixture.install_maas()
    return fixture


def main(args):
    # We tie everything onto one output stream so that we can capture
    # things for reporting.
    output_stream = utils.CachingOutputStream(sys.stdout)
    logging.basicConfig(
        level=logging.DEBUG, format='%(asctime)s %(levelname)s %(message)s',
        stream=output_stream)

    # TODO: Use Python 3's contextlib.ExitStack.  Not too hard to fake in 2.7.
    proxy_fixture = None
    machine_fixture = None
    maas_fixture = None
    maas_is_set_up = False
    try:
        check_for_root()
        check_for_virtualisation_support()
        check_against_virtual_host()
        check_against_dhcp_servers(args.interface)

        proxy_url, proxy_fixture = set_up_proxy(args)
        machine_fixture = create_vm(args, proxy_url)
        machine_fixture.setUp()
        maas_fixture = install_maas(args, machine_fixture, proxy_url)

        # Before we go on to download PXE images etc. use the virtual MAAS
        # installation to check for DHCP servers.
        check_against_dhcp_servers_from_vm(machine_fixture)

        # Now go through the rest of MAAS setup, and the actual tests.
        maas_fixture.setUp()
        maas_is_set_up = True

        from maastest.cases import TestOneNode

        class ConfiguredTestMAAS(TestOneNode):
            """A configured version of TestInteractiveOneNode.

            ConfiguredTestMAAS is a TestMAAS use to configure it by calling
            cls.configure.  We need to configure the class itself and not
            the instance because the KVMFixture TestMAAS instantiates
            and needs to configure is created at the class level.
            """
        ConfiguredTestMAAS.configure(args, maas_fixture)

        from maastest.console import run_console

        result = run_console(ConfiguredTestMAAS, output_stream)

        from maastest import report

        encoded_results = (
            output_stream.cache.getvalue().encode('utf-8'))

        if result.wasSuccessful():
            # If the test succeeded, we have to gather the details by
            # hand and include them with the output. Testtools won't do
            # it it for us.
            details = _details_to_str(
                ConfiguredTestMAAS.details).encode('utf-8')
            encoded_results = encoded_results + details

        if not args.no_reporting:
            report.write_test_results(encoded_results)

        if not args.no_reporting and not args.log_results_only:
            report.report_test_results(encoded_results, result.wasSuccessful())
        if result.wasSuccessful():
            return RETURN_CODES.SUCCESS
        else:
            return RETURN_CODES.TEST_FAILED

    except ProgramFailure as e:
        logging.error(e)
        return e.return_code
    finally:
        if maas_is_set_up:
            maas_fixture.cleanUp()
        if machine_fixture is not None:
            machine_fixture.cleanUp()
        if proxy_fixture is not None:
            proxy_fixture.cleanUp()
