/*
 *
 * (C) COPYRIGHT 2011-2013 ARM Limited. All rights reserved.
 *
 * This program is free software and is provided to you under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation, and any use by you of this program is subject to the terms
 * of such GNU licence.
 *
 * A copy of the licence is included with the program, and can also be obtained
 * from Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA  02110-1301, USA.
 *
 */

#ifdef CONFIG_MALI_PLATFORM_FAKE

#include <linux/errno.h>
#include <linux/export.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/string.h>

#ifdef CONFIG_MACH_MANTA
#include <plat/devs.h>
#endif

/* 
 * This file is included only for type definitions and functions belonging to
 * specific platform folders. Do not add dependencies with symbols that are
 * defined somewhere else.
 */
#include <kbase/mali_kbase_config.h>

#define PLATFORM_CONFIG_RESOURCE_COUNT 4
#define PLATFORM_CONFIG_IRQ_RES_COUNT  3

static struct platform_device *mali_device;

/**
 * @brief Gets the count of attributes in array
 *
 * Function gets the count of attributes in array. Note that end of list indicator is also included.
 *
 * @param[in]  attributes     Array of attributes
 *
 * @return  Number of attributes in the array including end of list indicator.
 */
static int kbasep_get_config_attribute_count(const kbase_attribute *attributes)
{
	int count = 1;

	if (!attributes)
		return -EINVAL;

	while (attributes->id != KBASE_CONFIG_ATTR_END) {
		attributes++;
		count++;
	}

	return count;
}

/**
 * @brief Convert data in kbase_io_resources struct to Linux-specific resources
 *
 * Function converts data in kbase_io_resources struct to an array of Linux resource structures. Note that function
 * assumes that size of linux_resource array is at least PLATFORM_CONFIG_RESOURCE_COUNT.
 * Resources are put in fixed order: I/O memory region, job IRQ, MMU IRQ, GPU IRQ.
 *
 * @param[in]  io_resource      Input IO resource data
 * @param[out] linux_resources  Pointer to output array of Linux resource structures
 */
static void kbasep_config_parse_io_resources(const kbase_io_resources *io_resources, struct resource *const linux_resources)
{
	if (!io_resources || !linux_resources) {
		printk(KERN_ERR "%s: couldn't find proper resources\n", __func__);
		return;
	}

	memset(linux_resources, 0, PLATFORM_CONFIG_RESOURCE_COUNT * sizeof(struct resource));

	linux_resources[0].start = io_resources->io_memory_region.start;
	linux_resources[0].end = io_resources->io_memory_region.end;
	linux_resources[0].flags = IORESOURCE_MEM;

	linux_resources[1].start = linux_resources[1].end = io_resources->job_irq_number;
	linux_resources[1].flags = IORESOURCE_IRQ | IORESOURCE_IRQ_HIGHLEVEL;

	linux_resources[2].start = linux_resources[2].end = io_resources->mmu_irq_number;
	linux_resources[2].flags = IORESOURCE_IRQ | IORESOURCE_IRQ_HIGHLEVEL;

	linux_resources[3].start = linux_resources[3].end = io_resources->gpu_irq_number;
	linux_resources[3].flags = IORESOURCE_IRQ | IORESOURCE_IRQ_HIGHLEVEL;
}

int kbase_platform_fake_register(void)
{
	kbase_platform_config *config;
	int attribute_count;
	struct resource resources[PLATFORM_CONFIG_RESOURCE_COUNT];
	int err;

	config = kbase_get_platform_config(); /* declared in kbase/mali_kbase_config.h but defined in platform folder */
	if (config == NULL)
	{
		printk(KERN_ERR "%s: couldn't get platform config\n", __func__);
		return -ENODEV;
	}

	attribute_count = kbasep_get_config_attribute_count(config->attributes);
#ifdef CONFIG_MACH_MANTA
	err = platform_device_add_data(&exynos5_device_g3d, config->attributes, attribute_count * sizeof(config->attributes[0]));
	if (err)
		return err;
#else

	mali_device = platform_device_alloc("mali", 0);
	if (mali_device == NULL)
		return -ENOMEM;

	kbasep_config_parse_io_resources(config->io_resources, resources);
	err = platform_device_add_resources(mali_device, resources, PLATFORM_CONFIG_RESOURCE_COUNT);
	if (err) {
		platform_device_put(mali_device);
		mali_device = NULL;
		return err;
	}

	err = platform_device_add_data(mali_device, config->attributes, attribute_count * sizeof(config->attributes[0]));
	if (err) {
		platform_device_unregister(mali_device);
		mali_device = NULL;
		return err;
	}

	err = platform_device_add(mali_device);
	if (err) {
		platform_device_unregister(mali_device);
		mali_device = NULL;
		return err;
	}
#endif /* CONFIG_CONFIG_MACH_MANTA */
	return 0;
}

void kbase_platform_fake_unregister(void)
{
	if (mali_device)
		platform_device_unregister(mali_device);
}

#ifdef MALI_PLATFORM_FAKE_MODULE
module_init(kbase_platform_fake_register);
module_exit(kbase_platform_fake_unregister);

MODULE_LICENSE("GPL");
MODULE_VERSION("mali_kbase_platform_fake");
#else
EXPORT_SYMBOL(kbase_platform_fake_register);
EXPORT_SYMBOL(kbase_platform_fake_unregister);
#endif

#endif /* CONFIG_MALI_PLATFORM_FAKE */

