/*
 * SVGDOcument.java
 */
package latexDraw.parsers.svg;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URI;

import javax.swing.JProgressBar;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import latexDraw.figures.Figure;
import latexDraw.parsers.svg.elements.OtherNSElement;
import latexDraw.parsers.svg.elements.SVGSVGElement;

import org.w3c.dom.Attr;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Comment;
import org.w3c.dom.DOMConfiguration;
import org.w3c.dom.DOMException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentFragment;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.EntityReference;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.ProcessingInstruction;
import org.w3c.dom.Text;
import org.w3c.dom.UserDataHandler;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;


/**
 * Defines an SVG document.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 09/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class SVGDocument implements Document
{
	/** The root of the SVG drawing. @since 0.1 */
	protected SVGSVGElement root;
	
    /** Defines if the document is standalone. @since 0.1 */
    protected boolean xmlStandalone;

    /** The document URI. @since 0.1 */
    protected String documentURI;
    
    /** The version of XML. @since 0.1 */
    protected String xmlVersion;
	
    /** The encoding of the document. @since 0.1 */ 
    protected String xmlEncoding;
    
    
    public final static String ACTION_NOT_IMPLEMENTED = "Action not implemented.";//$NON-NLS-1$
    
    
    public static final String SVG_NAMESPACE = "http://www.w3.org/2000/svg";//$NON-NLS-1$
    
    
	/**
	 * The constructor. An URI defines the location of the SVG document to parse. If the document is valid,
	 * the document is read an place in the <code>root</code> attribute. 
	 * @param uri The file to parse.
	 * @throws MalformedSVGDocument If an error occurs.
	 * @throws IllegalArgumentException If a n argument is not valid.
	 */
	public SVGDocument(URI uri) throws MalformedSVGDocument
	{
		if(uri==null)
			throw new IllegalArgumentException();
		
		try
		{
			DocumentBuilderFactory factory 	= DocumentBuilderFactory.newInstance();
	        DocumentBuilder builder 		= factory.newDocumentBuilder();
	        
	        builder.setEntityResolver(new EntityResolver()
			{
				public InputSource resolveEntity(String publicId, String systemId)
				{
					return new InputSource(new ByteArrayInputStream("<?xml version='1.0' encoding='UTF-8'?>".getBytes()));
				}
			});
	        
			Document doc = builder.parse(uri.getPath());
			NodeList nl;
			
			setDocumentURI(getDocumentURI());
			setXmlStandalone(doc.getXmlStandalone());
			setXmlVersion(doc.getXmlVersion());
			xmlEncoding = doc.getXmlEncoding();
			root 		= null;
			nl 			= doc.getChildNodes();
			Node n;
			
			for(int i=0, size = nl.getLength(); i<size && root==null; i++)
			{
				n = nl.item(i);
				
				if(n instanceof Element && n.getNodeName().endsWith(SVGElements.SVG_SVG))
					root = new SVGSVGElement(this, nl.item(i));
			}
				
			nl		= null;
			doc		= null;
			builder	= null;
			factory	= null;
		}
		catch(SAXException e)
		{
			e.printStackTrace();
			throw new MalformedSVGDocument();
		}
		catch(IOException e)
		{
			e.printStackTrace();
			throw new MalformedSVGDocument();
		}
		catch(ParserConfigurationException e1)
		{
			e1.printStackTrace();
			throw new MalformedSVGDocument();
		}
	}


	
	/**
	 * Creates an SVG document with an empty SVG element.
	 */
	public SVGDocument()
	{
		setDocumentURI(null);
		setXmlVersion("1.1");//$NON-NLS-1$
		setXmlStandalone(false);
		
		root = new SVGSVGElement(this);
	}




	@Override
	public String toString()
	{
		StringBuffer stringBuffer = new StringBuffer();
		
		stringBuffer.append("SVG Document:");//$NON-NLS-1$
		stringBuffer.append(root.toString());
		
		return stringBuffer.toString();
	}

	
	
	/**
	 * Allows to get the latexdraw version of the SVG drawing.
	 * @return A drawing containing all the figures or a unique figure or null.
	 * @since 0.1
	 */
	public Figure toLaTeXDraw(JProgressBar progressBar)
	{
		return root.toLaTeXDraw(progressBar);
	}

	
	
	public SVGSVGElement adoptNode(Node source) throws DOMException
	{
		if(source==null || !(source instanceof SVGSVGElement))
			throw new DOMException(DOMException.TYPE_MISMATCH_ERR, "SVGSVGElement expected here.");//$NON-NLS-1$
		
		root = (SVGSVGElement)source;
		root.setOwnerDocument(this);
		
		return root;
	}



	public SVGSVGElement getDocumentElement()
	{
		return root;
	}


	public String getDocumentURI()
	{
		return documentURI;
	}


	public String getXmlEncoding()
	{
		return xmlEncoding;
	}


	public boolean getXmlStandalone()
	{
		return xmlStandalone;
	}


	public String getXmlVersion()
	{
		return xmlVersion;
	}


	public void setDocumentURI(String documentURI)
	{
		this.documentURI = documentURI;
	}


	public void setXmlStandalone(boolean xmlStandalone) throws DOMException
	{
		this.xmlStandalone = xmlStandalone;
	}


	public void setXmlVersion(String xmlVersion) throws DOMException
	{
		this.xmlVersion = xmlVersion;
	}



	public NamedNodeMap getAttributes()
	{
		return null;
	}


	public NodeList getChildNodes()
	{
		return null;
	}


	public SVGSVGElement getFirstChild()
	{
		return root;
	}


	public SVGSVGElement getLastChild()
	{
		return root;
	}



	public Node getNextSibling()
	{
		return null;
	}


	public String getNodeName()
	{
		return "#document";//$NON-NLS-1$
	}


	public short getNodeType()
	{
		return Node.DOCUMENT_NODE;
	}


	public String getNodeValue() throws DOMException
	{
		return null;
	}


	public Document getOwnerDocument()
	{
		return null;
	}


	public Node getParentNode()
	{
		return null;
	}



	public Node getPreviousSibling()
	{
		return null;
	}



	public boolean hasAttributes()
	{
		return false;
	}


	public boolean hasChildNodes()
	{
		return root!=null;
	}



	public boolean isEqualNode(Node node)
	{
		if(node==null || !(node instanceof SVGDocument))
			return false;
		
		SVGDocument doc = (SVGDocument)node;
		boolean encod = xmlEncoding==null ? doc.xmlEncoding==null : xmlEncoding.equals(doc.xmlEncoding);
		boolean uri   = documentURI==null ? doc.documentURI==null : documentURI.equals(doc.documentURI);
		
		return xmlStandalone==doc.xmlStandalone && encod && root.isEqualNode(doc.root) && uri;
	}


	public boolean isSameNode(Node other)
	{ 
		return other!=null && other==this;
	}

	
	public DocumentType getDoctype()
	{
		return null;
	}
	
	
	
	public Element createElement(String tagName) throws DOMException
	{
		if(tagName==null)
			throw new DOMException(DOMException.INVALID_CHARACTER_ERR, "Invalid tagName.");//$NON-NLS-1$
		
		OtherNSElement elt = new OtherNSElement(this);
		elt.setNodeName(tagName);
		
		return elt;
	}
	
	
	public Text createTextNode(String data)
	{
		if(data==null)
			throw new DOMException(DOMException.INVALID_CHARACTER_ERR, "Invalid data.");//$NON-NLS-1$
		
		return new SVGText(data, this);
	}
	
	
	public Comment createComment(String data)
	{
		if(data==null)
			throw new DOMException(DOMException.INVALID_CHARACTER_ERR, "Invalid data.");//$NON-NLS-1$
		
		return new SVGComment(data, this);
	}
	
	
	public CDATASection createCDATASection(String data) throws DOMException
	{
		if(data==null)
			throw new DOMException(DOMException.INVALID_CHARACTER_ERR, "Invalid data.");//$NON-NLS-1$
		
		return new SVGCDATASection(data, this);
	}
	
	
	public DOMConfiguration getDomConfig()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public DOMImplementation getImplementation()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public String getInputEncoding()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public boolean getStrictErrorChecking()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public Node importNode(Node importedNode, boolean deep) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public void normalizeDocument()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public Node renameNode(Node n, String namespaceURI, String qualifiedName) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public void setStrictErrorChecking(boolean strictErrorChecking)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public Node appendChild(Node newChild) throws DOMException
	{ return null; }

	public Node cloneNode(boolean deep)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public short compareDocumentPosition(Node other) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public String getBaseURI()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public Object getFeature(String feature, String version)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public String getLocalName()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public String getNamespaceURI()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public String getPrefix()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public String getTextContent() throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public Object getUserData(String key)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public boolean isDefaultNamespace(String namespaceURI)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }
	
	public Attr createAttribute(String name) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public Attr createAttributeNS(String namespaceURI, String qualifiedName) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public DocumentFragment createDocumentFragment()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public Element createElementNS(String namespaceURI, String qualifiedName) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public EntityReference createEntityReference(String name) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public ProcessingInstruction createProcessingInstruction(String target, String data) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public boolean isSupported(String feature, String version)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public String lookupNamespaceURI(String prefix)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }
	
	public String lookupPrefix(String namespaceURI)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public void normalize()
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }

	public Object setUserData(String key, Object data, UserDataHandler handler)
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, ACTION_NOT_IMPLEMENTED); }
	
	
	
	public Node insertBefore(Node newChild, Node refChild) throws DOMException
	{ return null; }
	
	public void setPrefix(String prefix) throws DOMException
	{ /* Nothing to do. */ }
	
	public void setTextContent(String textContent) throws DOMException
	{ /* Nothing to do. */ }
	
	public Node removeChild(Node oldChild) throws DOMException
	{ return null; }

	public Node replaceChild(Node newChild, Node oldChild) throws DOMException
	{ return null; }

	public void setNodeValue(String nodeValue) throws DOMException
	{ /* Nothing to do. */ }
	
	public Element getElementById(String elementId)
	{ return null; }

	public NodeList getElementsByTagName(String tagname)
	{ return null; }

	public NodeList getElementsByTagNameNS(String namespaceURI, String localName)
	{ return null; }
}
