/* ----------------------------------------------------------------------
   LAMMPS - Large-scale Atomic/Molecular Massively Parallel Simulator
   https://www.lammps.org/, Sandia National Laboratories
   LAMMPS development team: developers@lammps.org

   Copyright (2003) Sandia Corporation.  Under the terms of Contract
   DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
   certain rights in this software.  This software is distributed under
   the GNU General Public License.

   See the README file in the top-level LAMMPS directory.
------------------------------------------------------------------------- */

#include "atom_vec_tdpd.h"

#include "atom.h"
#include "error.h"
#include "update.h"

#include <cstring>

using namespace LAMMPS_NS;

/* ---------------------------------------------------------------------- */

AtomVecTDPD::AtomVecTDPD(LAMMPS *lmp) : AtomVec(lmp)
{
  molecular = Atom::ATOMIC;
  mass_type = PER_TYPE;
  forceclearflag = 1;

  atom->tdpd_flag = 1;
  atom->vest_flag = 1;

  // strings with peratom variables to include in each AtomVec method
  // strings cannot contain fields in corresponding AtomVec default strings
  // order of fields in a string does not matter
  // except: fields_data_atom & fields_data_vel must match data file

  fields_grow = {"cc", "cc_flux", "vest"};
  fields_copy = {"cc", "vest"};
  fields_comm = {"cc", "vest"};
  fields_comm_vel = {"cc", "vest"};
  fields_reverse = {"cc_flux"};
  fields_border = {"cc", "vest"};
  fields_border_vel = {"cc", "vest"};
  fields_exchange = {"cc", "vest"};
  fields_restart = {"cc", "vest"};
  fields_create = {"cc", "vest"};
  fields_data_atom = {"id", "type", "x", "cc"};
  fields_data_vel = {"id", "v"};
}

/* ----------------------------------------------------------------------
   process additional args
   single arg = number of cc_species
------------------------------------------------------------------------- */

void AtomVecTDPD::process_args(int narg, char **arg)
{
  if (narg < 1) error->all(FLERR, "Invalid atom_style tdpd command");

  atom->cc_species = utils::inumeric(FLERR, arg[0], false, lmp);
  cc_species = atom->cc_species;

  atom->add_peratom_change_columns("cc", cc_species);
  atom->add_peratom_change_columns("cc_flux", cc_species);

  // delay setting up of fields until now

  setup_fields();
}

/* ---------------------------------------------------------------------- */

void AtomVecTDPD::init()
{
  AtomVec::init();

  if (strcmp(update->unit_style, "lj") != 0) error->all(FLERR, "Atom style tdpd requires lj units");
}

/* ----------------------------------------------------------------------
   set local copies of all grow ptrs used by this class, except defaults
   needed in replicate when 2 atom classes exist and it calls pack_restart()
------------------------------------------------------------------------- */

void AtomVecTDPD::grow_pointers()
{
  cc_flux = atom->cc_flux;
  vest = atom->vest;
}

/* ----------------------------------------------------------------------
   clear extra forces starting at atom N
   nbytes = # of bytes to clear for a per-atom vector
------------------------------------------------------------------------- */

void AtomVecTDPD::force_clear(int n, size_t nbytes)
{
  memset(&cc_flux[n][0], 0, cc_species * nbytes);
}

/* ----------------------------------------------------------------------
   modify what AtomVec::data_atom() just unpacked
   or initialize other atom quantities
------------------------------------------------------------------------- */

void AtomVecTDPD::data_atom_post(int ilocal)
{
  atom->vest[ilocal][0] = 0.0;
  atom->vest[ilocal][1] = 0.0;
  atom->vest[ilocal][2] = 0.0;
}
