import os
import inspect

from twisted.internet.defer import inlineCallbacks
from twisted.internet.threads import deferToThread

from juju.lib.lxc import (_lxc_start, _lxc_stop, _lxc_create,
                          _lxc_wait, _lxc_ls, _lxc_destroy,
                          LXCContainer, get_containers, LXCError,
                          DEFAULT_TEMPLATE, ensure_devtmpfs_fstab,
                          tests)
from juju.lib.testing import TestCase
from juju.providers.common.cloudinit import CloudInit
from juju.machine import ProviderMachine


def skip_sudo_tests():
    if os.environ.get("TEST_SUDO"):
        # Get user's password *now*, if needed, not mid-run
        os.system("sudo false")
        return False
    return "TEST_SUDO=1 to include tests which use sudo (including lxc tests)"


def uses_sudo(f):
    f.skip = skip_sudo_tests()
    return f

DATA_DIR = os.path.join(os.path.dirname(inspect.getabsfile(tests)), "data")

DEFAULT_SERIES = "precise"
DEFAULT_CONTAINER = "lxc_test"


@uses_sudo
class LXCTest(TestCase):
    timeout = 240

    def setUp(self):
        self.cloud_init_file = self.mktemp()
        with open(self.cloud_init_file, 'w') as cloud_init:
            cloud_init.write("# cloud-init\n")

    def tearDown(self):
        self.clean_container(DEFAULT_CONTAINER)

    def clean_container(self, container_name):
        if os.path.exists("/var/lib/lxc/%s" % container_name):
            _lxc_stop(container_name)
            _lxc_destroy(container_name)

    def test_lxc_create(self):
        self.addCleanup(self.clean_container, DEFAULT_CONTAINER)

        _lxc_create(DEFAULT_CONTAINER, DEFAULT_TEMPLATE, DEFAULT_SERIES,
                self.cloud_init_file)

        # verify we can find the container
        output = _lxc_ls()
        self.assertIn(DEFAULT_CONTAINER, output)

        # remove and verify the container was removed
        _lxc_destroy(DEFAULT_CONTAINER)
        output = _lxc_ls()
        self.assertNotIn(DEFAULT_CONTAINER, output)

    def test_lxc_start(self):
        self.addCleanup(self.clean_container, DEFAULT_CONTAINER)

        _lxc_create(DEFAULT_CONTAINER, DEFAULT_TEMPLATE, DEFAULT_SERIES,
                self.cloud_init_file)
        _lxc_start(DEFAULT_CONTAINER)
        _lxc_stop(DEFAULT_CONTAINER)

    @inlineCallbacks
    def test_lxc_deferred(self):
        self.addCleanup(self.clean_container, DEFAULT_CONTAINER)
        yield deferToThread(
            _lxc_create, DEFAULT_CONTAINER, DEFAULT_TEMPLATE, DEFAULT_SERIES,
            self.cloud_init_file)
        yield deferToThread(_lxc_start, DEFAULT_CONTAINER)

    @inlineCallbacks
    def test_lxc_container(self):
        self.addCleanup(self.clean_container, DEFAULT_CONTAINER)
        c = LXCContainer(DEFAULT_CONTAINER,
                         "precise")
        running = yield c.is_running()
        self.assertFalse(running)
        self.assertFalse(c.is_constructed())

        # verify we can't run a non-constructed container
        failure = c.run()
        yield self.assertFailure(failure, LXCError)

        yield c.create()

        self.assertFalse(running)
        self.assertTrue(c.is_constructed())
        yield c.run()

        running = yield c.is_running()
        self.assertTrue(running)
        self.assertTrue(c.is_constructed())

        output = _lxc_ls()
        self.assertIn(DEFAULT_CONTAINER, output)

        # Verify we have a path into the container
        self.assertTrue(os.path.exists(c.rootfs))
        self.assertTrue(c.is_constructed())

        self.verify_container(c, series="precise")

        # Verify that we are in containers
        containers = yield get_containers(None)
        self.assertEqual(containers[DEFAULT_CONTAINER], True)

        # tear it down
        yield c.destroy()
        running = yield c.is_running()
        self.assertFalse(running)

        containers = yield get_containers(None)
        self.assertNotIn(DEFAULT_CONTAINER, containers)

        # and its gone
        output = _lxc_ls()
        self.assertNotIn(DEFAULT_CONTAINER, output)

    @inlineCallbacks
    def test_lxc_wait(self):
        self.addCleanup(self.clean_container, DEFAULT_CONTAINER)

        _lxc_create(DEFAULT_CONTAINER, DEFAULT_TEMPLATE, DEFAULT_SERIES,
                self.cloud_init_file)

        _lxc_start(DEFAULT_CONTAINER)

        def waitForState(result):
            self.assertEqual(result, True)

        d = _lxc_wait(DEFAULT_CONTAINER, "RUNNING")
        d.addCallback(waitForState)
        yield d

        _lxc_stop(DEFAULT_CONTAINER)
        yield _lxc_wait(DEFAULT_CONTAINER, "STOPPED")
        _lxc_destroy(DEFAULT_CONTAINER)

    @inlineCallbacks
    def test_container_clone(self):
        self.addCleanup(self.clean_container, DEFAULT_CONTAINER)
        self.addCleanup(self.clean_container, DEFAULT_CONTAINER + "_child")

        cloud_init = CloudInit()
        cloud_init.add_ssh_key('dsa...')
        cloud_init.set_provider_type('local')
        cloud_init.set_zookeeper_machines([ProviderMachine('localhost','localhost','localhost')])

        master_container = LXCContainer(DEFAULT_CONTAINER,
                                        series="precise",
                                        cloud_init=cloud_init)

        # verify that we cannot clone an unconstructed container
        failure = master_container.clone("test_lxc_fail")
        yield self.assertFailure(failure, LXCError)

        yield master_container.create()

        # Clone a child container from the template
        child_name = DEFAULT_CONTAINER + "_child"
        c = yield master_container.clone(child_name)

        self.assertEqual(c.container_name, child_name)

        running = yield c.is_running()
        self.assertFalse(running)
        yield c.run()

        running = yield c.is_running()
        self.assertTrue(running)

        output = _lxc_ls()
        self.assertIn(DEFAULT_CONTAINER, output)

        self.verify_container(c, series="precise", cloud_init=cloud_init.render())

        # verify that we are in containers
        containers = yield get_containers(None)
        self.assertEqual(containers[child_name], True)

        # tear it down
        yield c.destroy()
        running = yield c.is_running()
        self.assertFalse(running)

        containers = yield get_containers(None)
        self.assertNotIn(child_name, containers)

        # and its gone
        output = _lxc_ls()
        self.assertNotIn(child_name, output)

        yield master_container.destroy()

    def test_create_wait(self):
        self.addCleanup(self.clean_container, DEFAULT_CONTAINER)

        cinit = CloudInit()
        cinit.add_ssh_key('dsa...')
        cinit.set_provider_type('local')
        cinit.set_zookeeper_machines([ProviderMachine('localhost','localhost','localhost')])

        c = LXCContainer(DEFAULT_CONTAINER, series="precise", cloud_init=cinit)
        c._create_wait()
        self.verify_container(c, "precise", cinit.render())


    def verify_container(self, c, series, cloud_init=None):
        """Verify properties of an LXCContainer"""

        def p(path):
            return os.path.join(c.rootfs, path)

        def sudo_get(path):
            # super get path (superuser priv)
            rc, output = c.execute(["cat", path])
            return output

        def run(cmd):
            try:
                rc, output = c.execute(cmd)
            except LXCError:
                rc = 1
            return rc

        # basic path checks
        self.assertTrue(os.path.exists(p('var/lib/cloud/seed/nocloud-net')))

        # ubuntu user
        self.assertEqual(run(["id", "ubuntu"]), 0)

        # Verify the container release series.
        with open(os.path.join(c.rootfs, "etc", "lsb-release")) as fh:
            lsb_info = fh.read()
        self.assertIn(series, lsb_info)


class LXCUtilTest(TestCase):
    def test_ensure_devtmpfs_fstab(self):
        without_dev_pts = self.makeDir()
        os.mkdir(os.path.join(without_dev_pts, 'rootfs'))
        ensure_devtmpfs_fstab(without_dev_pts)
        fstab_path = os.path.join(without_dev_pts, 'fstab')
        if os.path.exists(fstab_path):
            with open(fstab_path) as fstab:
                found = False
                for line in fstab:
                    if line.startswith('devtmpfs'):
                        found = True
                        break
                self.assertTrue(found)
        else:
            self.fail('fstab missing')

    def test_ensure_devtmpfs_fstab_pts_exists(self):
        with_dev_pts = self.makeDir()
        os.mkdir(os.path.join(with_dev_pts, 'rootfs'))
        os.mkdir(os.path.join(with_dev_pts, 'rootfs', 'dev'))
        os.mkdir(os.path.join(with_dev_pts, 'rootfs', 'dev', 'pts'))
        ensure_devtmpfs_fstab(with_dev_pts)
        fstab_path = os.path.join(with_dev_pts, 'fstab')
        if os.path.exists(fstab_path):
            self.fail('fstab created but pts already there')

    def test_ensure_devtmpfs_fstab_line_added(self):
        self._test_devtmpfs_line('sample_fstab')

    def test_ensure_devtmpfs_fstab_line_onlyone(self):
        self._test_devtmpfs_line('sample_fstab_withdevtmpfs')

    def _test_devtmpfs_line(self, datafile):
        without_dev_pts = self.makeDir()
        os.mkdir(os.path.join(without_dev_pts, 'rootfs'))
        fstab_path = os.path.join(without_dev_pts, 'fstab')
        with open(fstab_path, 'w') as fstab:
            source = os.path.join(DATA_DIR, datafile)
            with open(source) as src:
                fstab.write(src.read())
        ensure_devtmpfs_fstab(without_dev_pts)
        if os.path.exists(fstab_path):
            with open(fstab_path) as fstab:
                count = 0
                for line in fstab:
                    if line.startswith('devtmpfs'):
                        count += 1
                self.assertEquals(count, 1)
        else:
            self.fail('fstab missing')
