/*
 * Jalview - A Sequence Alignment Editor and Viewer (2.11.5.0)
 * Copyright (C) 2025 The Jalview Authors
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *  
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 * The Jalview Authors are detailed in the 'AUTHORS' file.
 */
package jalview.gui;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Vector;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;

import jalview.analysis.AlignmentUtils;
import jalview.analysis.Conservation;
import jalview.analysis.TreeModel;
import jalview.api.AlignViewportI;
import jalview.bin.Console;
import jalview.datamodel.AlignmentAnnotation;
import jalview.datamodel.BinaryNode;
import jalview.datamodel.ColumnSelection;
import jalview.datamodel.ContactMatrixI;
import jalview.datamodel.HiddenColumns;
import jalview.datamodel.Sequence;
import jalview.datamodel.SequenceGroup;
import jalview.datamodel.SequenceI;
import jalview.datamodel.SequenceNode;
import jalview.gui.JalviewColourChooser.ColourChooserListener;
import jalview.schemes.ColourSchemeI;
import jalview.structure.SelectionSource;
import jalview.util.ColorUtils;
import jalview.util.Format;
import jalview.util.MessageManager;
import jalview.ws.datamodel.MappableContactMatrixI;

/**
 * DOCUMENT ME!
 * 
 * @author $author$
 * @version $Revision$
 */
public class TreeCanvas extends JPanel implements MouseListener, Runnable,
        Printable, MouseMotionListener, SelectionSource
{
  /** DOCUMENT ME!! */
  public static final String PLACEHOLDER = " * ";
  
  private static final int DASHED_LINE_Y_OFFSET = 6;

  TreeModel tree;

  JScrollPane scrollPane;

  TreePanel tp;

  private AlignViewport av;

  private AlignmentPanel ap;

  Font font;

  FontMetrics fm;

  boolean fitToWindow = true;

  boolean showDistances = false;

  boolean showStructureProviderLabels = false;

  boolean showStructureProviderColouredLines = false;
  
  Map<String, Boolean> structureProviderColouredLineToggleState = new HashMap<String, Boolean>();

  boolean showBootstrap = false;

  boolean markPlaceholders = false;

  int offx = 20;

  int offy;

  private float threshold;
  
  int labelLengthThreshold = 4;

  String longestName;

  int labelLength = -1;
  
  private static final int COLOR_TRANSPARENCY_FOR_GROUP = 198; // approx 25% transparency
  private static final int COLOR_TRANSPARENCY_FOR_COLOURED_LINES = 80;
  
  // Variable to store index for tree group box dimensions
  private static final int TREE_GROUP_DIM_MIN_X_INDEX = 0;
  private static final int TREE_GROUP_DIM_MIN_Y_INDEX = 1;
  private static final int TREE_GROUP_DIM_MAX_X_INDEX = 2;
  private static final int TREE_GROUP_DIM_MAX_Y_INDEX = 3;

  /**
   * TODO - these rectangle-hash lookups should be optimised for big trees...
   */
  Map<BinaryNode, Rectangle> nameHash = new Hashtable<>();

  Map<BinaryNode, Rectangle> nodeHash = new Hashtable<>();

  BinaryNode highlightNode;

  boolean applyToAllViews = false;
  
  
  //Map to store label positions (BinaryNode -> List of bounding rectangles for the label)
  //private Map<BinaryNode, List<Rectangle>> labelBoundsMap = new HashMap<>();
  
  

  /**
   * Creates a new TreeCanvas object.
   * 
   * @param av
   *          DOCUMENT ME!
   * @param tree
   *          DOCUMENT ME!
   * @param scroller
   *          DOCUMENT ME!
   * @param label
   *          DOCUMENT ME!
   */
  public TreeCanvas(TreePanel tp, AlignmentPanel ap, JScrollPane scroller)
  {
    this.tp = tp;
    this.av = ap.av;
    this.setAssociatedPanel(ap);
    font = av.getFont();
    scrollPane = scroller;
    addMouseListener(this);
    addMouseMotionListener(this);

    ToolTipManager.sharedInstance().registerComponent(this);
  }

  public void clearSelectedLeaves()
  {
    Vector<BinaryNode> leaves = tp.getTree()
            .findLeaves(tp.getTree().getTopNode());
    if (tp.isColumnWise())
    {
      markColumnsFor(getAssociatedPanels(), leaves, Color.white, true);
    }
    else
    {
      for (AlignmentPanel ap : getAssociatedPanels())
      {
        SequenceGroup selected = ap.av.getSelectionGroup();
        if (selected != null)
        {
          {
            for (int i = 0; i < leaves.size(); i++)
            {
              SequenceI seq = (SequenceI) leaves.elementAt(i).element();
              if (selected.contains(seq))
              {
                selected.addOrRemove(seq, false);
              }
              
              
              if (leaves.elementAt(i).hasAlignmentAnnotation())
              {
                AlignmentAnnotation annot = leaves.elementAt(i).getAlignmentAnnotation();
                
                if (selected.containsAnnotation(annot)) 
                {
                  selected.addOrRemoveAnnotation(annot);
                }
              }
            }
            selected.recalcConservation();
          }
        }
        ap.av.sendSelection();
      }
    }
    PaintRefresher.Refresh(tp, av.getSequenceSetId());
    repaint();
  }

  /**
   * DOCUMENT ME!
   * 
   * @param sequence
   *          DOCUMENT ME!
   */
  public void treeSelectionChanged(SequenceI sequence, AlignmentAnnotation alignmentAnnotation)
  {
    boolean annotationSelected = false;
    AlignmentPanel[] aps = getAssociatedPanels();

    for (int a = 0; a < aps.length; a++)
    {
      SequenceGroup selected = aps[a].av.getSelectionGroup();

      if (selected == null)
      {
        selected = new SequenceGroup();
        aps[a].av.setSelectionGroup(selected);
      }
      
      if(alignmentAnnotation!=null) {
        annotationSelected = selected.addOrRemoveAnnotation(alignmentAnnotation);
      }

      selected.setEndRes(aps[a].av.getAlignment().getWidth() - 1);
      
      if(annotationSelected) {
        selected.addSequence(sequence, true);
      }
      else {
        selected.addOrRemove(sequence, true);
      }
    }
  }
  
  /**
   * DOCUMENT ME!
   * 
   * @param tree
   *          DOCUMENT ME!
   */
  public void setTree(TreeModel tree)
  {
    this.tree = tree;
    tree.findHeight(tree.getTopNode());

    // Now have to calculate longest name based on the leaves
    Vector<BinaryNode> leaves = tree.findLeaves(tree.getTopNode());
    boolean has_placeholders = false;
    longestName = "";

    AlignmentAnnotation aa = tp.getAssocAnnotation();
    ContactMatrixI cm = (aa != null) ? av.getContactMatrix(aa) : null;
    if (cm != null && cm.hasCutHeight())
    {
      threshold = (float) cm.getCutHeight();
    }

    for (int i = 0; i < leaves.size(); i++)
    {
      BinaryNode lf = leaves.elementAt(i);

      if (lf instanceof SequenceNode && ((SequenceNode) lf).isPlaceholder())
      {
        has_placeholders = true;
      }

      if (longestName.length() < lf.getDisplayName().length())
      {
        longestName = TreeCanvas.PLACEHOLDER + lf.getDisplayName();
      }
      if (tp.isColumnWise() && cm != null)
      {
        // get color from group colours, if they are set for the matrix
        try
        {
          Color col = cm.getGroupColorForPosition(parseColumnNode(lf));
          setColor(lf, col.brighter());
        } catch (NumberFormatException ex)
        {
        }
        ;
      }
    }

    setMarkPlaceholders(has_placeholders);
  }

  /**
   * DOCUMENT ME!
   * 
   * @param g
   *          DOCUMENT ME!
   * @param node
   *          DOCUMENT ME!
   * @param chunk
   *          DOCUMENT ME!
   * @param wscale
   *          DOCUMENT ME!
   * @param width
   *          DOCUMENT ME!
   * @param offx
   *          DOCUMENT ME!
   * @param offy
   *          DOCUMENT ME!
   */
  public void drawNode(Graphics g, BinaryNode node, double chunk,
          double wscale, int width, int offx, int offy)
  {

    if (node == null)
    {
      return;
    }

    SequenceGroup colourGroup = null;
    Vector<BinaryNode> leaves = tree.findLeaves(node);
    gatherLabelsTo(node, leaves);

    if ((node.left() == null) && (node.right() == null))
    {
      Color annotationColor = Color.WHITE;
      // Drawing leaf node
      double height = node.height;
      double dist = node.dist;

      int xstart = (int) ((height - dist) * wscale) + offx;
      int xend = (int) (height * wscale) + offx;

      int ypos = (int) (node.ycount * chunk) + offy;

      if (node.element() instanceof SequenceI)
      {

        SequenceI seq = (SequenceI) node.element();

        SequenceGroup[] groupsWithSequence = null;
        
        if(ap.getAlignment() != null)
        {
          groupsWithSequence = ap.getAlignment()
                .findAllGroups(seq);
        }

        // JAL-4537
        if (av.getSequenceColour(seq).equals(Color.white))
        {
          g.setColor(Color.black);
        }
        else if (node.hasLabel() && node.hasAlignmentAnnotation())
        {
          if(groupsWithSequence != null) 
          {
            for (SequenceGroup group : groupsWithSequence)
            {
  
              if (group.getAnnotationsFromTree()
                      .contains(node.getAlignmentAnnotation())
                      && group.getName().startsWith("JTreeGroup:"))
              {
  
                colourGroup = group;
  
              }
  
            }
          }

          annotationColor = av
                  .getAnnotationColour(node.getAlignmentAnnotation());
          if (annotationColor == Color.white)
          {
            annotationColor = av.getSequenceColour(seq);
          }
          g.setColor(annotationColor.darker());

          if (showStructureProviderColouredLines)
          {
            g.setColor(Color.black);
          }
        }
        else
        {
          if (node.hasLabel())
          {
            if(groupsWithSequence != null) 
            {
              for (SequenceGroup group : groupsWithSequence)
              {
  
                if (group.getName().startsWith("JTreeGroup:"))
                {
  
                  colourGroup = group;
  
                }
  
              }
            }
          }
          g.setColor(av.getSequenceColour(seq).darker());
        }
      }

      else
      {
        g.setColor(Color.black);
      }

      if (!(node.hasLabel() && showStructureProviderColouredLines))
      {
        // horizontal line
        g.drawLine(xstart, ypos, xend, ypos);
      }

      String nodeLabel = "";

      if (showDistances && (node.dist > 0))
      {
        nodeLabel = new Format("%g").form(node.dist);
      }

      if (showBootstrap && node.bootstrap > -1)
      {
        if (showDistances)
        {
          nodeLabel = nodeLabel + " : ";
        }

        nodeLabel = nodeLabel + String.valueOf(node.bootstrap);
      }

      if (node.hasLabel() && showStructureProviderColouredLines)
      {

        drawLinesAndLabelsForSecondaryStructureProvider(g, node, xstart,
                xend, ypos, nodeLabel);

      }

      else if (!nodeLabel.equals(""))
      {
        g.drawString(nodeLabel, xstart + 2, ypos - 2);
      }

      String name = (markPlaceholders && ((node instanceof SequenceNode
              && ((SequenceNode) node).isPlaceholder())))
                      ? (PLACEHOLDER + node.getDisplayName())
                      : node.getDisplayName();

      int charWidth = fm.stringWidth(name) + 3;
      int charHeight = font.getSize();

      Rectangle rect = new Rectangle(xend + 10, ypos - charHeight / 2,
              charWidth, charHeight);

      nameHash.put(node, rect);

      if (node.hasLabel() && showStructureProviderColouredLines)
      {
        g.setColor(Color.black);
      }

      // Colour selected leaves differently
      boolean isSelected = false;
      if (tp.isColumnWise())
      {
        isSelected = isColumnForNodeSelected(node);
      }
      else
      {
        SequenceGroup selected = av.getSelectionGroup();

        if ((selected != null)
                && selected.getSequences(null).contains(node.element())
                && (node.getAlignmentAnnotation() == null
                        || selected.getAnnotationsFromTree() == null
                        || selected.getAnnotationsFromTree().isEmpty() 
                        || selected.getAnnotationsFromTree()
                                .contains(node.getAlignmentAnnotation())))
        {
          isSelected = true;
        }
      }
      if (isSelected)
      {
        g.setColor(Color.gray);

        g.fillRect(xend + 10, ypos - charHeight / 2, charWidth, charHeight);
        g.setColor(Color.white);
      }

      g.drawString(name, xend + 10, ypos + fm.getDescent());
      g.setColor(Color.black);
    }
    else
    {
      drawNode(g, (BinaryNode) node.left(), chunk, wscale, width, offx,
              offy);
      drawNode(g, (BinaryNode) node.right(), chunk, wscale, width, offx,
              offy);

      double height = node.height;
      double dist = node.dist;

      int xstart = (int) ((height - dist) * wscale) + offx;
      int xend = (int) (height * wscale) + offx;
      int ypos = (int) (node.ycount * chunk) + offy;

      // g.setColor(node.color.darker());

      if (node.hasLabel() && showStructureProviderColouredLines)
      {
        g.setColor(Color.black);
      }
      else
      {
        // horizontal line
        g.setColor(node.color.darker());
        g.drawLine(xstart, ypos, xend, ypos);
      }

      int ystart = (node.left() == null ? 0
              : (int) (node.left().ycount * chunk)) + offy;
      int yend = (node.right() == null ? 0
              : (int) (node.right().ycount * chunk)) + offy;

      Rectangle pos = new Rectangle(xend - 2, ypos - 2, 5, 5);
      nodeHash.put(node, pos);

      g.drawLine((int) (height * wscale) + offx, ystart,
              (int) (height * wscale) + offx, yend);

      String nodeLabel = "";

      if (showDistances && (node.dist > 0))
      {
        nodeLabel = new Format("%g").form(node.dist);
      }

      if (showBootstrap && node.bootstrap > -1)
      {
        if (showDistances)
        {
          nodeLabel = nodeLabel + " : ";
        }

        nodeLabel = nodeLabel + String.valueOf(node.bootstrap);
      }

      // Display secondary structure providers only if:
      // ~ node has Label assigned (Secondary structure providers)
      // ~ either label or coloured lines option is selected by the user
      boolean labelHandled = false;
      if (node.hasLabel())
      {

        if (showStructureProviderColouredLines)
        {
          drawLinesAndLabelsForSecondaryStructureProvider(g, node, xstart,
                  xend, ypos, nodeLabel);

          labelHandled = true;

        }

        if (showStructureProviderLabels && node.count > 3
                && node.parent() != null && node.left() != null
                && node.right() != null && node.dist > 0)
        {

          String label = node.getLabel();

          if (label.length() > labelLengthThreshold)
          {

            // label = AlignmentUtils.reduceLabelLength(label);
          }

          nodeLabel = label + " | " + nodeLabel;

          // Split the nodeLabel by "|"
          String[] lines = nodeLabel.split("\\|");

          // Draw each string separately
          String longestLabelString = "";
          int i = 0;
          for (i = 0; i < lines.length; i++)
          {
            g.drawString(lines[i].trim(), xstart + 2,
                    ypos - 2 - (i * fm.getHeight()));
            if (longestLabelString.length() < lines[i].trim().length())
            {
              longestLabelString = lines[i].trim();
            }
          }

          labelHandled = true;
        }
      }

      if (!nodeLabel.equals("") && !labelHandled)
      {
        g.drawString(nodeLabel, xstart + 2, ypos - 2);
      }

      if (node == highlightNode)
      {
        g.fillRect(xend - 3, ypos - 3, 6, 6);
      }
      else
      {
        g.fillRect(xend - 2, ypos - 2, 4, 4);
      }
    }
  }

  /**
   * Updates the bounding box dimensions for sequence groups in a tree.
   *
   * @param g                      graphics object
   * @param node                   tree node
   * @param chunk                  chunk
   * @param wscale                 width
   * @param offx                   x offset
   * @param offy                   y offset
   * @param currentTreeGroups      set of current tree groups
   * @param boxGroupColouringDim   map from group hash to bounding box dimensions [minX, minY, maxX, maxY]
   */
  private void updateGroupBoxDimensions(Graphics g, BinaryNode node, double chunk, 
          double wscale, int offx, int offy, Set<SequenceGroup> currentTreeGroups, HashMap<Integer, ArrayList<Integer>> boxGroupColouringDim)
  {

    if (node == null)
    {
      return;
    }

    // Check if it is a leaf node
    if (node.left() == null && node.right() == null)
    {
      // Get the group with the node
      SequenceGroup colourGroup = findTreeGroupForNode(node);

      if (colourGroup != null)
      {

        // Calculate dimensions for the node

        double height = node.height;

        int xend = (int) (height * wscale) + offx;

        int ypos = (int) (node.ycount * chunk) + offy;

        String name = (markPlaceholders && ((node instanceof SequenceNode
                && ((SequenceNode) node).isPlaceholder())))
                        ? (PLACEHOLDER + node.getDisplayName())
                        : node.getDisplayName();

        int charWidth = fm.stringWidth(name) + 3;
        int charHeight = font.getSize();

        int newBoxMaxY = ypos - charHeight / 2 + charHeight;
        int newBoxMaxX = xend + 10 + charWidth;
        int newBoxMinX = xend + 10;
        int newBoxMinY = ypos - charHeight / 2;

        // Initialise the dimension for the group with default values
        // if not already set
        if (boxGroupColouringDim.get(colourGroup.hashCode()) == null)
        {
          ArrayList<Integer> boxDims = new ArrayList<>();
          for (int i = 0; i < 4; i++)
          {
            boxDims.add(-1);
          }
          boxGroupColouringDim.put(colourGroup.hashCode(), boxDims);
        }

        // Get the dimensions for the group bounding box
        ArrayList<Integer> boxDims = boxGroupColouringDim
                .get(colourGroup.hashCode());

        // Assign the box minX value to the least of all minX values of
        // nodes in the group
        boxDims.set(TREE_GROUP_DIM_MIN_X_INDEX,
                boxDims.get(TREE_GROUP_DIM_MIN_X_INDEX) == -1 ? xend + 1
                        : Math.min(boxDims.get(TREE_GROUP_DIM_MIN_X_INDEX),
                                newBoxMinX));
        // Assign the box minY value to the least of all minY values of
        // nodes in the group
        boxDims.set(TREE_GROUP_DIM_MIN_Y_INDEX,
                boxDims.get(TREE_GROUP_DIM_MIN_Y_INDEX) == -1 ? newBoxMinY
                        : Math.min(boxDims.get(TREE_GROUP_DIM_MIN_Y_INDEX),
                                newBoxMinY));
        // Assign the box maxX value to the max of all maxX values of
        // nodes in the group
        boxDims.set(TREE_GROUP_DIM_MAX_X_INDEX,
                boxDims.get(TREE_GROUP_DIM_MAX_X_INDEX) == -1 ? newBoxMaxX
                        : Math.max(boxDims.get(TREE_GROUP_DIM_MAX_X_INDEX),
                                newBoxMaxX));
        // Assign the box maxY value to the max of all maxY values of
        // nodes in the group
        boxDims.set(TREE_GROUP_DIM_MAX_Y_INDEX,
                boxDims.get(TREE_GROUP_DIM_MAX_Y_INDEX) == -1 ? newBoxMaxY
                        : Math.max(boxDims.get(TREE_GROUP_DIM_MAX_Y_INDEX),
                                newBoxMaxY));

        currentTreeGroups.add(colourGroup);
      }
    }

    else
    {
      // Recurse on both child nodes
      updateGroupBoxDimensions(g, (BinaryNode) node.left(), chunk, wscale,
              offx, offy, currentTreeGroups, boxGroupColouringDim);
      updateGroupBoxDimensions(g, (BinaryNode) node.right(), chunk, wscale,
              offx, offy, currentTreeGroups, boxGroupColouringDim);
    }
  }
  
  
  /**
   * Returns the tree group to be coloured for the node (in annotation based tree)
   * @param node current node
   */
  private SequenceGroup findTreeGroupForNode(BinaryNode node) 
  {

    if (!(node.element() instanceof SequenceI))
    {
      return null;
    }

    SequenceI seq = (SequenceI) node.element();

    // Get all groups of the sequence
    SequenceGroup[] groupsWithSequence = (ap.getAlignment() != null)
            ? ap.getAlignment().findAllGroups(seq)
            : null;

    // Identify the group
    if (groupsWithSequence != null && node.hasLabel())
    {
      // For nodes with annotation
      if (!av.getSequenceColour(seq).equals(Color.white)
              && node.hasAlignmentAnnotation())
      {
        for (SequenceGroup group : groupsWithSequence)
        {
          if (group.getAnnotationsFromTree()
                  .contains(node.getAlignmentAnnotation())
                  && group.getName().startsWith("JTreeGroup:"))
          {
            return group;
          }
        }
      }
      // For nodes without annotation
      else
      {
        for (SequenceGroup group : groupsWithSequence)
        {
          if (group.getName().startsWith("JTreeGroup:"))
          {
            return group;
          }
        }
      }
    }
    return null;
  }
  
  
  private void drawLinesAndLabelsForSecondaryStructureProvider(Graphics g, BinaryNode node, 
          int xstart, int xend, int ypos, String nodeLabel) {
    
    Graphics2D g2d = (Graphics2D) g.create();
    
    g2d.setStroke(new BasicStroke(
            4.0f,                 
            BasicStroke.CAP_BUTT, 
            BasicStroke.JOIN_ROUND 
    ));

    String label = node.getLabel();
    //String[] lines = label.split("\\|");
    String[] lines = Arrays.stream(label.split("\\|"))
            .map(String::trim)  // Remove spaces at the start and end
            .toArray(String[]::new);
    Arrays.sort(lines);
    
    int mid = lines.length / 2;
    
    // Reverse the order of elements from position 0 to mid - 1
    for (int i = 0; i < mid / 2; i++) {
        String temp = lines[i];
        lines[i] = lines[mid - 1 - i];
        lines[mid - 1 - i] = temp;
    }
    
    // Draw lines for the first half
    int firstHalfLinesCount = mid;

    drawSecondaryStructureProviderLinesSection(g2d, 
            lines, 0, mid, xstart, xend, 
            ypos - DASHED_LINE_Y_OFFSET,
            true);
    drawVerticalLineAndLabel(g, xstart, ypos, firstHalfLinesCount, true, nodeLabel); 

    // Draw lines for the second half
    int secondHalfLinesCount = lines.length - mid;
    drawSecondaryStructureProviderLinesSection(g2d, 
            lines, mid, lines.length, xstart, xend, 
            ypos,
            false);
    drawVerticalLineAndLabel(g, xstart, ypos, secondHalfLinesCount, false, nodeLabel); 

    g2d.dispose(); 
}

  private void drawSecondaryStructureProviderLinesSection(Graphics2D g2d, String[] lines, int start, int end, int xstart, int xend, int baseY, boolean above) {
            
    for (int i = start, j=0; i < end; i++, j++) {
          int adjustedY = above 
                  ? baseY - ((i - start) * DASHED_LINE_Y_OFFSET) 
                          : baseY +((i - start) * DASHED_LINE_Y_OFFSET);
          //Color providerColor = AlignmentUtils.getSecondaryStructureProviderColor(lines[i]);
          Boolean greyOutStateObj = structureProviderColouredLineToggleState.get(
                  lines[i].toUpperCase().trim()
              );

          boolean colourState = greyOutStateObj == null || !greyOutStateObj; // Coloured by default
          
          if(colourState) {
            Map<String, Color> secondaryStructureProviderColorMap = tp.getSecondaryStructureProviderColorMap();
            Color providerColor = secondaryStructureProviderColorMap.getOrDefault(lines[i].toUpperCase().trim(), Color.BLACK);
            g2d.setColor(providerColor);
          }
          else{
            g2d.setColor(new Color(Color.LIGHT_GRAY.getRed(), 
                    Color.LIGHT_GRAY.getGreen(), Color.LIGHT_GRAY.getBlue(), COLOR_TRANSPARENCY_FOR_COLOURED_LINES));
          }
          g2d.drawLine(xstart, adjustedY, xend, adjustedY);
      }
  }
  
  private void drawVerticalLineAndLabel(Graphics g, int xstart, int ypos, 
          int linesCount, boolean above, String nodeLabel) {    
    
    int adjustment = (linesCount * DASHED_LINE_Y_OFFSET) + (DASHED_LINE_Y_OFFSET / 3);
    int adjustedY = ypos + (above ? -adjustment : adjustment - DASHED_LINE_Y_OFFSET);

    // draw vertical line
    g.drawLine(xstart, ypos, xstart, adjustedY);

    // draw label
    if(above && !nodeLabel.equals(""))
      g.drawString(nodeLabel, xstart + 2, adjustedY - 2);
}


  /**
   * DOCUMENT ME!
   * 
   * @param x
   *          DOCUMENT ME!
   * @param y
   *          DOCUMENT ME!
   * 
   * @return DOCUMENT ME!
   */
  public Object findElement(int x, int y, boolean node)
  {
      for (Entry<BinaryNode, Rectangle> entry : nameHash.entrySet())
      {
        Rectangle rect = entry.getValue();
  
        if ((x >= rect.x) && (x <= (rect.x + rect.width)) && (y >= rect.y)
                && (y <= (rect.y + rect.height)))
        {
          if(!node) {
            return entry.getKey().element();
          }
          else {
            return entry.getKey();
          }
        }
      }

    for (Entry<BinaryNode, Rectangle> entry : nodeHash.entrySet())
    {
      Rectangle rect = entry.getValue();

      if ((x >= rect.x) && (x <= (rect.x + rect.width)) && (y >= rect.y)
              && (y <= (rect.y + rect.height)))
      {
        return entry.getKey();
      }
    }

    return null;
  }

  /**
   * DOCUMENT ME!
   * 
   * @param pickBox
   *          DOCUMENT ME!
   */
  public void pickNodes(Rectangle pickBox)
  {
    int width = getWidth();
    int height = getHeight();

    BinaryNode top = tree.getTopNode();

    double wscale = ((width * .8) - (offx * 2)) / tree.getMaxHeight();

    if (top.count == 0)
    {
      top.count = ((BinaryNode) top.left()).count
              + ((BinaryNode) top.right()).count;
    }

    float chunk = (float) (height - (offy)) / top.count;

    pickNode(pickBox, top, chunk, wscale, width, offx, offy);
  }

  /**
   * DOCUMENT ME!
   * 
   * @param pickBox
   *          DOCUMENT ME!
   * @param node
   *          DOCUMENT ME!
   * @param chunk
   *          DOCUMENT ME!
   * @param wscale
   *          DOCUMENT ME!
   * @param width
   *          DOCUMENT ME!
   * @param offx
   *          DOCUMENT ME!
   * @param offy
   *          DOCUMENT ME!
   */
  public void pickNode(Rectangle pickBox, BinaryNode node, float chunk,
          double wscale, int width, int offx, int offy)
  {
    if (node == null)
    {
      return;
    }

    if ((node.left() == null) && (node.right() == null))
    {
      double height = node.height;
      // double dist = node.dist;
      // int xstart = (int) ((height - dist) * wscale) + offx;
      int xend = (int) (height * wscale) + offx;

      int ypos = (int) (node.ycount * chunk) + offy;

      if (pickBox.contains(new Point(xend, ypos)))
      {
        if (node.element() instanceof SequenceI)
        {
          SequenceI seq = (SequenceI) node.element();
          SequenceGroup sg = av.getSelectionGroup();

          if (sg != null)
          {
            sg.addOrRemove(seq, true);
          }
        }
      }
    }
    else
    {
      pickNode(pickBox, (BinaryNode) node.left(), chunk, wscale, width,
              offx, offy);
      pickNode(pickBox, (BinaryNode) node.right(), chunk, wscale, width,
              offx, offy);
    }
  }

  /**
   * DOCUMENT ME!
   * 
   * @param node
   *          DOCUMENT ME!
   * @param c
   *          DOCUMENT ME!
   */
  public void setColor(BinaryNode node, Color c)
  {
    if (node == null)
    {
      return;
    }

    node.color = c;
    if (node.element() instanceof SequenceI)
    {
      final SequenceI seq = (SequenceI) node.element();
      AlignmentAnnotation annotation = node.getAlignmentAnnotation();
      AlignmentPanel[] aps = getAssociatedPanels();
      if (aps != null)
      {
        for (int a = 0; a < aps.length; a++)
        {
          aps[a].av.setSequenceColour(seq, c);
          
          //Assign color to annotation          
          if(annotation!=null) {
            aps[a].av.setAnnotationColour(annotation, c);
          }
        }
      }
    }
    setColor((BinaryNode) node.left(), c);
    setColor((BinaryNode) node.right(), c);
  }

  /**
   * DOCUMENT ME!
   */
  void startPrinting()
  {
    Thread thread = new Thread(this);
    thread.start();
  }

  // put printing in a thread to avoid painting problems
  @Override
  public void run()
  {
    PrinterJob printJob = PrinterJob.getPrinterJob();
    PageFormat defaultPage = printJob.defaultPage();
    PageFormat pf = printJob.pageDialog(defaultPage);

    if (defaultPage == pf)
    {
      /*
       * user cancelled
       */
      return;
    }

    printJob.setPrintable(this, pf);

    if (printJob.printDialog())
    {
      try
      {
        printJob.print();
      } catch (Exception PrintException)
      {
        PrintException.printStackTrace();
      }
    }
  }

  /**
   * DOCUMENT ME!
   * 
   * @param pg
   *          DOCUMENT ME!
   * @param pf
   *          DOCUMENT ME!
   * @param pi
   *          DOCUMENT ME!
   * 
   * @return DOCUMENT ME!
   * 
   * @throws PrinterException
   *           DOCUMENT ME!
   */
  @Override
  public int print(Graphics pg, PageFormat pf, int pi)
          throws PrinterException
  {
    pg.setFont(font);
    pg.translate((int) pf.getImageableX(), (int) pf.getImageableY());

    int pwidth = (int) pf.getImageableWidth();
    int pheight = (int) pf.getImageableHeight();

    int noPages = getHeight() / pheight;

    if (pi > noPages)
    {
      return Printable.NO_SUCH_PAGE;
    }

    if (pwidth > getWidth())
    {
      pwidth = getWidth();
    }

    if (fitToWindow)
    {
      if (pheight > getHeight())
      {
        pheight = getHeight();
      }

      noPages = 0;
    }
    else
    {
      FontMetrics fm = pg.getFontMetrics(font);
      int height = fm.getHeight() * nameHash.size();
      pg.translate(0, -pi * pheight);
      pg.setClip(0, pi * pheight, pwidth, (pi * pheight) + pheight);

      // translate number of pages,
      // height is screen size as this is the
      // non overlapping text size
      pheight = height;
    }

    draw(pg, pwidth, pheight);

    return Printable.PAGE_EXISTS;
  }

  /**
   * DOCUMENT ME!
   * 
   * @param g
   *          DOCUMENT ME!
   */
  @Override
  public void paintComponent(Graphics g)
  {
    super.paintComponent(g);
    g.setFont(font);

    if (tree == null)
    {
      g.drawString(
              MessageManager.getString("label.calculating_tree") + "....",
              20, getHeight() / 2);
    }
    else
    {
      fm = g.getFontMetrics(font);

      int nameCount = nameHash.size();
      if (nameCount == 0)
      {
        repaint();
      }

      if (fitToWindow || (!fitToWindow && (scrollPane
              .getHeight() > ((fm.getHeight() * nameCount) + offy))))
      {
        draw(g, scrollPane.getWidth(), scrollPane.getHeight());
        setPreferredSize(null);
      }
      else
      {
        setPreferredSize(new Dimension(scrollPane.getWidth(),
                fm.getHeight() * nameCount));
        draw(g, scrollPane.getWidth(), fm.getHeight() * nameCount);
      }

      scrollPane.revalidate();
    }
  }

  /**
   * DOCUMENT ME!
   * 
   * @param fontSize
   *          DOCUMENT ME!
   */
  @Override
  public void setFont(Font font)
  {
    this.font = font;
    repaint();
  }

  /**
   * DOCUMENT ME!
   * 
   * @param g1
   *          DOCUMENT ME!
   * @param width
   *          DOCUMENT ME!
   * @param height
   *          DOCUMENT ME!
   */
  public void draw(Graphics g1, int width, int height)
  {
    Graphics2D g2 = (Graphics2D) g1;
    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
            RenderingHints.VALUE_ANTIALIAS_ON);
    g2.setColor(Color.white);
    g2.fillRect(0, 0, width, height);
    g2.setFont(font);

    if (longestName == null || tree == null)
    {
      g2.drawString("Calculating tree.", 20, 20);
      return;
    }
    offy = font.getSize() + 10;

    fm = g2.getFontMetrics(font);

    labelLength = fm.stringWidth(longestName) + 20; // 20 allows for scrollbar

    double wscale = (width - labelLength - (offx * 2))
            / tree.getMaxHeight();

    BinaryNode top = tree.getTopNode();

    if (top.count == 0)
    {
      top.count = top.left().count + top.right().count;
    }

    float chunk = (float) (height - (offy)) / top.count;
    
    // Group colouring as coloured rectangular background
    // only for coloured line visualisation
    if (showStructureProviderColouredLines)
    {
      drawGroupColourBoxes(g2, tree.getTopNode(), chunk, wscale, offx, offy);
    }
    
    drawNode(g2, tree.getTopNode(), chunk, wscale, width, offx, offy);


    if (threshold != 0)
    {
      if (av.getCurrentTree() == tree)
      {
        g2.setColor(Color.red);
      }
      else
      {
        g2.setColor(Color.gray);
      }

      int x = (int) ((threshold * (getWidth() - labelLength - (2 * offx)))
              + offx);

      g2.drawLine(x, 0, x, getHeight());
    }
  }
  
  /**
   * Draws coloured background boxes for sequence groups in the tree.
   *
   * @param g2    Graphics object
   * @param node  Binary node
   * @param chunk Chunk 
   * @param wscale Width scale
   * @param offx  X offset
   * @param offy  Y offset
   */
  private void drawGroupColourBoxes(Graphics g2, BinaryNode node, double chunk, double wscale, int offx, int offy)
  {
    // Reset the object storing current groups and colouring dimensions
    Set<SequenceGroup> currentTreeGroups = new HashSet<SequenceGroup>();

    // Map that stores group hash code as key and the box dimensions for
    // background group colouring
    HashMap<Integer, ArrayList<Integer>> boxGroupColouringDim = new HashMap<Integer, ArrayList<Integer>>();

    // Gives the dimensions of the rectangular background group colouring
    updateGroupBoxDimensions(g2, tree.getTopNode(), chunk, wscale, offx,
            offy, currentTreeGroups, boxGroupColouringDim);

    // Store current colour temporarily to restore after group colouring
    Color prevColour = g2.getColor();

    // Iterate through each existing groups
    for (SequenceGroup treeGroup : currentTreeGroups)
    {

      // Get box dimensions for colouring
      ArrayList<Integer> boxDims = boxGroupColouringDim
              .get(treeGroup.hashCode());

      // Apply group colour if exists
      if (treeGroup.idColour != null)
      {

        // Set the brightness and transparency of the colour
        Color treeGroupColor = treeGroup.idColour.brighter();
        g2.setColor(new Color(treeGroupColor.getRed(),
                treeGroupColor.getGreen(), treeGroupColor.getBlue(),
                COLOR_TRANSPARENCY_FOR_GROUP));

        int x = boxDims.get(TREE_GROUP_DIM_MIN_X_INDEX);
        int y = boxDims.get(TREE_GROUP_DIM_MIN_Y_INDEX) - 2;
        int width = boxDims.get(TREE_GROUP_DIM_MAX_X_INDEX)
                - boxDims.get(TREE_GROUP_DIM_MIN_X_INDEX);
        int height = boxDims.get(TREE_GROUP_DIM_MAX_Y_INDEX)
                - boxDims.get(TREE_GROUP_DIM_MIN_Y_INDEX) + 4;

        // Apply group colour as a coloured rectangular background for the group
        g2.fillRect(x, y, width, height);

      }
    }
    // Restore the previous colour
    g2.setColor(prevColour);
  }
  

  /**
   * Empty method to satisfy the MouseListener interface
   * 
   * @param e
   */
  @Override
  public void mouseReleased(MouseEvent e)
  {
    /*
     * isPopupTrigger is set on mouseReleased on Windows
     */
    if (e.isPopupTrigger())
    {
      chooseSubtreeColour();
      e.consume(); // prevent mouseClicked happening
    }
  }

  /**
   * Empty method to satisfy the MouseListener interface
   * 
   * @param e
   */
  @Override
  public void mouseEntered(MouseEvent e)
  {
  }

  /**
   * Empty method to satisfy the MouseListener interface
   * 
   * @param e
   */
  @Override
  public void mouseExited(MouseEvent e)
  {
  }

  /**
   * Handles a mouse click on a tree node (clicks elsewhere are handled in
   * mousePressed). Click selects the sub-tree, double-click swaps leaf nodes
   * order, right-click opens a dialogue to choose colour for the sub-tree.
   * 
   * @param e
   */
  @Override
  public void mouseClicked(MouseEvent evt)
  {
    if (highlightNode == null)
    {
      return;
    }

    if (evt.getClickCount() > 1)
    {
      tree.swapNodes(highlightNode);
      tree.reCount(tree.getTopNode());
      tree.findHeight(tree.getTopNode());
    }
    else
    {
      Vector<BinaryNode> leaves = tree.findLeaves(highlightNode);
      if (tp.isColumnWise())
      {
        markColumnsFor(getAssociatedPanels(), leaves, Color.red, false);
      }

      else
      {
        for (int i = 0; i < leaves.size(); i++)
        {
          SequenceI seq = (SequenceI) leaves.elementAt(i).element();          
          if(leaves.get(i).hasAlignmentAnnotation()) 
          {
            treeSelectionChanged(seq, leaves.get(i).getAlignmentAnnotation());
            
          }
          else {
            treeSelectionChanged(seq, null);
          }
          
        }
      }
      av.sendSelection();
    }

    PaintRefresher.Refresh(tp, av.getSequenceSetId());
    repaint();
  }

  /**
   * Offer the user the option to choose a colour for the highlighted node and
   * its children; this colour is also applied to the corresponding sequence ids
   * in the alignment
   */
  void chooseSubtreeColour()
  {
    String ttl = MessageManager.getString("label.select_subtree_colour");
    ColourChooserListener listener = new ColourChooserListener()
    {
      @Override
      public void colourSelected(Color c)
      {
        setColor(highlightNode, c);
        PaintRefresher.Refresh(tp, ap.av.getSequenceSetId());
        repaint();
      }
    };
    JalviewColourChooser.showColourChooser(this, ttl, highlightNode.color,
            listener);
  }

  @Override
  public void mouseMoved(MouseEvent evt)
  {
    av.setCurrentTree(tree);

    Object ob = findElement(evt.getX(), evt.getY(), false);
    
    // Get mouse coordinates
    int mouseX = evt.getX();
    int mouseY = evt.getY();

    if (ob instanceof BinaryNode)
    {
      highlightNode = (BinaryNode) ob;
      this.setToolTipText(
              "<html>" + MessageManager.getString("label.highlightnode"));
      repaint();

    }
    else
    {     
      
      if (highlightNode != null)
      {
        highlightNode = null;
        setToolTipText(null);
        repaint();
      }
      
      // Iterate through the map of label bounding boxes
//      for (Map.Entry<BinaryNode, List<Rectangle>> entry : labelBoundsMap.entrySet()) {
//          BinaryNode node = entry.getKey();
//          List<Rectangle> boundsList = entry.getValue();
//
//          // Check each bounding box for this node
//          for (Rectangle labelBounds : boundsList) {
//              if (labelBounds.contains(mouseX, mouseY)) {
//                  // Show tooltip for this node's label
//                  String nodeLabel = node.getDisplayName();
//                  this.setToolTipText(nodeLabel);
//                  repaint();
//                  return; // Exit once we find a matching label
//              }
//          }
//      }
      // Clear tooltip if no label is hovered
      setToolTipText(null);
      repaint();

    }
  }

  @Override
  public void mouseDragged(MouseEvent ect)
  {
  }

  /**
   * Handles a mouse press on a sequence name or the tree background canvas
   * (click on a node is handled in mouseClicked). The action is to create
   * groups by partitioning the tree at the mouse position. Colours for the
   * groups (and sequence names) are generated randomly.
   * 
   * @param e
   */
  @Override
  public void mousePressed(MouseEvent e)
  {
    av.setCurrentTree(tree);

    /*
     * isPopupTrigger is set for mousePressed (Mac)
     * or mouseReleased (Windows)
     */
    if (e.isPopupTrigger())
    {
      if (highlightNode != null)
      {
        chooseSubtreeColour();
      }
      return;
    }

    /*
     * defer right-click handling on Windows to
     * mouseClicked; note isRightMouseButton
     * also matches Cmd-click on Mac which should do
     * nothing here
     */
    if (SwingUtilities.isRightMouseButton(e))
    {
      return;
    }

    int x = e.getX();
    int y = e.getY();

    Object ob = findElement(x, y, true);

    if (ob instanceof BinaryNode && (((BinaryNode) ob).element() != null))
    {
      if(((BinaryNode) ob).hasAlignmentAnnotation())
      {
        treeSelectionChanged((SequenceI) ((BinaryNode) ob).element(), 
                ((BinaryNode) ob).getAlignmentAnnotation());
      }
      else
      {
        treeSelectionChanged((SequenceI) ((BinaryNode) ob).element(), null);

      }
      PaintRefresher.Refresh(tp,
              getAssociatedPanel().av.getSequenceSetId());
      repaint();
      av.sendSelection();
      return;
    }
    else if (!(ob instanceof BinaryNode))
    {
      // Find threshold
      if (tree.getMaxHeight() != 0)
      {
        threshold = (float) (x - offx)
                / (float) (getWidth() - labelLength - (2 * offx));

        List<BinaryNode> groups = tree.groupNodes(threshold);
        setColor(tree.getTopNode(), Color.black);

        AlignmentPanel[] aps = getAssociatedPanels();

        // TODO push calls below into a single AlignViewportI method?
        // see also AlignViewController.deleteGroups
        for (int a = 0; a < aps.length; a++)
        {
          aps[a].av.setSelectionGroup(null);
          aps[a].av.getAlignment().deleteAllGroups();
          aps[a].av.clearSequenceColours();
          aps[a].av.clearAnnotationColours();
          if (aps[a].av.getCodingComplement() != null)
          {
            aps[a].av.getCodingComplement().setSelectionGroup(null);
            aps[a].av.getCodingComplement().getAlignment()
                    .deleteAllGroups();
            aps[a].av.getCodingComplement().clearSequenceColours();
          }
          aps[a].av.setUpdateStructures(true);
        }
        colourGroups(groups);

        /*
         * clear partition (don't show vertical line) if
         * it is to the right of all nodes
         */
        if (groups.isEmpty())
        {
          threshold = 0f;
        }
      }
      Console.log.debug("Tree cut threshold set at:" + threshold);
      PaintRefresher.Refresh(tp,
              getAssociatedPanel().av.getSequenceSetId());
      repaint();
    }

  }

  void colourGroups(List<BinaryNode> groups)
  {
    AlignmentPanel[] aps = getAssociatedPanels();
    List<BitSet> colGroups = new ArrayList<>();
    Map<BitSet, Color> colors = new HashMap();
    for (int i = 0; i < groups.size(); i++)
    {
      Color col = ColorUtils.getColorForIndex(i);
      setColor(groups.get(i), col.brighter());

      Vector<BinaryNode> l = tree.findLeaves(groups.get(i));
      //gatherLabelsTo(groups.get(i), l);
      if (!tp.isColumnWise())
      {
        createSeqGroupFor(aps, l, col, groups.get(i).getLabel());
      }
      else
      {
        BitSet gp = createColumnGroupFor(l, col);

        colGroups.add(gp);
        colors.put(gp, col);
      }
    }
    if (tp.isColumnWise())
    {
      AlignmentAnnotation aa = tp.getAssocAnnotation();
      if (aa != null)
      {
        ContactMatrixI cm = av.getContactMatrix(aa);
        if (cm != null)
        {
          cm.updateGroups(colGroups);
          for (BitSet gp : colors.keySet())
          {
            cm.setColorForGroup(gp, colors.get(gp));
          }
        }
        cm.transferGroupColorsTo(aa);
      }
    }

    // notify the panel(s) to redo any group specific stuff
    // also updates structure views if necessary
    for (int a = 0; a < aps.length; a++)
    {
      aps[a].updateAnnotation();
      final AlignViewportI codingComplement = aps[a].av
              .getCodingComplement();
      if (codingComplement != null)
      {
        ((AlignViewport) codingComplement).getAlignPanel()
                .updateAnnotation();
      }
    }
  }

  private void gatherLabelsTo(BinaryNode binaryNode, Vector<BinaryNode> l)
  {
    LinkedHashSet<String> labelsForNode = new LinkedHashSet<String>();
    for (BinaryNode leaf : l)
    {
      if (leaf.hasLabel())
      {
        labelsForNode.add(leaf.getLabel());
      }
    }
    StringBuilder sb = new StringBuilder();
    boolean first = true;
    for (String label : labelsForNode)
    {
      if (!first)
      {
        sb.append(" | ");
      }
      first = false;
//      if(labelsForNode.size()>1) {
//        String providerAbbreviation = AlignmentUtils.getProviderKey(label);
//        sb.append(providerAbbreviation);
//      }     
      sb.append(label);
      
    }
    binaryNode.setLabel(sb.toString());
  }

  private int parseColumnNode(BinaryNode bn) throws NumberFormatException
  {
    return Integer.parseInt(
            bn.getName().substring(bn.getName().indexOf("c") + 1));
  }

  private boolean isColumnForNodeSelected(BinaryNode bn)
  {
    SequenceI rseq = tp.assocAnnotation.sequenceRef;
    int colm = -1;
    try
    {
      colm = parseColumnNode(bn);
    } catch (Exception e)
    {
      return false;
    }
    if (av == null || av.getAlignment() == null)
    {
      // alignment is closed
      return false;
    }
    ColumnSelection cs = av.getColumnSelection();
    HiddenColumns hc = av.getAlignment().getHiddenColumns();
    AlignmentAnnotation aa = tp.getAssocAnnotation();
    int offp = -1;
    if (aa != null)
    {
      ContactMatrixI cm = av.getContactMatrix(aa);
      // generally, we assume cm has 1:1 mapping to annotation row - probably
      // wrong
      // but.. if
      if (cm instanceof MappableContactMatrixI)
      {
        int[] pos;
        // use the mappable's mapping - always the case for PAE Matrices so good
        // for 2.11.3
        MappableContactMatrixI mcm = (MappableContactMatrixI) cm;
        pos = mcm.getMappedPositionsFor(rseq, colm + 1);
        // finally, look up the position of the column
        if (pos != null)
        {
          offp = rseq.findIndex(pos[0]);
        }
      }
      else
      {
        offp = colm;
      }
    }
    if (offp <= 0)
    {
      return false;
    }

    offp -= 2;
    if (!av.hasHiddenColumns())
    {
      return cs.contains(offp);
    }
    if (hc.isVisible(offp))
    {
      return cs.contains(offp);
      // return cs.contains(hc.absoluteToVisibleColumn(offp));
    }
    return false;
  }

  private BitSet createColumnGroupFor(Vector<BinaryNode> l, Color col)
  {
    BitSet gp = new BitSet();
    for (BinaryNode bn : l)
    {
      int colm = -1;
      if (bn.element() != null && bn.element() instanceof Integer)
      {
        colm = (Integer) bn.element();
      }
      else
      {
        // parse out from nodename
        try
        {
          colm = parseColumnNode(bn);
        } catch (Exception e)
        {
          continue;
        }
      }
      gp.set(colm);
    }
    return gp;
  }

  private void markColumnsFor(AlignmentPanel[] aps, Vector<BinaryNode> l,
          Color col, boolean clearSelected)
  {
    SequenceI rseq = tp.assocAnnotation.sequenceRef;
    if (av == null || av.getAlignment() == null)
    {
      // alignment is closed
      return;
    }

    // TODO - sort indices for faster lookup
    ColumnSelection cs = av.getColumnSelection();
    HiddenColumns hc = av.getAlignment().getHiddenColumns();
    ContactMatrixI cm = av.getContactMatrix(tp.assocAnnotation);
    MappableContactMatrixI mcm = null;
    int offp;
    if (cm instanceof MappableContactMatrixI)
    {
      mcm = (MappableContactMatrixI) cm;
    }
    for (BinaryNode bn : l)
    {
      int colm = -1;
      try
      {
        colm = Integer.parseInt(
                bn.getName().substring(bn.getName().indexOf("c") + 1));
      } catch (Exception e)
      {
        continue;
      }
      if (mcm != null)
      {
        int[] seqpos = mcm.getMappedPositionsFor(rseq, colm);
        if (seqpos == null)
        {
          // no mapping for this column.
          continue;
        }
        // TODO: handle ranges...
        offp = rseq.findIndex(seqpos[0]) - 1;
      }
      else
      {
        offp = (rseq != null) ? rseq.findIndex(rseq.getStart() + colm)
                : colm;
      }
      if (!av.hasHiddenColumns() || hc.isVisible(offp))
      {
        if (clearSelected || cs.contains(offp))
        {
          cs.removeElement(offp);
        }
        else
        {
          cs.addElement(offp);
        }
      }
    }
    PaintRefresher.Refresh(tp, av.getSequenceSetId());
  }

  public void createSeqGroupFor(AlignmentPanel[] aps, Vector<BinaryNode> l,
          Color col, String label)
  {

    Vector<SequenceI> sequences = new Vector<>();
    List<AlignmentAnnotation> annotationInGroup = new ArrayList<AlignmentAnnotation>();

    for (int j = 0; j < l.size(); j++)
    {
      SequenceI s1 = (SequenceI) l.elementAt(j).element();

      if (!sequences.contains(s1))
      {
        sequences.addElement(s1);
      }
      
      if(l.elementAt(j).getAlignmentAnnotation()!=null 
              && !annotationInGroup.contains(l.elementAt(j).getAlignmentAnnotation())) {
        annotationInGroup.add(l.elementAt(j).getAlignmentAnnotation());
        
      }
      
    }

    ColourSchemeI cs = null;
    SequenceGroup _sg = new SequenceGroup(sequences, null, cs, true, true,
            false, 0, av.getAlignment().getWidth() - 1);
    

    for(AlignmentAnnotation annot:annotationInGroup) {
      
      _sg.addAnnotationFromTree(annot);
    }
    
    
    // Check if the label is not null and not empty
    if(label != null && !label.isEmpty()) {
      // Retrieve the existing groups from the alignment
      List<SequenceGroup> existingGroups = av.getAlignment().getGroups();
      
      // Reduce the label length
      label = AlignmentUtils.reduceLabelLength(label);
      
      // Create group name based on the label
      String newGroupName = "JTreeGroup:" + label;
      
      // Counter for groups with the same name
      int noOfGroupsWithSameName = 0;
      
      // Iterate through the existing groups to check for naming conflicts
      for (SequenceGroup sg : existingGroups) {
        if (sg.getName().equals(newGroupName) || sg.getName().matches(newGroupName + " \\d+")) {

          noOfGroupsWithSameName++;
          
          // If a group name matches exactly, update the group's name by appending the count
          if(sg.getName().equals(newGroupName) ) {
            String updatedGroupName = sg.getName() + " " + noOfGroupsWithSameName;
            sg.setName(updatedGroupName);
          }
        }
      }

      
      // If count > 0, increment the count and append it to the new group name
      if(noOfGroupsWithSameName>0) {
        noOfGroupsWithSameName++;
        newGroupName = newGroupName + " " + noOfGroupsWithSameName;
      }
      
      _sg.setName(newGroupName);
    }
    else {
      _sg.setName("JTreeGroup:" + _sg.hashCode());
    }
    _sg.setIdColour(col);

    for (int a = 0; a < aps.length; a++)
    {
      SequenceGroup sg = new SequenceGroup(_sg);
      AlignViewport viewport = aps[a].av;

      // Propagate group colours in each view
      if (viewport.getGlobalColourScheme() != null)
      {
        cs = viewport.getGlobalColourScheme().getInstance(viewport, sg);
        sg.setColourScheme(cs);
        sg.getGroupColourScheme().setThreshold(
                viewport.getResidueShading().getThreshold(),
                viewport.isIgnoreGapsConsensus());

        if (viewport.getResidueShading().conservationApplied())
        {
          Conservation c = new Conservation("Group", sg.getSequences(null),
                  sg.getStartRes(), sg.getEndRes());
          c.calculate();
          c.verdict(false, viewport.getConsPercGaps());
          sg.cs.setConservation(c);
        }
        if (viewport.getResidueShading()
                .isConsensusSecondaryStructureColouring())
        {
          sg.getGroupColourScheme().setConsensusSecondaryStructureThreshold(
                  viewport.getResidueShading().getThreshold());
          sg.getGroupColourScheme()
                  .setConsensusSecondaryStructureColouring(true);
        }
      }
      // indicate that associated structure views will need an update
      viewport.setUpdateStructures(true);
      // propagate structure view update and sequence group to complement view
      viewport.addSequenceGroup(sg);
    }
  }

  /**
   * DOCUMENT ME!
   * 
   * @param state
   *          DOCUMENT ME!
   */
  public void setShowDistances(boolean state)
  {
    this.showDistances = state;
    repaint();
  }
  
  public void hideStructureProviders(boolean state)
  {
    if(state) {
      this.showStructureProviderColouredLines = false;
      this.showStructureProviderLabels = false;
      repaint();
    }
  }
  
  public void setShowStructureProviderColouredLines(boolean state)
  {
    this.showStructureProviderColouredLines = state;
    if(state) {
      this.showStructureProviderLabels = false;
    }
    repaint();
  }
  
  public void setShowStructureProviderLabels(boolean state)
  {
    this.showStructureProviderLabels = state;
    if(state) {
      this.showStructureProviderColouredLines = false;
    }
    repaint();
  }
  
  
  public void toggleStructureProviderColouredLine(String provider, boolean state)
  {
    this.structureProviderColouredLineToggleState.put(provider.toUpperCase().trim(), state);
    repaint();
  }
 
  

  /**
   * DOCUMENT ME!
   * 
   * @param state
   *          DOCUMENT ME!
   */
  public void setShowBootstrap(boolean state)
  {
    this.showBootstrap = state;
    repaint();
  }

  /**
   * DOCUMENT ME!
   * 
   * @param state
   *          DOCUMENT ME!
   */
  public void setMarkPlaceholders(boolean state)
  {
    this.markPlaceholders = state;
    repaint();
  }

  AlignmentPanel[] getAssociatedPanels()
  {
    if (applyToAllViews)
    {
      return PaintRefresher.getAssociatedPanels(av.getSequenceSetId());
    }
    else
    {
      return new AlignmentPanel[] { getAssociatedPanel() };
    }
  }

  public AlignmentPanel getAssociatedPanel()
  {
    return ap;
  }

  public void setAssociatedPanel(AlignmentPanel ap)
  {
    this.ap = ap;
  }

  public AlignViewport getViewport()
  {
    return av;
  }

  public void setViewport(AlignViewport av)
  {
    this.av = av;
  }

  public float getThreshold()
  {
    return threshold;
  }

  public void setThreshold(float threshold)
  {
    this.threshold = threshold;
  }

  public boolean isApplyToAllViews()
  {
    return this.applyToAllViews;
  }

  public void setApplyToAllViews(boolean applyToAllViews)
  {
    this.applyToAllViews = applyToAllViews;
  }
}
