#!/usr/bin/env bash

# see https://api.adoptopenjdk.net/swagger-ui/#/Binary/get_v3_binary_latest__feature_version___release_type___os___arch___image_type___jvm_impl___heap_size___vendor_


### bs 2020-01-22
### This is a script to download and update the JREs used in the windows, mac (and maybe linux) installations, and update channel
### It creates a structure with
### ./jre-VERSION-OS-ARCH/jre/...
### as used by getdown
### and
### ./tgz/jre-VERSION-OS-ARCH.tgz
### which is an archive of the _contents_ of ./jre-VERSION-OS-ARCH/jre/ and used by install4j for the installer

BASE=https://api.adoptopenjdk.net/v3/binary/latest
RELEASE_TYPE=ga
JVM_IMPL=hotspot
HEAP_SIZE=normal
VENDOR=adoptopenjdk
IMAGE_TYPE=jre
TAR=tar
ZIP=zip
UNZIP=unzip

STRIP_MAC_APP_BUNDLING=false
CREATE_ARCHIVES="zip tgz"
# need zip with top-level jre dir for getdown updates. need tgz without top-level jre dir for install4j bundling

RM=/bin/rm

# unzip-strip from https://superuser.com/questions/518347/equivalent-to-tars-strip-components-1-in-unzip
unzip-strip() (
  local zip=$1
  local dest=${2:-.}
  local temp=$(mktemp -d) && $UNZIP -qq -d "$temp" "$zip" && mkdir -p "$dest" &&
  shopt -s dotglob && local f=("$temp"/*) &&
  if (( ${#f[@]} == 1 )) && [[ -d "${f[0]}" ]] ; then
    mv "$temp"/*/* "$dest"
  else
    mv "$temp"/* "$dest"
  fi && rmdir "$temp"/* "$temp"
)

for FEATURE_VERSION in 8 11; do
  for OS in linux mac windows; do
    for ARCH in x64 x32 arm; do
      #for ARCH in aarch64 arm x32 x64; do
      NAME="${IMAGE_TYPE}-${FEATURE_VERSION}-${OS}-${ARCH}"
      TARFILE="${NAME}.tgz"
      echo "* Downloading ${TARFILE}"
      URL="${BASE}/${FEATURE_VERSION}/${RELEASE_TYPE}/${OS}/${ARCH}/${IMAGE_TYPE}/${JVM_IMPL}/${HEAP_SIZE}/${VENDOR}"
      wget -q -O "${TARFILE}" "${URL}"
      if [ "$?" != 0 ]; then
        echo "- No ${IMAGE_TYPE}-${FEATURE_VERSION} download for ${OS}-${ARCH} '${URL}'"
        $RM -f "${TARFILE}"
        continue;
      fi
      echo "Unpacking ${TARFILE}"
      JREDIR="${NAME}/${IMAGE_TYPE}"
      [ x$NAME != x -a -e "${JREDIR}" ] && $RM -rf "${JREDIR}"
      mkdir -p "${JREDIR}"
      if [ x$OS = xwindows ]; then
        echo "using unzip"
        unzip-strip "${TARFILE}" "${JREDIR}"
        RET=$?
      else
        echo "using tar"
        if [ x$OS = xmac -a x$STRIP_MAC_APP_BUNDLING = xtrue ]; then
          $TAR --strip-components=3 -C "${JREDIR}" -zxf "${TARFILE}" "*/Contents/Home"
          RET=$?
        else
          $TAR --strip-components=1 -C "${JREDIR}" -zxf "${TARFILE}"
          RET=$?
        fi
      fi
      if [ "$RET" != 0 ]; then
        echo "Error unpacking ${TARFILE}"
        exit 1
      fi
      $RM "${TARFILE}"
      if [ \! -z "$CREATE_ARCHIVES" ]; then
        for CREATEARCHIVE in ${CREATE_ARCHIVES}; do
          ARCHIVEDIR=$CREATEARCHIVE
          case $CREATEARCHIVE in
            zip)
              EXT=${CREATEARCHIVE}
              echo "Creating ${NAME}.${EXT} for getdown updates"
              [ \! -d ${ARCHIVEDIR} ] && mkdir -p "${ARCHIVEDIR}"
              ABSARCHIVEDIR="${PWD}/$ARCHIVEDIR"
              ZIPFILE="${ABSARCHIVEDIR}/${NAME}.${CREATEARCHIVE}"
              [ -e "${ZIPFILE}" ] && $RM "${ZIPFILE}"
              cd ${NAME}
              $ZIP -X -r "${ZIPFILE}" "${IMAGE_TYPE}"
              cd -
              ;;
            tgz)
              EXT=tar.gz
              echo "Creating ${NAME}.${EXT} for install4j bundling"
              [ \! -d ${ARCHIVEDIR} ] && mkdir -p "${ARCHIVEDIR}"
              $TAR -C "${JREDIR}" -zcf "${ARCHIVEDIR}/${NAME}.${EXT}" .
              ;;
            *)
              echo "Archiving as '${CREATEARCHIVE}' file not supported"
              ;;
          esac
        done
      fi
    done
  done
done

