/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Samuel Gagnon <samuel.gagnon92@gmail.com>
 *
 *  Copyright:
 *     Samuel Gagnon, 2018
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include <gecode/support/config.hpp>

#ifdef GECODE_HAS_CBS

#include <gecode/int/distinct.hh>

namespace Gecode { namespace Int { namespace Distinct {

  /*
   * Program used to precompute both factors using GNU gmp and the c++ MPFR
   * interface. Compile with: g++ linear.cpp -lmpfr -lgmp
   *
   * #include <gmp.h>
   * #include <mpreal.h>
   * #include <iostream>
   *
   * int main() {
   *  std::cout << "const double mincFactors[] = {" << std::endl;
   *  for (unsigned int _r_i=1; _r_i<400; _r_i++) {
   *    mpfr::mpreal r_i(_r_i, 256);
   *    std::cout.precision(17);
   *    std::cout << std::fixed << mpfr::exp(1/r_i * mpfr::log(mpfr::fac_ui(_r_i)))
   *              << "," << std::endl;
   *  }
   *  std::cout << "};";
   *  return 0;
   * }
   */

  const double mincFactors[] = {
    1.00000000000000000,
    1.41421356237309503,
    1.81712059283213963,
    2.21336383940064326,
    2.60517108469735180,
    2.99379516552390906,
    3.38001515914129663,
    3.76435059950312890,
    4.14716627439691264,
    4.52872868811676503,
    4.90923877958439568,
    5.28885199410244722,
    5.66769117625176115,
    6.04585517141850113,
    6.42342474977976057,
    6.80046679826764207,
    7.17703735702477519,
    7.55318386410680540,
    7.92894684486515053,
    8.30436120373934332,
    8.67945722613286892,
    9.05426136539794004,
    9.42879686817524423,
    9.80308427650907665,
    10.17714183488018406,
    10.55098582305415047,
    10.92463083045900753,
    11.29808998404420402,
    11.67137513880885247,
    12.04449703813164141,
    12.41746544948987035,
    12.79028927998164187,
    13.16297667516615132,
    13.53553510404150478,
    13.90797143243727189,
    14.28029198667325177,
    14.65250260899859376,
    15.02460870605700941,
    15.39661529140837855,
    15.76852702296325553,
    16.14034823604591501,
    16.51208297268635648,
    16.88373500764762055,
    17.25530787161673181,
    17.62680487192353526,
    17.99822911109791413,
    18.36958350353131161,
    18.74087079047107193,
    19.11209355354424360,
    19.48325422698127684,
    19.85435510868717075,
    20.22539837028827925,
    20.59638606626692106,
    20.96732014228171861,
    21.33820244275927456,
    21.70903471783284423,
    22.07981862969444027,
    22.45055575841889780,
    22.82124760731200354,
    23.19189560782874148,
    23.56250112410235463,
    23.93306545712105503,
    24.30358984858447164,
    24.67407548446931992,
    25.04452349832989180,
    25.41493497435707758,
    25.78531095021668256,
    26.15565241968596579,
    26.52596033510539678,
    26.89623560966105829,
    27.26647911951150683,
    27.63669170577170811,
    28.00687417636541139,
    28.37702730775638450,
    28.74715184656783522,
    29.11724851109866875,
    29.48731799274432324,
    29.85736095732936654,
    30.22737804635839208,
    30.59736987819097079,
    30.96733704914646171,
    31.33728013454339074,
    31.70719968967806141,
    32.07709625074676090,
    32.44697033571526771,
    32.81682244513918772,
    33.18665306293876446,
    33.55646265713073500,
    33.92625168052037206,
    34.29602057135609012,
    34.66576975394924908,
    35.03549963926110159,
    35.40521062545914593,
    35.77490309844468766,
    36.14457743235348558,
    36.51423399003088432,
    36.88387312348332365,
    37.25349517430730321,
    37.62310047409740785,
    37.99268934483428467,
    38.36226209925419058,
    38.73181904120067863,
    39.10136046595984082,
    39.47088666057973506,
    39.84039790417516111,
    40.20989446821820901,
    40.57937661681586418,
    40.94884460697482086,
    41.31829868885470704,
    41.68773910600994073,
    42.05716609562099287,
    42.42657988871569549,
    42.79598071038087266,
    43.16536877996494880,
    43.53474431127211210,
    43.90410751274811857,
    44.27345858765850139,
    44.64279773425932063,
    45.01212514596099104,
    45.38144101148543546,
    45.75074551501685603,
    46.12003883634655552,
    46.48932115101187068,
    46.85859263042992190,
    47.22785344202578439,
    47.59710374935594824,
    47.96634371222699862,
    48.33557348680966424,
    48.70479322574863139,
    49.07400307826826374,
    49.44320319027426906,
    49.81239370445177532,
    50.18157476035956679,
    50.55074649452119467,
    50.91990904051245372,
    51.28906252904602409,
    51.65820708805285255,
    52.02734284276085039,
    52.39646991577064011,
    52.76558842712884119,
    53.13469849439861556,
    53.50380023272806732,
    53.87289375491583663,
    54.24197917147515649,
    54.61105659069504157,
    54.98012611869983070,
    55.34918785950675601,
    55.71824191508138725,
    56.08728838539141306,
    56.45632736845873036,
    56.82535896040967421,
    57.19438325552368889,
    57.56340034628049008,
    57.93241032340582396,
    58.30141327591547908,
    58.67040929115821791,
    59.03939845485717393,
    59.40838085115024066,
    59.77735656262869922,
    60.14632567037530614,
    60.51528825400072539,
    60.88424439167913076,
    61.25319416018254594,
    61.62213763491437214,
    61.99107488994177307,
    62.36000599802717914,
    62.72893103065869982,
    63.09785005807991765,
    63.46676314931864485,
    63.83567037221495759,
    64.20457179344815074,
    64.57346747856338002,
    64.94235749199715310,
    65.31124189710230274,
    65.68012075617224324,
    66.04899413046437952,
    66.41786208022320524,
    66.78672466470250401,
    67.15558194218686757,
    67.52443397001290661,
    67.89328080458979775,
    68.26212250141910424,
    68.63095911511423117,
    68.99979069941927762,
    69.36861730722760295,
    69.73743899059944570,
    70.10625580077968513,
    70.47506778821458074,
    70.84387500256837793,
    71.21267749273960970,
    71.58147530687642201,
    71.95026849239226373,
    72.31905709598060851,
    72.68784116362947672,
    73.05662074063560163,
    73.42539587161839685,
    73.79416660053330142,
    74.16293297068486223,
    74.53169502473967539,
    74.90045280473881546,
    75.26920635211012469,
    75.63795570767994538,
    76.00670091168483621,
    76.37544200378294049,
    76.74417902306495232,
    77.11291200806499027,
    77.48164099677103035,
    77.85036602663534096,
    78.21908713458445072,
    78.58780435702901428,
    78.95651772987328654,
    79.32522728852462293,
    79.69393306790251438,
    80.06263510244762968,
    80.43133342613051458,
    80.80002807246005155,
    81.16871907449189777,
    81.53740646483668077,
    81.90609027566776094,
    82.27477053872929354,
    82.64344728534365236,
    83.01212054641892440,
    83.38079035245636547,
    83.74945673355718617,
    84.11811971942994005,
    84.48677933939692741,
    84.85543562240120687,
    85.22408859701299143,
    85.59273829143593980,
    85.96138473351363179,
    86.33002795073548830,
    86.69866797024287543,
    87.06730481883484891,
    87.43593852297413504,
    87.80456910879249147,
    88.17319660209628363,
    88.54182102837195806,
    88.91044241279134188,
    89.27906078021652120,
    89.64767615520540334,
    90.01628856201625873,
    90.38489802461258973,
    90.75350456666830848,
    91.12210821157192641,
    91.49070898243125469,
    91.85930690207815631,
    92.22790199307263007,
    92.59649427770711554,
    92.96508377801105311,
    93.33367051575445529,
    93.70225451245255168,
    94.07083578936932036,
    94.43941436752178890,
    94.80799026768333373,
    95.17656351038802180,
    95.54513411593379066,
    95.91370210438633274,
    96.28226749558257550,
    96.65083030913417272,
    97.01939056443073951,
    97.38794828064340887,
    97.75650347672818056,
    98.12505617142869991,
    98.49360638328005161,
    98.86215413061107849,
    99.23069943154833111,
    99.59924230401813377,
    99.96778276575037567,
    100.33632083428067676,
    100.70485652695351310,
    101.07338986092532084,
    101.44192085316634121,
    101.81044952046437632,
    102.17897587942649904,
    102.54749994648213594,
    102.91602173788551236,
    103.28454126971807521,
    103.65305855789091307,
    104.02157361814736058,
    104.39008646606486982,
    104.75859711705802663,
    105.12710558638034934,
    105.49561188912648940,
    105.86411604023467143,
    106.23261805448878405,
    106.60111794652054883,
    106.96961573081142924,
    107.33811142169478783,
    107.70660503335786548,
    108.07509657984400255,
    108.44358607505416399,
    108.81207353274911710,
    109.18055896655140555,
    109.54904238994693985,
    109.91752381628703962,
    110.28600325879006691,
    110.65448073054322885,
    111.02295624450436380,
    111.39142981350369346,
    111.75990145024535234,
    112.12837116730909812,
    112.49683897715184213,
    112.86530489210957024,
    113.23376892439852899,
    113.60223108611683728,
    113.97069138924632800,
    114.33914984565349095,
    114.70760646709160547,
    115.07606126520157731,
    115.44451425151365050,
    115.81296543744886156,
    116.18141483432037246,
    116.54986245333464847,
    116.91830830559295409,
    117.28675240209287705,
    117.65519475372908194,
    118.02363537129503351,
    118.39207426548429654,
    118.76051144689120369,
    119.12894692601288268,
    119.49738071324972350,
    119.86581281890703731,
    120.23424325319591735,
    120.60267202623469729,
    120.97109914804969330,
    121.33952462857653903,
    121.70794847766147518,
    122.07637070506195502,
    122.44479132044804202,
    122.81321033340336693,
    123.18162775342610364,
    123.55004358993016856,
    123.91845785224600143,
    124.28687054962146948,
    124.65528169122335162,
    125.02369128613757413,
    125.39209934337076876,
    125.76050587185090044,
    126.12891088042809589,
    126.49731437787578844,
    126.86571637289161696,
    127.23411687409785482,
    127.60251589004292074,
    127.97091342920171602,
    128.33930949997667740,
    128.70770411069855804,
    129.07609726962747125,
    129.44448898495312414,
    129.81287926479629734,
    130.18126811720897449,
    130.54965555017568437,
    130.91804157161396583,
    131.28642618937507986,
    131.65480941124473874,
    132.02319124494412988,
    132.39157169813032723,
    132.75995077839695881,
    133.12832849327515278,
    133.49670485023419223,
    133.86507985668198497,
    134.23345351996579252,
    134.60182584737315454,
    134.97019684613209956,
    135.33856652341218398,
    135.70693488632494215,
    136.07530194192452795,
    136.44366769720831376,
    136.81203215911755015,
    137.18039533453793605,
    137.54875723030011775,
    137.91711785318055747,
    138.28547720990177492,
    138.65383530713307836,
    139.02219215149121049,
    139.39054774954067273,
    139.75890210779442064,
    140.12725523271443988,
    140.49560713071221876,
    140.86395780814921780,
    141.23230727133760559,
    141.60065552654034661,
    141.96900257997217738,
    142.33734843779979573,
    142.70569310614262985,
    143.07403659107284969,
    143.44237889861649385,
    143.81072003475327547,
    144.17906000541770043,
    144.54739881649889194,
    144.91573647384150086,
    145.28407298324610941,
    145.65240835046939485,
    146.02074258122482610,
    146.38907568118306120,
    146.75740765597231129,
    147.12573851117868335,
    147.49406825234693610,
    147.86239688498047816,
    148.23072441454197175,
  };


  //#include <iostream>
  //#include <vector>
  //#include <algorithm>
  //#include <numeric>
  //#include <cmath>
  //
  //
  //using namespace std;
  //
  //
  //int main() {
  //
  //  int WIDTH = 400;
  //  double liangBaiFactors[WIDTH][WIDTH];
  //
  //  for (int i = 1; i <= WIDTH; i++) {
  //    double b = std::ceil(i/2.0);
  //    for (int j=1; j <= WIDTH; j++) {
  //      double a = std::ceil((j+1)/2.0);
  //      double q = std::min(a,b);
  //      liangBaiFactors[i-1][j-1] = q*(j-q+1);
  //    }
  //  }
  //
  //  cout << "const double liangBaiFactors[] = {" << endl;
  //
  //  for (int i=0; i<WIDTH; i++) {
  //    for (int j=0; j<WIDTH; j++)
  //      cout << liangBaiFactors[i][j] << ",";
  //    cout << endl;
  //  }
  //
  //  cout << "};";
  //
  //  return 0;
  //}

  const double liangBaiFactors[] = {
    1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,64,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,91,92,93,94,95,96,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255,256,257,258,259,260,261,262,263,264,265,266,267,268,269,270,271,272,273,274,275,276,277,278,279,280,281,282,283,284,285,286,287,288,289,290,291,292,293,294,295,296,297,298,299,300,301,302,303,304,305,306,307,308,309,310,311,312,313,314,315,316,317,318,319,320,321,322,323,324,325,326,327,328,329,330,331,332,333,334,335,336,337,338,339,340,341,342,343,344,345,346,347,348,349,350,351,352,353,354,355,356,357,358,359,360,361,362,363,364,365,366,367,368,369,370,371,372,373,374,375,376,377,378,379,380,381,382,383,384,385,386,387,388,389,390,391,392,393,394,395,396,397,398,399,400,
    1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,64,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,91,92,93,94,95,96,97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255,256,257,258,259,260,261,262,263,264,265,266,267,268,269,270,271,272,273,274,275,276,277,278,279,280,281,282,283,284,285,286,287,288,289,290,291,292,293,294,295,296,297,298,299,300,301,302,303,304,305,306,307,308,309,310,311,312,313,314,315,316,317,318,319,320,321,322,323,324,325,326,327,328,329,330,331,332,333,334,335,336,337,338,339,340,341,342,343,344,345,346,347,348,349,350,351,352,353,354,355,356,357,358,359,360,361,362,363,364,365,366,367,368,369,370,371,372,373,374,375,376,377,378,379,380,381,382,383,384,385,386,387,388,389,390,391,392,393,394,395,396,397,398,399,400,
    1,2,4,6,8,10,12,14,16,18,20,22,24,26,28,30,32,34,36,38,40,42,44,46,48,50,52,54,56,58,60,62,64,66,68,70,72,74,76,78,80,82,84,86,88,90,92,94,96,98,100,102,104,106,108,110,112,114,116,118,120,122,124,126,128,130,132,134,136,138,140,142,144,146,148,150,152,154,156,158,160,162,164,166,168,170,172,174,176,178,180,182,184,186,188,190,192,194,196,198,200,202,204,206,208,210,212,214,216,218,220,222,224,226,228,230,232,234,236,238,240,242,244,246,248,250,252,254,256,258,260,262,264,266,268,270,272,274,276,278,280,282,284,286,288,290,292,294,296,298,300,302,304,306,308,310,312,314,316,318,320,322,324,326,328,330,332,334,336,338,340,342,344,346,348,350,352,354,356,358,360,362,364,366,368,370,372,374,376,378,380,382,384,386,388,390,392,394,396,398,400,402,404,406,408,410,412,414,416,418,420,422,424,426,428,430,432,434,436,438,440,442,444,446,448,450,452,454,456,458,460,462,464,466,468,470,472,474,476,478,480,482,484,486,488,490,492,494,496,498,500,502,504,506,508,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,540,542,544,546,548,550,552,554,556,558,560,562,564,566,568,570,572,574,576,578,580,582,584,586,588,590,592,594,596,598,600,602,604,606,608,610,612,614,616,618,620,622,624,626,628,630,632,634,636,638,640,642,644,646,648,650,652,654,656,658,660,662,664,666,668,670,672,674,676,678,680,682,684,686,688,690,692,694,696,698,700,702,704,706,708,710,712,714,716,718,720,722,724,726,728,730,732,734,736,738,740,742,744,746,748,750,752,754,756,758,760,762,764,766,768,770,772,774,776,778,780,782,784,786,788,790,792,794,796,798,
    1,2,4,6,8,10,12,14,16,18,20,22,24,26,28,30,32,34,36,38,40,42,44,46,48,50,52,54,56,58,60,62,64,66,68,70,72,74,76,78,80,82,84,86,88,90,92,94,96,98,100,102,104,106,108,110,112,114,116,118,120,122,124,126,128,130,132,134,136,138,140,142,144,146,148,150,152,154,156,158,160,162,164,166,168,170,172,174,176,178,180,182,184,186,188,190,192,194,196,198,200,202,204,206,208,210,212,214,216,218,220,222,224,226,228,230,232,234,236,238,240,242,244,246,248,250,252,254,256,258,260,262,264,266,268,270,272,274,276,278,280,282,284,286,288,290,292,294,296,298,300,302,304,306,308,310,312,314,316,318,320,322,324,326,328,330,332,334,336,338,340,342,344,346,348,350,352,354,356,358,360,362,364,366,368,370,372,374,376,378,380,382,384,386,388,390,392,394,396,398,400,402,404,406,408,410,412,414,416,418,420,422,424,426,428,430,432,434,436,438,440,442,444,446,448,450,452,454,456,458,460,462,464,466,468,470,472,474,476,478,480,482,484,486,488,490,492,494,496,498,500,502,504,506,508,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,540,542,544,546,548,550,552,554,556,558,560,562,564,566,568,570,572,574,576,578,580,582,584,586,588,590,592,594,596,598,600,602,604,606,608,610,612,614,616,618,620,622,624,626,628,630,632,634,636,638,640,642,644,646,648,650,652,654,656,658,660,662,664,666,668,670,672,674,676,678,680,682,684,686,688,690,692,694,696,698,700,702,704,706,708,710,712,714,716,718,720,722,724,726,728,730,732,734,736,738,740,742,744,746,748,750,752,754,756,758,760,762,764,766,768,770,772,774,776,778,780,782,784,786,788,790,792,794,796,798,
    1,2,4,6,9,12,15,18,21,24,27,30,33,36,39,42,45,48,51,54,57,60,63,66,69,72,75,78,81,84,87,90,93,96,99,102,105,108,111,114,117,120,123,126,129,132,135,138,141,144,147,150,153,156,159,162,165,168,171,174,177,180,183,186,189,192,195,198,201,204,207,210,213,216,219,222,225,228,231,234,237,240,243,246,249,252,255,258,261,264,267,270,273,276,279,282,285,288,291,294,297,300,303,306,309,312,315,318,321,324,327,330,333,336,339,342,345,348,351,354,357,360,363,366,369,372,375,378,381,384,387,390,393,396,399,402,405,408,411,414,417,420,423,426,429,432,435,438,441,444,447,450,453,456,459,462,465,468,471,474,477,480,483,486,489,492,495,498,501,504,507,510,513,516,519,522,525,528,531,534,537,540,543,546,549,552,555,558,561,564,567,570,573,576,579,582,585,588,591,594,597,600,603,606,609,612,615,618,621,624,627,630,633,636,639,642,645,648,651,654,657,660,663,666,669,672,675,678,681,684,687,690,693,696,699,702,705,708,711,714,717,720,723,726,729,732,735,738,741,744,747,750,753,756,759,762,765,768,771,774,777,780,783,786,789,792,795,798,801,804,807,810,813,816,819,822,825,828,831,834,837,840,843,846,849,852,855,858,861,864,867,870,873,876,879,882,885,888,891,894,897,900,903,906,909,912,915,918,921,924,927,930,933,936,939,942,945,948,951,954,957,960,963,966,969,972,975,978,981,984,987,990,993,996,999,1002,1005,1008,1011,1014,1017,1020,1023,1026,1029,1032,1035,1038,1041,1044,1047,1050,1053,1056,1059,1062,1065,1068,1071,1074,1077,1080,1083,1086,1089,1092,1095,1098,1101,1104,1107,1110,1113,1116,1119,1122,1125,1128,1131,1134,1137,1140,1143,1146,1149,1152,1155,1158,1161,1164,1167,1170,1173,1176,1179,1182,1185,1188,1191,1194,
    1,2,4,6,9,12,15,18,21,24,27,30,33,36,39,42,45,48,51,54,57,60,63,66,69,72,75,78,81,84,87,90,93,96,99,102,105,108,111,114,117,120,123,126,129,132,135,138,141,144,147,150,153,156,159,162,165,168,171,174,177,180,183,186,189,192,195,198,201,204,207,210,213,216,219,222,225,228,231,234,237,240,243,246,249,252,255,258,261,264,267,270,273,276,279,282,285,288,291,294,297,300,303,306,309,312,315,318,321,324,327,330,333,336,339,342,345,348,351,354,357,360,363,366,369,372,375,378,381,384,387,390,393,396,399,402,405,408,411,414,417,420,423,426,429,432,435,438,441,444,447,450,453,456,459,462,465,468,471,474,477,480,483,486,489,492,495,498,501,504,507,510,513,516,519,522,525,528,531,534,537,540,543,546,549,552,555,558,561,564,567,570,573,576,579,582,585,588,591,594,597,600,603,606,609,612,615,618,621,624,627,630,633,636,639,642,645,648,651,654,657,660,663,666,669,672,675,678,681,684,687,690,693,696,699,702,705,708,711,714,717,720,723,726,729,732,735,738,741,744,747,750,753,756,759,762,765,768,771,774,777,780,783,786,789,792,795,798,801,804,807,810,813,816,819,822,825,828,831,834,837,840,843,846,849,852,855,858,861,864,867,870,873,876,879,882,885,888,891,894,897,900,903,906,909,912,915,918,921,924,927,930,933,936,939,942,945,948,951,954,957,960,963,966,969,972,975,978,981,984,987,990,993,996,999,1002,1005,1008,1011,1014,1017,1020,1023,1026,1029,1032,1035,1038,1041,1044,1047,1050,1053,1056,1059,1062,1065,1068,1071,1074,1077,1080,1083,1086,1089,1092,1095,1098,1101,1104,1107,1110,1113,1116,1119,1122,1125,1128,1131,1134,1137,1140,1143,1146,1149,1152,1155,1158,1161,1164,1167,1170,1173,1176,1179,1182,1185,1188,1191,1194,
    1,2,4,6,9,12,16,20,24,28,32,36,40,44,48,52,56,60,64,68,72,76,80,84,88,92,96,100,104,108,112,116,120,124,128,132,136,140,144,148,152,156,160,164,168,172,176,180,184,188,192,196,200,204,208,212,216,220,224,228,232,236,240,244,248,252,256,260,264,268,272,276,280,284,288,292,296,300,304,308,312,316,320,324,328,332,336,340,344,348,352,356,360,364,368,372,376,380,384,388,392,396,400,404,408,412,416,420,424,428,432,436,440,444,448,452,456,460,464,468,472,476,480,484,488,492,496,500,504,508,512,516,520,524,528,532,536,540,544,548,552,556,560,564,568,572,576,580,584,588,592,596,600,604,608,612,616,620,624,628,632,636,640,644,648,652,656,660,664,668,672,676,680,684,688,692,696,700,704,708,712,716,720,724,728,732,736,740,744,748,752,756,760,764,768,772,776,780,784,788,792,796,800,804,808,812,816,820,824,828,832,836,840,844,848,852,856,860,864,868,872,876,880,884,888,892,896,900,904,908,912,916,920,924,928,932,936,940,944,948,952,956,960,964,968,972,976,980,984,988,992,996,1000,1004,1008,1012,1016,1020,1024,1028,1032,1036,1040,1044,1048,1052,1056,1060,1064,1068,1072,1076,1080,1084,1088,1092,1096,1100,1104,1108,1112,1116,1120,1124,1128,1132,1136,1140,1144,1148,1152,1156,1160,1164,1168,1172,1176,1180,1184,1188,1192,1196,1200,1204,1208,1212,1216,1220,1224,1228,1232,1236,1240,1244,1248,1252,1256,1260,1264,1268,1272,1276,1280,1284,1288,1292,1296,1300,1304,1308,1312,1316,1320,1324,1328,1332,1336,1340,1344,1348,1352,1356,1360,1364,1368,1372,1376,1380,1384,1388,1392,1396,1400,1404,1408,1412,1416,1420,1424,1428,1432,1436,1440,1444,1448,1452,1456,1460,1464,1468,1472,1476,1480,1484,1488,1492,1496,1500,1504,1508,1512,1516,1520,1524,1528,1532,1536,1540,1544,1548,1552,1556,1560,1564,1568,1572,1576,1580,1584,1588,
    1,2,4,6,9,12,16,20,24,28,32,36,40,44,48,52,56,60,64,68,72,76,80,84,88,92,96,100,104,108,112,116,120,124,128,132,136,140,144,148,152,156,160,164,168,172,176,180,184,188,192,196,200,204,208,212,216,220,224,228,232,236,240,244,248,252,256,260,264,268,272,276,280,284,288,292,296,300,304,308,312,316,320,324,328,332,336,340,344,348,352,356,360,364,368,372,376,380,384,388,392,396,400,404,408,412,416,420,424,428,432,436,440,444,448,452,456,460,464,468,472,476,480,484,488,492,496,500,504,508,512,516,520,524,528,532,536,540,544,548,552,556,560,564,568,572,576,580,584,588,592,596,600,604,608,612,616,620,624,628,632,636,640,644,648,652,656,660,664,668,672,676,680,684,688,692,696,700,704,708,712,716,720,724,728,732,736,740,744,748,752,756,760,764,768,772,776,780,784,788,792,796,800,804,808,812,816,820,824,828,832,836,840,844,848,852,856,860,864,868,872,876,880,884,888,892,896,900,904,908,912,916,920,924,928,932,936,940,944,948,952,956,960,964,968,972,976,980,984,988,992,996,1000,1004,1008,1012,1016,1020,1024,1028,1032,1036,1040,1044,1048,1052,1056,1060,1064,1068,1072,1076,1080,1084,1088,1092,1096,1100,1104,1108,1112,1116,1120,1124,1128,1132,1136,1140,1144,1148,1152,1156,1160,1164,1168,1172,1176,1180,1184,1188,1192,1196,1200,1204,1208,1212,1216,1220,1224,1228,1232,1236,1240,1244,1248,1252,1256,1260,1264,1268,1272,1276,1280,1284,1288,1292,1296,1300,1304,1308,1312,1316,1320,1324,1328,1332,1336,1340,1344,1348,1352,1356,1360,1364,1368,1372,1376,1380,1384,1388,1392,1396,1400,1404,1408,1412,1416,1420,1424,1428,1432,1436,1440,1444,1448,1452,1456,1460,1464,1468,1472,1476,1480,1484,1488,1492,1496,1500,1504,1508,1512,1516,1520,1524,1528,1532,1536,1540,1544,1548,1552,1556,1560,1564,1568,1572,1576,1580,1584,1588,
    1,2,4,6,9,12,16,20,25,30,35,40,45,50,55,60,65,70,75,80,85,90,95,100,105,110,115,120,125,130,135,140,145,150,155,160,165,170,175,180,185,190,195,200,205,210,215,220,225,230,235,240,245,250,255,260,265,270,275,280,285,290,295,300,305,310,315,320,325,330,335,340,345,350,355,360,365,370,375,380,385,390,395,400,405,410,415,420,425,430,435,440,445,450,455,460,465,470,475,480,485,490,495,500,505,510,515,520,525,530,535,540,545,550,555,560,565,570,575,580,585,590,595,600,605,610,615,620,625,630,635,640,645,650,655,660,665,670,675,680,685,690,695,700,705,710,715,720,725,730,735,740,745,750,755,760,765,770,775,780,785,790,795,800,805,810,815,820,825,830,835,840,845,850,855,860,865,870,875,880,885,890,895,900,905,910,915,920,925,930,935,940,945,950,955,960,965,970,975,980,985,990,995,1000,1005,1010,1015,1020,1025,1030,1035,1040,1045,1050,1055,1060,1065,1070,1075,1080,1085,1090,1095,1100,1105,1110,1115,1120,1125,1130,1135,1140,1145,1150,1155,1160,1165,1170,1175,1180,1185,1190,1195,1200,1205,1210,1215,1220,1225,1230,1235,1240,1245,1250,1255,1260,1265,1270,1275,1280,1285,1290,1295,1300,1305,1310,1315,1320,1325,1330,1335,1340,1345,1350,1355,1360,1365,1370,1375,1380,1385,1390,1395,1400,1405,1410,1415,1420,1425,1430,1435,1440,1445,1450,1455,1460,1465,1470,1475,1480,1485,1490,1495,1500,1505,1510,1515,1520,1525,1530,1535,1540,1545,1550,1555,1560,1565,1570,1575,1580,1585,1590,1595,1600,1605,1610,1615,1620,1625,1630,1635,1640,1645,1650,1655,1660,1665,1670,1675,1680,1685,1690,1695,1700,1705,1710,1715,1720,1725,1730,1735,1740,1745,1750,1755,1760,1765,1770,1775,1780,1785,1790,1795,1800,1805,1810,1815,1820,1825,1830,1835,1840,1845,1850,1855,1860,1865,1870,1875,1880,1885,1890,1895,1900,1905,1910,1915,1920,1925,1930,1935,1940,1945,1950,1955,1960,1965,1970,1975,1980,
    1,2,4,6,9,12,16,20,25,30,35,40,45,50,55,60,65,70,75,80,85,90,95,100,105,110,115,120,125,130,135,140,145,150,155,160,165,170,175,180,185,190,195,200,205,210,215,220,225,230,235,240,245,250,255,260,265,270,275,280,285,290,295,300,305,310,315,320,325,330,335,340,345,350,355,360,365,370,375,380,385,390,395,400,405,410,415,420,425,430,435,440,445,450,455,460,465,470,475,480,485,490,495,500,505,510,515,520,525,530,535,540,545,550,555,560,565,570,575,580,585,590,595,600,605,610,615,620,625,630,635,640,645,650,655,660,665,670,675,680,685,690,695,700,705,710,715,720,725,730,735,740,745,750,755,760,765,770,775,780,785,790,795,800,805,810,815,820,825,830,835,840,845,850,855,860,865,870,875,880,885,890,895,900,905,910,915,920,925,930,935,940,945,950,955,960,965,970,975,980,985,990,995,1000,1005,1010,1015,1020,1025,1030,1035,1040,1045,1050,1055,1060,1065,1070,1075,1080,1085,1090,1095,1100,1105,1110,1115,1120,1125,1130,1135,1140,1145,1150,1155,1160,1165,1170,1175,1180,1185,1190,1195,1200,1205,1210,1215,1220,1225,1230,1235,1240,1245,1250,1255,1260,1265,1270,1275,1280,1285,1290,1295,1300,1305,1310,1315,1320,1325,1330,1335,1340,1345,1350,1355,1360,1365,1370,1375,1380,1385,1390,1395,1400,1405,1410,1415,1420,1425,1430,1435,1440,1445,1450,1455,1460,1465,1470,1475,1480,1485,1490,1495,1500,1505,1510,1515,1520,1525,1530,1535,1540,1545,1550,1555,1560,1565,1570,1575,1580,1585,1590,1595,1600,1605,1610,1615,1620,1625,1630,1635,1640,1645,1650,1655,1660,1665,1670,1675,1680,1685,1690,1695,1700,1705,1710,1715,1720,1725,1730,1735,1740,1745,1750,1755,1760,1765,1770,1775,1780,1785,1790,1795,1800,1805,1810,1815,1820,1825,1830,1835,1840,1845,1850,1855,1860,1865,1870,1875,1880,1885,1890,1895,1900,1905,1910,1915,1920,1925,1930,1935,1940,1945,1950,1955,1960,1965,1970,1975,1980,
    1,2,4,6,9,12,16,20,25,30,36,42,48,54,60,66,72,78,84,90,96,102,108,114,120,126,132,138,144,150,156,162,168,174,180,186,192,198,204,210,216,222,228,234,240,246,252,258,264,270,276,282,288,294,300,306,312,318,324,330,336,342,348,354,360,366,372,378,384,390,396,402,408,414,420,426,432,438,444,450,456,462,468,474,480,486,492,498,504,510,516,522,528,534,540,546,552,558,564,570,576,582,588,594,600,606,612,618,624,630,636,642,648,654,660,666,672,678,684,690,696,702,708,714,720,726,732,738,744,750,756,762,768,774,780,786,792,798,804,810,816,822,828,834,840,846,852,858,864,870,876,882,888,894,900,906,912,918,924,930,936,942,948,954,960,966,972,978,984,990,996,1002,1008,1014,1020,1026,1032,1038,1044,1050,1056,1062,1068,1074,1080,1086,1092,1098,1104,1110,1116,1122,1128,1134,1140,1146,1152,1158,1164,1170,1176,1182,1188,1194,1200,1206,1212,1218,1224,1230,1236,1242,1248,1254,1260,1266,1272,1278,1284,1290,1296,1302,1308,1314,1320,1326,1332,1338,1344,1350,1356,1362,1368,1374,1380,1386,1392,1398,1404,1410,1416,1422,1428,1434,1440,1446,1452,1458,1464,1470,1476,1482,1488,1494,1500,1506,1512,1518,1524,1530,1536,1542,1548,1554,1560,1566,1572,1578,1584,1590,1596,1602,1608,1614,1620,1626,1632,1638,1644,1650,1656,1662,1668,1674,1680,1686,1692,1698,1704,1710,1716,1722,1728,1734,1740,1746,1752,1758,1764,1770,1776,1782,1788,1794,1800,1806,1812,1818,1824,1830,1836,1842,1848,1854,1860,1866,1872,1878,1884,1890,1896,1902,1908,1914,1920,1926,1932,1938,1944,1950,1956,1962,1968,1974,1980,1986,1992,1998,2004,2010,2016,2022,2028,2034,2040,2046,2052,2058,2064,2070,2076,2082,2088,2094,2100,2106,2112,2118,2124,2130,2136,2142,2148,2154,2160,2166,2172,2178,2184,2190,2196,2202,2208,2214,2220,2226,2232,2238,2244,2250,2256,2262,2268,2274,2280,2286,2292,2298,2304,2310,2316,2322,2328,2334,2340,2346,2352,2358,2364,2370,
    1,2,4,6,9,12,16,20,25,30,36,42,48,54,60,66,72,78,84,90,96,102,108,114,120,126,132,138,144,150,156,162,168,174,180,186,192,198,204,210,216,222,228,234,240,246,252,258,264,270,276,282,288,294,300,306,312,318,324,330,336,342,348,354,360,366,372,378,384,390,396,402,408,414,420,426,432,438,444,450,456,462,468,474,480,486,492,498,504,510,516,522,528,534,540,546,552,558,564,570,576,582,588,594,600,606,612,618,624,630,636,642,648,654,660,666,672,678,684,690,696,702,708,714,720,726,732,738,744,750,756,762,768,774,780,786,792,798,804,810,816,822,828,834,840,846,852,858,864,870,876,882,888,894,900,906,912,918,924,930,936,942,948,954,960,966,972,978,984,990,996,1002,1008,1014,1020,1026,1032,1038,1044,1050,1056,1062,1068,1074,1080,1086,1092,1098,1104,1110,1116,1122,1128,1134,1140,1146,1152,1158,1164,1170,1176,1182,1188,1194,1200,1206,1212,1218,1224,1230,1236,1242,1248,1254,1260,1266,1272,1278,1284,1290,1296,1302,1308,1314,1320,1326,1332,1338,1344,1350,1356,1362,1368,1374,1380,1386,1392,1398,1404,1410,1416,1422,1428,1434,1440,1446,1452,1458,1464,1470,1476,1482,1488,1494,1500,1506,1512,1518,1524,1530,1536,1542,1548,1554,1560,1566,1572,1578,1584,1590,1596,1602,1608,1614,1620,1626,1632,1638,1644,1650,1656,1662,1668,1674,1680,1686,1692,1698,1704,1710,1716,1722,1728,1734,1740,1746,1752,1758,1764,1770,1776,1782,1788,1794,1800,1806,1812,1818,1824,1830,1836,1842,1848,1854,1860,1866,1872,1878,1884,1890,1896,1902,1908,1914,1920,1926,1932,1938,1944,1950,1956,1962,1968,1974,1980,1986,1992,1998,2004,2010,2016,2022,2028,2034,2040,2046,2052,2058,2064,2070,2076,2082,2088,2094,2100,2106,2112,2118,2124,2130,2136,2142,2148,2154,2160,2166,2172,2178,2184,2190,2196,2202,2208,2214,2220,2226,2232,2238,2244,2250,2256,2262,2268,2274,2280,2286,2292,2298,2304,2310,2316,2322,2328,2334,2340,2346,2352,2358,2364,2370,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,63,70,77,84,91,98,105,112,119,126,133,140,147,154,161,168,175,182,189,196,203,210,217,224,231,238,245,252,259,266,273,280,287,294,301,308,315,322,329,336,343,350,357,364,371,378,385,392,399,406,413,420,427,434,441,448,455,462,469,476,483,490,497,504,511,518,525,532,539,546,553,560,567,574,581,588,595,602,609,616,623,630,637,644,651,658,665,672,679,686,693,700,707,714,721,728,735,742,749,756,763,770,777,784,791,798,805,812,819,826,833,840,847,854,861,868,875,882,889,896,903,910,917,924,931,938,945,952,959,966,973,980,987,994,1001,1008,1015,1022,1029,1036,1043,1050,1057,1064,1071,1078,1085,1092,1099,1106,1113,1120,1127,1134,1141,1148,1155,1162,1169,1176,1183,1190,1197,1204,1211,1218,1225,1232,1239,1246,1253,1260,1267,1274,1281,1288,1295,1302,1309,1316,1323,1330,1337,1344,1351,1358,1365,1372,1379,1386,1393,1400,1407,1414,1421,1428,1435,1442,1449,1456,1463,1470,1477,1484,1491,1498,1505,1512,1519,1526,1533,1540,1547,1554,1561,1568,1575,1582,1589,1596,1603,1610,1617,1624,1631,1638,1645,1652,1659,1666,1673,1680,1687,1694,1701,1708,1715,1722,1729,1736,1743,1750,1757,1764,1771,1778,1785,1792,1799,1806,1813,1820,1827,1834,1841,1848,1855,1862,1869,1876,1883,1890,1897,1904,1911,1918,1925,1932,1939,1946,1953,1960,1967,1974,1981,1988,1995,2002,2009,2016,2023,2030,2037,2044,2051,2058,2065,2072,2079,2086,2093,2100,2107,2114,2121,2128,2135,2142,2149,2156,2163,2170,2177,2184,2191,2198,2205,2212,2219,2226,2233,2240,2247,2254,2261,2268,2275,2282,2289,2296,2303,2310,2317,2324,2331,2338,2345,2352,2359,2366,2373,2380,2387,2394,2401,2408,2415,2422,2429,2436,2443,2450,2457,2464,2471,2478,2485,2492,2499,2506,2513,2520,2527,2534,2541,2548,2555,2562,2569,2576,2583,2590,2597,2604,2611,2618,2625,2632,2639,2646,2653,2660,2667,2674,2681,2688,2695,2702,2709,2716,2723,2730,2737,2744,2751,2758,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,63,70,77,84,91,98,105,112,119,126,133,140,147,154,161,168,175,182,189,196,203,210,217,224,231,238,245,252,259,266,273,280,287,294,301,308,315,322,329,336,343,350,357,364,371,378,385,392,399,406,413,420,427,434,441,448,455,462,469,476,483,490,497,504,511,518,525,532,539,546,553,560,567,574,581,588,595,602,609,616,623,630,637,644,651,658,665,672,679,686,693,700,707,714,721,728,735,742,749,756,763,770,777,784,791,798,805,812,819,826,833,840,847,854,861,868,875,882,889,896,903,910,917,924,931,938,945,952,959,966,973,980,987,994,1001,1008,1015,1022,1029,1036,1043,1050,1057,1064,1071,1078,1085,1092,1099,1106,1113,1120,1127,1134,1141,1148,1155,1162,1169,1176,1183,1190,1197,1204,1211,1218,1225,1232,1239,1246,1253,1260,1267,1274,1281,1288,1295,1302,1309,1316,1323,1330,1337,1344,1351,1358,1365,1372,1379,1386,1393,1400,1407,1414,1421,1428,1435,1442,1449,1456,1463,1470,1477,1484,1491,1498,1505,1512,1519,1526,1533,1540,1547,1554,1561,1568,1575,1582,1589,1596,1603,1610,1617,1624,1631,1638,1645,1652,1659,1666,1673,1680,1687,1694,1701,1708,1715,1722,1729,1736,1743,1750,1757,1764,1771,1778,1785,1792,1799,1806,1813,1820,1827,1834,1841,1848,1855,1862,1869,1876,1883,1890,1897,1904,1911,1918,1925,1932,1939,1946,1953,1960,1967,1974,1981,1988,1995,2002,2009,2016,2023,2030,2037,2044,2051,2058,2065,2072,2079,2086,2093,2100,2107,2114,2121,2128,2135,2142,2149,2156,2163,2170,2177,2184,2191,2198,2205,2212,2219,2226,2233,2240,2247,2254,2261,2268,2275,2282,2289,2296,2303,2310,2317,2324,2331,2338,2345,2352,2359,2366,2373,2380,2387,2394,2401,2408,2415,2422,2429,2436,2443,2450,2457,2464,2471,2478,2485,2492,2499,2506,2513,2520,2527,2534,2541,2548,2555,2562,2569,2576,2583,2590,2597,2604,2611,2618,2625,2632,2639,2646,2653,2660,2667,2674,2681,2688,2695,2702,2709,2716,2723,2730,2737,2744,2751,2758,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,80,88,96,104,112,120,128,136,144,152,160,168,176,184,192,200,208,216,224,232,240,248,256,264,272,280,288,296,304,312,320,328,336,344,352,360,368,376,384,392,400,408,416,424,432,440,448,456,464,472,480,488,496,504,512,520,528,536,544,552,560,568,576,584,592,600,608,616,624,632,640,648,656,664,672,680,688,696,704,712,720,728,736,744,752,760,768,776,784,792,800,808,816,824,832,840,848,856,864,872,880,888,896,904,912,920,928,936,944,952,960,968,976,984,992,1000,1008,1016,1024,1032,1040,1048,1056,1064,1072,1080,1088,1096,1104,1112,1120,1128,1136,1144,1152,1160,1168,1176,1184,1192,1200,1208,1216,1224,1232,1240,1248,1256,1264,1272,1280,1288,1296,1304,1312,1320,1328,1336,1344,1352,1360,1368,1376,1384,1392,1400,1408,1416,1424,1432,1440,1448,1456,1464,1472,1480,1488,1496,1504,1512,1520,1528,1536,1544,1552,1560,1568,1576,1584,1592,1600,1608,1616,1624,1632,1640,1648,1656,1664,1672,1680,1688,1696,1704,1712,1720,1728,1736,1744,1752,1760,1768,1776,1784,1792,1800,1808,1816,1824,1832,1840,1848,1856,1864,1872,1880,1888,1896,1904,1912,1920,1928,1936,1944,1952,1960,1968,1976,1984,1992,2000,2008,2016,2024,2032,2040,2048,2056,2064,2072,2080,2088,2096,2104,2112,2120,2128,2136,2144,2152,2160,2168,2176,2184,2192,2200,2208,2216,2224,2232,2240,2248,2256,2264,2272,2280,2288,2296,2304,2312,2320,2328,2336,2344,2352,2360,2368,2376,2384,2392,2400,2408,2416,2424,2432,2440,2448,2456,2464,2472,2480,2488,2496,2504,2512,2520,2528,2536,2544,2552,2560,2568,2576,2584,2592,2600,2608,2616,2624,2632,2640,2648,2656,2664,2672,2680,2688,2696,2704,2712,2720,2728,2736,2744,2752,2760,2768,2776,2784,2792,2800,2808,2816,2824,2832,2840,2848,2856,2864,2872,2880,2888,2896,2904,2912,2920,2928,2936,2944,2952,2960,2968,2976,2984,2992,3000,3008,3016,3024,3032,3040,3048,3056,3064,3072,3080,3088,3096,3104,3112,3120,3128,3136,3144,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,80,88,96,104,112,120,128,136,144,152,160,168,176,184,192,200,208,216,224,232,240,248,256,264,272,280,288,296,304,312,320,328,336,344,352,360,368,376,384,392,400,408,416,424,432,440,448,456,464,472,480,488,496,504,512,520,528,536,544,552,560,568,576,584,592,600,608,616,624,632,640,648,656,664,672,680,688,696,704,712,720,728,736,744,752,760,768,776,784,792,800,808,816,824,832,840,848,856,864,872,880,888,896,904,912,920,928,936,944,952,960,968,976,984,992,1000,1008,1016,1024,1032,1040,1048,1056,1064,1072,1080,1088,1096,1104,1112,1120,1128,1136,1144,1152,1160,1168,1176,1184,1192,1200,1208,1216,1224,1232,1240,1248,1256,1264,1272,1280,1288,1296,1304,1312,1320,1328,1336,1344,1352,1360,1368,1376,1384,1392,1400,1408,1416,1424,1432,1440,1448,1456,1464,1472,1480,1488,1496,1504,1512,1520,1528,1536,1544,1552,1560,1568,1576,1584,1592,1600,1608,1616,1624,1632,1640,1648,1656,1664,1672,1680,1688,1696,1704,1712,1720,1728,1736,1744,1752,1760,1768,1776,1784,1792,1800,1808,1816,1824,1832,1840,1848,1856,1864,1872,1880,1888,1896,1904,1912,1920,1928,1936,1944,1952,1960,1968,1976,1984,1992,2000,2008,2016,2024,2032,2040,2048,2056,2064,2072,2080,2088,2096,2104,2112,2120,2128,2136,2144,2152,2160,2168,2176,2184,2192,2200,2208,2216,2224,2232,2240,2248,2256,2264,2272,2280,2288,2296,2304,2312,2320,2328,2336,2344,2352,2360,2368,2376,2384,2392,2400,2408,2416,2424,2432,2440,2448,2456,2464,2472,2480,2488,2496,2504,2512,2520,2528,2536,2544,2552,2560,2568,2576,2584,2592,2600,2608,2616,2624,2632,2640,2648,2656,2664,2672,2680,2688,2696,2704,2712,2720,2728,2736,2744,2752,2760,2768,2776,2784,2792,2800,2808,2816,2824,2832,2840,2848,2856,2864,2872,2880,2888,2896,2904,2912,2920,2928,2936,2944,2952,2960,2968,2976,2984,2992,3000,3008,3016,3024,3032,3040,3048,3056,3064,3072,3080,3088,3096,3104,3112,3120,3128,3136,3144,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,99,108,117,126,135,144,153,162,171,180,189,198,207,216,225,234,243,252,261,270,279,288,297,306,315,324,333,342,351,360,369,378,387,396,405,414,423,432,441,450,459,468,477,486,495,504,513,522,531,540,549,558,567,576,585,594,603,612,621,630,639,648,657,666,675,684,693,702,711,720,729,738,747,756,765,774,783,792,801,810,819,828,837,846,855,864,873,882,891,900,909,918,927,936,945,954,963,972,981,990,999,1008,1017,1026,1035,1044,1053,1062,1071,1080,1089,1098,1107,1116,1125,1134,1143,1152,1161,1170,1179,1188,1197,1206,1215,1224,1233,1242,1251,1260,1269,1278,1287,1296,1305,1314,1323,1332,1341,1350,1359,1368,1377,1386,1395,1404,1413,1422,1431,1440,1449,1458,1467,1476,1485,1494,1503,1512,1521,1530,1539,1548,1557,1566,1575,1584,1593,1602,1611,1620,1629,1638,1647,1656,1665,1674,1683,1692,1701,1710,1719,1728,1737,1746,1755,1764,1773,1782,1791,1800,1809,1818,1827,1836,1845,1854,1863,1872,1881,1890,1899,1908,1917,1926,1935,1944,1953,1962,1971,1980,1989,1998,2007,2016,2025,2034,2043,2052,2061,2070,2079,2088,2097,2106,2115,2124,2133,2142,2151,2160,2169,2178,2187,2196,2205,2214,2223,2232,2241,2250,2259,2268,2277,2286,2295,2304,2313,2322,2331,2340,2349,2358,2367,2376,2385,2394,2403,2412,2421,2430,2439,2448,2457,2466,2475,2484,2493,2502,2511,2520,2529,2538,2547,2556,2565,2574,2583,2592,2601,2610,2619,2628,2637,2646,2655,2664,2673,2682,2691,2700,2709,2718,2727,2736,2745,2754,2763,2772,2781,2790,2799,2808,2817,2826,2835,2844,2853,2862,2871,2880,2889,2898,2907,2916,2925,2934,2943,2952,2961,2970,2979,2988,2997,3006,3015,3024,3033,3042,3051,3060,3069,3078,3087,3096,3105,3114,3123,3132,3141,3150,3159,3168,3177,3186,3195,3204,3213,3222,3231,3240,3249,3258,3267,3276,3285,3294,3303,3312,3321,3330,3339,3348,3357,3366,3375,3384,3393,3402,3411,3420,3429,3438,3447,3456,3465,3474,3483,3492,3501,3510,3519,3528,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,99,108,117,126,135,144,153,162,171,180,189,198,207,216,225,234,243,252,261,270,279,288,297,306,315,324,333,342,351,360,369,378,387,396,405,414,423,432,441,450,459,468,477,486,495,504,513,522,531,540,549,558,567,576,585,594,603,612,621,630,639,648,657,666,675,684,693,702,711,720,729,738,747,756,765,774,783,792,801,810,819,828,837,846,855,864,873,882,891,900,909,918,927,936,945,954,963,972,981,990,999,1008,1017,1026,1035,1044,1053,1062,1071,1080,1089,1098,1107,1116,1125,1134,1143,1152,1161,1170,1179,1188,1197,1206,1215,1224,1233,1242,1251,1260,1269,1278,1287,1296,1305,1314,1323,1332,1341,1350,1359,1368,1377,1386,1395,1404,1413,1422,1431,1440,1449,1458,1467,1476,1485,1494,1503,1512,1521,1530,1539,1548,1557,1566,1575,1584,1593,1602,1611,1620,1629,1638,1647,1656,1665,1674,1683,1692,1701,1710,1719,1728,1737,1746,1755,1764,1773,1782,1791,1800,1809,1818,1827,1836,1845,1854,1863,1872,1881,1890,1899,1908,1917,1926,1935,1944,1953,1962,1971,1980,1989,1998,2007,2016,2025,2034,2043,2052,2061,2070,2079,2088,2097,2106,2115,2124,2133,2142,2151,2160,2169,2178,2187,2196,2205,2214,2223,2232,2241,2250,2259,2268,2277,2286,2295,2304,2313,2322,2331,2340,2349,2358,2367,2376,2385,2394,2403,2412,2421,2430,2439,2448,2457,2466,2475,2484,2493,2502,2511,2520,2529,2538,2547,2556,2565,2574,2583,2592,2601,2610,2619,2628,2637,2646,2655,2664,2673,2682,2691,2700,2709,2718,2727,2736,2745,2754,2763,2772,2781,2790,2799,2808,2817,2826,2835,2844,2853,2862,2871,2880,2889,2898,2907,2916,2925,2934,2943,2952,2961,2970,2979,2988,2997,3006,3015,3024,3033,3042,3051,3060,3069,3078,3087,3096,3105,3114,3123,3132,3141,3150,3159,3168,3177,3186,3195,3204,3213,3222,3231,3240,3249,3258,3267,3276,3285,3294,3303,3312,3321,3330,3339,3348,3357,3366,3375,3384,3393,3402,3411,3420,3429,3438,3447,3456,3465,3474,3483,3492,3501,3510,3519,3528,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,120,130,140,150,160,170,180,190,200,210,220,230,240,250,260,270,280,290,300,310,320,330,340,350,360,370,380,390,400,410,420,430,440,450,460,470,480,490,500,510,520,530,540,550,560,570,580,590,600,610,620,630,640,650,660,670,680,690,700,710,720,730,740,750,760,770,780,790,800,810,820,830,840,850,860,870,880,890,900,910,920,930,940,950,960,970,980,990,1000,1010,1020,1030,1040,1050,1060,1070,1080,1090,1100,1110,1120,1130,1140,1150,1160,1170,1180,1190,1200,1210,1220,1230,1240,1250,1260,1270,1280,1290,1300,1310,1320,1330,1340,1350,1360,1370,1380,1390,1400,1410,1420,1430,1440,1450,1460,1470,1480,1490,1500,1510,1520,1530,1540,1550,1560,1570,1580,1590,1600,1610,1620,1630,1640,1650,1660,1670,1680,1690,1700,1710,1720,1730,1740,1750,1760,1770,1780,1790,1800,1810,1820,1830,1840,1850,1860,1870,1880,1890,1900,1910,1920,1930,1940,1950,1960,1970,1980,1990,2000,2010,2020,2030,2040,2050,2060,2070,2080,2090,2100,2110,2120,2130,2140,2150,2160,2170,2180,2190,2200,2210,2220,2230,2240,2250,2260,2270,2280,2290,2300,2310,2320,2330,2340,2350,2360,2370,2380,2390,2400,2410,2420,2430,2440,2450,2460,2470,2480,2490,2500,2510,2520,2530,2540,2550,2560,2570,2580,2590,2600,2610,2620,2630,2640,2650,2660,2670,2680,2690,2700,2710,2720,2730,2740,2750,2760,2770,2780,2790,2800,2810,2820,2830,2840,2850,2860,2870,2880,2890,2900,2910,2920,2930,2940,2950,2960,2970,2980,2990,3000,3010,3020,3030,3040,3050,3060,3070,3080,3090,3100,3110,3120,3130,3140,3150,3160,3170,3180,3190,3200,3210,3220,3230,3240,3250,3260,3270,3280,3290,3300,3310,3320,3330,3340,3350,3360,3370,3380,3390,3400,3410,3420,3430,3440,3450,3460,3470,3480,3490,3500,3510,3520,3530,3540,3550,3560,3570,3580,3590,3600,3610,3620,3630,3640,3650,3660,3670,3680,3690,3700,3710,3720,3730,3740,3750,3760,3770,3780,3790,3800,3810,3820,3830,3840,3850,3860,3870,3880,3890,3900,3910,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,120,130,140,150,160,170,180,190,200,210,220,230,240,250,260,270,280,290,300,310,320,330,340,350,360,370,380,390,400,410,420,430,440,450,460,470,480,490,500,510,520,530,540,550,560,570,580,590,600,610,620,630,640,650,660,670,680,690,700,710,720,730,740,750,760,770,780,790,800,810,820,830,840,850,860,870,880,890,900,910,920,930,940,950,960,970,980,990,1000,1010,1020,1030,1040,1050,1060,1070,1080,1090,1100,1110,1120,1130,1140,1150,1160,1170,1180,1190,1200,1210,1220,1230,1240,1250,1260,1270,1280,1290,1300,1310,1320,1330,1340,1350,1360,1370,1380,1390,1400,1410,1420,1430,1440,1450,1460,1470,1480,1490,1500,1510,1520,1530,1540,1550,1560,1570,1580,1590,1600,1610,1620,1630,1640,1650,1660,1670,1680,1690,1700,1710,1720,1730,1740,1750,1760,1770,1780,1790,1800,1810,1820,1830,1840,1850,1860,1870,1880,1890,1900,1910,1920,1930,1940,1950,1960,1970,1980,1990,2000,2010,2020,2030,2040,2050,2060,2070,2080,2090,2100,2110,2120,2130,2140,2150,2160,2170,2180,2190,2200,2210,2220,2230,2240,2250,2260,2270,2280,2290,2300,2310,2320,2330,2340,2350,2360,2370,2380,2390,2400,2410,2420,2430,2440,2450,2460,2470,2480,2490,2500,2510,2520,2530,2540,2550,2560,2570,2580,2590,2600,2610,2620,2630,2640,2650,2660,2670,2680,2690,2700,2710,2720,2730,2740,2750,2760,2770,2780,2790,2800,2810,2820,2830,2840,2850,2860,2870,2880,2890,2900,2910,2920,2930,2940,2950,2960,2970,2980,2990,3000,3010,3020,3030,3040,3050,3060,3070,3080,3090,3100,3110,3120,3130,3140,3150,3160,3170,3180,3190,3200,3210,3220,3230,3240,3250,3260,3270,3280,3290,3300,3310,3320,3330,3340,3350,3360,3370,3380,3390,3400,3410,3420,3430,3440,3450,3460,3470,3480,3490,3500,3510,3520,3530,3540,3550,3560,3570,3580,3590,3600,3610,3620,3630,3640,3650,3660,3670,3680,3690,3700,3710,3720,3730,3740,3750,3760,3770,3780,3790,3800,3810,3820,3830,3840,3850,3860,3870,3880,3890,3900,3910,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,143,154,165,176,187,198,209,220,231,242,253,264,275,286,297,308,319,330,341,352,363,374,385,396,407,418,429,440,451,462,473,484,495,506,517,528,539,550,561,572,583,594,605,616,627,638,649,660,671,682,693,704,715,726,737,748,759,770,781,792,803,814,825,836,847,858,869,880,891,902,913,924,935,946,957,968,979,990,1001,1012,1023,1034,1045,1056,1067,1078,1089,1100,1111,1122,1133,1144,1155,1166,1177,1188,1199,1210,1221,1232,1243,1254,1265,1276,1287,1298,1309,1320,1331,1342,1353,1364,1375,1386,1397,1408,1419,1430,1441,1452,1463,1474,1485,1496,1507,1518,1529,1540,1551,1562,1573,1584,1595,1606,1617,1628,1639,1650,1661,1672,1683,1694,1705,1716,1727,1738,1749,1760,1771,1782,1793,1804,1815,1826,1837,1848,1859,1870,1881,1892,1903,1914,1925,1936,1947,1958,1969,1980,1991,2002,2013,2024,2035,2046,2057,2068,2079,2090,2101,2112,2123,2134,2145,2156,2167,2178,2189,2200,2211,2222,2233,2244,2255,2266,2277,2288,2299,2310,2321,2332,2343,2354,2365,2376,2387,2398,2409,2420,2431,2442,2453,2464,2475,2486,2497,2508,2519,2530,2541,2552,2563,2574,2585,2596,2607,2618,2629,2640,2651,2662,2673,2684,2695,2706,2717,2728,2739,2750,2761,2772,2783,2794,2805,2816,2827,2838,2849,2860,2871,2882,2893,2904,2915,2926,2937,2948,2959,2970,2981,2992,3003,3014,3025,3036,3047,3058,3069,3080,3091,3102,3113,3124,3135,3146,3157,3168,3179,3190,3201,3212,3223,3234,3245,3256,3267,3278,3289,3300,3311,3322,3333,3344,3355,3366,3377,3388,3399,3410,3421,3432,3443,3454,3465,3476,3487,3498,3509,3520,3531,3542,3553,3564,3575,3586,3597,3608,3619,3630,3641,3652,3663,3674,3685,3696,3707,3718,3729,3740,3751,3762,3773,3784,3795,3806,3817,3828,3839,3850,3861,3872,3883,3894,3905,3916,3927,3938,3949,3960,3971,3982,3993,4004,4015,4026,4037,4048,4059,4070,4081,4092,4103,4114,4125,4136,4147,4158,4169,4180,4191,4202,4213,4224,4235,4246,4257,4268,4279,4290,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,143,154,165,176,187,198,209,220,231,242,253,264,275,286,297,308,319,330,341,352,363,374,385,396,407,418,429,440,451,462,473,484,495,506,517,528,539,550,561,572,583,594,605,616,627,638,649,660,671,682,693,704,715,726,737,748,759,770,781,792,803,814,825,836,847,858,869,880,891,902,913,924,935,946,957,968,979,990,1001,1012,1023,1034,1045,1056,1067,1078,1089,1100,1111,1122,1133,1144,1155,1166,1177,1188,1199,1210,1221,1232,1243,1254,1265,1276,1287,1298,1309,1320,1331,1342,1353,1364,1375,1386,1397,1408,1419,1430,1441,1452,1463,1474,1485,1496,1507,1518,1529,1540,1551,1562,1573,1584,1595,1606,1617,1628,1639,1650,1661,1672,1683,1694,1705,1716,1727,1738,1749,1760,1771,1782,1793,1804,1815,1826,1837,1848,1859,1870,1881,1892,1903,1914,1925,1936,1947,1958,1969,1980,1991,2002,2013,2024,2035,2046,2057,2068,2079,2090,2101,2112,2123,2134,2145,2156,2167,2178,2189,2200,2211,2222,2233,2244,2255,2266,2277,2288,2299,2310,2321,2332,2343,2354,2365,2376,2387,2398,2409,2420,2431,2442,2453,2464,2475,2486,2497,2508,2519,2530,2541,2552,2563,2574,2585,2596,2607,2618,2629,2640,2651,2662,2673,2684,2695,2706,2717,2728,2739,2750,2761,2772,2783,2794,2805,2816,2827,2838,2849,2860,2871,2882,2893,2904,2915,2926,2937,2948,2959,2970,2981,2992,3003,3014,3025,3036,3047,3058,3069,3080,3091,3102,3113,3124,3135,3146,3157,3168,3179,3190,3201,3212,3223,3234,3245,3256,3267,3278,3289,3300,3311,3322,3333,3344,3355,3366,3377,3388,3399,3410,3421,3432,3443,3454,3465,3476,3487,3498,3509,3520,3531,3542,3553,3564,3575,3586,3597,3608,3619,3630,3641,3652,3663,3674,3685,3696,3707,3718,3729,3740,3751,3762,3773,3784,3795,3806,3817,3828,3839,3850,3861,3872,3883,3894,3905,3916,3927,3938,3949,3960,3971,3982,3993,4004,4015,4026,4037,4048,4059,4070,4081,4092,4103,4114,4125,4136,4147,4158,4169,4180,4191,4202,4213,4224,4235,4246,4257,4268,4279,4290,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,168,180,192,204,216,228,240,252,264,276,288,300,312,324,336,348,360,372,384,396,408,420,432,444,456,468,480,492,504,516,528,540,552,564,576,588,600,612,624,636,648,660,672,684,696,708,720,732,744,756,768,780,792,804,816,828,840,852,864,876,888,900,912,924,936,948,960,972,984,996,1008,1020,1032,1044,1056,1068,1080,1092,1104,1116,1128,1140,1152,1164,1176,1188,1200,1212,1224,1236,1248,1260,1272,1284,1296,1308,1320,1332,1344,1356,1368,1380,1392,1404,1416,1428,1440,1452,1464,1476,1488,1500,1512,1524,1536,1548,1560,1572,1584,1596,1608,1620,1632,1644,1656,1668,1680,1692,1704,1716,1728,1740,1752,1764,1776,1788,1800,1812,1824,1836,1848,1860,1872,1884,1896,1908,1920,1932,1944,1956,1968,1980,1992,2004,2016,2028,2040,2052,2064,2076,2088,2100,2112,2124,2136,2148,2160,2172,2184,2196,2208,2220,2232,2244,2256,2268,2280,2292,2304,2316,2328,2340,2352,2364,2376,2388,2400,2412,2424,2436,2448,2460,2472,2484,2496,2508,2520,2532,2544,2556,2568,2580,2592,2604,2616,2628,2640,2652,2664,2676,2688,2700,2712,2724,2736,2748,2760,2772,2784,2796,2808,2820,2832,2844,2856,2868,2880,2892,2904,2916,2928,2940,2952,2964,2976,2988,3000,3012,3024,3036,3048,3060,3072,3084,3096,3108,3120,3132,3144,3156,3168,3180,3192,3204,3216,3228,3240,3252,3264,3276,3288,3300,3312,3324,3336,3348,3360,3372,3384,3396,3408,3420,3432,3444,3456,3468,3480,3492,3504,3516,3528,3540,3552,3564,3576,3588,3600,3612,3624,3636,3648,3660,3672,3684,3696,3708,3720,3732,3744,3756,3768,3780,3792,3804,3816,3828,3840,3852,3864,3876,3888,3900,3912,3924,3936,3948,3960,3972,3984,3996,4008,4020,4032,4044,4056,4068,4080,4092,4104,4116,4128,4140,4152,4164,4176,4188,4200,4212,4224,4236,4248,4260,4272,4284,4296,4308,4320,4332,4344,4356,4368,4380,4392,4404,4416,4428,4440,4452,4464,4476,4488,4500,4512,4524,4536,4548,4560,4572,4584,4596,4608,4620,4632,4644,4656,4668,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,168,180,192,204,216,228,240,252,264,276,288,300,312,324,336,348,360,372,384,396,408,420,432,444,456,468,480,492,504,516,528,540,552,564,576,588,600,612,624,636,648,660,672,684,696,708,720,732,744,756,768,780,792,804,816,828,840,852,864,876,888,900,912,924,936,948,960,972,984,996,1008,1020,1032,1044,1056,1068,1080,1092,1104,1116,1128,1140,1152,1164,1176,1188,1200,1212,1224,1236,1248,1260,1272,1284,1296,1308,1320,1332,1344,1356,1368,1380,1392,1404,1416,1428,1440,1452,1464,1476,1488,1500,1512,1524,1536,1548,1560,1572,1584,1596,1608,1620,1632,1644,1656,1668,1680,1692,1704,1716,1728,1740,1752,1764,1776,1788,1800,1812,1824,1836,1848,1860,1872,1884,1896,1908,1920,1932,1944,1956,1968,1980,1992,2004,2016,2028,2040,2052,2064,2076,2088,2100,2112,2124,2136,2148,2160,2172,2184,2196,2208,2220,2232,2244,2256,2268,2280,2292,2304,2316,2328,2340,2352,2364,2376,2388,2400,2412,2424,2436,2448,2460,2472,2484,2496,2508,2520,2532,2544,2556,2568,2580,2592,2604,2616,2628,2640,2652,2664,2676,2688,2700,2712,2724,2736,2748,2760,2772,2784,2796,2808,2820,2832,2844,2856,2868,2880,2892,2904,2916,2928,2940,2952,2964,2976,2988,3000,3012,3024,3036,3048,3060,3072,3084,3096,3108,3120,3132,3144,3156,3168,3180,3192,3204,3216,3228,3240,3252,3264,3276,3288,3300,3312,3324,3336,3348,3360,3372,3384,3396,3408,3420,3432,3444,3456,3468,3480,3492,3504,3516,3528,3540,3552,3564,3576,3588,3600,3612,3624,3636,3648,3660,3672,3684,3696,3708,3720,3732,3744,3756,3768,3780,3792,3804,3816,3828,3840,3852,3864,3876,3888,3900,3912,3924,3936,3948,3960,3972,3984,3996,4008,4020,4032,4044,4056,4068,4080,4092,4104,4116,4128,4140,4152,4164,4176,4188,4200,4212,4224,4236,4248,4260,4272,4284,4296,4308,4320,4332,4344,4356,4368,4380,4392,4404,4416,4428,4440,4452,4464,4476,4488,4500,4512,4524,4536,4548,4560,4572,4584,4596,4608,4620,4632,4644,4656,4668,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,195,208,221,234,247,260,273,286,299,312,325,338,351,364,377,390,403,416,429,442,455,468,481,494,507,520,533,546,559,572,585,598,611,624,637,650,663,676,689,702,715,728,741,754,767,780,793,806,819,832,845,858,871,884,897,910,923,936,949,962,975,988,1001,1014,1027,1040,1053,1066,1079,1092,1105,1118,1131,1144,1157,1170,1183,1196,1209,1222,1235,1248,1261,1274,1287,1300,1313,1326,1339,1352,1365,1378,1391,1404,1417,1430,1443,1456,1469,1482,1495,1508,1521,1534,1547,1560,1573,1586,1599,1612,1625,1638,1651,1664,1677,1690,1703,1716,1729,1742,1755,1768,1781,1794,1807,1820,1833,1846,1859,1872,1885,1898,1911,1924,1937,1950,1963,1976,1989,2002,2015,2028,2041,2054,2067,2080,2093,2106,2119,2132,2145,2158,2171,2184,2197,2210,2223,2236,2249,2262,2275,2288,2301,2314,2327,2340,2353,2366,2379,2392,2405,2418,2431,2444,2457,2470,2483,2496,2509,2522,2535,2548,2561,2574,2587,2600,2613,2626,2639,2652,2665,2678,2691,2704,2717,2730,2743,2756,2769,2782,2795,2808,2821,2834,2847,2860,2873,2886,2899,2912,2925,2938,2951,2964,2977,2990,3003,3016,3029,3042,3055,3068,3081,3094,3107,3120,3133,3146,3159,3172,3185,3198,3211,3224,3237,3250,3263,3276,3289,3302,3315,3328,3341,3354,3367,3380,3393,3406,3419,3432,3445,3458,3471,3484,3497,3510,3523,3536,3549,3562,3575,3588,3601,3614,3627,3640,3653,3666,3679,3692,3705,3718,3731,3744,3757,3770,3783,3796,3809,3822,3835,3848,3861,3874,3887,3900,3913,3926,3939,3952,3965,3978,3991,4004,4017,4030,4043,4056,4069,4082,4095,4108,4121,4134,4147,4160,4173,4186,4199,4212,4225,4238,4251,4264,4277,4290,4303,4316,4329,4342,4355,4368,4381,4394,4407,4420,4433,4446,4459,4472,4485,4498,4511,4524,4537,4550,4563,4576,4589,4602,4615,4628,4641,4654,4667,4680,4693,4706,4719,4732,4745,4758,4771,4784,4797,4810,4823,4836,4849,4862,4875,4888,4901,4914,4927,4940,4953,4966,4979,4992,5005,5018,5031,5044,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,195,208,221,234,247,260,273,286,299,312,325,338,351,364,377,390,403,416,429,442,455,468,481,494,507,520,533,546,559,572,585,598,611,624,637,650,663,676,689,702,715,728,741,754,767,780,793,806,819,832,845,858,871,884,897,910,923,936,949,962,975,988,1001,1014,1027,1040,1053,1066,1079,1092,1105,1118,1131,1144,1157,1170,1183,1196,1209,1222,1235,1248,1261,1274,1287,1300,1313,1326,1339,1352,1365,1378,1391,1404,1417,1430,1443,1456,1469,1482,1495,1508,1521,1534,1547,1560,1573,1586,1599,1612,1625,1638,1651,1664,1677,1690,1703,1716,1729,1742,1755,1768,1781,1794,1807,1820,1833,1846,1859,1872,1885,1898,1911,1924,1937,1950,1963,1976,1989,2002,2015,2028,2041,2054,2067,2080,2093,2106,2119,2132,2145,2158,2171,2184,2197,2210,2223,2236,2249,2262,2275,2288,2301,2314,2327,2340,2353,2366,2379,2392,2405,2418,2431,2444,2457,2470,2483,2496,2509,2522,2535,2548,2561,2574,2587,2600,2613,2626,2639,2652,2665,2678,2691,2704,2717,2730,2743,2756,2769,2782,2795,2808,2821,2834,2847,2860,2873,2886,2899,2912,2925,2938,2951,2964,2977,2990,3003,3016,3029,3042,3055,3068,3081,3094,3107,3120,3133,3146,3159,3172,3185,3198,3211,3224,3237,3250,3263,3276,3289,3302,3315,3328,3341,3354,3367,3380,3393,3406,3419,3432,3445,3458,3471,3484,3497,3510,3523,3536,3549,3562,3575,3588,3601,3614,3627,3640,3653,3666,3679,3692,3705,3718,3731,3744,3757,3770,3783,3796,3809,3822,3835,3848,3861,3874,3887,3900,3913,3926,3939,3952,3965,3978,3991,4004,4017,4030,4043,4056,4069,4082,4095,4108,4121,4134,4147,4160,4173,4186,4199,4212,4225,4238,4251,4264,4277,4290,4303,4316,4329,4342,4355,4368,4381,4394,4407,4420,4433,4446,4459,4472,4485,4498,4511,4524,4537,4550,4563,4576,4589,4602,4615,4628,4641,4654,4667,4680,4693,4706,4719,4732,4745,4758,4771,4784,4797,4810,4823,4836,4849,4862,4875,4888,4901,4914,4927,4940,4953,4966,4979,4992,5005,5018,5031,5044,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,224,238,252,266,280,294,308,322,336,350,364,378,392,406,420,434,448,462,476,490,504,518,532,546,560,574,588,602,616,630,644,658,672,686,700,714,728,742,756,770,784,798,812,826,840,854,868,882,896,910,924,938,952,966,980,994,1008,1022,1036,1050,1064,1078,1092,1106,1120,1134,1148,1162,1176,1190,1204,1218,1232,1246,1260,1274,1288,1302,1316,1330,1344,1358,1372,1386,1400,1414,1428,1442,1456,1470,1484,1498,1512,1526,1540,1554,1568,1582,1596,1610,1624,1638,1652,1666,1680,1694,1708,1722,1736,1750,1764,1778,1792,1806,1820,1834,1848,1862,1876,1890,1904,1918,1932,1946,1960,1974,1988,2002,2016,2030,2044,2058,2072,2086,2100,2114,2128,2142,2156,2170,2184,2198,2212,2226,2240,2254,2268,2282,2296,2310,2324,2338,2352,2366,2380,2394,2408,2422,2436,2450,2464,2478,2492,2506,2520,2534,2548,2562,2576,2590,2604,2618,2632,2646,2660,2674,2688,2702,2716,2730,2744,2758,2772,2786,2800,2814,2828,2842,2856,2870,2884,2898,2912,2926,2940,2954,2968,2982,2996,3010,3024,3038,3052,3066,3080,3094,3108,3122,3136,3150,3164,3178,3192,3206,3220,3234,3248,3262,3276,3290,3304,3318,3332,3346,3360,3374,3388,3402,3416,3430,3444,3458,3472,3486,3500,3514,3528,3542,3556,3570,3584,3598,3612,3626,3640,3654,3668,3682,3696,3710,3724,3738,3752,3766,3780,3794,3808,3822,3836,3850,3864,3878,3892,3906,3920,3934,3948,3962,3976,3990,4004,4018,4032,4046,4060,4074,4088,4102,4116,4130,4144,4158,4172,4186,4200,4214,4228,4242,4256,4270,4284,4298,4312,4326,4340,4354,4368,4382,4396,4410,4424,4438,4452,4466,4480,4494,4508,4522,4536,4550,4564,4578,4592,4606,4620,4634,4648,4662,4676,4690,4704,4718,4732,4746,4760,4774,4788,4802,4816,4830,4844,4858,4872,4886,4900,4914,4928,4942,4956,4970,4984,4998,5012,5026,5040,5054,5068,5082,5096,5110,5124,5138,5152,5166,5180,5194,5208,5222,5236,5250,5264,5278,5292,5306,5320,5334,5348,5362,5376,5390,5404,5418,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,224,238,252,266,280,294,308,322,336,350,364,378,392,406,420,434,448,462,476,490,504,518,532,546,560,574,588,602,616,630,644,658,672,686,700,714,728,742,756,770,784,798,812,826,840,854,868,882,896,910,924,938,952,966,980,994,1008,1022,1036,1050,1064,1078,1092,1106,1120,1134,1148,1162,1176,1190,1204,1218,1232,1246,1260,1274,1288,1302,1316,1330,1344,1358,1372,1386,1400,1414,1428,1442,1456,1470,1484,1498,1512,1526,1540,1554,1568,1582,1596,1610,1624,1638,1652,1666,1680,1694,1708,1722,1736,1750,1764,1778,1792,1806,1820,1834,1848,1862,1876,1890,1904,1918,1932,1946,1960,1974,1988,2002,2016,2030,2044,2058,2072,2086,2100,2114,2128,2142,2156,2170,2184,2198,2212,2226,2240,2254,2268,2282,2296,2310,2324,2338,2352,2366,2380,2394,2408,2422,2436,2450,2464,2478,2492,2506,2520,2534,2548,2562,2576,2590,2604,2618,2632,2646,2660,2674,2688,2702,2716,2730,2744,2758,2772,2786,2800,2814,2828,2842,2856,2870,2884,2898,2912,2926,2940,2954,2968,2982,2996,3010,3024,3038,3052,3066,3080,3094,3108,3122,3136,3150,3164,3178,3192,3206,3220,3234,3248,3262,3276,3290,3304,3318,3332,3346,3360,3374,3388,3402,3416,3430,3444,3458,3472,3486,3500,3514,3528,3542,3556,3570,3584,3598,3612,3626,3640,3654,3668,3682,3696,3710,3724,3738,3752,3766,3780,3794,3808,3822,3836,3850,3864,3878,3892,3906,3920,3934,3948,3962,3976,3990,4004,4018,4032,4046,4060,4074,4088,4102,4116,4130,4144,4158,4172,4186,4200,4214,4228,4242,4256,4270,4284,4298,4312,4326,4340,4354,4368,4382,4396,4410,4424,4438,4452,4466,4480,4494,4508,4522,4536,4550,4564,4578,4592,4606,4620,4634,4648,4662,4676,4690,4704,4718,4732,4746,4760,4774,4788,4802,4816,4830,4844,4858,4872,4886,4900,4914,4928,4942,4956,4970,4984,4998,5012,5026,5040,5054,5068,5082,5096,5110,5124,5138,5152,5166,5180,5194,5208,5222,5236,5250,5264,5278,5292,5306,5320,5334,5348,5362,5376,5390,5404,5418,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,255,270,285,300,315,330,345,360,375,390,405,420,435,450,465,480,495,510,525,540,555,570,585,600,615,630,645,660,675,690,705,720,735,750,765,780,795,810,825,840,855,870,885,900,915,930,945,960,975,990,1005,1020,1035,1050,1065,1080,1095,1110,1125,1140,1155,1170,1185,1200,1215,1230,1245,1260,1275,1290,1305,1320,1335,1350,1365,1380,1395,1410,1425,1440,1455,1470,1485,1500,1515,1530,1545,1560,1575,1590,1605,1620,1635,1650,1665,1680,1695,1710,1725,1740,1755,1770,1785,1800,1815,1830,1845,1860,1875,1890,1905,1920,1935,1950,1965,1980,1995,2010,2025,2040,2055,2070,2085,2100,2115,2130,2145,2160,2175,2190,2205,2220,2235,2250,2265,2280,2295,2310,2325,2340,2355,2370,2385,2400,2415,2430,2445,2460,2475,2490,2505,2520,2535,2550,2565,2580,2595,2610,2625,2640,2655,2670,2685,2700,2715,2730,2745,2760,2775,2790,2805,2820,2835,2850,2865,2880,2895,2910,2925,2940,2955,2970,2985,3000,3015,3030,3045,3060,3075,3090,3105,3120,3135,3150,3165,3180,3195,3210,3225,3240,3255,3270,3285,3300,3315,3330,3345,3360,3375,3390,3405,3420,3435,3450,3465,3480,3495,3510,3525,3540,3555,3570,3585,3600,3615,3630,3645,3660,3675,3690,3705,3720,3735,3750,3765,3780,3795,3810,3825,3840,3855,3870,3885,3900,3915,3930,3945,3960,3975,3990,4005,4020,4035,4050,4065,4080,4095,4110,4125,4140,4155,4170,4185,4200,4215,4230,4245,4260,4275,4290,4305,4320,4335,4350,4365,4380,4395,4410,4425,4440,4455,4470,4485,4500,4515,4530,4545,4560,4575,4590,4605,4620,4635,4650,4665,4680,4695,4710,4725,4740,4755,4770,4785,4800,4815,4830,4845,4860,4875,4890,4905,4920,4935,4950,4965,4980,4995,5010,5025,5040,5055,5070,5085,5100,5115,5130,5145,5160,5175,5190,5205,5220,5235,5250,5265,5280,5295,5310,5325,5340,5355,5370,5385,5400,5415,5430,5445,5460,5475,5490,5505,5520,5535,5550,5565,5580,5595,5610,5625,5640,5655,5670,5685,5700,5715,5730,5745,5760,5775,5790,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,255,270,285,300,315,330,345,360,375,390,405,420,435,450,465,480,495,510,525,540,555,570,585,600,615,630,645,660,675,690,705,720,735,750,765,780,795,810,825,840,855,870,885,900,915,930,945,960,975,990,1005,1020,1035,1050,1065,1080,1095,1110,1125,1140,1155,1170,1185,1200,1215,1230,1245,1260,1275,1290,1305,1320,1335,1350,1365,1380,1395,1410,1425,1440,1455,1470,1485,1500,1515,1530,1545,1560,1575,1590,1605,1620,1635,1650,1665,1680,1695,1710,1725,1740,1755,1770,1785,1800,1815,1830,1845,1860,1875,1890,1905,1920,1935,1950,1965,1980,1995,2010,2025,2040,2055,2070,2085,2100,2115,2130,2145,2160,2175,2190,2205,2220,2235,2250,2265,2280,2295,2310,2325,2340,2355,2370,2385,2400,2415,2430,2445,2460,2475,2490,2505,2520,2535,2550,2565,2580,2595,2610,2625,2640,2655,2670,2685,2700,2715,2730,2745,2760,2775,2790,2805,2820,2835,2850,2865,2880,2895,2910,2925,2940,2955,2970,2985,3000,3015,3030,3045,3060,3075,3090,3105,3120,3135,3150,3165,3180,3195,3210,3225,3240,3255,3270,3285,3300,3315,3330,3345,3360,3375,3390,3405,3420,3435,3450,3465,3480,3495,3510,3525,3540,3555,3570,3585,3600,3615,3630,3645,3660,3675,3690,3705,3720,3735,3750,3765,3780,3795,3810,3825,3840,3855,3870,3885,3900,3915,3930,3945,3960,3975,3990,4005,4020,4035,4050,4065,4080,4095,4110,4125,4140,4155,4170,4185,4200,4215,4230,4245,4260,4275,4290,4305,4320,4335,4350,4365,4380,4395,4410,4425,4440,4455,4470,4485,4500,4515,4530,4545,4560,4575,4590,4605,4620,4635,4650,4665,4680,4695,4710,4725,4740,4755,4770,4785,4800,4815,4830,4845,4860,4875,4890,4905,4920,4935,4950,4965,4980,4995,5010,5025,5040,5055,5070,5085,5100,5115,5130,5145,5160,5175,5190,5205,5220,5235,5250,5265,5280,5295,5310,5325,5340,5355,5370,5385,5400,5415,5430,5445,5460,5475,5490,5505,5520,5535,5550,5565,5580,5595,5610,5625,5640,5655,5670,5685,5700,5715,5730,5745,5760,5775,5790,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,288,304,320,336,352,368,384,400,416,432,448,464,480,496,512,528,544,560,576,592,608,624,640,656,672,688,704,720,736,752,768,784,800,816,832,848,864,880,896,912,928,944,960,976,992,1008,1024,1040,1056,1072,1088,1104,1120,1136,1152,1168,1184,1200,1216,1232,1248,1264,1280,1296,1312,1328,1344,1360,1376,1392,1408,1424,1440,1456,1472,1488,1504,1520,1536,1552,1568,1584,1600,1616,1632,1648,1664,1680,1696,1712,1728,1744,1760,1776,1792,1808,1824,1840,1856,1872,1888,1904,1920,1936,1952,1968,1984,2000,2016,2032,2048,2064,2080,2096,2112,2128,2144,2160,2176,2192,2208,2224,2240,2256,2272,2288,2304,2320,2336,2352,2368,2384,2400,2416,2432,2448,2464,2480,2496,2512,2528,2544,2560,2576,2592,2608,2624,2640,2656,2672,2688,2704,2720,2736,2752,2768,2784,2800,2816,2832,2848,2864,2880,2896,2912,2928,2944,2960,2976,2992,3008,3024,3040,3056,3072,3088,3104,3120,3136,3152,3168,3184,3200,3216,3232,3248,3264,3280,3296,3312,3328,3344,3360,3376,3392,3408,3424,3440,3456,3472,3488,3504,3520,3536,3552,3568,3584,3600,3616,3632,3648,3664,3680,3696,3712,3728,3744,3760,3776,3792,3808,3824,3840,3856,3872,3888,3904,3920,3936,3952,3968,3984,4000,4016,4032,4048,4064,4080,4096,4112,4128,4144,4160,4176,4192,4208,4224,4240,4256,4272,4288,4304,4320,4336,4352,4368,4384,4400,4416,4432,4448,4464,4480,4496,4512,4528,4544,4560,4576,4592,4608,4624,4640,4656,4672,4688,4704,4720,4736,4752,4768,4784,4800,4816,4832,4848,4864,4880,4896,4912,4928,4944,4960,4976,4992,5008,5024,5040,5056,5072,5088,5104,5120,5136,5152,5168,5184,5200,5216,5232,5248,5264,5280,5296,5312,5328,5344,5360,5376,5392,5408,5424,5440,5456,5472,5488,5504,5520,5536,5552,5568,5584,5600,5616,5632,5648,5664,5680,5696,5712,5728,5744,5760,5776,5792,5808,5824,5840,5856,5872,5888,5904,5920,5936,5952,5968,5984,6000,6016,6032,6048,6064,6080,6096,6112,6128,6144,6160,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,288,304,320,336,352,368,384,400,416,432,448,464,480,496,512,528,544,560,576,592,608,624,640,656,672,688,704,720,736,752,768,784,800,816,832,848,864,880,896,912,928,944,960,976,992,1008,1024,1040,1056,1072,1088,1104,1120,1136,1152,1168,1184,1200,1216,1232,1248,1264,1280,1296,1312,1328,1344,1360,1376,1392,1408,1424,1440,1456,1472,1488,1504,1520,1536,1552,1568,1584,1600,1616,1632,1648,1664,1680,1696,1712,1728,1744,1760,1776,1792,1808,1824,1840,1856,1872,1888,1904,1920,1936,1952,1968,1984,2000,2016,2032,2048,2064,2080,2096,2112,2128,2144,2160,2176,2192,2208,2224,2240,2256,2272,2288,2304,2320,2336,2352,2368,2384,2400,2416,2432,2448,2464,2480,2496,2512,2528,2544,2560,2576,2592,2608,2624,2640,2656,2672,2688,2704,2720,2736,2752,2768,2784,2800,2816,2832,2848,2864,2880,2896,2912,2928,2944,2960,2976,2992,3008,3024,3040,3056,3072,3088,3104,3120,3136,3152,3168,3184,3200,3216,3232,3248,3264,3280,3296,3312,3328,3344,3360,3376,3392,3408,3424,3440,3456,3472,3488,3504,3520,3536,3552,3568,3584,3600,3616,3632,3648,3664,3680,3696,3712,3728,3744,3760,3776,3792,3808,3824,3840,3856,3872,3888,3904,3920,3936,3952,3968,3984,4000,4016,4032,4048,4064,4080,4096,4112,4128,4144,4160,4176,4192,4208,4224,4240,4256,4272,4288,4304,4320,4336,4352,4368,4384,4400,4416,4432,4448,4464,4480,4496,4512,4528,4544,4560,4576,4592,4608,4624,4640,4656,4672,4688,4704,4720,4736,4752,4768,4784,4800,4816,4832,4848,4864,4880,4896,4912,4928,4944,4960,4976,4992,5008,5024,5040,5056,5072,5088,5104,5120,5136,5152,5168,5184,5200,5216,5232,5248,5264,5280,5296,5312,5328,5344,5360,5376,5392,5408,5424,5440,5456,5472,5488,5504,5520,5536,5552,5568,5584,5600,5616,5632,5648,5664,5680,5696,5712,5728,5744,5760,5776,5792,5808,5824,5840,5856,5872,5888,5904,5920,5936,5952,5968,5984,6000,6016,6032,6048,6064,6080,6096,6112,6128,6144,6160,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,323,340,357,374,391,408,425,442,459,476,493,510,527,544,561,578,595,612,629,646,663,680,697,714,731,748,765,782,799,816,833,850,867,884,901,918,935,952,969,986,1003,1020,1037,1054,1071,1088,1105,1122,1139,1156,1173,1190,1207,1224,1241,1258,1275,1292,1309,1326,1343,1360,1377,1394,1411,1428,1445,1462,1479,1496,1513,1530,1547,1564,1581,1598,1615,1632,1649,1666,1683,1700,1717,1734,1751,1768,1785,1802,1819,1836,1853,1870,1887,1904,1921,1938,1955,1972,1989,2006,2023,2040,2057,2074,2091,2108,2125,2142,2159,2176,2193,2210,2227,2244,2261,2278,2295,2312,2329,2346,2363,2380,2397,2414,2431,2448,2465,2482,2499,2516,2533,2550,2567,2584,2601,2618,2635,2652,2669,2686,2703,2720,2737,2754,2771,2788,2805,2822,2839,2856,2873,2890,2907,2924,2941,2958,2975,2992,3009,3026,3043,3060,3077,3094,3111,3128,3145,3162,3179,3196,3213,3230,3247,3264,3281,3298,3315,3332,3349,3366,3383,3400,3417,3434,3451,3468,3485,3502,3519,3536,3553,3570,3587,3604,3621,3638,3655,3672,3689,3706,3723,3740,3757,3774,3791,3808,3825,3842,3859,3876,3893,3910,3927,3944,3961,3978,3995,4012,4029,4046,4063,4080,4097,4114,4131,4148,4165,4182,4199,4216,4233,4250,4267,4284,4301,4318,4335,4352,4369,4386,4403,4420,4437,4454,4471,4488,4505,4522,4539,4556,4573,4590,4607,4624,4641,4658,4675,4692,4709,4726,4743,4760,4777,4794,4811,4828,4845,4862,4879,4896,4913,4930,4947,4964,4981,4998,5015,5032,5049,5066,5083,5100,5117,5134,5151,5168,5185,5202,5219,5236,5253,5270,5287,5304,5321,5338,5355,5372,5389,5406,5423,5440,5457,5474,5491,5508,5525,5542,5559,5576,5593,5610,5627,5644,5661,5678,5695,5712,5729,5746,5763,5780,5797,5814,5831,5848,5865,5882,5899,5916,5933,5950,5967,5984,6001,6018,6035,6052,6069,6086,6103,6120,6137,6154,6171,6188,6205,6222,6239,6256,6273,6290,6307,6324,6341,6358,6375,6392,6409,6426,6443,6460,6477,6494,6511,6528,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,323,340,357,374,391,408,425,442,459,476,493,510,527,544,561,578,595,612,629,646,663,680,697,714,731,748,765,782,799,816,833,850,867,884,901,918,935,952,969,986,1003,1020,1037,1054,1071,1088,1105,1122,1139,1156,1173,1190,1207,1224,1241,1258,1275,1292,1309,1326,1343,1360,1377,1394,1411,1428,1445,1462,1479,1496,1513,1530,1547,1564,1581,1598,1615,1632,1649,1666,1683,1700,1717,1734,1751,1768,1785,1802,1819,1836,1853,1870,1887,1904,1921,1938,1955,1972,1989,2006,2023,2040,2057,2074,2091,2108,2125,2142,2159,2176,2193,2210,2227,2244,2261,2278,2295,2312,2329,2346,2363,2380,2397,2414,2431,2448,2465,2482,2499,2516,2533,2550,2567,2584,2601,2618,2635,2652,2669,2686,2703,2720,2737,2754,2771,2788,2805,2822,2839,2856,2873,2890,2907,2924,2941,2958,2975,2992,3009,3026,3043,3060,3077,3094,3111,3128,3145,3162,3179,3196,3213,3230,3247,3264,3281,3298,3315,3332,3349,3366,3383,3400,3417,3434,3451,3468,3485,3502,3519,3536,3553,3570,3587,3604,3621,3638,3655,3672,3689,3706,3723,3740,3757,3774,3791,3808,3825,3842,3859,3876,3893,3910,3927,3944,3961,3978,3995,4012,4029,4046,4063,4080,4097,4114,4131,4148,4165,4182,4199,4216,4233,4250,4267,4284,4301,4318,4335,4352,4369,4386,4403,4420,4437,4454,4471,4488,4505,4522,4539,4556,4573,4590,4607,4624,4641,4658,4675,4692,4709,4726,4743,4760,4777,4794,4811,4828,4845,4862,4879,4896,4913,4930,4947,4964,4981,4998,5015,5032,5049,5066,5083,5100,5117,5134,5151,5168,5185,5202,5219,5236,5253,5270,5287,5304,5321,5338,5355,5372,5389,5406,5423,5440,5457,5474,5491,5508,5525,5542,5559,5576,5593,5610,5627,5644,5661,5678,5695,5712,5729,5746,5763,5780,5797,5814,5831,5848,5865,5882,5899,5916,5933,5950,5967,5984,6001,6018,6035,6052,6069,6086,6103,6120,6137,6154,6171,6188,6205,6222,6239,6256,6273,6290,6307,6324,6341,6358,6375,6392,6409,6426,6443,6460,6477,6494,6511,6528,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,360,378,396,414,432,450,468,486,504,522,540,558,576,594,612,630,648,666,684,702,720,738,756,774,792,810,828,846,864,882,900,918,936,954,972,990,1008,1026,1044,1062,1080,1098,1116,1134,1152,1170,1188,1206,1224,1242,1260,1278,1296,1314,1332,1350,1368,1386,1404,1422,1440,1458,1476,1494,1512,1530,1548,1566,1584,1602,1620,1638,1656,1674,1692,1710,1728,1746,1764,1782,1800,1818,1836,1854,1872,1890,1908,1926,1944,1962,1980,1998,2016,2034,2052,2070,2088,2106,2124,2142,2160,2178,2196,2214,2232,2250,2268,2286,2304,2322,2340,2358,2376,2394,2412,2430,2448,2466,2484,2502,2520,2538,2556,2574,2592,2610,2628,2646,2664,2682,2700,2718,2736,2754,2772,2790,2808,2826,2844,2862,2880,2898,2916,2934,2952,2970,2988,3006,3024,3042,3060,3078,3096,3114,3132,3150,3168,3186,3204,3222,3240,3258,3276,3294,3312,3330,3348,3366,3384,3402,3420,3438,3456,3474,3492,3510,3528,3546,3564,3582,3600,3618,3636,3654,3672,3690,3708,3726,3744,3762,3780,3798,3816,3834,3852,3870,3888,3906,3924,3942,3960,3978,3996,4014,4032,4050,4068,4086,4104,4122,4140,4158,4176,4194,4212,4230,4248,4266,4284,4302,4320,4338,4356,4374,4392,4410,4428,4446,4464,4482,4500,4518,4536,4554,4572,4590,4608,4626,4644,4662,4680,4698,4716,4734,4752,4770,4788,4806,4824,4842,4860,4878,4896,4914,4932,4950,4968,4986,5004,5022,5040,5058,5076,5094,5112,5130,5148,5166,5184,5202,5220,5238,5256,5274,5292,5310,5328,5346,5364,5382,5400,5418,5436,5454,5472,5490,5508,5526,5544,5562,5580,5598,5616,5634,5652,5670,5688,5706,5724,5742,5760,5778,5796,5814,5832,5850,5868,5886,5904,5922,5940,5958,5976,5994,6012,6030,6048,6066,6084,6102,6120,6138,6156,6174,6192,6210,6228,6246,6264,6282,6300,6318,6336,6354,6372,6390,6408,6426,6444,6462,6480,6498,6516,6534,6552,6570,6588,6606,6624,6642,6660,6678,6696,6714,6732,6750,6768,6786,6804,6822,6840,6858,6876,6894,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,360,378,396,414,432,450,468,486,504,522,540,558,576,594,612,630,648,666,684,702,720,738,756,774,792,810,828,846,864,882,900,918,936,954,972,990,1008,1026,1044,1062,1080,1098,1116,1134,1152,1170,1188,1206,1224,1242,1260,1278,1296,1314,1332,1350,1368,1386,1404,1422,1440,1458,1476,1494,1512,1530,1548,1566,1584,1602,1620,1638,1656,1674,1692,1710,1728,1746,1764,1782,1800,1818,1836,1854,1872,1890,1908,1926,1944,1962,1980,1998,2016,2034,2052,2070,2088,2106,2124,2142,2160,2178,2196,2214,2232,2250,2268,2286,2304,2322,2340,2358,2376,2394,2412,2430,2448,2466,2484,2502,2520,2538,2556,2574,2592,2610,2628,2646,2664,2682,2700,2718,2736,2754,2772,2790,2808,2826,2844,2862,2880,2898,2916,2934,2952,2970,2988,3006,3024,3042,3060,3078,3096,3114,3132,3150,3168,3186,3204,3222,3240,3258,3276,3294,3312,3330,3348,3366,3384,3402,3420,3438,3456,3474,3492,3510,3528,3546,3564,3582,3600,3618,3636,3654,3672,3690,3708,3726,3744,3762,3780,3798,3816,3834,3852,3870,3888,3906,3924,3942,3960,3978,3996,4014,4032,4050,4068,4086,4104,4122,4140,4158,4176,4194,4212,4230,4248,4266,4284,4302,4320,4338,4356,4374,4392,4410,4428,4446,4464,4482,4500,4518,4536,4554,4572,4590,4608,4626,4644,4662,4680,4698,4716,4734,4752,4770,4788,4806,4824,4842,4860,4878,4896,4914,4932,4950,4968,4986,5004,5022,5040,5058,5076,5094,5112,5130,5148,5166,5184,5202,5220,5238,5256,5274,5292,5310,5328,5346,5364,5382,5400,5418,5436,5454,5472,5490,5508,5526,5544,5562,5580,5598,5616,5634,5652,5670,5688,5706,5724,5742,5760,5778,5796,5814,5832,5850,5868,5886,5904,5922,5940,5958,5976,5994,6012,6030,6048,6066,6084,6102,6120,6138,6156,6174,6192,6210,6228,6246,6264,6282,6300,6318,6336,6354,6372,6390,6408,6426,6444,6462,6480,6498,6516,6534,6552,6570,6588,6606,6624,6642,6660,6678,6696,6714,6732,6750,6768,6786,6804,6822,6840,6858,6876,6894,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,399,418,437,456,475,494,513,532,551,570,589,608,627,646,665,684,703,722,741,760,779,798,817,836,855,874,893,912,931,950,969,988,1007,1026,1045,1064,1083,1102,1121,1140,1159,1178,1197,1216,1235,1254,1273,1292,1311,1330,1349,1368,1387,1406,1425,1444,1463,1482,1501,1520,1539,1558,1577,1596,1615,1634,1653,1672,1691,1710,1729,1748,1767,1786,1805,1824,1843,1862,1881,1900,1919,1938,1957,1976,1995,2014,2033,2052,2071,2090,2109,2128,2147,2166,2185,2204,2223,2242,2261,2280,2299,2318,2337,2356,2375,2394,2413,2432,2451,2470,2489,2508,2527,2546,2565,2584,2603,2622,2641,2660,2679,2698,2717,2736,2755,2774,2793,2812,2831,2850,2869,2888,2907,2926,2945,2964,2983,3002,3021,3040,3059,3078,3097,3116,3135,3154,3173,3192,3211,3230,3249,3268,3287,3306,3325,3344,3363,3382,3401,3420,3439,3458,3477,3496,3515,3534,3553,3572,3591,3610,3629,3648,3667,3686,3705,3724,3743,3762,3781,3800,3819,3838,3857,3876,3895,3914,3933,3952,3971,3990,4009,4028,4047,4066,4085,4104,4123,4142,4161,4180,4199,4218,4237,4256,4275,4294,4313,4332,4351,4370,4389,4408,4427,4446,4465,4484,4503,4522,4541,4560,4579,4598,4617,4636,4655,4674,4693,4712,4731,4750,4769,4788,4807,4826,4845,4864,4883,4902,4921,4940,4959,4978,4997,5016,5035,5054,5073,5092,5111,5130,5149,5168,5187,5206,5225,5244,5263,5282,5301,5320,5339,5358,5377,5396,5415,5434,5453,5472,5491,5510,5529,5548,5567,5586,5605,5624,5643,5662,5681,5700,5719,5738,5757,5776,5795,5814,5833,5852,5871,5890,5909,5928,5947,5966,5985,6004,6023,6042,6061,6080,6099,6118,6137,6156,6175,6194,6213,6232,6251,6270,6289,6308,6327,6346,6365,6384,6403,6422,6441,6460,6479,6498,6517,6536,6555,6574,6593,6612,6631,6650,6669,6688,6707,6726,6745,6764,6783,6802,6821,6840,6859,6878,6897,6916,6935,6954,6973,6992,7011,7030,7049,7068,7087,7106,7125,7144,7163,7182,7201,7220,7239,7258,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,399,418,437,456,475,494,513,532,551,570,589,608,627,646,665,684,703,722,741,760,779,798,817,836,855,874,893,912,931,950,969,988,1007,1026,1045,1064,1083,1102,1121,1140,1159,1178,1197,1216,1235,1254,1273,1292,1311,1330,1349,1368,1387,1406,1425,1444,1463,1482,1501,1520,1539,1558,1577,1596,1615,1634,1653,1672,1691,1710,1729,1748,1767,1786,1805,1824,1843,1862,1881,1900,1919,1938,1957,1976,1995,2014,2033,2052,2071,2090,2109,2128,2147,2166,2185,2204,2223,2242,2261,2280,2299,2318,2337,2356,2375,2394,2413,2432,2451,2470,2489,2508,2527,2546,2565,2584,2603,2622,2641,2660,2679,2698,2717,2736,2755,2774,2793,2812,2831,2850,2869,2888,2907,2926,2945,2964,2983,3002,3021,3040,3059,3078,3097,3116,3135,3154,3173,3192,3211,3230,3249,3268,3287,3306,3325,3344,3363,3382,3401,3420,3439,3458,3477,3496,3515,3534,3553,3572,3591,3610,3629,3648,3667,3686,3705,3724,3743,3762,3781,3800,3819,3838,3857,3876,3895,3914,3933,3952,3971,3990,4009,4028,4047,4066,4085,4104,4123,4142,4161,4180,4199,4218,4237,4256,4275,4294,4313,4332,4351,4370,4389,4408,4427,4446,4465,4484,4503,4522,4541,4560,4579,4598,4617,4636,4655,4674,4693,4712,4731,4750,4769,4788,4807,4826,4845,4864,4883,4902,4921,4940,4959,4978,4997,5016,5035,5054,5073,5092,5111,5130,5149,5168,5187,5206,5225,5244,5263,5282,5301,5320,5339,5358,5377,5396,5415,5434,5453,5472,5491,5510,5529,5548,5567,5586,5605,5624,5643,5662,5681,5700,5719,5738,5757,5776,5795,5814,5833,5852,5871,5890,5909,5928,5947,5966,5985,6004,6023,6042,6061,6080,6099,6118,6137,6156,6175,6194,6213,6232,6251,6270,6289,6308,6327,6346,6365,6384,6403,6422,6441,6460,6479,6498,6517,6536,6555,6574,6593,6612,6631,6650,6669,6688,6707,6726,6745,6764,6783,6802,6821,6840,6859,6878,6897,6916,6935,6954,6973,6992,7011,7030,7049,7068,7087,7106,7125,7144,7163,7182,7201,7220,7239,7258,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,440,460,480,500,520,540,560,580,600,620,640,660,680,700,720,740,760,780,800,820,840,860,880,900,920,940,960,980,1000,1020,1040,1060,1080,1100,1120,1140,1160,1180,1200,1220,1240,1260,1280,1300,1320,1340,1360,1380,1400,1420,1440,1460,1480,1500,1520,1540,1560,1580,1600,1620,1640,1660,1680,1700,1720,1740,1760,1780,1800,1820,1840,1860,1880,1900,1920,1940,1960,1980,2000,2020,2040,2060,2080,2100,2120,2140,2160,2180,2200,2220,2240,2260,2280,2300,2320,2340,2360,2380,2400,2420,2440,2460,2480,2500,2520,2540,2560,2580,2600,2620,2640,2660,2680,2700,2720,2740,2760,2780,2800,2820,2840,2860,2880,2900,2920,2940,2960,2980,3000,3020,3040,3060,3080,3100,3120,3140,3160,3180,3200,3220,3240,3260,3280,3300,3320,3340,3360,3380,3400,3420,3440,3460,3480,3500,3520,3540,3560,3580,3600,3620,3640,3660,3680,3700,3720,3740,3760,3780,3800,3820,3840,3860,3880,3900,3920,3940,3960,3980,4000,4020,4040,4060,4080,4100,4120,4140,4160,4180,4200,4220,4240,4260,4280,4300,4320,4340,4360,4380,4400,4420,4440,4460,4480,4500,4520,4540,4560,4580,4600,4620,4640,4660,4680,4700,4720,4740,4760,4780,4800,4820,4840,4860,4880,4900,4920,4940,4960,4980,5000,5020,5040,5060,5080,5100,5120,5140,5160,5180,5200,5220,5240,5260,5280,5300,5320,5340,5360,5380,5400,5420,5440,5460,5480,5500,5520,5540,5560,5580,5600,5620,5640,5660,5680,5700,5720,5740,5760,5780,5800,5820,5840,5860,5880,5900,5920,5940,5960,5980,6000,6020,6040,6060,6080,6100,6120,6140,6160,6180,6200,6220,6240,6260,6280,6300,6320,6340,6360,6380,6400,6420,6440,6460,6480,6500,6520,6540,6560,6580,6600,6620,6640,6660,6680,6700,6720,6740,6760,6780,6800,6820,6840,6860,6880,6900,6920,6940,6960,6980,7000,7020,7040,7060,7080,7100,7120,7140,7160,7180,7200,7220,7240,7260,7280,7300,7320,7340,7360,7380,7400,7420,7440,7460,7480,7500,7520,7540,7560,7580,7600,7620,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,440,460,480,500,520,540,560,580,600,620,640,660,680,700,720,740,760,780,800,820,840,860,880,900,920,940,960,980,1000,1020,1040,1060,1080,1100,1120,1140,1160,1180,1200,1220,1240,1260,1280,1300,1320,1340,1360,1380,1400,1420,1440,1460,1480,1500,1520,1540,1560,1580,1600,1620,1640,1660,1680,1700,1720,1740,1760,1780,1800,1820,1840,1860,1880,1900,1920,1940,1960,1980,2000,2020,2040,2060,2080,2100,2120,2140,2160,2180,2200,2220,2240,2260,2280,2300,2320,2340,2360,2380,2400,2420,2440,2460,2480,2500,2520,2540,2560,2580,2600,2620,2640,2660,2680,2700,2720,2740,2760,2780,2800,2820,2840,2860,2880,2900,2920,2940,2960,2980,3000,3020,3040,3060,3080,3100,3120,3140,3160,3180,3200,3220,3240,3260,3280,3300,3320,3340,3360,3380,3400,3420,3440,3460,3480,3500,3520,3540,3560,3580,3600,3620,3640,3660,3680,3700,3720,3740,3760,3780,3800,3820,3840,3860,3880,3900,3920,3940,3960,3980,4000,4020,4040,4060,4080,4100,4120,4140,4160,4180,4200,4220,4240,4260,4280,4300,4320,4340,4360,4380,4400,4420,4440,4460,4480,4500,4520,4540,4560,4580,4600,4620,4640,4660,4680,4700,4720,4740,4760,4780,4800,4820,4840,4860,4880,4900,4920,4940,4960,4980,5000,5020,5040,5060,5080,5100,5120,5140,5160,5180,5200,5220,5240,5260,5280,5300,5320,5340,5360,5380,5400,5420,5440,5460,5480,5500,5520,5540,5560,5580,5600,5620,5640,5660,5680,5700,5720,5740,5760,5780,5800,5820,5840,5860,5880,5900,5920,5940,5960,5980,6000,6020,6040,6060,6080,6100,6120,6140,6160,6180,6200,6220,6240,6260,6280,6300,6320,6340,6360,6380,6400,6420,6440,6460,6480,6500,6520,6540,6560,6580,6600,6620,6640,6660,6680,6700,6720,6740,6760,6780,6800,6820,6840,6860,6880,6900,6920,6940,6960,6980,7000,7020,7040,7060,7080,7100,7120,7140,7160,7180,7200,7220,7240,7260,7280,7300,7320,7340,7360,7380,7400,7420,7440,7460,7480,7500,7520,7540,7560,7580,7600,7620,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,483,504,525,546,567,588,609,630,651,672,693,714,735,756,777,798,819,840,861,882,903,924,945,966,987,1008,1029,1050,1071,1092,1113,1134,1155,1176,1197,1218,1239,1260,1281,1302,1323,1344,1365,1386,1407,1428,1449,1470,1491,1512,1533,1554,1575,1596,1617,1638,1659,1680,1701,1722,1743,1764,1785,1806,1827,1848,1869,1890,1911,1932,1953,1974,1995,2016,2037,2058,2079,2100,2121,2142,2163,2184,2205,2226,2247,2268,2289,2310,2331,2352,2373,2394,2415,2436,2457,2478,2499,2520,2541,2562,2583,2604,2625,2646,2667,2688,2709,2730,2751,2772,2793,2814,2835,2856,2877,2898,2919,2940,2961,2982,3003,3024,3045,3066,3087,3108,3129,3150,3171,3192,3213,3234,3255,3276,3297,3318,3339,3360,3381,3402,3423,3444,3465,3486,3507,3528,3549,3570,3591,3612,3633,3654,3675,3696,3717,3738,3759,3780,3801,3822,3843,3864,3885,3906,3927,3948,3969,3990,4011,4032,4053,4074,4095,4116,4137,4158,4179,4200,4221,4242,4263,4284,4305,4326,4347,4368,4389,4410,4431,4452,4473,4494,4515,4536,4557,4578,4599,4620,4641,4662,4683,4704,4725,4746,4767,4788,4809,4830,4851,4872,4893,4914,4935,4956,4977,4998,5019,5040,5061,5082,5103,5124,5145,5166,5187,5208,5229,5250,5271,5292,5313,5334,5355,5376,5397,5418,5439,5460,5481,5502,5523,5544,5565,5586,5607,5628,5649,5670,5691,5712,5733,5754,5775,5796,5817,5838,5859,5880,5901,5922,5943,5964,5985,6006,6027,6048,6069,6090,6111,6132,6153,6174,6195,6216,6237,6258,6279,6300,6321,6342,6363,6384,6405,6426,6447,6468,6489,6510,6531,6552,6573,6594,6615,6636,6657,6678,6699,6720,6741,6762,6783,6804,6825,6846,6867,6888,6909,6930,6951,6972,6993,7014,7035,7056,7077,7098,7119,7140,7161,7182,7203,7224,7245,7266,7287,7308,7329,7350,7371,7392,7413,7434,7455,7476,7497,7518,7539,7560,7581,7602,7623,7644,7665,7686,7707,7728,7749,7770,7791,7812,7833,7854,7875,7896,7917,7938,7959,7980,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,483,504,525,546,567,588,609,630,651,672,693,714,735,756,777,798,819,840,861,882,903,924,945,966,987,1008,1029,1050,1071,1092,1113,1134,1155,1176,1197,1218,1239,1260,1281,1302,1323,1344,1365,1386,1407,1428,1449,1470,1491,1512,1533,1554,1575,1596,1617,1638,1659,1680,1701,1722,1743,1764,1785,1806,1827,1848,1869,1890,1911,1932,1953,1974,1995,2016,2037,2058,2079,2100,2121,2142,2163,2184,2205,2226,2247,2268,2289,2310,2331,2352,2373,2394,2415,2436,2457,2478,2499,2520,2541,2562,2583,2604,2625,2646,2667,2688,2709,2730,2751,2772,2793,2814,2835,2856,2877,2898,2919,2940,2961,2982,3003,3024,3045,3066,3087,3108,3129,3150,3171,3192,3213,3234,3255,3276,3297,3318,3339,3360,3381,3402,3423,3444,3465,3486,3507,3528,3549,3570,3591,3612,3633,3654,3675,3696,3717,3738,3759,3780,3801,3822,3843,3864,3885,3906,3927,3948,3969,3990,4011,4032,4053,4074,4095,4116,4137,4158,4179,4200,4221,4242,4263,4284,4305,4326,4347,4368,4389,4410,4431,4452,4473,4494,4515,4536,4557,4578,4599,4620,4641,4662,4683,4704,4725,4746,4767,4788,4809,4830,4851,4872,4893,4914,4935,4956,4977,4998,5019,5040,5061,5082,5103,5124,5145,5166,5187,5208,5229,5250,5271,5292,5313,5334,5355,5376,5397,5418,5439,5460,5481,5502,5523,5544,5565,5586,5607,5628,5649,5670,5691,5712,5733,5754,5775,5796,5817,5838,5859,5880,5901,5922,5943,5964,5985,6006,6027,6048,6069,6090,6111,6132,6153,6174,6195,6216,6237,6258,6279,6300,6321,6342,6363,6384,6405,6426,6447,6468,6489,6510,6531,6552,6573,6594,6615,6636,6657,6678,6699,6720,6741,6762,6783,6804,6825,6846,6867,6888,6909,6930,6951,6972,6993,7014,7035,7056,7077,7098,7119,7140,7161,7182,7203,7224,7245,7266,7287,7308,7329,7350,7371,7392,7413,7434,7455,7476,7497,7518,7539,7560,7581,7602,7623,7644,7665,7686,7707,7728,7749,7770,7791,7812,7833,7854,7875,7896,7917,7938,7959,7980,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,528,550,572,594,616,638,660,682,704,726,748,770,792,814,836,858,880,902,924,946,968,990,1012,1034,1056,1078,1100,1122,1144,1166,1188,1210,1232,1254,1276,1298,1320,1342,1364,1386,1408,1430,1452,1474,1496,1518,1540,1562,1584,1606,1628,1650,1672,1694,1716,1738,1760,1782,1804,1826,1848,1870,1892,1914,1936,1958,1980,2002,2024,2046,2068,2090,2112,2134,2156,2178,2200,2222,2244,2266,2288,2310,2332,2354,2376,2398,2420,2442,2464,2486,2508,2530,2552,2574,2596,2618,2640,2662,2684,2706,2728,2750,2772,2794,2816,2838,2860,2882,2904,2926,2948,2970,2992,3014,3036,3058,3080,3102,3124,3146,3168,3190,3212,3234,3256,3278,3300,3322,3344,3366,3388,3410,3432,3454,3476,3498,3520,3542,3564,3586,3608,3630,3652,3674,3696,3718,3740,3762,3784,3806,3828,3850,3872,3894,3916,3938,3960,3982,4004,4026,4048,4070,4092,4114,4136,4158,4180,4202,4224,4246,4268,4290,4312,4334,4356,4378,4400,4422,4444,4466,4488,4510,4532,4554,4576,4598,4620,4642,4664,4686,4708,4730,4752,4774,4796,4818,4840,4862,4884,4906,4928,4950,4972,4994,5016,5038,5060,5082,5104,5126,5148,5170,5192,5214,5236,5258,5280,5302,5324,5346,5368,5390,5412,5434,5456,5478,5500,5522,5544,5566,5588,5610,5632,5654,5676,5698,5720,5742,5764,5786,5808,5830,5852,5874,5896,5918,5940,5962,5984,6006,6028,6050,6072,6094,6116,6138,6160,6182,6204,6226,6248,6270,6292,6314,6336,6358,6380,6402,6424,6446,6468,6490,6512,6534,6556,6578,6600,6622,6644,6666,6688,6710,6732,6754,6776,6798,6820,6842,6864,6886,6908,6930,6952,6974,6996,7018,7040,7062,7084,7106,7128,7150,7172,7194,7216,7238,7260,7282,7304,7326,7348,7370,7392,7414,7436,7458,7480,7502,7524,7546,7568,7590,7612,7634,7656,7678,7700,7722,7744,7766,7788,7810,7832,7854,7876,7898,7920,7942,7964,7986,8008,8030,8052,8074,8096,8118,8140,8162,8184,8206,8228,8250,8272,8294,8316,8338,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,528,550,572,594,616,638,660,682,704,726,748,770,792,814,836,858,880,902,924,946,968,990,1012,1034,1056,1078,1100,1122,1144,1166,1188,1210,1232,1254,1276,1298,1320,1342,1364,1386,1408,1430,1452,1474,1496,1518,1540,1562,1584,1606,1628,1650,1672,1694,1716,1738,1760,1782,1804,1826,1848,1870,1892,1914,1936,1958,1980,2002,2024,2046,2068,2090,2112,2134,2156,2178,2200,2222,2244,2266,2288,2310,2332,2354,2376,2398,2420,2442,2464,2486,2508,2530,2552,2574,2596,2618,2640,2662,2684,2706,2728,2750,2772,2794,2816,2838,2860,2882,2904,2926,2948,2970,2992,3014,3036,3058,3080,3102,3124,3146,3168,3190,3212,3234,3256,3278,3300,3322,3344,3366,3388,3410,3432,3454,3476,3498,3520,3542,3564,3586,3608,3630,3652,3674,3696,3718,3740,3762,3784,3806,3828,3850,3872,3894,3916,3938,3960,3982,4004,4026,4048,4070,4092,4114,4136,4158,4180,4202,4224,4246,4268,4290,4312,4334,4356,4378,4400,4422,4444,4466,4488,4510,4532,4554,4576,4598,4620,4642,4664,4686,4708,4730,4752,4774,4796,4818,4840,4862,4884,4906,4928,4950,4972,4994,5016,5038,5060,5082,5104,5126,5148,5170,5192,5214,5236,5258,5280,5302,5324,5346,5368,5390,5412,5434,5456,5478,5500,5522,5544,5566,5588,5610,5632,5654,5676,5698,5720,5742,5764,5786,5808,5830,5852,5874,5896,5918,5940,5962,5984,6006,6028,6050,6072,6094,6116,6138,6160,6182,6204,6226,6248,6270,6292,6314,6336,6358,6380,6402,6424,6446,6468,6490,6512,6534,6556,6578,6600,6622,6644,6666,6688,6710,6732,6754,6776,6798,6820,6842,6864,6886,6908,6930,6952,6974,6996,7018,7040,7062,7084,7106,7128,7150,7172,7194,7216,7238,7260,7282,7304,7326,7348,7370,7392,7414,7436,7458,7480,7502,7524,7546,7568,7590,7612,7634,7656,7678,7700,7722,7744,7766,7788,7810,7832,7854,7876,7898,7920,7942,7964,7986,8008,8030,8052,8074,8096,8118,8140,8162,8184,8206,8228,8250,8272,8294,8316,8338,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,575,598,621,644,667,690,713,736,759,782,805,828,851,874,897,920,943,966,989,1012,1035,1058,1081,1104,1127,1150,1173,1196,1219,1242,1265,1288,1311,1334,1357,1380,1403,1426,1449,1472,1495,1518,1541,1564,1587,1610,1633,1656,1679,1702,1725,1748,1771,1794,1817,1840,1863,1886,1909,1932,1955,1978,2001,2024,2047,2070,2093,2116,2139,2162,2185,2208,2231,2254,2277,2300,2323,2346,2369,2392,2415,2438,2461,2484,2507,2530,2553,2576,2599,2622,2645,2668,2691,2714,2737,2760,2783,2806,2829,2852,2875,2898,2921,2944,2967,2990,3013,3036,3059,3082,3105,3128,3151,3174,3197,3220,3243,3266,3289,3312,3335,3358,3381,3404,3427,3450,3473,3496,3519,3542,3565,3588,3611,3634,3657,3680,3703,3726,3749,3772,3795,3818,3841,3864,3887,3910,3933,3956,3979,4002,4025,4048,4071,4094,4117,4140,4163,4186,4209,4232,4255,4278,4301,4324,4347,4370,4393,4416,4439,4462,4485,4508,4531,4554,4577,4600,4623,4646,4669,4692,4715,4738,4761,4784,4807,4830,4853,4876,4899,4922,4945,4968,4991,5014,5037,5060,5083,5106,5129,5152,5175,5198,5221,5244,5267,5290,5313,5336,5359,5382,5405,5428,5451,5474,5497,5520,5543,5566,5589,5612,5635,5658,5681,5704,5727,5750,5773,5796,5819,5842,5865,5888,5911,5934,5957,5980,6003,6026,6049,6072,6095,6118,6141,6164,6187,6210,6233,6256,6279,6302,6325,6348,6371,6394,6417,6440,6463,6486,6509,6532,6555,6578,6601,6624,6647,6670,6693,6716,6739,6762,6785,6808,6831,6854,6877,6900,6923,6946,6969,6992,7015,7038,7061,7084,7107,7130,7153,7176,7199,7222,7245,7268,7291,7314,7337,7360,7383,7406,7429,7452,7475,7498,7521,7544,7567,7590,7613,7636,7659,7682,7705,7728,7751,7774,7797,7820,7843,7866,7889,7912,7935,7958,7981,8004,8027,8050,8073,8096,8119,8142,8165,8188,8211,8234,8257,8280,8303,8326,8349,8372,8395,8418,8441,8464,8487,8510,8533,8556,8579,8602,8625,8648,8671,8694,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,575,598,621,644,667,690,713,736,759,782,805,828,851,874,897,920,943,966,989,1012,1035,1058,1081,1104,1127,1150,1173,1196,1219,1242,1265,1288,1311,1334,1357,1380,1403,1426,1449,1472,1495,1518,1541,1564,1587,1610,1633,1656,1679,1702,1725,1748,1771,1794,1817,1840,1863,1886,1909,1932,1955,1978,2001,2024,2047,2070,2093,2116,2139,2162,2185,2208,2231,2254,2277,2300,2323,2346,2369,2392,2415,2438,2461,2484,2507,2530,2553,2576,2599,2622,2645,2668,2691,2714,2737,2760,2783,2806,2829,2852,2875,2898,2921,2944,2967,2990,3013,3036,3059,3082,3105,3128,3151,3174,3197,3220,3243,3266,3289,3312,3335,3358,3381,3404,3427,3450,3473,3496,3519,3542,3565,3588,3611,3634,3657,3680,3703,3726,3749,3772,3795,3818,3841,3864,3887,3910,3933,3956,3979,4002,4025,4048,4071,4094,4117,4140,4163,4186,4209,4232,4255,4278,4301,4324,4347,4370,4393,4416,4439,4462,4485,4508,4531,4554,4577,4600,4623,4646,4669,4692,4715,4738,4761,4784,4807,4830,4853,4876,4899,4922,4945,4968,4991,5014,5037,5060,5083,5106,5129,5152,5175,5198,5221,5244,5267,5290,5313,5336,5359,5382,5405,5428,5451,5474,5497,5520,5543,5566,5589,5612,5635,5658,5681,5704,5727,5750,5773,5796,5819,5842,5865,5888,5911,5934,5957,5980,6003,6026,6049,6072,6095,6118,6141,6164,6187,6210,6233,6256,6279,6302,6325,6348,6371,6394,6417,6440,6463,6486,6509,6532,6555,6578,6601,6624,6647,6670,6693,6716,6739,6762,6785,6808,6831,6854,6877,6900,6923,6946,6969,6992,7015,7038,7061,7084,7107,7130,7153,7176,7199,7222,7245,7268,7291,7314,7337,7360,7383,7406,7429,7452,7475,7498,7521,7544,7567,7590,7613,7636,7659,7682,7705,7728,7751,7774,7797,7820,7843,7866,7889,7912,7935,7958,7981,8004,8027,8050,8073,8096,8119,8142,8165,8188,8211,8234,8257,8280,8303,8326,8349,8372,8395,8418,8441,8464,8487,8510,8533,8556,8579,8602,8625,8648,8671,8694,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,624,648,672,696,720,744,768,792,816,840,864,888,912,936,960,984,1008,1032,1056,1080,1104,1128,1152,1176,1200,1224,1248,1272,1296,1320,1344,1368,1392,1416,1440,1464,1488,1512,1536,1560,1584,1608,1632,1656,1680,1704,1728,1752,1776,1800,1824,1848,1872,1896,1920,1944,1968,1992,2016,2040,2064,2088,2112,2136,2160,2184,2208,2232,2256,2280,2304,2328,2352,2376,2400,2424,2448,2472,2496,2520,2544,2568,2592,2616,2640,2664,2688,2712,2736,2760,2784,2808,2832,2856,2880,2904,2928,2952,2976,3000,3024,3048,3072,3096,3120,3144,3168,3192,3216,3240,3264,3288,3312,3336,3360,3384,3408,3432,3456,3480,3504,3528,3552,3576,3600,3624,3648,3672,3696,3720,3744,3768,3792,3816,3840,3864,3888,3912,3936,3960,3984,4008,4032,4056,4080,4104,4128,4152,4176,4200,4224,4248,4272,4296,4320,4344,4368,4392,4416,4440,4464,4488,4512,4536,4560,4584,4608,4632,4656,4680,4704,4728,4752,4776,4800,4824,4848,4872,4896,4920,4944,4968,4992,5016,5040,5064,5088,5112,5136,5160,5184,5208,5232,5256,5280,5304,5328,5352,5376,5400,5424,5448,5472,5496,5520,5544,5568,5592,5616,5640,5664,5688,5712,5736,5760,5784,5808,5832,5856,5880,5904,5928,5952,5976,6000,6024,6048,6072,6096,6120,6144,6168,6192,6216,6240,6264,6288,6312,6336,6360,6384,6408,6432,6456,6480,6504,6528,6552,6576,6600,6624,6648,6672,6696,6720,6744,6768,6792,6816,6840,6864,6888,6912,6936,6960,6984,7008,7032,7056,7080,7104,7128,7152,7176,7200,7224,7248,7272,7296,7320,7344,7368,7392,7416,7440,7464,7488,7512,7536,7560,7584,7608,7632,7656,7680,7704,7728,7752,7776,7800,7824,7848,7872,7896,7920,7944,7968,7992,8016,8040,8064,8088,8112,8136,8160,8184,8208,8232,8256,8280,8304,8328,8352,8376,8400,8424,8448,8472,8496,8520,8544,8568,8592,8616,8640,8664,8688,8712,8736,8760,8784,8808,8832,8856,8880,8904,8928,8952,8976,9000,9024,9048,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,624,648,672,696,720,744,768,792,816,840,864,888,912,936,960,984,1008,1032,1056,1080,1104,1128,1152,1176,1200,1224,1248,1272,1296,1320,1344,1368,1392,1416,1440,1464,1488,1512,1536,1560,1584,1608,1632,1656,1680,1704,1728,1752,1776,1800,1824,1848,1872,1896,1920,1944,1968,1992,2016,2040,2064,2088,2112,2136,2160,2184,2208,2232,2256,2280,2304,2328,2352,2376,2400,2424,2448,2472,2496,2520,2544,2568,2592,2616,2640,2664,2688,2712,2736,2760,2784,2808,2832,2856,2880,2904,2928,2952,2976,3000,3024,3048,3072,3096,3120,3144,3168,3192,3216,3240,3264,3288,3312,3336,3360,3384,3408,3432,3456,3480,3504,3528,3552,3576,3600,3624,3648,3672,3696,3720,3744,3768,3792,3816,3840,3864,3888,3912,3936,3960,3984,4008,4032,4056,4080,4104,4128,4152,4176,4200,4224,4248,4272,4296,4320,4344,4368,4392,4416,4440,4464,4488,4512,4536,4560,4584,4608,4632,4656,4680,4704,4728,4752,4776,4800,4824,4848,4872,4896,4920,4944,4968,4992,5016,5040,5064,5088,5112,5136,5160,5184,5208,5232,5256,5280,5304,5328,5352,5376,5400,5424,5448,5472,5496,5520,5544,5568,5592,5616,5640,5664,5688,5712,5736,5760,5784,5808,5832,5856,5880,5904,5928,5952,5976,6000,6024,6048,6072,6096,6120,6144,6168,6192,6216,6240,6264,6288,6312,6336,6360,6384,6408,6432,6456,6480,6504,6528,6552,6576,6600,6624,6648,6672,6696,6720,6744,6768,6792,6816,6840,6864,6888,6912,6936,6960,6984,7008,7032,7056,7080,7104,7128,7152,7176,7200,7224,7248,7272,7296,7320,7344,7368,7392,7416,7440,7464,7488,7512,7536,7560,7584,7608,7632,7656,7680,7704,7728,7752,7776,7800,7824,7848,7872,7896,7920,7944,7968,7992,8016,8040,8064,8088,8112,8136,8160,8184,8208,8232,8256,8280,8304,8328,8352,8376,8400,8424,8448,8472,8496,8520,8544,8568,8592,8616,8640,8664,8688,8712,8736,8760,8784,8808,8832,8856,8880,8904,8928,8952,8976,9000,9024,9048,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,675,700,725,750,775,800,825,850,875,900,925,950,975,1000,1025,1050,1075,1100,1125,1150,1175,1200,1225,1250,1275,1300,1325,1350,1375,1400,1425,1450,1475,1500,1525,1550,1575,1600,1625,1650,1675,1700,1725,1750,1775,1800,1825,1850,1875,1900,1925,1950,1975,2000,2025,2050,2075,2100,2125,2150,2175,2200,2225,2250,2275,2300,2325,2350,2375,2400,2425,2450,2475,2500,2525,2550,2575,2600,2625,2650,2675,2700,2725,2750,2775,2800,2825,2850,2875,2900,2925,2950,2975,3000,3025,3050,3075,3100,3125,3150,3175,3200,3225,3250,3275,3300,3325,3350,3375,3400,3425,3450,3475,3500,3525,3550,3575,3600,3625,3650,3675,3700,3725,3750,3775,3800,3825,3850,3875,3900,3925,3950,3975,4000,4025,4050,4075,4100,4125,4150,4175,4200,4225,4250,4275,4300,4325,4350,4375,4400,4425,4450,4475,4500,4525,4550,4575,4600,4625,4650,4675,4700,4725,4750,4775,4800,4825,4850,4875,4900,4925,4950,4975,5000,5025,5050,5075,5100,5125,5150,5175,5200,5225,5250,5275,5300,5325,5350,5375,5400,5425,5450,5475,5500,5525,5550,5575,5600,5625,5650,5675,5700,5725,5750,5775,5800,5825,5850,5875,5900,5925,5950,5975,6000,6025,6050,6075,6100,6125,6150,6175,6200,6225,6250,6275,6300,6325,6350,6375,6400,6425,6450,6475,6500,6525,6550,6575,6600,6625,6650,6675,6700,6725,6750,6775,6800,6825,6850,6875,6900,6925,6950,6975,7000,7025,7050,7075,7100,7125,7150,7175,7200,7225,7250,7275,7300,7325,7350,7375,7400,7425,7450,7475,7500,7525,7550,7575,7600,7625,7650,7675,7700,7725,7750,7775,7800,7825,7850,7875,7900,7925,7950,7975,8000,8025,8050,8075,8100,8125,8150,8175,8200,8225,8250,8275,8300,8325,8350,8375,8400,8425,8450,8475,8500,8525,8550,8575,8600,8625,8650,8675,8700,8725,8750,8775,8800,8825,8850,8875,8900,8925,8950,8975,9000,9025,9050,9075,9100,9125,9150,9175,9200,9225,9250,9275,9300,9325,9350,9375,9400,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,675,700,725,750,775,800,825,850,875,900,925,950,975,1000,1025,1050,1075,1100,1125,1150,1175,1200,1225,1250,1275,1300,1325,1350,1375,1400,1425,1450,1475,1500,1525,1550,1575,1600,1625,1650,1675,1700,1725,1750,1775,1800,1825,1850,1875,1900,1925,1950,1975,2000,2025,2050,2075,2100,2125,2150,2175,2200,2225,2250,2275,2300,2325,2350,2375,2400,2425,2450,2475,2500,2525,2550,2575,2600,2625,2650,2675,2700,2725,2750,2775,2800,2825,2850,2875,2900,2925,2950,2975,3000,3025,3050,3075,3100,3125,3150,3175,3200,3225,3250,3275,3300,3325,3350,3375,3400,3425,3450,3475,3500,3525,3550,3575,3600,3625,3650,3675,3700,3725,3750,3775,3800,3825,3850,3875,3900,3925,3950,3975,4000,4025,4050,4075,4100,4125,4150,4175,4200,4225,4250,4275,4300,4325,4350,4375,4400,4425,4450,4475,4500,4525,4550,4575,4600,4625,4650,4675,4700,4725,4750,4775,4800,4825,4850,4875,4900,4925,4950,4975,5000,5025,5050,5075,5100,5125,5150,5175,5200,5225,5250,5275,5300,5325,5350,5375,5400,5425,5450,5475,5500,5525,5550,5575,5600,5625,5650,5675,5700,5725,5750,5775,5800,5825,5850,5875,5900,5925,5950,5975,6000,6025,6050,6075,6100,6125,6150,6175,6200,6225,6250,6275,6300,6325,6350,6375,6400,6425,6450,6475,6500,6525,6550,6575,6600,6625,6650,6675,6700,6725,6750,6775,6800,6825,6850,6875,6900,6925,6950,6975,7000,7025,7050,7075,7100,7125,7150,7175,7200,7225,7250,7275,7300,7325,7350,7375,7400,7425,7450,7475,7500,7525,7550,7575,7600,7625,7650,7675,7700,7725,7750,7775,7800,7825,7850,7875,7900,7925,7950,7975,8000,8025,8050,8075,8100,8125,8150,8175,8200,8225,8250,8275,8300,8325,8350,8375,8400,8425,8450,8475,8500,8525,8550,8575,8600,8625,8650,8675,8700,8725,8750,8775,8800,8825,8850,8875,8900,8925,8950,8975,9000,9025,9050,9075,9100,9125,9150,9175,9200,9225,9250,9275,9300,9325,9350,9375,9400,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,728,754,780,806,832,858,884,910,936,962,988,1014,1040,1066,1092,1118,1144,1170,1196,1222,1248,1274,1300,1326,1352,1378,1404,1430,1456,1482,1508,1534,1560,1586,1612,1638,1664,1690,1716,1742,1768,1794,1820,1846,1872,1898,1924,1950,1976,2002,2028,2054,2080,2106,2132,2158,2184,2210,2236,2262,2288,2314,2340,2366,2392,2418,2444,2470,2496,2522,2548,2574,2600,2626,2652,2678,2704,2730,2756,2782,2808,2834,2860,2886,2912,2938,2964,2990,3016,3042,3068,3094,3120,3146,3172,3198,3224,3250,3276,3302,3328,3354,3380,3406,3432,3458,3484,3510,3536,3562,3588,3614,3640,3666,3692,3718,3744,3770,3796,3822,3848,3874,3900,3926,3952,3978,4004,4030,4056,4082,4108,4134,4160,4186,4212,4238,4264,4290,4316,4342,4368,4394,4420,4446,4472,4498,4524,4550,4576,4602,4628,4654,4680,4706,4732,4758,4784,4810,4836,4862,4888,4914,4940,4966,4992,5018,5044,5070,5096,5122,5148,5174,5200,5226,5252,5278,5304,5330,5356,5382,5408,5434,5460,5486,5512,5538,5564,5590,5616,5642,5668,5694,5720,5746,5772,5798,5824,5850,5876,5902,5928,5954,5980,6006,6032,6058,6084,6110,6136,6162,6188,6214,6240,6266,6292,6318,6344,6370,6396,6422,6448,6474,6500,6526,6552,6578,6604,6630,6656,6682,6708,6734,6760,6786,6812,6838,6864,6890,6916,6942,6968,6994,7020,7046,7072,7098,7124,7150,7176,7202,7228,7254,7280,7306,7332,7358,7384,7410,7436,7462,7488,7514,7540,7566,7592,7618,7644,7670,7696,7722,7748,7774,7800,7826,7852,7878,7904,7930,7956,7982,8008,8034,8060,8086,8112,8138,8164,8190,8216,8242,8268,8294,8320,8346,8372,8398,8424,8450,8476,8502,8528,8554,8580,8606,8632,8658,8684,8710,8736,8762,8788,8814,8840,8866,8892,8918,8944,8970,8996,9022,9048,9074,9100,9126,9152,9178,9204,9230,9256,9282,9308,9334,9360,9386,9412,9438,9464,9490,9516,9542,9568,9594,9620,9646,9672,9698,9724,9750,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,728,754,780,806,832,858,884,910,936,962,988,1014,1040,1066,1092,1118,1144,1170,1196,1222,1248,1274,1300,1326,1352,1378,1404,1430,1456,1482,1508,1534,1560,1586,1612,1638,1664,1690,1716,1742,1768,1794,1820,1846,1872,1898,1924,1950,1976,2002,2028,2054,2080,2106,2132,2158,2184,2210,2236,2262,2288,2314,2340,2366,2392,2418,2444,2470,2496,2522,2548,2574,2600,2626,2652,2678,2704,2730,2756,2782,2808,2834,2860,2886,2912,2938,2964,2990,3016,3042,3068,3094,3120,3146,3172,3198,3224,3250,3276,3302,3328,3354,3380,3406,3432,3458,3484,3510,3536,3562,3588,3614,3640,3666,3692,3718,3744,3770,3796,3822,3848,3874,3900,3926,3952,3978,4004,4030,4056,4082,4108,4134,4160,4186,4212,4238,4264,4290,4316,4342,4368,4394,4420,4446,4472,4498,4524,4550,4576,4602,4628,4654,4680,4706,4732,4758,4784,4810,4836,4862,4888,4914,4940,4966,4992,5018,5044,5070,5096,5122,5148,5174,5200,5226,5252,5278,5304,5330,5356,5382,5408,5434,5460,5486,5512,5538,5564,5590,5616,5642,5668,5694,5720,5746,5772,5798,5824,5850,5876,5902,5928,5954,5980,6006,6032,6058,6084,6110,6136,6162,6188,6214,6240,6266,6292,6318,6344,6370,6396,6422,6448,6474,6500,6526,6552,6578,6604,6630,6656,6682,6708,6734,6760,6786,6812,6838,6864,6890,6916,6942,6968,6994,7020,7046,7072,7098,7124,7150,7176,7202,7228,7254,7280,7306,7332,7358,7384,7410,7436,7462,7488,7514,7540,7566,7592,7618,7644,7670,7696,7722,7748,7774,7800,7826,7852,7878,7904,7930,7956,7982,8008,8034,8060,8086,8112,8138,8164,8190,8216,8242,8268,8294,8320,8346,8372,8398,8424,8450,8476,8502,8528,8554,8580,8606,8632,8658,8684,8710,8736,8762,8788,8814,8840,8866,8892,8918,8944,8970,8996,9022,9048,9074,9100,9126,9152,9178,9204,9230,9256,9282,9308,9334,9360,9386,9412,9438,9464,9490,9516,9542,9568,9594,9620,9646,9672,9698,9724,9750,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,783,810,837,864,891,918,945,972,999,1026,1053,1080,1107,1134,1161,1188,1215,1242,1269,1296,1323,1350,1377,1404,1431,1458,1485,1512,1539,1566,1593,1620,1647,1674,1701,1728,1755,1782,1809,1836,1863,1890,1917,1944,1971,1998,2025,2052,2079,2106,2133,2160,2187,2214,2241,2268,2295,2322,2349,2376,2403,2430,2457,2484,2511,2538,2565,2592,2619,2646,2673,2700,2727,2754,2781,2808,2835,2862,2889,2916,2943,2970,2997,3024,3051,3078,3105,3132,3159,3186,3213,3240,3267,3294,3321,3348,3375,3402,3429,3456,3483,3510,3537,3564,3591,3618,3645,3672,3699,3726,3753,3780,3807,3834,3861,3888,3915,3942,3969,3996,4023,4050,4077,4104,4131,4158,4185,4212,4239,4266,4293,4320,4347,4374,4401,4428,4455,4482,4509,4536,4563,4590,4617,4644,4671,4698,4725,4752,4779,4806,4833,4860,4887,4914,4941,4968,4995,5022,5049,5076,5103,5130,5157,5184,5211,5238,5265,5292,5319,5346,5373,5400,5427,5454,5481,5508,5535,5562,5589,5616,5643,5670,5697,5724,5751,5778,5805,5832,5859,5886,5913,5940,5967,5994,6021,6048,6075,6102,6129,6156,6183,6210,6237,6264,6291,6318,6345,6372,6399,6426,6453,6480,6507,6534,6561,6588,6615,6642,6669,6696,6723,6750,6777,6804,6831,6858,6885,6912,6939,6966,6993,7020,7047,7074,7101,7128,7155,7182,7209,7236,7263,7290,7317,7344,7371,7398,7425,7452,7479,7506,7533,7560,7587,7614,7641,7668,7695,7722,7749,7776,7803,7830,7857,7884,7911,7938,7965,7992,8019,8046,8073,8100,8127,8154,8181,8208,8235,8262,8289,8316,8343,8370,8397,8424,8451,8478,8505,8532,8559,8586,8613,8640,8667,8694,8721,8748,8775,8802,8829,8856,8883,8910,8937,8964,8991,9018,9045,9072,9099,9126,9153,9180,9207,9234,9261,9288,9315,9342,9369,9396,9423,9450,9477,9504,9531,9558,9585,9612,9639,9666,9693,9720,9747,9774,9801,9828,9855,9882,9909,9936,9963,9990,10017,10044,10071,10098,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,783,810,837,864,891,918,945,972,999,1026,1053,1080,1107,1134,1161,1188,1215,1242,1269,1296,1323,1350,1377,1404,1431,1458,1485,1512,1539,1566,1593,1620,1647,1674,1701,1728,1755,1782,1809,1836,1863,1890,1917,1944,1971,1998,2025,2052,2079,2106,2133,2160,2187,2214,2241,2268,2295,2322,2349,2376,2403,2430,2457,2484,2511,2538,2565,2592,2619,2646,2673,2700,2727,2754,2781,2808,2835,2862,2889,2916,2943,2970,2997,3024,3051,3078,3105,3132,3159,3186,3213,3240,3267,3294,3321,3348,3375,3402,3429,3456,3483,3510,3537,3564,3591,3618,3645,3672,3699,3726,3753,3780,3807,3834,3861,3888,3915,3942,3969,3996,4023,4050,4077,4104,4131,4158,4185,4212,4239,4266,4293,4320,4347,4374,4401,4428,4455,4482,4509,4536,4563,4590,4617,4644,4671,4698,4725,4752,4779,4806,4833,4860,4887,4914,4941,4968,4995,5022,5049,5076,5103,5130,5157,5184,5211,5238,5265,5292,5319,5346,5373,5400,5427,5454,5481,5508,5535,5562,5589,5616,5643,5670,5697,5724,5751,5778,5805,5832,5859,5886,5913,5940,5967,5994,6021,6048,6075,6102,6129,6156,6183,6210,6237,6264,6291,6318,6345,6372,6399,6426,6453,6480,6507,6534,6561,6588,6615,6642,6669,6696,6723,6750,6777,6804,6831,6858,6885,6912,6939,6966,6993,7020,7047,7074,7101,7128,7155,7182,7209,7236,7263,7290,7317,7344,7371,7398,7425,7452,7479,7506,7533,7560,7587,7614,7641,7668,7695,7722,7749,7776,7803,7830,7857,7884,7911,7938,7965,7992,8019,8046,8073,8100,8127,8154,8181,8208,8235,8262,8289,8316,8343,8370,8397,8424,8451,8478,8505,8532,8559,8586,8613,8640,8667,8694,8721,8748,8775,8802,8829,8856,8883,8910,8937,8964,8991,9018,9045,9072,9099,9126,9153,9180,9207,9234,9261,9288,9315,9342,9369,9396,9423,9450,9477,9504,9531,9558,9585,9612,9639,9666,9693,9720,9747,9774,9801,9828,9855,9882,9909,9936,9963,9990,10017,10044,10071,10098,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,840,868,896,924,952,980,1008,1036,1064,1092,1120,1148,1176,1204,1232,1260,1288,1316,1344,1372,1400,1428,1456,1484,1512,1540,1568,1596,1624,1652,1680,1708,1736,1764,1792,1820,1848,1876,1904,1932,1960,1988,2016,2044,2072,2100,2128,2156,2184,2212,2240,2268,2296,2324,2352,2380,2408,2436,2464,2492,2520,2548,2576,2604,2632,2660,2688,2716,2744,2772,2800,2828,2856,2884,2912,2940,2968,2996,3024,3052,3080,3108,3136,3164,3192,3220,3248,3276,3304,3332,3360,3388,3416,3444,3472,3500,3528,3556,3584,3612,3640,3668,3696,3724,3752,3780,3808,3836,3864,3892,3920,3948,3976,4004,4032,4060,4088,4116,4144,4172,4200,4228,4256,4284,4312,4340,4368,4396,4424,4452,4480,4508,4536,4564,4592,4620,4648,4676,4704,4732,4760,4788,4816,4844,4872,4900,4928,4956,4984,5012,5040,5068,5096,5124,5152,5180,5208,5236,5264,5292,5320,5348,5376,5404,5432,5460,5488,5516,5544,5572,5600,5628,5656,5684,5712,5740,5768,5796,5824,5852,5880,5908,5936,5964,5992,6020,6048,6076,6104,6132,6160,6188,6216,6244,6272,6300,6328,6356,6384,6412,6440,6468,6496,6524,6552,6580,6608,6636,6664,6692,6720,6748,6776,6804,6832,6860,6888,6916,6944,6972,7000,7028,7056,7084,7112,7140,7168,7196,7224,7252,7280,7308,7336,7364,7392,7420,7448,7476,7504,7532,7560,7588,7616,7644,7672,7700,7728,7756,7784,7812,7840,7868,7896,7924,7952,7980,8008,8036,8064,8092,8120,8148,8176,8204,8232,8260,8288,8316,8344,8372,8400,8428,8456,8484,8512,8540,8568,8596,8624,8652,8680,8708,8736,8764,8792,8820,8848,8876,8904,8932,8960,8988,9016,9044,9072,9100,9128,9156,9184,9212,9240,9268,9296,9324,9352,9380,9408,9436,9464,9492,9520,9548,9576,9604,9632,9660,9688,9716,9744,9772,9800,9828,9856,9884,9912,9940,9968,9996,10024,10052,10080,10108,10136,10164,10192,10220,10248,10276,10304,10332,10360,10388,10416,10444,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,840,868,896,924,952,980,1008,1036,1064,1092,1120,1148,1176,1204,1232,1260,1288,1316,1344,1372,1400,1428,1456,1484,1512,1540,1568,1596,1624,1652,1680,1708,1736,1764,1792,1820,1848,1876,1904,1932,1960,1988,2016,2044,2072,2100,2128,2156,2184,2212,2240,2268,2296,2324,2352,2380,2408,2436,2464,2492,2520,2548,2576,2604,2632,2660,2688,2716,2744,2772,2800,2828,2856,2884,2912,2940,2968,2996,3024,3052,3080,3108,3136,3164,3192,3220,3248,3276,3304,3332,3360,3388,3416,3444,3472,3500,3528,3556,3584,3612,3640,3668,3696,3724,3752,3780,3808,3836,3864,3892,3920,3948,3976,4004,4032,4060,4088,4116,4144,4172,4200,4228,4256,4284,4312,4340,4368,4396,4424,4452,4480,4508,4536,4564,4592,4620,4648,4676,4704,4732,4760,4788,4816,4844,4872,4900,4928,4956,4984,5012,5040,5068,5096,5124,5152,5180,5208,5236,5264,5292,5320,5348,5376,5404,5432,5460,5488,5516,5544,5572,5600,5628,5656,5684,5712,5740,5768,5796,5824,5852,5880,5908,5936,5964,5992,6020,6048,6076,6104,6132,6160,6188,6216,6244,6272,6300,6328,6356,6384,6412,6440,6468,6496,6524,6552,6580,6608,6636,6664,6692,6720,6748,6776,6804,6832,6860,6888,6916,6944,6972,7000,7028,7056,7084,7112,7140,7168,7196,7224,7252,7280,7308,7336,7364,7392,7420,7448,7476,7504,7532,7560,7588,7616,7644,7672,7700,7728,7756,7784,7812,7840,7868,7896,7924,7952,7980,8008,8036,8064,8092,8120,8148,8176,8204,8232,8260,8288,8316,8344,8372,8400,8428,8456,8484,8512,8540,8568,8596,8624,8652,8680,8708,8736,8764,8792,8820,8848,8876,8904,8932,8960,8988,9016,9044,9072,9100,9128,9156,9184,9212,9240,9268,9296,9324,9352,9380,9408,9436,9464,9492,9520,9548,9576,9604,9632,9660,9688,9716,9744,9772,9800,9828,9856,9884,9912,9940,9968,9996,10024,10052,10080,10108,10136,10164,10192,10220,10248,10276,10304,10332,10360,10388,10416,10444,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,899,928,957,986,1015,1044,1073,1102,1131,1160,1189,1218,1247,1276,1305,1334,1363,1392,1421,1450,1479,1508,1537,1566,1595,1624,1653,1682,1711,1740,1769,1798,1827,1856,1885,1914,1943,1972,2001,2030,2059,2088,2117,2146,2175,2204,2233,2262,2291,2320,2349,2378,2407,2436,2465,2494,2523,2552,2581,2610,2639,2668,2697,2726,2755,2784,2813,2842,2871,2900,2929,2958,2987,3016,3045,3074,3103,3132,3161,3190,3219,3248,3277,3306,3335,3364,3393,3422,3451,3480,3509,3538,3567,3596,3625,3654,3683,3712,3741,3770,3799,3828,3857,3886,3915,3944,3973,4002,4031,4060,4089,4118,4147,4176,4205,4234,4263,4292,4321,4350,4379,4408,4437,4466,4495,4524,4553,4582,4611,4640,4669,4698,4727,4756,4785,4814,4843,4872,4901,4930,4959,4988,5017,5046,5075,5104,5133,5162,5191,5220,5249,5278,5307,5336,5365,5394,5423,5452,5481,5510,5539,5568,5597,5626,5655,5684,5713,5742,5771,5800,5829,5858,5887,5916,5945,5974,6003,6032,6061,6090,6119,6148,6177,6206,6235,6264,6293,6322,6351,6380,6409,6438,6467,6496,6525,6554,6583,6612,6641,6670,6699,6728,6757,6786,6815,6844,6873,6902,6931,6960,6989,7018,7047,7076,7105,7134,7163,7192,7221,7250,7279,7308,7337,7366,7395,7424,7453,7482,7511,7540,7569,7598,7627,7656,7685,7714,7743,7772,7801,7830,7859,7888,7917,7946,7975,8004,8033,8062,8091,8120,8149,8178,8207,8236,8265,8294,8323,8352,8381,8410,8439,8468,8497,8526,8555,8584,8613,8642,8671,8700,8729,8758,8787,8816,8845,8874,8903,8932,8961,8990,9019,9048,9077,9106,9135,9164,9193,9222,9251,9280,9309,9338,9367,9396,9425,9454,9483,9512,9541,9570,9599,9628,9657,9686,9715,9744,9773,9802,9831,9860,9889,9918,9947,9976,10005,10034,10063,10092,10121,10150,10179,10208,10237,10266,10295,10324,10353,10382,10411,10440,10469,10498,10527,10556,10585,10614,10643,10672,10701,10730,10759,10788,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,899,928,957,986,1015,1044,1073,1102,1131,1160,1189,1218,1247,1276,1305,1334,1363,1392,1421,1450,1479,1508,1537,1566,1595,1624,1653,1682,1711,1740,1769,1798,1827,1856,1885,1914,1943,1972,2001,2030,2059,2088,2117,2146,2175,2204,2233,2262,2291,2320,2349,2378,2407,2436,2465,2494,2523,2552,2581,2610,2639,2668,2697,2726,2755,2784,2813,2842,2871,2900,2929,2958,2987,3016,3045,3074,3103,3132,3161,3190,3219,3248,3277,3306,3335,3364,3393,3422,3451,3480,3509,3538,3567,3596,3625,3654,3683,3712,3741,3770,3799,3828,3857,3886,3915,3944,3973,4002,4031,4060,4089,4118,4147,4176,4205,4234,4263,4292,4321,4350,4379,4408,4437,4466,4495,4524,4553,4582,4611,4640,4669,4698,4727,4756,4785,4814,4843,4872,4901,4930,4959,4988,5017,5046,5075,5104,5133,5162,5191,5220,5249,5278,5307,5336,5365,5394,5423,5452,5481,5510,5539,5568,5597,5626,5655,5684,5713,5742,5771,5800,5829,5858,5887,5916,5945,5974,6003,6032,6061,6090,6119,6148,6177,6206,6235,6264,6293,6322,6351,6380,6409,6438,6467,6496,6525,6554,6583,6612,6641,6670,6699,6728,6757,6786,6815,6844,6873,6902,6931,6960,6989,7018,7047,7076,7105,7134,7163,7192,7221,7250,7279,7308,7337,7366,7395,7424,7453,7482,7511,7540,7569,7598,7627,7656,7685,7714,7743,7772,7801,7830,7859,7888,7917,7946,7975,8004,8033,8062,8091,8120,8149,8178,8207,8236,8265,8294,8323,8352,8381,8410,8439,8468,8497,8526,8555,8584,8613,8642,8671,8700,8729,8758,8787,8816,8845,8874,8903,8932,8961,8990,9019,9048,9077,9106,9135,9164,9193,9222,9251,9280,9309,9338,9367,9396,9425,9454,9483,9512,9541,9570,9599,9628,9657,9686,9715,9744,9773,9802,9831,9860,9889,9918,9947,9976,10005,10034,10063,10092,10121,10150,10179,10208,10237,10266,10295,10324,10353,10382,10411,10440,10469,10498,10527,10556,10585,10614,10643,10672,10701,10730,10759,10788,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,960,990,1020,1050,1080,1110,1140,1170,1200,1230,1260,1290,1320,1350,1380,1410,1440,1470,1500,1530,1560,1590,1620,1650,1680,1710,1740,1770,1800,1830,1860,1890,1920,1950,1980,2010,2040,2070,2100,2130,2160,2190,2220,2250,2280,2310,2340,2370,2400,2430,2460,2490,2520,2550,2580,2610,2640,2670,2700,2730,2760,2790,2820,2850,2880,2910,2940,2970,3000,3030,3060,3090,3120,3150,3180,3210,3240,3270,3300,3330,3360,3390,3420,3450,3480,3510,3540,3570,3600,3630,3660,3690,3720,3750,3780,3810,3840,3870,3900,3930,3960,3990,4020,4050,4080,4110,4140,4170,4200,4230,4260,4290,4320,4350,4380,4410,4440,4470,4500,4530,4560,4590,4620,4650,4680,4710,4740,4770,4800,4830,4860,4890,4920,4950,4980,5010,5040,5070,5100,5130,5160,5190,5220,5250,5280,5310,5340,5370,5400,5430,5460,5490,5520,5550,5580,5610,5640,5670,5700,5730,5760,5790,5820,5850,5880,5910,5940,5970,6000,6030,6060,6090,6120,6150,6180,6210,6240,6270,6300,6330,6360,6390,6420,6450,6480,6510,6540,6570,6600,6630,6660,6690,6720,6750,6780,6810,6840,6870,6900,6930,6960,6990,7020,7050,7080,7110,7140,7170,7200,7230,7260,7290,7320,7350,7380,7410,7440,7470,7500,7530,7560,7590,7620,7650,7680,7710,7740,7770,7800,7830,7860,7890,7920,7950,7980,8010,8040,8070,8100,8130,8160,8190,8220,8250,8280,8310,8340,8370,8400,8430,8460,8490,8520,8550,8580,8610,8640,8670,8700,8730,8760,8790,8820,8850,8880,8910,8940,8970,9000,9030,9060,9090,9120,9150,9180,9210,9240,9270,9300,9330,9360,9390,9420,9450,9480,9510,9540,9570,9600,9630,9660,9690,9720,9750,9780,9810,9840,9870,9900,9930,9960,9990,10020,10050,10080,10110,10140,10170,10200,10230,10260,10290,10320,10350,10380,10410,10440,10470,10500,10530,10560,10590,10620,10650,10680,10710,10740,10770,10800,10830,10860,10890,10920,10950,10980,11010,11040,11070,11100,11130,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,960,990,1020,1050,1080,1110,1140,1170,1200,1230,1260,1290,1320,1350,1380,1410,1440,1470,1500,1530,1560,1590,1620,1650,1680,1710,1740,1770,1800,1830,1860,1890,1920,1950,1980,2010,2040,2070,2100,2130,2160,2190,2220,2250,2280,2310,2340,2370,2400,2430,2460,2490,2520,2550,2580,2610,2640,2670,2700,2730,2760,2790,2820,2850,2880,2910,2940,2970,3000,3030,3060,3090,3120,3150,3180,3210,3240,3270,3300,3330,3360,3390,3420,3450,3480,3510,3540,3570,3600,3630,3660,3690,3720,3750,3780,3810,3840,3870,3900,3930,3960,3990,4020,4050,4080,4110,4140,4170,4200,4230,4260,4290,4320,4350,4380,4410,4440,4470,4500,4530,4560,4590,4620,4650,4680,4710,4740,4770,4800,4830,4860,4890,4920,4950,4980,5010,5040,5070,5100,5130,5160,5190,5220,5250,5280,5310,5340,5370,5400,5430,5460,5490,5520,5550,5580,5610,5640,5670,5700,5730,5760,5790,5820,5850,5880,5910,5940,5970,6000,6030,6060,6090,6120,6150,6180,6210,6240,6270,6300,6330,6360,6390,6420,6450,6480,6510,6540,6570,6600,6630,6660,6690,6720,6750,6780,6810,6840,6870,6900,6930,6960,6990,7020,7050,7080,7110,7140,7170,7200,7230,7260,7290,7320,7350,7380,7410,7440,7470,7500,7530,7560,7590,7620,7650,7680,7710,7740,7770,7800,7830,7860,7890,7920,7950,7980,8010,8040,8070,8100,8130,8160,8190,8220,8250,8280,8310,8340,8370,8400,8430,8460,8490,8520,8550,8580,8610,8640,8670,8700,8730,8760,8790,8820,8850,8880,8910,8940,8970,9000,9030,9060,9090,9120,9150,9180,9210,9240,9270,9300,9330,9360,9390,9420,9450,9480,9510,9540,9570,9600,9630,9660,9690,9720,9750,9780,9810,9840,9870,9900,9930,9960,9990,10020,10050,10080,10110,10140,10170,10200,10230,10260,10290,10320,10350,10380,10410,10440,10470,10500,10530,10560,10590,10620,10650,10680,10710,10740,10770,10800,10830,10860,10890,10920,10950,10980,11010,11040,11070,11100,11130,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1023,1054,1085,1116,1147,1178,1209,1240,1271,1302,1333,1364,1395,1426,1457,1488,1519,1550,1581,1612,1643,1674,1705,1736,1767,1798,1829,1860,1891,1922,1953,1984,2015,2046,2077,2108,2139,2170,2201,2232,2263,2294,2325,2356,2387,2418,2449,2480,2511,2542,2573,2604,2635,2666,2697,2728,2759,2790,2821,2852,2883,2914,2945,2976,3007,3038,3069,3100,3131,3162,3193,3224,3255,3286,3317,3348,3379,3410,3441,3472,3503,3534,3565,3596,3627,3658,3689,3720,3751,3782,3813,3844,3875,3906,3937,3968,3999,4030,4061,4092,4123,4154,4185,4216,4247,4278,4309,4340,4371,4402,4433,4464,4495,4526,4557,4588,4619,4650,4681,4712,4743,4774,4805,4836,4867,4898,4929,4960,4991,5022,5053,5084,5115,5146,5177,5208,5239,5270,5301,5332,5363,5394,5425,5456,5487,5518,5549,5580,5611,5642,5673,5704,5735,5766,5797,5828,5859,5890,5921,5952,5983,6014,6045,6076,6107,6138,6169,6200,6231,6262,6293,6324,6355,6386,6417,6448,6479,6510,6541,6572,6603,6634,6665,6696,6727,6758,6789,6820,6851,6882,6913,6944,6975,7006,7037,7068,7099,7130,7161,7192,7223,7254,7285,7316,7347,7378,7409,7440,7471,7502,7533,7564,7595,7626,7657,7688,7719,7750,7781,7812,7843,7874,7905,7936,7967,7998,8029,8060,8091,8122,8153,8184,8215,8246,8277,8308,8339,8370,8401,8432,8463,8494,8525,8556,8587,8618,8649,8680,8711,8742,8773,8804,8835,8866,8897,8928,8959,8990,9021,9052,9083,9114,9145,9176,9207,9238,9269,9300,9331,9362,9393,9424,9455,9486,9517,9548,9579,9610,9641,9672,9703,9734,9765,9796,9827,9858,9889,9920,9951,9982,10013,10044,10075,10106,10137,10168,10199,10230,10261,10292,10323,10354,10385,10416,10447,10478,10509,10540,10571,10602,10633,10664,10695,10726,10757,10788,10819,10850,10881,10912,10943,10974,11005,11036,11067,11098,11129,11160,11191,11222,11253,11284,11315,11346,11377,11408,11439,11470,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1023,1054,1085,1116,1147,1178,1209,1240,1271,1302,1333,1364,1395,1426,1457,1488,1519,1550,1581,1612,1643,1674,1705,1736,1767,1798,1829,1860,1891,1922,1953,1984,2015,2046,2077,2108,2139,2170,2201,2232,2263,2294,2325,2356,2387,2418,2449,2480,2511,2542,2573,2604,2635,2666,2697,2728,2759,2790,2821,2852,2883,2914,2945,2976,3007,3038,3069,3100,3131,3162,3193,3224,3255,3286,3317,3348,3379,3410,3441,3472,3503,3534,3565,3596,3627,3658,3689,3720,3751,3782,3813,3844,3875,3906,3937,3968,3999,4030,4061,4092,4123,4154,4185,4216,4247,4278,4309,4340,4371,4402,4433,4464,4495,4526,4557,4588,4619,4650,4681,4712,4743,4774,4805,4836,4867,4898,4929,4960,4991,5022,5053,5084,5115,5146,5177,5208,5239,5270,5301,5332,5363,5394,5425,5456,5487,5518,5549,5580,5611,5642,5673,5704,5735,5766,5797,5828,5859,5890,5921,5952,5983,6014,6045,6076,6107,6138,6169,6200,6231,6262,6293,6324,6355,6386,6417,6448,6479,6510,6541,6572,6603,6634,6665,6696,6727,6758,6789,6820,6851,6882,6913,6944,6975,7006,7037,7068,7099,7130,7161,7192,7223,7254,7285,7316,7347,7378,7409,7440,7471,7502,7533,7564,7595,7626,7657,7688,7719,7750,7781,7812,7843,7874,7905,7936,7967,7998,8029,8060,8091,8122,8153,8184,8215,8246,8277,8308,8339,8370,8401,8432,8463,8494,8525,8556,8587,8618,8649,8680,8711,8742,8773,8804,8835,8866,8897,8928,8959,8990,9021,9052,9083,9114,9145,9176,9207,9238,9269,9300,9331,9362,9393,9424,9455,9486,9517,9548,9579,9610,9641,9672,9703,9734,9765,9796,9827,9858,9889,9920,9951,9982,10013,10044,10075,10106,10137,10168,10199,10230,10261,10292,10323,10354,10385,10416,10447,10478,10509,10540,10571,10602,10633,10664,10695,10726,10757,10788,10819,10850,10881,10912,10943,10974,11005,11036,11067,11098,11129,11160,11191,11222,11253,11284,11315,11346,11377,11408,11439,11470,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1088,1120,1152,1184,1216,1248,1280,1312,1344,1376,1408,1440,1472,1504,1536,1568,1600,1632,1664,1696,1728,1760,1792,1824,1856,1888,1920,1952,1984,2016,2048,2080,2112,2144,2176,2208,2240,2272,2304,2336,2368,2400,2432,2464,2496,2528,2560,2592,2624,2656,2688,2720,2752,2784,2816,2848,2880,2912,2944,2976,3008,3040,3072,3104,3136,3168,3200,3232,3264,3296,3328,3360,3392,3424,3456,3488,3520,3552,3584,3616,3648,3680,3712,3744,3776,3808,3840,3872,3904,3936,3968,4000,4032,4064,4096,4128,4160,4192,4224,4256,4288,4320,4352,4384,4416,4448,4480,4512,4544,4576,4608,4640,4672,4704,4736,4768,4800,4832,4864,4896,4928,4960,4992,5024,5056,5088,5120,5152,5184,5216,5248,5280,5312,5344,5376,5408,5440,5472,5504,5536,5568,5600,5632,5664,5696,5728,5760,5792,5824,5856,5888,5920,5952,5984,6016,6048,6080,6112,6144,6176,6208,6240,6272,6304,6336,6368,6400,6432,6464,6496,6528,6560,6592,6624,6656,6688,6720,6752,6784,6816,6848,6880,6912,6944,6976,7008,7040,7072,7104,7136,7168,7200,7232,7264,7296,7328,7360,7392,7424,7456,7488,7520,7552,7584,7616,7648,7680,7712,7744,7776,7808,7840,7872,7904,7936,7968,8000,8032,8064,8096,8128,8160,8192,8224,8256,8288,8320,8352,8384,8416,8448,8480,8512,8544,8576,8608,8640,8672,8704,8736,8768,8800,8832,8864,8896,8928,8960,8992,9024,9056,9088,9120,9152,9184,9216,9248,9280,9312,9344,9376,9408,9440,9472,9504,9536,9568,9600,9632,9664,9696,9728,9760,9792,9824,9856,9888,9920,9952,9984,10016,10048,10080,10112,10144,10176,10208,10240,10272,10304,10336,10368,10400,10432,10464,10496,10528,10560,10592,10624,10656,10688,10720,10752,10784,10816,10848,10880,10912,10944,10976,11008,11040,11072,11104,11136,11168,11200,11232,11264,11296,11328,11360,11392,11424,11456,11488,11520,11552,11584,11616,11648,11680,11712,11744,11776,11808,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1088,1120,1152,1184,1216,1248,1280,1312,1344,1376,1408,1440,1472,1504,1536,1568,1600,1632,1664,1696,1728,1760,1792,1824,1856,1888,1920,1952,1984,2016,2048,2080,2112,2144,2176,2208,2240,2272,2304,2336,2368,2400,2432,2464,2496,2528,2560,2592,2624,2656,2688,2720,2752,2784,2816,2848,2880,2912,2944,2976,3008,3040,3072,3104,3136,3168,3200,3232,3264,3296,3328,3360,3392,3424,3456,3488,3520,3552,3584,3616,3648,3680,3712,3744,3776,3808,3840,3872,3904,3936,3968,4000,4032,4064,4096,4128,4160,4192,4224,4256,4288,4320,4352,4384,4416,4448,4480,4512,4544,4576,4608,4640,4672,4704,4736,4768,4800,4832,4864,4896,4928,4960,4992,5024,5056,5088,5120,5152,5184,5216,5248,5280,5312,5344,5376,5408,5440,5472,5504,5536,5568,5600,5632,5664,5696,5728,5760,5792,5824,5856,5888,5920,5952,5984,6016,6048,6080,6112,6144,6176,6208,6240,6272,6304,6336,6368,6400,6432,6464,6496,6528,6560,6592,6624,6656,6688,6720,6752,6784,6816,6848,6880,6912,6944,6976,7008,7040,7072,7104,7136,7168,7200,7232,7264,7296,7328,7360,7392,7424,7456,7488,7520,7552,7584,7616,7648,7680,7712,7744,7776,7808,7840,7872,7904,7936,7968,8000,8032,8064,8096,8128,8160,8192,8224,8256,8288,8320,8352,8384,8416,8448,8480,8512,8544,8576,8608,8640,8672,8704,8736,8768,8800,8832,8864,8896,8928,8960,8992,9024,9056,9088,9120,9152,9184,9216,9248,9280,9312,9344,9376,9408,9440,9472,9504,9536,9568,9600,9632,9664,9696,9728,9760,9792,9824,9856,9888,9920,9952,9984,10016,10048,10080,10112,10144,10176,10208,10240,10272,10304,10336,10368,10400,10432,10464,10496,10528,10560,10592,10624,10656,10688,10720,10752,10784,10816,10848,10880,10912,10944,10976,11008,11040,11072,11104,11136,11168,11200,11232,11264,11296,11328,11360,11392,11424,11456,11488,11520,11552,11584,11616,11648,11680,11712,11744,11776,11808,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1155,1188,1221,1254,1287,1320,1353,1386,1419,1452,1485,1518,1551,1584,1617,1650,1683,1716,1749,1782,1815,1848,1881,1914,1947,1980,2013,2046,2079,2112,2145,2178,2211,2244,2277,2310,2343,2376,2409,2442,2475,2508,2541,2574,2607,2640,2673,2706,2739,2772,2805,2838,2871,2904,2937,2970,3003,3036,3069,3102,3135,3168,3201,3234,3267,3300,3333,3366,3399,3432,3465,3498,3531,3564,3597,3630,3663,3696,3729,3762,3795,3828,3861,3894,3927,3960,3993,4026,4059,4092,4125,4158,4191,4224,4257,4290,4323,4356,4389,4422,4455,4488,4521,4554,4587,4620,4653,4686,4719,4752,4785,4818,4851,4884,4917,4950,4983,5016,5049,5082,5115,5148,5181,5214,5247,5280,5313,5346,5379,5412,5445,5478,5511,5544,5577,5610,5643,5676,5709,5742,5775,5808,5841,5874,5907,5940,5973,6006,6039,6072,6105,6138,6171,6204,6237,6270,6303,6336,6369,6402,6435,6468,6501,6534,6567,6600,6633,6666,6699,6732,6765,6798,6831,6864,6897,6930,6963,6996,7029,7062,7095,7128,7161,7194,7227,7260,7293,7326,7359,7392,7425,7458,7491,7524,7557,7590,7623,7656,7689,7722,7755,7788,7821,7854,7887,7920,7953,7986,8019,8052,8085,8118,8151,8184,8217,8250,8283,8316,8349,8382,8415,8448,8481,8514,8547,8580,8613,8646,8679,8712,8745,8778,8811,8844,8877,8910,8943,8976,9009,9042,9075,9108,9141,9174,9207,9240,9273,9306,9339,9372,9405,9438,9471,9504,9537,9570,9603,9636,9669,9702,9735,9768,9801,9834,9867,9900,9933,9966,9999,10032,10065,10098,10131,10164,10197,10230,10263,10296,10329,10362,10395,10428,10461,10494,10527,10560,10593,10626,10659,10692,10725,10758,10791,10824,10857,10890,10923,10956,10989,11022,11055,11088,11121,11154,11187,11220,11253,11286,11319,11352,11385,11418,11451,11484,11517,11550,11583,11616,11649,11682,11715,11748,11781,11814,11847,11880,11913,11946,11979,12012,12045,12078,12111,12144,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1155,1188,1221,1254,1287,1320,1353,1386,1419,1452,1485,1518,1551,1584,1617,1650,1683,1716,1749,1782,1815,1848,1881,1914,1947,1980,2013,2046,2079,2112,2145,2178,2211,2244,2277,2310,2343,2376,2409,2442,2475,2508,2541,2574,2607,2640,2673,2706,2739,2772,2805,2838,2871,2904,2937,2970,3003,3036,3069,3102,3135,3168,3201,3234,3267,3300,3333,3366,3399,3432,3465,3498,3531,3564,3597,3630,3663,3696,3729,3762,3795,3828,3861,3894,3927,3960,3993,4026,4059,4092,4125,4158,4191,4224,4257,4290,4323,4356,4389,4422,4455,4488,4521,4554,4587,4620,4653,4686,4719,4752,4785,4818,4851,4884,4917,4950,4983,5016,5049,5082,5115,5148,5181,5214,5247,5280,5313,5346,5379,5412,5445,5478,5511,5544,5577,5610,5643,5676,5709,5742,5775,5808,5841,5874,5907,5940,5973,6006,6039,6072,6105,6138,6171,6204,6237,6270,6303,6336,6369,6402,6435,6468,6501,6534,6567,6600,6633,6666,6699,6732,6765,6798,6831,6864,6897,6930,6963,6996,7029,7062,7095,7128,7161,7194,7227,7260,7293,7326,7359,7392,7425,7458,7491,7524,7557,7590,7623,7656,7689,7722,7755,7788,7821,7854,7887,7920,7953,7986,8019,8052,8085,8118,8151,8184,8217,8250,8283,8316,8349,8382,8415,8448,8481,8514,8547,8580,8613,8646,8679,8712,8745,8778,8811,8844,8877,8910,8943,8976,9009,9042,9075,9108,9141,9174,9207,9240,9273,9306,9339,9372,9405,9438,9471,9504,9537,9570,9603,9636,9669,9702,9735,9768,9801,9834,9867,9900,9933,9966,9999,10032,10065,10098,10131,10164,10197,10230,10263,10296,10329,10362,10395,10428,10461,10494,10527,10560,10593,10626,10659,10692,10725,10758,10791,10824,10857,10890,10923,10956,10989,11022,11055,11088,11121,11154,11187,11220,11253,11286,11319,11352,11385,11418,11451,11484,11517,11550,11583,11616,11649,11682,11715,11748,11781,11814,11847,11880,11913,11946,11979,12012,12045,12078,12111,12144,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1224,1258,1292,1326,1360,1394,1428,1462,1496,1530,1564,1598,1632,1666,1700,1734,1768,1802,1836,1870,1904,1938,1972,2006,2040,2074,2108,2142,2176,2210,2244,2278,2312,2346,2380,2414,2448,2482,2516,2550,2584,2618,2652,2686,2720,2754,2788,2822,2856,2890,2924,2958,2992,3026,3060,3094,3128,3162,3196,3230,3264,3298,3332,3366,3400,3434,3468,3502,3536,3570,3604,3638,3672,3706,3740,3774,3808,3842,3876,3910,3944,3978,4012,4046,4080,4114,4148,4182,4216,4250,4284,4318,4352,4386,4420,4454,4488,4522,4556,4590,4624,4658,4692,4726,4760,4794,4828,4862,4896,4930,4964,4998,5032,5066,5100,5134,5168,5202,5236,5270,5304,5338,5372,5406,5440,5474,5508,5542,5576,5610,5644,5678,5712,5746,5780,5814,5848,5882,5916,5950,5984,6018,6052,6086,6120,6154,6188,6222,6256,6290,6324,6358,6392,6426,6460,6494,6528,6562,6596,6630,6664,6698,6732,6766,6800,6834,6868,6902,6936,6970,7004,7038,7072,7106,7140,7174,7208,7242,7276,7310,7344,7378,7412,7446,7480,7514,7548,7582,7616,7650,7684,7718,7752,7786,7820,7854,7888,7922,7956,7990,8024,8058,8092,8126,8160,8194,8228,8262,8296,8330,8364,8398,8432,8466,8500,8534,8568,8602,8636,8670,8704,8738,8772,8806,8840,8874,8908,8942,8976,9010,9044,9078,9112,9146,9180,9214,9248,9282,9316,9350,9384,9418,9452,9486,9520,9554,9588,9622,9656,9690,9724,9758,9792,9826,9860,9894,9928,9962,9996,10030,10064,10098,10132,10166,10200,10234,10268,10302,10336,10370,10404,10438,10472,10506,10540,10574,10608,10642,10676,10710,10744,10778,10812,10846,10880,10914,10948,10982,11016,11050,11084,11118,11152,11186,11220,11254,11288,11322,11356,11390,11424,11458,11492,11526,11560,11594,11628,11662,11696,11730,11764,11798,11832,11866,11900,11934,11968,12002,12036,12070,12104,12138,12172,12206,12240,12274,12308,12342,12376,12410,12444,12478,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1224,1258,1292,1326,1360,1394,1428,1462,1496,1530,1564,1598,1632,1666,1700,1734,1768,1802,1836,1870,1904,1938,1972,2006,2040,2074,2108,2142,2176,2210,2244,2278,2312,2346,2380,2414,2448,2482,2516,2550,2584,2618,2652,2686,2720,2754,2788,2822,2856,2890,2924,2958,2992,3026,3060,3094,3128,3162,3196,3230,3264,3298,3332,3366,3400,3434,3468,3502,3536,3570,3604,3638,3672,3706,3740,3774,3808,3842,3876,3910,3944,3978,4012,4046,4080,4114,4148,4182,4216,4250,4284,4318,4352,4386,4420,4454,4488,4522,4556,4590,4624,4658,4692,4726,4760,4794,4828,4862,4896,4930,4964,4998,5032,5066,5100,5134,5168,5202,5236,5270,5304,5338,5372,5406,5440,5474,5508,5542,5576,5610,5644,5678,5712,5746,5780,5814,5848,5882,5916,5950,5984,6018,6052,6086,6120,6154,6188,6222,6256,6290,6324,6358,6392,6426,6460,6494,6528,6562,6596,6630,6664,6698,6732,6766,6800,6834,6868,6902,6936,6970,7004,7038,7072,7106,7140,7174,7208,7242,7276,7310,7344,7378,7412,7446,7480,7514,7548,7582,7616,7650,7684,7718,7752,7786,7820,7854,7888,7922,7956,7990,8024,8058,8092,8126,8160,8194,8228,8262,8296,8330,8364,8398,8432,8466,8500,8534,8568,8602,8636,8670,8704,8738,8772,8806,8840,8874,8908,8942,8976,9010,9044,9078,9112,9146,9180,9214,9248,9282,9316,9350,9384,9418,9452,9486,9520,9554,9588,9622,9656,9690,9724,9758,9792,9826,9860,9894,9928,9962,9996,10030,10064,10098,10132,10166,10200,10234,10268,10302,10336,10370,10404,10438,10472,10506,10540,10574,10608,10642,10676,10710,10744,10778,10812,10846,10880,10914,10948,10982,11016,11050,11084,11118,11152,11186,11220,11254,11288,11322,11356,11390,11424,11458,11492,11526,11560,11594,11628,11662,11696,11730,11764,11798,11832,11866,11900,11934,11968,12002,12036,12070,12104,12138,12172,12206,12240,12274,12308,12342,12376,12410,12444,12478,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1295,1330,1365,1400,1435,1470,1505,1540,1575,1610,1645,1680,1715,1750,1785,1820,1855,1890,1925,1960,1995,2030,2065,2100,2135,2170,2205,2240,2275,2310,2345,2380,2415,2450,2485,2520,2555,2590,2625,2660,2695,2730,2765,2800,2835,2870,2905,2940,2975,3010,3045,3080,3115,3150,3185,3220,3255,3290,3325,3360,3395,3430,3465,3500,3535,3570,3605,3640,3675,3710,3745,3780,3815,3850,3885,3920,3955,3990,4025,4060,4095,4130,4165,4200,4235,4270,4305,4340,4375,4410,4445,4480,4515,4550,4585,4620,4655,4690,4725,4760,4795,4830,4865,4900,4935,4970,5005,5040,5075,5110,5145,5180,5215,5250,5285,5320,5355,5390,5425,5460,5495,5530,5565,5600,5635,5670,5705,5740,5775,5810,5845,5880,5915,5950,5985,6020,6055,6090,6125,6160,6195,6230,6265,6300,6335,6370,6405,6440,6475,6510,6545,6580,6615,6650,6685,6720,6755,6790,6825,6860,6895,6930,6965,7000,7035,7070,7105,7140,7175,7210,7245,7280,7315,7350,7385,7420,7455,7490,7525,7560,7595,7630,7665,7700,7735,7770,7805,7840,7875,7910,7945,7980,8015,8050,8085,8120,8155,8190,8225,8260,8295,8330,8365,8400,8435,8470,8505,8540,8575,8610,8645,8680,8715,8750,8785,8820,8855,8890,8925,8960,8995,9030,9065,9100,9135,9170,9205,9240,9275,9310,9345,9380,9415,9450,9485,9520,9555,9590,9625,9660,9695,9730,9765,9800,9835,9870,9905,9940,9975,10010,10045,10080,10115,10150,10185,10220,10255,10290,10325,10360,10395,10430,10465,10500,10535,10570,10605,10640,10675,10710,10745,10780,10815,10850,10885,10920,10955,10990,11025,11060,11095,11130,11165,11200,11235,11270,11305,11340,11375,11410,11445,11480,11515,11550,11585,11620,11655,11690,11725,11760,11795,11830,11865,11900,11935,11970,12005,12040,12075,12110,12145,12180,12215,12250,12285,12320,12355,12390,12425,12460,12495,12530,12565,12600,12635,12670,12705,12740,12775,12810,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1295,1330,1365,1400,1435,1470,1505,1540,1575,1610,1645,1680,1715,1750,1785,1820,1855,1890,1925,1960,1995,2030,2065,2100,2135,2170,2205,2240,2275,2310,2345,2380,2415,2450,2485,2520,2555,2590,2625,2660,2695,2730,2765,2800,2835,2870,2905,2940,2975,3010,3045,3080,3115,3150,3185,3220,3255,3290,3325,3360,3395,3430,3465,3500,3535,3570,3605,3640,3675,3710,3745,3780,3815,3850,3885,3920,3955,3990,4025,4060,4095,4130,4165,4200,4235,4270,4305,4340,4375,4410,4445,4480,4515,4550,4585,4620,4655,4690,4725,4760,4795,4830,4865,4900,4935,4970,5005,5040,5075,5110,5145,5180,5215,5250,5285,5320,5355,5390,5425,5460,5495,5530,5565,5600,5635,5670,5705,5740,5775,5810,5845,5880,5915,5950,5985,6020,6055,6090,6125,6160,6195,6230,6265,6300,6335,6370,6405,6440,6475,6510,6545,6580,6615,6650,6685,6720,6755,6790,6825,6860,6895,6930,6965,7000,7035,7070,7105,7140,7175,7210,7245,7280,7315,7350,7385,7420,7455,7490,7525,7560,7595,7630,7665,7700,7735,7770,7805,7840,7875,7910,7945,7980,8015,8050,8085,8120,8155,8190,8225,8260,8295,8330,8365,8400,8435,8470,8505,8540,8575,8610,8645,8680,8715,8750,8785,8820,8855,8890,8925,8960,8995,9030,9065,9100,9135,9170,9205,9240,9275,9310,9345,9380,9415,9450,9485,9520,9555,9590,9625,9660,9695,9730,9765,9800,9835,9870,9905,9940,9975,10010,10045,10080,10115,10150,10185,10220,10255,10290,10325,10360,10395,10430,10465,10500,10535,10570,10605,10640,10675,10710,10745,10780,10815,10850,10885,10920,10955,10990,11025,11060,11095,11130,11165,11200,11235,11270,11305,11340,11375,11410,11445,11480,11515,11550,11585,11620,11655,11690,11725,11760,11795,11830,11865,11900,11935,11970,12005,12040,12075,12110,12145,12180,12215,12250,12285,12320,12355,12390,12425,12460,12495,12530,12565,12600,12635,12670,12705,12740,12775,12810,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1368,1404,1440,1476,1512,1548,1584,1620,1656,1692,1728,1764,1800,1836,1872,1908,1944,1980,2016,2052,2088,2124,2160,2196,2232,2268,2304,2340,2376,2412,2448,2484,2520,2556,2592,2628,2664,2700,2736,2772,2808,2844,2880,2916,2952,2988,3024,3060,3096,3132,3168,3204,3240,3276,3312,3348,3384,3420,3456,3492,3528,3564,3600,3636,3672,3708,3744,3780,3816,3852,3888,3924,3960,3996,4032,4068,4104,4140,4176,4212,4248,4284,4320,4356,4392,4428,4464,4500,4536,4572,4608,4644,4680,4716,4752,4788,4824,4860,4896,4932,4968,5004,5040,5076,5112,5148,5184,5220,5256,5292,5328,5364,5400,5436,5472,5508,5544,5580,5616,5652,5688,5724,5760,5796,5832,5868,5904,5940,5976,6012,6048,6084,6120,6156,6192,6228,6264,6300,6336,6372,6408,6444,6480,6516,6552,6588,6624,6660,6696,6732,6768,6804,6840,6876,6912,6948,6984,7020,7056,7092,7128,7164,7200,7236,7272,7308,7344,7380,7416,7452,7488,7524,7560,7596,7632,7668,7704,7740,7776,7812,7848,7884,7920,7956,7992,8028,8064,8100,8136,8172,8208,8244,8280,8316,8352,8388,8424,8460,8496,8532,8568,8604,8640,8676,8712,8748,8784,8820,8856,8892,8928,8964,9000,9036,9072,9108,9144,9180,9216,9252,9288,9324,9360,9396,9432,9468,9504,9540,9576,9612,9648,9684,9720,9756,9792,9828,9864,9900,9936,9972,10008,10044,10080,10116,10152,10188,10224,10260,10296,10332,10368,10404,10440,10476,10512,10548,10584,10620,10656,10692,10728,10764,10800,10836,10872,10908,10944,10980,11016,11052,11088,11124,11160,11196,11232,11268,11304,11340,11376,11412,11448,11484,11520,11556,11592,11628,11664,11700,11736,11772,11808,11844,11880,11916,11952,11988,12024,12060,12096,12132,12168,12204,12240,12276,12312,12348,12384,12420,12456,12492,12528,12564,12600,12636,12672,12708,12744,12780,12816,12852,12888,12924,12960,12996,13032,13068,13104,13140,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1368,1404,1440,1476,1512,1548,1584,1620,1656,1692,1728,1764,1800,1836,1872,1908,1944,1980,2016,2052,2088,2124,2160,2196,2232,2268,2304,2340,2376,2412,2448,2484,2520,2556,2592,2628,2664,2700,2736,2772,2808,2844,2880,2916,2952,2988,3024,3060,3096,3132,3168,3204,3240,3276,3312,3348,3384,3420,3456,3492,3528,3564,3600,3636,3672,3708,3744,3780,3816,3852,3888,3924,3960,3996,4032,4068,4104,4140,4176,4212,4248,4284,4320,4356,4392,4428,4464,4500,4536,4572,4608,4644,4680,4716,4752,4788,4824,4860,4896,4932,4968,5004,5040,5076,5112,5148,5184,5220,5256,5292,5328,5364,5400,5436,5472,5508,5544,5580,5616,5652,5688,5724,5760,5796,5832,5868,5904,5940,5976,6012,6048,6084,6120,6156,6192,6228,6264,6300,6336,6372,6408,6444,6480,6516,6552,6588,6624,6660,6696,6732,6768,6804,6840,6876,6912,6948,6984,7020,7056,7092,7128,7164,7200,7236,7272,7308,7344,7380,7416,7452,7488,7524,7560,7596,7632,7668,7704,7740,7776,7812,7848,7884,7920,7956,7992,8028,8064,8100,8136,8172,8208,8244,8280,8316,8352,8388,8424,8460,8496,8532,8568,8604,8640,8676,8712,8748,8784,8820,8856,8892,8928,8964,9000,9036,9072,9108,9144,9180,9216,9252,9288,9324,9360,9396,9432,9468,9504,9540,9576,9612,9648,9684,9720,9756,9792,9828,9864,9900,9936,9972,10008,10044,10080,10116,10152,10188,10224,10260,10296,10332,10368,10404,10440,10476,10512,10548,10584,10620,10656,10692,10728,10764,10800,10836,10872,10908,10944,10980,11016,11052,11088,11124,11160,11196,11232,11268,11304,11340,11376,11412,11448,11484,11520,11556,11592,11628,11664,11700,11736,11772,11808,11844,11880,11916,11952,11988,12024,12060,12096,12132,12168,12204,12240,12276,12312,12348,12384,12420,12456,12492,12528,12564,12600,12636,12672,12708,12744,12780,12816,12852,12888,12924,12960,12996,13032,13068,13104,13140,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1443,1480,1517,1554,1591,1628,1665,1702,1739,1776,1813,1850,1887,1924,1961,1998,2035,2072,2109,2146,2183,2220,2257,2294,2331,2368,2405,2442,2479,2516,2553,2590,2627,2664,2701,2738,2775,2812,2849,2886,2923,2960,2997,3034,3071,3108,3145,3182,3219,3256,3293,3330,3367,3404,3441,3478,3515,3552,3589,3626,3663,3700,3737,3774,3811,3848,3885,3922,3959,3996,4033,4070,4107,4144,4181,4218,4255,4292,4329,4366,4403,4440,4477,4514,4551,4588,4625,4662,4699,4736,4773,4810,4847,4884,4921,4958,4995,5032,5069,5106,5143,5180,5217,5254,5291,5328,5365,5402,5439,5476,5513,5550,5587,5624,5661,5698,5735,5772,5809,5846,5883,5920,5957,5994,6031,6068,6105,6142,6179,6216,6253,6290,6327,6364,6401,6438,6475,6512,6549,6586,6623,6660,6697,6734,6771,6808,6845,6882,6919,6956,6993,7030,7067,7104,7141,7178,7215,7252,7289,7326,7363,7400,7437,7474,7511,7548,7585,7622,7659,7696,7733,7770,7807,7844,7881,7918,7955,7992,8029,8066,8103,8140,8177,8214,8251,8288,8325,8362,8399,8436,8473,8510,8547,8584,8621,8658,8695,8732,8769,8806,8843,8880,8917,8954,8991,9028,9065,9102,9139,9176,9213,9250,9287,9324,9361,9398,9435,9472,9509,9546,9583,9620,9657,9694,9731,9768,9805,9842,9879,9916,9953,9990,10027,10064,10101,10138,10175,10212,10249,10286,10323,10360,10397,10434,10471,10508,10545,10582,10619,10656,10693,10730,10767,10804,10841,10878,10915,10952,10989,11026,11063,11100,11137,11174,11211,11248,11285,11322,11359,11396,11433,11470,11507,11544,11581,11618,11655,11692,11729,11766,11803,11840,11877,11914,11951,11988,12025,12062,12099,12136,12173,12210,12247,12284,12321,12358,12395,12432,12469,12506,12543,12580,12617,12654,12691,12728,12765,12802,12839,12876,12913,12950,12987,13024,13061,13098,13135,13172,13209,13246,13283,13320,13357,13394,13431,13468,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1443,1480,1517,1554,1591,1628,1665,1702,1739,1776,1813,1850,1887,1924,1961,1998,2035,2072,2109,2146,2183,2220,2257,2294,2331,2368,2405,2442,2479,2516,2553,2590,2627,2664,2701,2738,2775,2812,2849,2886,2923,2960,2997,3034,3071,3108,3145,3182,3219,3256,3293,3330,3367,3404,3441,3478,3515,3552,3589,3626,3663,3700,3737,3774,3811,3848,3885,3922,3959,3996,4033,4070,4107,4144,4181,4218,4255,4292,4329,4366,4403,4440,4477,4514,4551,4588,4625,4662,4699,4736,4773,4810,4847,4884,4921,4958,4995,5032,5069,5106,5143,5180,5217,5254,5291,5328,5365,5402,5439,5476,5513,5550,5587,5624,5661,5698,5735,5772,5809,5846,5883,5920,5957,5994,6031,6068,6105,6142,6179,6216,6253,6290,6327,6364,6401,6438,6475,6512,6549,6586,6623,6660,6697,6734,6771,6808,6845,6882,6919,6956,6993,7030,7067,7104,7141,7178,7215,7252,7289,7326,7363,7400,7437,7474,7511,7548,7585,7622,7659,7696,7733,7770,7807,7844,7881,7918,7955,7992,8029,8066,8103,8140,8177,8214,8251,8288,8325,8362,8399,8436,8473,8510,8547,8584,8621,8658,8695,8732,8769,8806,8843,8880,8917,8954,8991,9028,9065,9102,9139,9176,9213,9250,9287,9324,9361,9398,9435,9472,9509,9546,9583,9620,9657,9694,9731,9768,9805,9842,9879,9916,9953,9990,10027,10064,10101,10138,10175,10212,10249,10286,10323,10360,10397,10434,10471,10508,10545,10582,10619,10656,10693,10730,10767,10804,10841,10878,10915,10952,10989,11026,11063,11100,11137,11174,11211,11248,11285,11322,11359,11396,11433,11470,11507,11544,11581,11618,11655,11692,11729,11766,11803,11840,11877,11914,11951,11988,12025,12062,12099,12136,12173,12210,12247,12284,12321,12358,12395,12432,12469,12506,12543,12580,12617,12654,12691,12728,12765,12802,12839,12876,12913,12950,12987,13024,13061,13098,13135,13172,13209,13246,13283,13320,13357,13394,13431,13468,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1520,1558,1596,1634,1672,1710,1748,1786,1824,1862,1900,1938,1976,2014,2052,2090,2128,2166,2204,2242,2280,2318,2356,2394,2432,2470,2508,2546,2584,2622,2660,2698,2736,2774,2812,2850,2888,2926,2964,3002,3040,3078,3116,3154,3192,3230,3268,3306,3344,3382,3420,3458,3496,3534,3572,3610,3648,3686,3724,3762,3800,3838,3876,3914,3952,3990,4028,4066,4104,4142,4180,4218,4256,4294,4332,4370,4408,4446,4484,4522,4560,4598,4636,4674,4712,4750,4788,4826,4864,4902,4940,4978,5016,5054,5092,5130,5168,5206,5244,5282,5320,5358,5396,5434,5472,5510,5548,5586,5624,5662,5700,5738,5776,5814,5852,5890,5928,5966,6004,6042,6080,6118,6156,6194,6232,6270,6308,6346,6384,6422,6460,6498,6536,6574,6612,6650,6688,6726,6764,6802,6840,6878,6916,6954,6992,7030,7068,7106,7144,7182,7220,7258,7296,7334,7372,7410,7448,7486,7524,7562,7600,7638,7676,7714,7752,7790,7828,7866,7904,7942,7980,8018,8056,8094,8132,8170,8208,8246,8284,8322,8360,8398,8436,8474,8512,8550,8588,8626,8664,8702,8740,8778,8816,8854,8892,8930,8968,9006,9044,9082,9120,9158,9196,9234,9272,9310,9348,9386,9424,9462,9500,9538,9576,9614,9652,9690,9728,9766,9804,9842,9880,9918,9956,9994,10032,10070,10108,10146,10184,10222,10260,10298,10336,10374,10412,10450,10488,10526,10564,10602,10640,10678,10716,10754,10792,10830,10868,10906,10944,10982,11020,11058,11096,11134,11172,11210,11248,11286,11324,11362,11400,11438,11476,11514,11552,11590,11628,11666,11704,11742,11780,11818,11856,11894,11932,11970,12008,12046,12084,12122,12160,12198,12236,12274,12312,12350,12388,12426,12464,12502,12540,12578,12616,12654,12692,12730,12768,12806,12844,12882,12920,12958,12996,13034,13072,13110,13148,13186,13224,13262,13300,13338,13376,13414,13452,13490,13528,13566,13604,13642,13680,13718,13756,13794,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1520,1558,1596,1634,1672,1710,1748,1786,1824,1862,1900,1938,1976,2014,2052,2090,2128,2166,2204,2242,2280,2318,2356,2394,2432,2470,2508,2546,2584,2622,2660,2698,2736,2774,2812,2850,2888,2926,2964,3002,3040,3078,3116,3154,3192,3230,3268,3306,3344,3382,3420,3458,3496,3534,3572,3610,3648,3686,3724,3762,3800,3838,3876,3914,3952,3990,4028,4066,4104,4142,4180,4218,4256,4294,4332,4370,4408,4446,4484,4522,4560,4598,4636,4674,4712,4750,4788,4826,4864,4902,4940,4978,5016,5054,5092,5130,5168,5206,5244,5282,5320,5358,5396,5434,5472,5510,5548,5586,5624,5662,5700,5738,5776,5814,5852,5890,5928,5966,6004,6042,6080,6118,6156,6194,6232,6270,6308,6346,6384,6422,6460,6498,6536,6574,6612,6650,6688,6726,6764,6802,6840,6878,6916,6954,6992,7030,7068,7106,7144,7182,7220,7258,7296,7334,7372,7410,7448,7486,7524,7562,7600,7638,7676,7714,7752,7790,7828,7866,7904,7942,7980,8018,8056,8094,8132,8170,8208,8246,8284,8322,8360,8398,8436,8474,8512,8550,8588,8626,8664,8702,8740,8778,8816,8854,8892,8930,8968,9006,9044,9082,9120,9158,9196,9234,9272,9310,9348,9386,9424,9462,9500,9538,9576,9614,9652,9690,9728,9766,9804,9842,9880,9918,9956,9994,10032,10070,10108,10146,10184,10222,10260,10298,10336,10374,10412,10450,10488,10526,10564,10602,10640,10678,10716,10754,10792,10830,10868,10906,10944,10982,11020,11058,11096,11134,11172,11210,11248,11286,11324,11362,11400,11438,11476,11514,11552,11590,11628,11666,11704,11742,11780,11818,11856,11894,11932,11970,12008,12046,12084,12122,12160,12198,12236,12274,12312,12350,12388,12426,12464,12502,12540,12578,12616,12654,12692,12730,12768,12806,12844,12882,12920,12958,12996,13034,13072,13110,13148,13186,13224,13262,13300,13338,13376,13414,13452,13490,13528,13566,13604,13642,13680,13718,13756,13794,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1599,1638,1677,1716,1755,1794,1833,1872,1911,1950,1989,2028,2067,2106,2145,2184,2223,2262,2301,2340,2379,2418,2457,2496,2535,2574,2613,2652,2691,2730,2769,2808,2847,2886,2925,2964,3003,3042,3081,3120,3159,3198,3237,3276,3315,3354,3393,3432,3471,3510,3549,3588,3627,3666,3705,3744,3783,3822,3861,3900,3939,3978,4017,4056,4095,4134,4173,4212,4251,4290,4329,4368,4407,4446,4485,4524,4563,4602,4641,4680,4719,4758,4797,4836,4875,4914,4953,4992,5031,5070,5109,5148,5187,5226,5265,5304,5343,5382,5421,5460,5499,5538,5577,5616,5655,5694,5733,5772,5811,5850,5889,5928,5967,6006,6045,6084,6123,6162,6201,6240,6279,6318,6357,6396,6435,6474,6513,6552,6591,6630,6669,6708,6747,6786,6825,6864,6903,6942,6981,7020,7059,7098,7137,7176,7215,7254,7293,7332,7371,7410,7449,7488,7527,7566,7605,7644,7683,7722,7761,7800,7839,7878,7917,7956,7995,8034,8073,8112,8151,8190,8229,8268,8307,8346,8385,8424,8463,8502,8541,8580,8619,8658,8697,8736,8775,8814,8853,8892,8931,8970,9009,9048,9087,9126,9165,9204,9243,9282,9321,9360,9399,9438,9477,9516,9555,9594,9633,9672,9711,9750,9789,9828,9867,9906,9945,9984,10023,10062,10101,10140,10179,10218,10257,10296,10335,10374,10413,10452,10491,10530,10569,10608,10647,10686,10725,10764,10803,10842,10881,10920,10959,10998,11037,11076,11115,11154,11193,11232,11271,11310,11349,11388,11427,11466,11505,11544,11583,11622,11661,11700,11739,11778,11817,11856,11895,11934,11973,12012,12051,12090,12129,12168,12207,12246,12285,12324,12363,12402,12441,12480,12519,12558,12597,12636,12675,12714,12753,12792,12831,12870,12909,12948,12987,13026,13065,13104,13143,13182,13221,13260,13299,13338,13377,13416,13455,13494,13533,13572,13611,13650,13689,13728,13767,13806,13845,13884,13923,13962,14001,14040,14079,14118,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1599,1638,1677,1716,1755,1794,1833,1872,1911,1950,1989,2028,2067,2106,2145,2184,2223,2262,2301,2340,2379,2418,2457,2496,2535,2574,2613,2652,2691,2730,2769,2808,2847,2886,2925,2964,3003,3042,3081,3120,3159,3198,3237,3276,3315,3354,3393,3432,3471,3510,3549,3588,3627,3666,3705,3744,3783,3822,3861,3900,3939,3978,4017,4056,4095,4134,4173,4212,4251,4290,4329,4368,4407,4446,4485,4524,4563,4602,4641,4680,4719,4758,4797,4836,4875,4914,4953,4992,5031,5070,5109,5148,5187,5226,5265,5304,5343,5382,5421,5460,5499,5538,5577,5616,5655,5694,5733,5772,5811,5850,5889,5928,5967,6006,6045,6084,6123,6162,6201,6240,6279,6318,6357,6396,6435,6474,6513,6552,6591,6630,6669,6708,6747,6786,6825,6864,6903,6942,6981,7020,7059,7098,7137,7176,7215,7254,7293,7332,7371,7410,7449,7488,7527,7566,7605,7644,7683,7722,7761,7800,7839,7878,7917,7956,7995,8034,8073,8112,8151,8190,8229,8268,8307,8346,8385,8424,8463,8502,8541,8580,8619,8658,8697,8736,8775,8814,8853,8892,8931,8970,9009,9048,9087,9126,9165,9204,9243,9282,9321,9360,9399,9438,9477,9516,9555,9594,9633,9672,9711,9750,9789,9828,9867,9906,9945,9984,10023,10062,10101,10140,10179,10218,10257,10296,10335,10374,10413,10452,10491,10530,10569,10608,10647,10686,10725,10764,10803,10842,10881,10920,10959,10998,11037,11076,11115,11154,11193,11232,11271,11310,11349,11388,11427,11466,11505,11544,11583,11622,11661,11700,11739,11778,11817,11856,11895,11934,11973,12012,12051,12090,12129,12168,12207,12246,12285,12324,12363,12402,12441,12480,12519,12558,12597,12636,12675,12714,12753,12792,12831,12870,12909,12948,12987,13026,13065,13104,13143,13182,13221,13260,13299,13338,13377,13416,13455,13494,13533,13572,13611,13650,13689,13728,13767,13806,13845,13884,13923,13962,14001,14040,14079,14118,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1680,1720,1760,1800,1840,1880,1920,1960,2000,2040,2080,2120,2160,2200,2240,2280,2320,2360,2400,2440,2480,2520,2560,2600,2640,2680,2720,2760,2800,2840,2880,2920,2960,3000,3040,3080,3120,3160,3200,3240,3280,3320,3360,3400,3440,3480,3520,3560,3600,3640,3680,3720,3760,3800,3840,3880,3920,3960,4000,4040,4080,4120,4160,4200,4240,4280,4320,4360,4400,4440,4480,4520,4560,4600,4640,4680,4720,4760,4800,4840,4880,4920,4960,5000,5040,5080,5120,5160,5200,5240,5280,5320,5360,5400,5440,5480,5520,5560,5600,5640,5680,5720,5760,5800,5840,5880,5920,5960,6000,6040,6080,6120,6160,6200,6240,6280,6320,6360,6400,6440,6480,6520,6560,6600,6640,6680,6720,6760,6800,6840,6880,6920,6960,7000,7040,7080,7120,7160,7200,7240,7280,7320,7360,7400,7440,7480,7520,7560,7600,7640,7680,7720,7760,7800,7840,7880,7920,7960,8000,8040,8080,8120,8160,8200,8240,8280,8320,8360,8400,8440,8480,8520,8560,8600,8640,8680,8720,8760,8800,8840,8880,8920,8960,9000,9040,9080,9120,9160,9200,9240,9280,9320,9360,9400,9440,9480,9520,9560,9600,9640,9680,9720,9760,9800,9840,9880,9920,9960,10000,10040,10080,10120,10160,10200,10240,10280,10320,10360,10400,10440,10480,10520,10560,10600,10640,10680,10720,10760,10800,10840,10880,10920,10960,11000,11040,11080,11120,11160,11200,11240,11280,11320,11360,11400,11440,11480,11520,11560,11600,11640,11680,11720,11760,11800,11840,11880,11920,11960,12000,12040,12080,12120,12160,12200,12240,12280,12320,12360,12400,12440,12480,12520,12560,12600,12640,12680,12720,12760,12800,12840,12880,12920,12960,13000,13040,13080,13120,13160,13200,13240,13280,13320,13360,13400,13440,13480,13520,13560,13600,13640,13680,13720,13760,13800,13840,13880,13920,13960,14000,14040,14080,14120,14160,14200,14240,14280,14320,14360,14400,14440,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1680,1720,1760,1800,1840,1880,1920,1960,2000,2040,2080,2120,2160,2200,2240,2280,2320,2360,2400,2440,2480,2520,2560,2600,2640,2680,2720,2760,2800,2840,2880,2920,2960,3000,3040,3080,3120,3160,3200,3240,3280,3320,3360,3400,3440,3480,3520,3560,3600,3640,3680,3720,3760,3800,3840,3880,3920,3960,4000,4040,4080,4120,4160,4200,4240,4280,4320,4360,4400,4440,4480,4520,4560,4600,4640,4680,4720,4760,4800,4840,4880,4920,4960,5000,5040,5080,5120,5160,5200,5240,5280,5320,5360,5400,5440,5480,5520,5560,5600,5640,5680,5720,5760,5800,5840,5880,5920,5960,6000,6040,6080,6120,6160,6200,6240,6280,6320,6360,6400,6440,6480,6520,6560,6600,6640,6680,6720,6760,6800,6840,6880,6920,6960,7000,7040,7080,7120,7160,7200,7240,7280,7320,7360,7400,7440,7480,7520,7560,7600,7640,7680,7720,7760,7800,7840,7880,7920,7960,8000,8040,8080,8120,8160,8200,8240,8280,8320,8360,8400,8440,8480,8520,8560,8600,8640,8680,8720,8760,8800,8840,8880,8920,8960,9000,9040,9080,9120,9160,9200,9240,9280,9320,9360,9400,9440,9480,9520,9560,9600,9640,9680,9720,9760,9800,9840,9880,9920,9960,10000,10040,10080,10120,10160,10200,10240,10280,10320,10360,10400,10440,10480,10520,10560,10600,10640,10680,10720,10760,10800,10840,10880,10920,10960,11000,11040,11080,11120,11160,11200,11240,11280,11320,11360,11400,11440,11480,11520,11560,11600,11640,11680,11720,11760,11800,11840,11880,11920,11960,12000,12040,12080,12120,12160,12200,12240,12280,12320,12360,12400,12440,12480,12520,12560,12600,12640,12680,12720,12760,12800,12840,12880,12920,12960,13000,13040,13080,13120,13160,13200,13240,13280,13320,13360,13400,13440,13480,13520,13560,13600,13640,13680,13720,13760,13800,13840,13880,13920,13960,14000,14040,14080,14120,14160,14200,14240,14280,14320,14360,14400,14440,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1763,1804,1845,1886,1927,1968,2009,2050,2091,2132,2173,2214,2255,2296,2337,2378,2419,2460,2501,2542,2583,2624,2665,2706,2747,2788,2829,2870,2911,2952,2993,3034,3075,3116,3157,3198,3239,3280,3321,3362,3403,3444,3485,3526,3567,3608,3649,3690,3731,3772,3813,3854,3895,3936,3977,4018,4059,4100,4141,4182,4223,4264,4305,4346,4387,4428,4469,4510,4551,4592,4633,4674,4715,4756,4797,4838,4879,4920,4961,5002,5043,5084,5125,5166,5207,5248,5289,5330,5371,5412,5453,5494,5535,5576,5617,5658,5699,5740,5781,5822,5863,5904,5945,5986,6027,6068,6109,6150,6191,6232,6273,6314,6355,6396,6437,6478,6519,6560,6601,6642,6683,6724,6765,6806,6847,6888,6929,6970,7011,7052,7093,7134,7175,7216,7257,7298,7339,7380,7421,7462,7503,7544,7585,7626,7667,7708,7749,7790,7831,7872,7913,7954,7995,8036,8077,8118,8159,8200,8241,8282,8323,8364,8405,8446,8487,8528,8569,8610,8651,8692,8733,8774,8815,8856,8897,8938,8979,9020,9061,9102,9143,9184,9225,9266,9307,9348,9389,9430,9471,9512,9553,9594,9635,9676,9717,9758,9799,9840,9881,9922,9963,10004,10045,10086,10127,10168,10209,10250,10291,10332,10373,10414,10455,10496,10537,10578,10619,10660,10701,10742,10783,10824,10865,10906,10947,10988,11029,11070,11111,11152,11193,11234,11275,11316,11357,11398,11439,11480,11521,11562,11603,11644,11685,11726,11767,11808,11849,11890,11931,11972,12013,12054,12095,12136,12177,12218,12259,12300,12341,12382,12423,12464,12505,12546,12587,12628,12669,12710,12751,12792,12833,12874,12915,12956,12997,13038,13079,13120,13161,13202,13243,13284,13325,13366,13407,13448,13489,13530,13571,13612,13653,13694,13735,13776,13817,13858,13899,13940,13981,14022,14063,14104,14145,14186,14227,14268,14309,14350,14391,14432,14473,14514,14555,14596,14637,14678,14719,14760,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1763,1804,1845,1886,1927,1968,2009,2050,2091,2132,2173,2214,2255,2296,2337,2378,2419,2460,2501,2542,2583,2624,2665,2706,2747,2788,2829,2870,2911,2952,2993,3034,3075,3116,3157,3198,3239,3280,3321,3362,3403,3444,3485,3526,3567,3608,3649,3690,3731,3772,3813,3854,3895,3936,3977,4018,4059,4100,4141,4182,4223,4264,4305,4346,4387,4428,4469,4510,4551,4592,4633,4674,4715,4756,4797,4838,4879,4920,4961,5002,5043,5084,5125,5166,5207,5248,5289,5330,5371,5412,5453,5494,5535,5576,5617,5658,5699,5740,5781,5822,5863,5904,5945,5986,6027,6068,6109,6150,6191,6232,6273,6314,6355,6396,6437,6478,6519,6560,6601,6642,6683,6724,6765,6806,6847,6888,6929,6970,7011,7052,7093,7134,7175,7216,7257,7298,7339,7380,7421,7462,7503,7544,7585,7626,7667,7708,7749,7790,7831,7872,7913,7954,7995,8036,8077,8118,8159,8200,8241,8282,8323,8364,8405,8446,8487,8528,8569,8610,8651,8692,8733,8774,8815,8856,8897,8938,8979,9020,9061,9102,9143,9184,9225,9266,9307,9348,9389,9430,9471,9512,9553,9594,9635,9676,9717,9758,9799,9840,9881,9922,9963,10004,10045,10086,10127,10168,10209,10250,10291,10332,10373,10414,10455,10496,10537,10578,10619,10660,10701,10742,10783,10824,10865,10906,10947,10988,11029,11070,11111,11152,11193,11234,11275,11316,11357,11398,11439,11480,11521,11562,11603,11644,11685,11726,11767,11808,11849,11890,11931,11972,12013,12054,12095,12136,12177,12218,12259,12300,12341,12382,12423,12464,12505,12546,12587,12628,12669,12710,12751,12792,12833,12874,12915,12956,12997,13038,13079,13120,13161,13202,13243,13284,13325,13366,13407,13448,13489,13530,13571,13612,13653,13694,13735,13776,13817,13858,13899,13940,13981,14022,14063,14104,14145,14186,14227,14268,14309,14350,14391,14432,14473,14514,14555,14596,14637,14678,14719,14760,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1848,1890,1932,1974,2016,2058,2100,2142,2184,2226,2268,2310,2352,2394,2436,2478,2520,2562,2604,2646,2688,2730,2772,2814,2856,2898,2940,2982,3024,3066,3108,3150,3192,3234,3276,3318,3360,3402,3444,3486,3528,3570,3612,3654,3696,3738,3780,3822,3864,3906,3948,3990,4032,4074,4116,4158,4200,4242,4284,4326,4368,4410,4452,4494,4536,4578,4620,4662,4704,4746,4788,4830,4872,4914,4956,4998,5040,5082,5124,5166,5208,5250,5292,5334,5376,5418,5460,5502,5544,5586,5628,5670,5712,5754,5796,5838,5880,5922,5964,6006,6048,6090,6132,6174,6216,6258,6300,6342,6384,6426,6468,6510,6552,6594,6636,6678,6720,6762,6804,6846,6888,6930,6972,7014,7056,7098,7140,7182,7224,7266,7308,7350,7392,7434,7476,7518,7560,7602,7644,7686,7728,7770,7812,7854,7896,7938,7980,8022,8064,8106,8148,8190,8232,8274,8316,8358,8400,8442,8484,8526,8568,8610,8652,8694,8736,8778,8820,8862,8904,8946,8988,9030,9072,9114,9156,9198,9240,9282,9324,9366,9408,9450,9492,9534,9576,9618,9660,9702,9744,9786,9828,9870,9912,9954,9996,10038,10080,10122,10164,10206,10248,10290,10332,10374,10416,10458,10500,10542,10584,10626,10668,10710,10752,10794,10836,10878,10920,10962,11004,11046,11088,11130,11172,11214,11256,11298,11340,11382,11424,11466,11508,11550,11592,11634,11676,11718,11760,11802,11844,11886,11928,11970,12012,12054,12096,12138,12180,12222,12264,12306,12348,12390,12432,12474,12516,12558,12600,12642,12684,12726,12768,12810,12852,12894,12936,12978,13020,13062,13104,13146,13188,13230,13272,13314,13356,13398,13440,13482,13524,13566,13608,13650,13692,13734,13776,13818,13860,13902,13944,13986,14028,14070,14112,14154,14196,14238,14280,14322,14364,14406,14448,14490,14532,14574,14616,14658,14700,14742,14784,14826,14868,14910,14952,14994,15036,15078,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1848,1890,1932,1974,2016,2058,2100,2142,2184,2226,2268,2310,2352,2394,2436,2478,2520,2562,2604,2646,2688,2730,2772,2814,2856,2898,2940,2982,3024,3066,3108,3150,3192,3234,3276,3318,3360,3402,3444,3486,3528,3570,3612,3654,3696,3738,3780,3822,3864,3906,3948,3990,4032,4074,4116,4158,4200,4242,4284,4326,4368,4410,4452,4494,4536,4578,4620,4662,4704,4746,4788,4830,4872,4914,4956,4998,5040,5082,5124,5166,5208,5250,5292,5334,5376,5418,5460,5502,5544,5586,5628,5670,5712,5754,5796,5838,5880,5922,5964,6006,6048,6090,6132,6174,6216,6258,6300,6342,6384,6426,6468,6510,6552,6594,6636,6678,6720,6762,6804,6846,6888,6930,6972,7014,7056,7098,7140,7182,7224,7266,7308,7350,7392,7434,7476,7518,7560,7602,7644,7686,7728,7770,7812,7854,7896,7938,7980,8022,8064,8106,8148,8190,8232,8274,8316,8358,8400,8442,8484,8526,8568,8610,8652,8694,8736,8778,8820,8862,8904,8946,8988,9030,9072,9114,9156,9198,9240,9282,9324,9366,9408,9450,9492,9534,9576,9618,9660,9702,9744,9786,9828,9870,9912,9954,9996,10038,10080,10122,10164,10206,10248,10290,10332,10374,10416,10458,10500,10542,10584,10626,10668,10710,10752,10794,10836,10878,10920,10962,11004,11046,11088,11130,11172,11214,11256,11298,11340,11382,11424,11466,11508,11550,11592,11634,11676,11718,11760,11802,11844,11886,11928,11970,12012,12054,12096,12138,12180,12222,12264,12306,12348,12390,12432,12474,12516,12558,12600,12642,12684,12726,12768,12810,12852,12894,12936,12978,13020,13062,13104,13146,13188,13230,13272,13314,13356,13398,13440,13482,13524,13566,13608,13650,13692,13734,13776,13818,13860,13902,13944,13986,14028,14070,14112,14154,14196,14238,14280,14322,14364,14406,14448,14490,14532,14574,14616,14658,14700,14742,14784,14826,14868,14910,14952,14994,15036,15078,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1935,1978,2021,2064,2107,2150,2193,2236,2279,2322,2365,2408,2451,2494,2537,2580,2623,2666,2709,2752,2795,2838,2881,2924,2967,3010,3053,3096,3139,3182,3225,3268,3311,3354,3397,3440,3483,3526,3569,3612,3655,3698,3741,3784,3827,3870,3913,3956,3999,4042,4085,4128,4171,4214,4257,4300,4343,4386,4429,4472,4515,4558,4601,4644,4687,4730,4773,4816,4859,4902,4945,4988,5031,5074,5117,5160,5203,5246,5289,5332,5375,5418,5461,5504,5547,5590,5633,5676,5719,5762,5805,5848,5891,5934,5977,6020,6063,6106,6149,6192,6235,6278,6321,6364,6407,6450,6493,6536,6579,6622,6665,6708,6751,6794,6837,6880,6923,6966,7009,7052,7095,7138,7181,7224,7267,7310,7353,7396,7439,7482,7525,7568,7611,7654,7697,7740,7783,7826,7869,7912,7955,7998,8041,8084,8127,8170,8213,8256,8299,8342,8385,8428,8471,8514,8557,8600,8643,8686,8729,8772,8815,8858,8901,8944,8987,9030,9073,9116,9159,9202,9245,9288,9331,9374,9417,9460,9503,9546,9589,9632,9675,9718,9761,9804,9847,9890,9933,9976,10019,10062,10105,10148,10191,10234,10277,10320,10363,10406,10449,10492,10535,10578,10621,10664,10707,10750,10793,10836,10879,10922,10965,11008,11051,11094,11137,11180,11223,11266,11309,11352,11395,11438,11481,11524,11567,11610,11653,11696,11739,11782,11825,11868,11911,11954,11997,12040,12083,12126,12169,12212,12255,12298,12341,12384,12427,12470,12513,12556,12599,12642,12685,12728,12771,12814,12857,12900,12943,12986,13029,13072,13115,13158,13201,13244,13287,13330,13373,13416,13459,13502,13545,13588,13631,13674,13717,13760,13803,13846,13889,13932,13975,14018,14061,14104,14147,14190,14233,14276,14319,14362,14405,14448,14491,14534,14577,14620,14663,14706,14749,14792,14835,14878,14921,14964,15007,15050,15093,15136,15179,15222,15265,15308,15351,15394,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1935,1978,2021,2064,2107,2150,2193,2236,2279,2322,2365,2408,2451,2494,2537,2580,2623,2666,2709,2752,2795,2838,2881,2924,2967,3010,3053,3096,3139,3182,3225,3268,3311,3354,3397,3440,3483,3526,3569,3612,3655,3698,3741,3784,3827,3870,3913,3956,3999,4042,4085,4128,4171,4214,4257,4300,4343,4386,4429,4472,4515,4558,4601,4644,4687,4730,4773,4816,4859,4902,4945,4988,5031,5074,5117,5160,5203,5246,5289,5332,5375,5418,5461,5504,5547,5590,5633,5676,5719,5762,5805,5848,5891,5934,5977,6020,6063,6106,6149,6192,6235,6278,6321,6364,6407,6450,6493,6536,6579,6622,6665,6708,6751,6794,6837,6880,6923,6966,7009,7052,7095,7138,7181,7224,7267,7310,7353,7396,7439,7482,7525,7568,7611,7654,7697,7740,7783,7826,7869,7912,7955,7998,8041,8084,8127,8170,8213,8256,8299,8342,8385,8428,8471,8514,8557,8600,8643,8686,8729,8772,8815,8858,8901,8944,8987,9030,9073,9116,9159,9202,9245,9288,9331,9374,9417,9460,9503,9546,9589,9632,9675,9718,9761,9804,9847,9890,9933,9976,10019,10062,10105,10148,10191,10234,10277,10320,10363,10406,10449,10492,10535,10578,10621,10664,10707,10750,10793,10836,10879,10922,10965,11008,11051,11094,11137,11180,11223,11266,11309,11352,11395,11438,11481,11524,11567,11610,11653,11696,11739,11782,11825,11868,11911,11954,11997,12040,12083,12126,12169,12212,12255,12298,12341,12384,12427,12470,12513,12556,12599,12642,12685,12728,12771,12814,12857,12900,12943,12986,13029,13072,13115,13158,13201,13244,13287,13330,13373,13416,13459,13502,13545,13588,13631,13674,13717,13760,13803,13846,13889,13932,13975,14018,14061,14104,14147,14190,14233,14276,14319,14362,14405,14448,14491,14534,14577,14620,14663,14706,14749,14792,14835,14878,14921,14964,15007,15050,15093,15136,15179,15222,15265,15308,15351,15394,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2024,2068,2112,2156,2200,2244,2288,2332,2376,2420,2464,2508,2552,2596,2640,2684,2728,2772,2816,2860,2904,2948,2992,3036,3080,3124,3168,3212,3256,3300,3344,3388,3432,3476,3520,3564,3608,3652,3696,3740,3784,3828,3872,3916,3960,4004,4048,4092,4136,4180,4224,4268,4312,4356,4400,4444,4488,4532,4576,4620,4664,4708,4752,4796,4840,4884,4928,4972,5016,5060,5104,5148,5192,5236,5280,5324,5368,5412,5456,5500,5544,5588,5632,5676,5720,5764,5808,5852,5896,5940,5984,6028,6072,6116,6160,6204,6248,6292,6336,6380,6424,6468,6512,6556,6600,6644,6688,6732,6776,6820,6864,6908,6952,6996,7040,7084,7128,7172,7216,7260,7304,7348,7392,7436,7480,7524,7568,7612,7656,7700,7744,7788,7832,7876,7920,7964,8008,8052,8096,8140,8184,8228,8272,8316,8360,8404,8448,8492,8536,8580,8624,8668,8712,8756,8800,8844,8888,8932,8976,9020,9064,9108,9152,9196,9240,9284,9328,9372,9416,9460,9504,9548,9592,9636,9680,9724,9768,9812,9856,9900,9944,9988,10032,10076,10120,10164,10208,10252,10296,10340,10384,10428,10472,10516,10560,10604,10648,10692,10736,10780,10824,10868,10912,10956,11000,11044,11088,11132,11176,11220,11264,11308,11352,11396,11440,11484,11528,11572,11616,11660,11704,11748,11792,11836,11880,11924,11968,12012,12056,12100,12144,12188,12232,12276,12320,12364,12408,12452,12496,12540,12584,12628,12672,12716,12760,12804,12848,12892,12936,12980,13024,13068,13112,13156,13200,13244,13288,13332,13376,13420,13464,13508,13552,13596,13640,13684,13728,13772,13816,13860,13904,13948,13992,14036,14080,14124,14168,14212,14256,14300,14344,14388,14432,14476,14520,14564,14608,14652,14696,14740,14784,14828,14872,14916,14960,15004,15048,15092,15136,15180,15224,15268,15312,15356,15400,15444,15488,15532,15576,15620,15664,15708,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2024,2068,2112,2156,2200,2244,2288,2332,2376,2420,2464,2508,2552,2596,2640,2684,2728,2772,2816,2860,2904,2948,2992,3036,3080,3124,3168,3212,3256,3300,3344,3388,3432,3476,3520,3564,3608,3652,3696,3740,3784,3828,3872,3916,3960,4004,4048,4092,4136,4180,4224,4268,4312,4356,4400,4444,4488,4532,4576,4620,4664,4708,4752,4796,4840,4884,4928,4972,5016,5060,5104,5148,5192,5236,5280,5324,5368,5412,5456,5500,5544,5588,5632,5676,5720,5764,5808,5852,5896,5940,5984,6028,6072,6116,6160,6204,6248,6292,6336,6380,6424,6468,6512,6556,6600,6644,6688,6732,6776,6820,6864,6908,6952,6996,7040,7084,7128,7172,7216,7260,7304,7348,7392,7436,7480,7524,7568,7612,7656,7700,7744,7788,7832,7876,7920,7964,8008,8052,8096,8140,8184,8228,8272,8316,8360,8404,8448,8492,8536,8580,8624,8668,8712,8756,8800,8844,8888,8932,8976,9020,9064,9108,9152,9196,9240,9284,9328,9372,9416,9460,9504,9548,9592,9636,9680,9724,9768,9812,9856,9900,9944,9988,10032,10076,10120,10164,10208,10252,10296,10340,10384,10428,10472,10516,10560,10604,10648,10692,10736,10780,10824,10868,10912,10956,11000,11044,11088,11132,11176,11220,11264,11308,11352,11396,11440,11484,11528,11572,11616,11660,11704,11748,11792,11836,11880,11924,11968,12012,12056,12100,12144,12188,12232,12276,12320,12364,12408,12452,12496,12540,12584,12628,12672,12716,12760,12804,12848,12892,12936,12980,13024,13068,13112,13156,13200,13244,13288,13332,13376,13420,13464,13508,13552,13596,13640,13684,13728,13772,13816,13860,13904,13948,13992,14036,14080,14124,14168,14212,14256,14300,14344,14388,14432,14476,14520,14564,14608,14652,14696,14740,14784,14828,14872,14916,14960,15004,15048,15092,15136,15180,15224,15268,15312,15356,15400,15444,15488,15532,15576,15620,15664,15708,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2115,2160,2205,2250,2295,2340,2385,2430,2475,2520,2565,2610,2655,2700,2745,2790,2835,2880,2925,2970,3015,3060,3105,3150,3195,3240,3285,3330,3375,3420,3465,3510,3555,3600,3645,3690,3735,3780,3825,3870,3915,3960,4005,4050,4095,4140,4185,4230,4275,4320,4365,4410,4455,4500,4545,4590,4635,4680,4725,4770,4815,4860,4905,4950,4995,5040,5085,5130,5175,5220,5265,5310,5355,5400,5445,5490,5535,5580,5625,5670,5715,5760,5805,5850,5895,5940,5985,6030,6075,6120,6165,6210,6255,6300,6345,6390,6435,6480,6525,6570,6615,6660,6705,6750,6795,6840,6885,6930,6975,7020,7065,7110,7155,7200,7245,7290,7335,7380,7425,7470,7515,7560,7605,7650,7695,7740,7785,7830,7875,7920,7965,8010,8055,8100,8145,8190,8235,8280,8325,8370,8415,8460,8505,8550,8595,8640,8685,8730,8775,8820,8865,8910,8955,9000,9045,9090,9135,9180,9225,9270,9315,9360,9405,9450,9495,9540,9585,9630,9675,9720,9765,9810,9855,9900,9945,9990,10035,10080,10125,10170,10215,10260,10305,10350,10395,10440,10485,10530,10575,10620,10665,10710,10755,10800,10845,10890,10935,10980,11025,11070,11115,11160,11205,11250,11295,11340,11385,11430,11475,11520,11565,11610,11655,11700,11745,11790,11835,11880,11925,11970,12015,12060,12105,12150,12195,12240,12285,12330,12375,12420,12465,12510,12555,12600,12645,12690,12735,12780,12825,12870,12915,12960,13005,13050,13095,13140,13185,13230,13275,13320,13365,13410,13455,13500,13545,13590,13635,13680,13725,13770,13815,13860,13905,13950,13995,14040,14085,14130,14175,14220,14265,14310,14355,14400,14445,14490,14535,14580,14625,14670,14715,14760,14805,14850,14895,14940,14985,15030,15075,15120,15165,15210,15255,15300,15345,15390,15435,15480,15525,15570,15615,15660,15705,15750,15795,15840,15885,15930,15975,16020,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2115,2160,2205,2250,2295,2340,2385,2430,2475,2520,2565,2610,2655,2700,2745,2790,2835,2880,2925,2970,3015,3060,3105,3150,3195,3240,3285,3330,3375,3420,3465,3510,3555,3600,3645,3690,3735,3780,3825,3870,3915,3960,4005,4050,4095,4140,4185,4230,4275,4320,4365,4410,4455,4500,4545,4590,4635,4680,4725,4770,4815,4860,4905,4950,4995,5040,5085,5130,5175,5220,5265,5310,5355,5400,5445,5490,5535,5580,5625,5670,5715,5760,5805,5850,5895,5940,5985,6030,6075,6120,6165,6210,6255,6300,6345,6390,6435,6480,6525,6570,6615,6660,6705,6750,6795,6840,6885,6930,6975,7020,7065,7110,7155,7200,7245,7290,7335,7380,7425,7470,7515,7560,7605,7650,7695,7740,7785,7830,7875,7920,7965,8010,8055,8100,8145,8190,8235,8280,8325,8370,8415,8460,8505,8550,8595,8640,8685,8730,8775,8820,8865,8910,8955,9000,9045,9090,9135,9180,9225,9270,9315,9360,9405,9450,9495,9540,9585,9630,9675,9720,9765,9810,9855,9900,9945,9990,10035,10080,10125,10170,10215,10260,10305,10350,10395,10440,10485,10530,10575,10620,10665,10710,10755,10800,10845,10890,10935,10980,11025,11070,11115,11160,11205,11250,11295,11340,11385,11430,11475,11520,11565,11610,11655,11700,11745,11790,11835,11880,11925,11970,12015,12060,12105,12150,12195,12240,12285,12330,12375,12420,12465,12510,12555,12600,12645,12690,12735,12780,12825,12870,12915,12960,13005,13050,13095,13140,13185,13230,13275,13320,13365,13410,13455,13500,13545,13590,13635,13680,13725,13770,13815,13860,13905,13950,13995,14040,14085,14130,14175,14220,14265,14310,14355,14400,14445,14490,14535,14580,14625,14670,14715,14760,14805,14850,14895,14940,14985,15030,15075,15120,15165,15210,15255,15300,15345,15390,15435,15480,15525,15570,15615,15660,15705,15750,15795,15840,15885,15930,15975,16020,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2208,2254,2300,2346,2392,2438,2484,2530,2576,2622,2668,2714,2760,2806,2852,2898,2944,2990,3036,3082,3128,3174,3220,3266,3312,3358,3404,3450,3496,3542,3588,3634,3680,3726,3772,3818,3864,3910,3956,4002,4048,4094,4140,4186,4232,4278,4324,4370,4416,4462,4508,4554,4600,4646,4692,4738,4784,4830,4876,4922,4968,5014,5060,5106,5152,5198,5244,5290,5336,5382,5428,5474,5520,5566,5612,5658,5704,5750,5796,5842,5888,5934,5980,6026,6072,6118,6164,6210,6256,6302,6348,6394,6440,6486,6532,6578,6624,6670,6716,6762,6808,6854,6900,6946,6992,7038,7084,7130,7176,7222,7268,7314,7360,7406,7452,7498,7544,7590,7636,7682,7728,7774,7820,7866,7912,7958,8004,8050,8096,8142,8188,8234,8280,8326,8372,8418,8464,8510,8556,8602,8648,8694,8740,8786,8832,8878,8924,8970,9016,9062,9108,9154,9200,9246,9292,9338,9384,9430,9476,9522,9568,9614,9660,9706,9752,9798,9844,9890,9936,9982,10028,10074,10120,10166,10212,10258,10304,10350,10396,10442,10488,10534,10580,10626,10672,10718,10764,10810,10856,10902,10948,10994,11040,11086,11132,11178,11224,11270,11316,11362,11408,11454,11500,11546,11592,11638,11684,11730,11776,11822,11868,11914,11960,12006,12052,12098,12144,12190,12236,12282,12328,12374,12420,12466,12512,12558,12604,12650,12696,12742,12788,12834,12880,12926,12972,13018,13064,13110,13156,13202,13248,13294,13340,13386,13432,13478,13524,13570,13616,13662,13708,13754,13800,13846,13892,13938,13984,14030,14076,14122,14168,14214,14260,14306,14352,14398,14444,14490,14536,14582,14628,14674,14720,14766,14812,14858,14904,14950,14996,15042,15088,15134,15180,15226,15272,15318,15364,15410,15456,15502,15548,15594,15640,15686,15732,15778,15824,15870,15916,15962,16008,16054,16100,16146,16192,16238,16284,16330,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2208,2254,2300,2346,2392,2438,2484,2530,2576,2622,2668,2714,2760,2806,2852,2898,2944,2990,3036,3082,3128,3174,3220,3266,3312,3358,3404,3450,3496,3542,3588,3634,3680,3726,3772,3818,3864,3910,3956,4002,4048,4094,4140,4186,4232,4278,4324,4370,4416,4462,4508,4554,4600,4646,4692,4738,4784,4830,4876,4922,4968,5014,5060,5106,5152,5198,5244,5290,5336,5382,5428,5474,5520,5566,5612,5658,5704,5750,5796,5842,5888,5934,5980,6026,6072,6118,6164,6210,6256,6302,6348,6394,6440,6486,6532,6578,6624,6670,6716,6762,6808,6854,6900,6946,6992,7038,7084,7130,7176,7222,7268,7314,7360,7406,7452,7498,7544,7590,7636,7682,7728,7774,7820,7866,7912,7958,8004,8050,8096,8142,8188,8234,8280,8326,8372,8418,8464,8510,8556,8602,8648,8694,8740,8786,8832,8878,8924,8970,9016,9062,9108,9154,9200,9246,9292,9338,9384,9430,9476,9522,9568,9614,9660,9706,9752,9798,9844,9890,9936,9982,10028,10074,10120,10166,10212,10258,10304,10350,10396,10442,10488,10534,10580,10626,10672,10718,10764,10810,10856,10902,10948,10994,11040,11086,11132,11178,11224,11270,11316,11362,11408,11454,11500,11546,11592,11638,11684,11730,11776,11822,11868,11914,11960,12006,12052,12098,12144,12190,12236,12282,12328,12374,12420,12466,12512,12558,12604,12650,12696,12742,12788,12834,12880,12926,12972,13018,13064,13110,13156,13202,13248,13294,13340,13386,13432,13478,13524,13570,13616,13662,13708,13754,13800,13846,13892,13938,13984,14030,14076,14122,14168,14214,14260,14306,14352,14398,14444,14490,14536,14582,14628,14674,14720,14766,14812,14858,14904,14950,14996,15042,15088,15134,15180,15226,15272,15318,15364,15410,15456,15502,15548,15594,15640,15686,15732,15778,15824,15870,15916,15962,16008,16054,16100,16146,16192,16238,16284,16330,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2303,2350,2397,2444,2491,2538,2585,2632,2679,2726,2773,2820,2867,2914,2961,3008,3055,3102,3149,3196,3243,3290,3337,3384,3431,3478,3525,3572,3619,3666,3713,3760,3807,3854,3901,3948,3995,4042,4089,4136,4183,4230,4277,4324,4371,4418,4465,4512,4559,4606,4653,4700,4747,4794,4841,4888,4935,4982,5029,5076,5123,5170,5217,5264,5311,5358,5405,5452,5499,5546,5593,5640,5687,5734,5781,5828,5875,5922,5969,6016,6063,6110,6157,6204,6251,6298,6345,6392,6439,6486,6533,6580,6627,6674,6721,6768,6815,6862,6909,6956,7003,7050,7097,7144,7191,7238,7285,7332,7379,7426,7473,7520,7567,7614,7661,7708,7755,7802,7849,7896,7943,7990,8037,8084,8131,8178,8225,8272,8319,8366,8413,8460,8507,8554,8601,8648,8695,8742,8789,8836,8883,8930,8977,9024,9071,9118,9165,9212,9259,9306,9353,9400,9447,9494,9541,9588,9635,9682,9729,9776,9823,9870,9917,9964,10011,10058,10105,10152,10199,10246,10293,10340,10387,10434,10481,10528,10575,10622,10669,10716,10763,10810,10857,10904,10951,10998,11045,11092,11139,11186,11233,11280,11327,11374,11421,11468,11515,11562,11609,11656,11703,11750,11797,11844,11891,11938,11985,12032,12079,12126,12173,12220,12267,12314,12361,12408,12455,12502,12549,12596,12643,12690,12737,12784,12831,12878,12925,12972,13019,13066,13113,13160,13207,13254,13301,13348,13395,13442,13489,13536,13583,13630,13677,13724,13771,13818,13865,13912,13959,14006,14053,14100,14147,14194,14241,14288,14335,14382,14429,14476,14523,14570,14617,14664,14711,14758,14805,14852,14899,14946,14993,15040,15087,15134,15181,15228,15275,15322,15369,15416,15463,15510,15557,15604,15651,15698,15745,15792,15839,15886,15933,15980,16027,16074,16121,16168,16215,16262,16309,16356,16403,16450,16497,16544,16591,16638,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2303,2350,2397,2444,2491,2538,2585,2632,2679,2726,2773,2820,2867,2914,2961,3008,3055,3102,3149,3196,3243,3290,3337,3384,3431,3478,3525,3572,3619,3666,3713,3760,3807,3854,3901,3948,3995,4042,4089,4136,4183,4230,4277,4324,4371,4418,4465,4512,4559,4606,4653,4700,4747,4794,4841,4888,4935,4982,5029,5076,5123,5170,5217,5264,5311,5358,5405,5452,5499,5546,5593,5640,5687,5734,5781,5828,5875,5922,5969,6016,6063,6110,6157,6204,6251,6298,6345,6392,6439,6486,6533,6580,6627,6674,6721,6768,6815,6862,6909,6956,7003,7050,7097,7144,7191,7238,7285,7332,7379,7426,7473,7520,7567,7614,7661,7708,7755,7802,7849,7896,7943,7990,8037,8084,8131,8178,8225,8272,8319,8366,8413,8460,8507,8554,8601,8648,8695,8742,8789,8836,8883,8930,8977,9024,9071,9118,9165,9212,9259,9306,9353,9400,9447,9494,9541,9588,9635,9682,9729,9776,9823,9870,9917,9964,10011,10058,10105,10152,10199,10246,10293,10340,10387,10434,10481,10528,10575,10622,10669,10716,10763,10810,10857,10904,10951,10998,11045,11092,11139,11186,11233,11280,11327,11374,11421,11468,11515,11562,11609,11656,11703,11750,11797,11844,11891,11938,11985,12032,12079,12126,12173,12220,12267,12314,12361,12408,12455,12502,12549,12596,12643,12690,12737,12784,12831,12878,12925,12972,13019,13066,13113,13160,13207,13254,13301,13348,13395,13442,13489,13536,13583,13630,13677,13724,13771,13818,13865,13912,13959,14006,14053,14100,14147,14194,14241,14288,14335,14382,14429,14476,14523,14570,14617,14664,14711,14758,14805,14852,14899,14946,14993,15040,15087,15134,15181,15228,15275,15322,15369,15416,15463,15510,15557,15604,15651,15698,15745,15792,15839,15886,15933,15980,16027,16074,16121,16168,16215,16262,16309,16356,16403,16450,16497,16544,16591,16638,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2400,2448,2496,2544,2592,2640,2688,2736,2784,2832,2880,2928,2976,3024,3072,3120,3168,3216,3264,3312,3360,3408,3456,3504,3552,3600,3648,3696,3744,3792,3840,3888,3936,3984,4032,4080,4128,4176,4224,4272,4320,4368,4416,4464,4512,4560,4608,4656,4704,4752,4800,4848,4896,4944,4992,5040,5088,5136,5184,5232,5280,5328,5376,5424,5472,5520,5568,5616,5664,5712,5760,5808,5856,5904,5952,6000,6048,6096,6144,6192,6240,6288,6336,6384,6432,6480,6528,6576,6624,6672,6720,6768,6816,6864,6912,6960,7008,7056,7104,7152,7200,7248,7296,7344,7392,7440,7488,7536,7584,7632,7680,7728,7776,7824,7872,7920,7968,8016,8064,8112,8160,8208,8256,8304,8352,8400,8448,8496,8544,8592,8640,8688,8736,8784,8832,8880,8928,8976,9024,9072,9120,9168,9216,9264,9312,9360,9408,9456,9504,9552,9600,9648,9696,9744,9792,9840,9888,9936,9984,10032,10080,10128,10176,10224,10272,10320,10368,10416,10464,10512,10560,10608,10656,10704,10752,10800,10848,10896,10944,10992,11040,11088,11136,11184,11232,11280,11328,11376,11424,11472,11520,11568,11616,11664,11712,11760,11808,11856,11904,11952,12000,12048,12096,12144,12192,12240,12288,12336,12384,12432,12480,12528,12576,12624,12672,12720,12768,12816,12864,12912,12960,13008,13056,13104,13152,13200,13248,13296,13344,13392,13440,13488,13536,13584,13632,13680,13728,13776,13824,13872,13920,13968,14016,14064,14112,14160,14208,14256,14304,14352,14400,14448,14496,14544,14592,14640,14688,14736,14784,14832,14880,14928,14976,15024,15072,15120,15168,15216,15264,15312,15360,15408,15456,15504,15552,15600,15648,15696,15744,15792,15840,15888,15936,15984,16032,16080,16128,16176,16224,16272,16320,16368,16416,16464,16512,16560,16608,16656,16704,16752,16800,16848,16896,16944,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2400,2448,2496,2544,2592,2640,2688,2736,2784,2832,2880,2928,2976,3024,3072,3120,3168,3216,3264,3312,3360,3408,3456,3504,3552,3600,3648,3696,3744,3792,3840,3888,3936,3984,4032,4080,4128,4176,4224,4272,4320,4368,4416,4464,4512,4560,4608,4656,4704,4752,4800,4848,4896,4944,4992,5040,5088,5136,5184,5232,5280,5328,5376,5424,5472,5520,5568,5616,5664,5712,5760,5808,5856,5904,5952,6000,6048,6096,6144,6192,6240,6288,6336,6384,6432,6480,6528,6576,6624,6672,6720,6768,6816,6864,6912,6960,7008,7056,7104,7152,7200,7248,7296,7344,7392,7440,7488,7536,7584,7632,7680,7728,7776,7824,7872,7920,7968,8016,8064,8112,8160,8208,8256,8304,8352,8400,8448,8496,8544,8592,8640,8688,8736,8784,8832,8880,8928,8976,9024,9072,9120,9168,9216,9264,9312,9360,9408,9456,9504,9552,9600,9648,9696,9744,9792,9840,9888,9936,9984,10032,10080,10128,10176,10224,10272,10320,10368,10416,10464,10512,10560,10608,10656,10704,10752,10800,10848,10896,10944,10992,11040,11088,11136,11184,11232,11280,11328,11376,11424,11472,11520,11568,11616,11664,11712,11760,11808,11856,11904,11952,12000,12048,12096,12144,12192,12240,12288,12336,12384,12432,12480,12528,12576,12624,12672,12720,12768,12816,12864,12912,12960,13008,13056,13104,13152,13200,13248,13296,13344,13392,13440,13488,13536,13584,13632,13680,13728,13776,13824,13872,13920,13968,14016,14064,14112,14160,14208,14256,14304,14352,14400,14448,14496,14544,14592,14640,14688,14736,14784,14832,14880,14928,14976,15024,15072,15120,15168,15216,15264,15312,15360,15408,15456,15504,15552,15600,15648,15696,15744,15792,15840,15888,15936,15984,16032,16080,16128,16176,16224,16272,16320,16368,16416,16464,16512,16560,16608,16656,16704,16752,16800,16848,16896,16944,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2499,2548,2597,2646,2695,2744,2793,2842,2891,2940,2989,3038,3087,3136,3185,3234,3283,3332,3381,3430,3479,3528,3577,3626,3675,3724,3773,3822,3871,3920,3969,4018,4067,4116,4165,4214,4263,4312,4361,4410,4459,4508,4557,4606,4655,4704,4753,4802,4851,4900,4949,4998,5047,5096,5145,5194,5243,5292,5341,5390,5439,5488,5537,5586,5635,5684,5733,5782,5831,5880,5929,5978,6027,6076,6125,6174,6223,6272,6321,6370,6419,6468,6517,6566,6615,6664,6713,6762,6811,6860,6909,6958,7007,7056,7105,7154,7203,7252,7301,7350,7399,7448,7497,7546,7595,7644,7693,7742,7791,7840,7889,7938,7987,8036,8085,8134,8183,8232,8281,8330,8379,8428,8477,8526,8575,8624,8673,8722,8771,8820,8869,8918,8967,9016,9065,9114,9163,9212,9261,9310,9359,9408,9457,9506,9555,9604,9653,9702,9751,9800,9849,9898,9947,9996,10045,10094,10143,10192,10241,10290,10339,10388,10437,10486,10535,10584,10633,10682,10731,10780,10829,10878,10927,10976,11025,11074,11123,11172,11221,11270,11319,11368,11417,11466,11515,11564,11613,11662,11711,11760,11809,11858,11907,11956,12005,12054,12103,12152,12201,12250,12299,12348,12397,12446,12495,12544,12593,12642,12691,12740,12789,12838,12887,12936,12985,13034,13083,13132,13181,13230,13279,13328,13377,13426,13475,13524,13573,13622,13671,13720,13769,13818,13867,13916,13965,14014,14063,14112,14161,14210,14259,14308,14357,14406,14455,14504,14553,14602,14651,14700,14749,14798,14847,14896,14945,14994,15043,15092,15141,15190,15239,15288,15337,15386,15435,15484,15533,15582,15631,15680,15729,15778,15827,15876,15925,15974,16023,16072,16121,16170,16219,16268,16317,16366,16415,16464,16513,16562,16611,16660,16709,16758,16807,16856,16905,16954,17003,17052,17101,17150,17199,17248,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2499,2548,2597,2646,2695,2744,2793,2842,2891,2940,2989,3038,3087,3136,3185,3234,3283,3332,3381,3430,3479,3528,3577,3626,3675,3724,3773,3822,3871,3920,3969,4018,4067,4116,4165,4214,4263,4312,4361,4410,4459,4508,4557,4606,4655,4704,4753,4802,4851,4900,4949,4998,5047,5096,5145,5194,5243,5292,5341,5390,5439,5488,5537,5586,5635,5684,5733,5782,5831,5880,5929,5978,6027,6076,6125,6174,6223,6272,6321,6370,6419,6468,6517,6566,6615,6664,6713,6762,6811,6860,6909,6958,7007,7056,7105,7154,7203,7252,7301,7350,7399,7448,7497,7546,7595,7644,7693,7742,7791,7840,7889,7938,7987,8036,8085,8134,8183,8232,8281,8330,8379,8428,8477,8526,8575,8624,8673,8722,8771,8820,8869,8918,8967,9016,9065,9114,9163,9212,9261,9310,9359,9408,9457,9506,9555,9604,9653,9702,9751,9800,9849,9898,9947,9996,10045,10094,10143,10192,10241,10290,10339,10388,10437,10486,10535,10584,10633,10682,10731,10780,10829,10878,10927,10976,11025,11074,11123,11172,11221,11270,11319,11368,11417,11466,11515,11564,11613,11662,11711,11760,11809,11858,11907,11956,12005,12054,12103,12152,12201,12250,12299,12348,12397,12446,12495,12544,12593,12642,12691,12740,12789,12838,12887,12936,12985,13034,13083,13132,13181,13230,13279,13328,13377,13426,13475,13524,13573,13622,13671,13720,13769,13818,13867,13916,13965,14014,14063,14112,14161,14210,14259,14308,14357,14406,14455,14504,14553,14602,14651,14700,14749,14798,14847,14896,14945,14994,15043,15092,15141,15190,15239,15288,15337,15386,15435,15484,15533,15582,15631,15680,15729,15778,15827,15876,15925,15974,16023,16072,16121,16170,16219,16268,16317,16366,16415,16464,16513,16562,16611,16660,16709,16758,16807,16856,16905,16954,17003,17052,17101,17150,17199,17248,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2600,2650,2700,2750,2800,2850,2900,2950,3000,3050,3100,3150,3200,3250,3300,3350,3400,3450,3500,3550,3600,3650,3700,3750,3800,3850,3900,3950,4000,4050,4100,4150,4200,4250,4300,4350,4400,4450,4500,4550,4600,4650,4700,4750,4800,4850,4900,4950,5000,5050,5100,5150,5200,5250,5300,5350,5400,5450,5500,5550,5600,5650,5700,5750,5800,5850,5900,5950,6000,6050,6100,6150,6200,6250,6300,6350,6400,6450,6500,6550,6600,6650,6700,6750,6800,6850,6900,6950,7000,7050,7100,7150,7200,7250,7300,7350,7400,7450,7500,7550,7600,7650,7700,7750,7800,7850,7900,7950,8000,8050,8100,8150,8200,8250,8300,8350,8400,8450,8500,8550,8600,8650,8700,8750,8800,8850,8900,8950,9000,9050,9100,9150,9200,9250,9300,9350,9400,9450,9500,9550,9600,9650,9700,9750,9800,9850,9900,9950,10000,10050,10100,10150,10200,10250,10300,10350,10400,10450,10500,10550,10600,10650,10700,10750,10800,10850,10900,10950,11000,11050,11100,11150,11200,11250,11300,11350,11400,11450,11500,11550,11600,11650,11700,11750,11800,11850,11900,11950,12000,12050,12100,12150,12200,12250,12300,12350,12400,12450,12500,12550,12600,12650,12700,12750,12800,12850,12900,12950,13000,13050,13100,13150,13200,13250,13300,13350,13400,13450,13500,13550,13600,13650,13700,13750,13800,13850,13900,13950,14000,14050,14100,14150,14200,14250,14300,14350,14400,14450,14500,14550,14600,14650,14700,14750,14800,14850,14900,14950,15000,15050,15100,15150,15200,15250,15300,15350,15400,15450,15500,15550,15600,15650,15700,15750,15800,15850,15900,15950,16000,16050,16100,16150,16200,16250,16300,16350,16400,16450,16500,16550,16600,16650,16700,16750,16800,16850,16900,16950,17000,17050,17100,17150,17200,17250,17300,17350,17400,17450,17500,17550,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2600,2650,2700,2750,2800,2850,2900,2950,3000,3050,3100,3150,3200,3250,3300,3350,3400,3450,3500,3550,3600,3650,3700,3750,3800,3850,3900,3950,4000,4050,4100,4150,4200,4250,4300,4350,4400,4450,4500,4550,4600,4650,4700,4750,4800,4850,4900,4950,5000,5050,5100,5150,5200,5250,5300,5350,5400,5450,5500,5550,5600,5650,5700,5750,5800,5850,5900,5950,6000,6050,6100,6150,6200,6250,6300,6350,6400,6450,6500,6550,6600,6650,6700,6750,6800,6850,6900,6950,7000,7050,7100,7150,7200,7250,7300,7350,7400,7450,7500,7550,7600,7650,7700,7750,7800,7850,7900,7950,8000,8050,8100,8150,8200,8250,8300,8350,8400,8450,8500,8550,8600,8650,8700,8750,8800,8850,8900,8950,9000,9050,9100,9150,9200,9250,9300,9350,9400,9450,9500,9550,9600,9650,9700,9750,9800,9850,9900,9950,10000,10050,10100,10150,10200,10250,10300,10350,10400,10450,10500,10550,10600,10650,10700,10750,10800,10850,10900,10950,11000,11050,11100,11150,11200,11250,11300,11350,11400,11450,11500,11550,11600,11650,11700,11750,11800,11850,11900,11950,12000,12050,12100,12150,12200,12250,12300,12350,12400,12450,12500,12550,12600,12650,12700,12750,12800,12850,12900,12950,13000,13050,13100,13150,13200,13250,13300,13350,13400,13450,13500,13550,13600,13650,13700,13750,13800,13850,13900,13950,14000,14050,14100,14150,14200,14250,14300,14350,14400,14450,14500,14550,14600,14650,14700,14750,14800,14850,14900,14950,15000,15050,15100,15150,15200,15250,15300,15350,15400,15450,15500,15550,15600,15650,15700,15750,15800,15850,15900,15950,16000,16050,16100,16150,16200,16250,16300,16350,16400,16450,16500,16550,16600,16650,16700,16750,16800,16850,16900,16950,17000,17050,17100,17150,17200,17250,17300,17350,17400,17450,17500,17550,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2703,2754,2805,2856,2907,2958,3009,3060,3111,3162,3213,3264,3315,3366,3417,3468,3519,3570,3621,3672,3723,3774,3825,3876,3927,3978,4029,4080,4131,4182,4233,4284,4335,4386,4437,4488,4539,4590,4641,4692,4743,4794,4845,4896,4947,4998,5049,5100,5151,5202,5253,5304,5355,5406,5457,5508,5559,5610,5661,5712,5763,5814,5865,5916,5967,6018,6069,6120,6171,6222,6273,6324,6375,6426,6477,6528,6579,6630,6681,6732,6783,6834,6885,6936,6987,7038,7089,7140,7191,7242,7293,7344,7395,7446,7497,7548,7599,7650,7701,7752,7803,7854,7905,7956,8007,8058,8109,8160,8211,8262,8313,8364,8415,8466,8517,8568,8619,8670,8721,8772,8823,8874,8925,8976,9027,9078,9129,9180,9231,9282,9333,9384,9435,9486,9537,9588,9639,9690,9741,9792,9843,9894,9945,9996,10047,10098,10149,10200,10251,10302,10353,10404,10455,10506,10557,10608,10659,10710,10761,10812,10863,10914,10965,11016,11067,11118,11169,11220,11271,11322,11373,11424,11475,11526,11577,11628,11679,11730,11781,11832,11883,11934,11985,12036,12087,12138,12189,12240,12291,12342,12393,12444,12495,12546,12597,12648,12699,12750,12801,12852,12903,12954,13005,13056,13107,13158,13209,13260,13311,13362,13413,13464,13515,13566,13617,13668,13719,13770,13821,13872,13923,13974,14025,14076,14127,14178,14229,14280,14331,14382,14433,14484,14535,14586,14637,14688,14739,14790,14841,14892,14943,14994,15045,15096,15147,15198,15249,15300,15351,15402,15453,15504,15555,15606,15657,15708,15759,15810,15861,15912,15963,16014,16065,16116,16167,16218,16269,16320,16371,16422,16473,16524,16575,16626,16677,16728,16779,16830,16881,16932,16983,17034,17085,17136,17187,17238,17289,17340,17391,17442,17493,17544,17595,17646,17697,17748,17799,17850,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2703,2754,2805,2856,2907,2958,3009,3060,3111,3162,3213,3264,3315,3366,3417,3468,3519,3570,3621,3672,3723,3774,3825,3876,3927,3978,4029,4080,4131,4182,4233,4284,4335,4386,4437,4488,4539,4590,4641,4692,4743,4794,4845,4896,4947,4998,5049,5100,5151,5202,5253,5304,5355,5406,5457,5508,5559,5610,5661,5712,5763,5814,5865,5916,5967,6018,6069,6120,6171,6222,6273,6324,6375,6426,6477,6528,6579,6630,6681,6732,6783,6834,6885,6936,6987,7038,7089,7140,7191,7242,7293,7344,7395,7446,7497,7548,7599,7650,7701,7752,7803,7854,7905,7956,8007,8058,8109,8160,8211,8262,8313,8364,8415,8466,8517,8568,8619,8670,8721,8772,8823,8874,8925,8976,9027,9078,9129,9180,9231,9282,9333,9384,9435,9486,9537,9588,9639,9690,9741,9792,9843,9894,9945,9996,10047,10098,10149,10200,10251,10302,10353,10404,10455,10506,10557,10608,10659,10710,10761,10812,10863,10914,10965,11016,11067,11118,11169,11220,11271,11322,11373,11424,11475,11526,11577,11628,11679,11730,11781,11832,11883,11934,11985,12036,12087,12138,12189,12240,12291,12342,12393,12444,12495,12546,12597,12648,12699,12750,12801,12852,12903,12954,13005,13056,13107,13158,13209,13260,13311,13362,13413,13464,13515,13566,13617,13668,13719,13770,13821,13872,13923,13974,14025,14076,14127,14178,14229,14280,14331,14382,14433,14484,14535,14586,14637,14688,14739,14790,14841,14892,14943,14994,15045,15096,15147,15198,15249,15300,15351,15402,15453,15504,15555,15606,15657,15708,15759,15810,15861,15912,15963,16014,16065,16116,16167,16218,16269,16320,16371,16422,16473,16524,16575,16626,16677,16728,16779,16830,16881,16932,16983,17034,17085,17136,17187,17238,17289,17340,17391,17442,17493,17544,17595,17646,17697,17748,17799,17850,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2808,2860,2912,2964,3016,3068,3120,3172,3224,3276,3328,3380,3432,3484,3536,3588,3640,3692,3744,3796,3848,3900,3952,4004,4056,4108,4160,4212,4264,4316,4368,4420,4472,4524,4576,4628,4680,4732,4784,4836,4888,4940,4992,5044,5096,5148,5200,5252,5304,5356,5408,5460,5512,5564,5616,5668,5720,5772,5824,5876,5928,5980,6032,6084,6136,6188,6240,6292,6344,6396,6448,6500,6552,6604,6656,6708,6760,6812,6864,6916,6968,7020,7072,7124,7176,7228,7280,7332,7384,7436,7488,7540,7592,7644,7696,7748,7800,7852,7904,7956,8008,8060,8112,8164,8216,8268,8320,8372,8424,8476,8528,8580,8632,8684,8736,8788,8840,8892,8944,8996,9048,9100,9152,9204,9256,9308,9360,9412,9464,9516,9568,9620,9672,9724,9776,9828,9880,9932,9984,10036,10088,10140,10192,10244,10296,10348,10400,10452,10504,10556,10608,10660,10712,10764,10816,10868,10920,10972,11024,11076,11128,11180,11232,11284,11336,11388,11440,11492,11544,11596,11648,11700,11752,11804,11856,11908,11960,12012,12064,12116,12168,12220,12272,12324,12376,12428,12480,12532,12584,12636,12688,12740,12792,12844,12896,12948,13000,13052,13104,13156,13208,13260,13312,13364,13416,13468,13520,13572,13624,13676,13728,13780,13832,13884,13936,13988,14040,14092,14144,14196,14248,14300,14352,14404,14456,14508,14560,14612,14664,14716,14768,14820,14872,14924,14976,15028,15080,15132,15184,15236,15288,15340,15392,15444,15496,15548,15600,15652,15704,15756,15808,15860,15912,15964,16016,16068,16120,16172,16224,16276,16328,16380,16432,16484,16536,16588,16640,16692,16744,16796,16848,16900,16952,17004,17056,17108,17160,17212,17264,17316,17368,17420,17472,17524,17576,17628,17680,17732,17784,17836,17888,17940,17992,18044,18096,18148,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2808,2860,2912,2964,3016,3068,3120,3172,3224,3276,3328,3380,3432,3484,3536,3588,3640,3692,3744,3796,3848,3900,3952,4004,4056,4108,4160,4212,4264,4316,4368,4420,4472,4524,4576,4628,4680,4732,4784,4836,4888,4940,4992,5044,5096,5148,5200,5252,5304,5356,5408,5460,5512,5564,5616,5668,5720,5772,5824,5876,5928,5980,6032,6084,6136,6188,6240,6292,6344,6396,6448,6500,6552,6604,6656,6708,6760,6812,6864,6916,6968,7020,7072,7124,7176,7228,7280,7332,7384,7436,7488,7540,7592,7644,7696,7748,7800,7852,7904,7956,8008,8060,8112,8164,8216,8268,8320,8372,8424,8476,8528,8580,8632,8684,8736,8788,8840,8892,8944,8996,9048,9100,9152,9204,9256,9308,9360,9412,9464,9516,9568,9620,9672,9724,9776,9828,9880,9932,9984,10036,10088,10140,10192,10244,10296,10348,10400,10452,10504,10556,10608,10660,10712,10764,10816,10868,10920,10972,11024,11076,11128,11180,11232,11284,11336,11388,11440,11492,11544,11596,11648,11700,11752,11804,11856,11908,11960,12012,12064,12116,12168,12220,12272,12324,12376,12428,12480,12532,12584,12636,12688,12740,12792,12844,12896,12948,13000,13052,13104,13156,13208,13260,13312,13364,13416,13468,13520,13572,13624,13676,13728,13780,13832,13884,13936,13988,14040,14092,14144,14196,14248,14300,14352,14404,14456,14508,14560,14612,14664,14716,14768,14820,14872,14924,14976,15028,15080,15132,15184,15236,15288,15340,15392,15444,15496,15548,15600,15652,15704,15756,15808,15860,15912,15964,16016,16068,16120,16172,16224,16276,16328,16380,16432,16484,16536,16588,16640,16692,16744,16796,16848,16900,16952,17004,17056,17108,17160,17212,17264,17316,17368,17420,17472,17524,17576,17628,17680,17732,17784,17836,17888,17940,17992,18044,18096,18148,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2915,2968,3021,3074,3127,3180,3233,3286,3339,3392,3445,3498,3551,3604,3657,3710,3763,3816,3869,3922,3975,4028,4081,4134,4187,4240,4293,4346,4399,4452,4505,4558,4611,4664,4717,4770,4823,4876,4929,4982,5035,5088,5141,5194,5247,5300,5353,5406,5459,5512,5565,5618,5671,5724,5777,5830,5883,5936,5989,6042,6095,6148,6201,6254,6307,6360,6413,6466,6519,6572,6625,6678,6731,6784,6837,6890,6943,6996,7049,7102,7155,7208,7261,7314,7367,7420,7473,7526,7579,7632,7685,7738,7791,7844,7897,7950,8003,8056,8109,8162,8215,8268,8321,8374,8427,8480,8533,8586,8639,8692,8745,8798,8851,8904,8957,9010,9063,9116,9169,9222,9275,9328,9381,9434,9487,9540,9593,9646,9699,9752,9805,9858,9911,9964,10017,10070,10123,10176,10229,10282,10335,10388,10441,10494,10547,10600,10653,10706,10759,10812,10865,10918,10971,11024,11077,11130,11183,11236,11289,11342,11395,11448,11501,11554,11607,11660,11713,11766,11819,11872,11925,11978,12031,12084,12137,12190,12243,12296,12349,12402,12455,12508,12561,12614,12667,12720,12773,12826,12879,12932,12985,13038,13091,13144,13197,13250,13303,13356,13409,13462,13515,13568,13621,13674,13727,13780,13833,13886,13939,13992,14045,14098,14151,14204,14257,14310,14363,14416,14469,14522,14575,14628,14681,14734,14787,14840,14893,14946,14999,15052,15105,15158,15211,15264,15317,15370,15423,15476,15529,15582,15635,15688,15741,15794,15847,15900,15953,16006,16059,16112,16165,16218,16271,16324,16377,16430,16483,16536,16589,16642,16695,16748,16801,16854,16907,16960,17013,17066,17119,17172,17225,17278,17331,17384,17437,17490,17543,17596,17649,17702,17755,17808,17861,17914,17967,18020,18073,18126,18179,18232,18285,18338,18391,18444,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2915,2968,3021,3074,3127,3180,3233,3286,3339,3392,3445,3498,3551,3604,3657,3710,3763,3816,3869,3922,3975,4028,4081,4134,4187,4240,4293,4346,4399,4452,4505,4558,4611,4664,4717,4770,4823,4876,4929,4982,5035,5088,5141,5194,5247,5300,5353,5406,5459,5512,5565,5618,5671,5724,5777,5830,5883,5936,5989,6042,6095,6148,6201,6254,6307,6360,6413,6466,6519,6572,6625,6678,6731,6784,6837,6890,6943,6996,7049,7102,7155,7208,7261,7314,7367,7420,7473,7526,7579,7632,7685,7738,7791,7844,7897,7950,8003,8056,8109,8162,8215,8268,8321,8374,8427,8480,8533,8586,8639,8692,8745,8798,8851,8904,8957,9010,9063,9116,9169,9222,9275,9328,9381,9434,9487,9540,9593,9646,9699,9752,9805,9858,9911,9964,10017,10070,10123,10176,10229,10282,10335,10388,10441,10494,10547,10600,10653,10706,10759,10812,10865,10918,10971,11024,11077,11130,11183,11236,11289,11342,11395,11448,11501,11554,11607,11660,11713,11766,11819,11872,11925,11978,12031,12084,12137,12190,12243,12296,12349,12402,12455,12508,12561,12614,12667,12720,12773,12826,12879,12932,12985,13038,13091,13144,13197,13250,13303,13356,13409,13462,13515,13568,13621,13674,13727,13780,13833,13886,13939,13992,14045,14098,14151,14204,14257,14310,14363,14416,14469,14522,14575,14628,14681,14734,14787,14840,14893,14946,14999,15052,15105,15158,15211,15264,15317,15370,15423,15476,15529,15582,15635,15688,15741,15794,15847,15900,15953,16006,16059,16112,16165,16218,16271,16324,16377,16430,16483,16536,16589,16642,16695,16748,16801,16854,16907,16960,17013,17066,17119,17172,17225,17278,17331,17384,17437,17490,17543,17596,17649,17702,17755,17808,17861,17914,17967,18020,18073,18126,18179,18232,18285,18338,18391,18444,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3024,3078,3132,3186,3240,3294,3348,3402,3456,3510,3564,3618,3672,3726,3780,3834,3888,3942,3996,4050,4104,4158,4212,4266,4320,4374,4428,4482,4536,4590,4644,4698,4752,4806,4860,4914,4968,5022,5076,5130,5184,5238,5292,5346,5400,5454,5508,5562,5616,5670,5724,5778,5832,5886,5940,5994,6048,6102,6156,6210,6264,6318,6372,6426,6480,6534,6588,6642,6696,6750,6804,6858,6912,6966,7020,7074,7128,7182,7236,7290,7344,7398,7452,7506,7560,7614,7668,7722,7776,7830,7884,7938,7992,8046,8100,8154,8208,8262,8316,8370,8424,8478,8532,8586,8640,8694,8748,8802,8856,8910,8964,9018,9072,9126,9180,9234,9288,9342,9396,9450,9504,9558,9612,9666,9720,9774,9828,9882,9936,9990,10044,10098,10152,10206,10260,10314,10368,10422,10476,10530,10584,10638,10692,10746,10800,10854,10908,10962,11016,11070,11124,11178,11232,11286,11340,11394,11448,11502,11556,11610,11664,11718,11772,11826,11880,11934,11988,12042,12096,12150,12204,12258,12312,12366,12420,12474,12528,12582,12636,12690,12744,12798,12852,12906,12960,13014,13068,13122,13176,13230,13284,13338,13392,13446,13500,13554,13608,13662,13716,13770,13824,13878,13932,13986,14040,14094,14148,14202,14256,14310,14364,14418,14472,14526,14580,14634,14688,14742,14796,14850,14904,14958,15012,15066,15120,15174,15228,15282,15336,15390,15444,15498,15552,15606,15660,15714,15768,15822,15876,15930,15984,16038,16092,16146,16200,16254,16308,16362,16416,16470,16524,16578,16632,16686,16740,16794,16848,16902,16956,17010,17064,17118,17172,17226,17280,17334,17388,17442,17496,17550,17604,17658,17712,17766,17820,17874,17928,17982,18036,18090,18144,18198,18252,18306,18360,18414,18468,18522,18576,18630,18684,18738,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3024,3078,3132,3186,3240,3294,3348,3402,3456,3510,3564,3618,3672,3726,3780,3834,3888,3942,3996,4050,4104,4158,4212,4266,4320,4374,4428,4482,4536,4590,4644,4698,4752,4806,4860,4914,4968,5022,5076,5130,5184,5238,5292,5346,5400,5454,5508,5562,5616,5670,5724,5778,5832,5886,5940,5994,6048,6102,6156,6210,6264,6318,6372,6426,6480,6534,6588,6642,6696,6750,6804,6858,6912,6966,7020,7074,7128,7182,7236,7290,7344,7398,7452,7506,7560,7614,7668,7722,7776,7830,7884,7938,7992,8046,8100,8154,8208,8262,8316,8370,8424,8478,8532,8586,8640,8694,8748,8802,8856,8910,8964,9018,9072,9126,9180,9234,9288,9342,9396,9450,9504,9558,9612,9666,9720,9774,9828,9882,9936,9990,10044,10098,10152,10206,10260,10314,10368,10422,10476,10530,10584,10638,10692,10746,10800,10854,10908,10962,11016,11070,11124,11178,11232,11286,11340,11394,11448,11502,11556,11610,11664,11718,11772,11826,11880,11934,11988,12042,12096,12150,12204,12258,12312,12366,12420,12474,12528,12582,12636,12690,12744,12798,12852,12906,12960,13014,13068,13122,13176,13230,13284,13338,13392,13446,13500,13554,13608,13662,13716,13770,13824,13878,13932,13986,14040,14094,14148,14202,14256,14310,14364,14418,14472,14526,14580,14634,14688,14742,14796,14850,14904,14958,15012,15066,15120,15174,15228,15282,15336,15390,15444,15498,15552,15606,15660,15714,15768,15822,15876,15930,15984,16038,16092,16146,16200,16254,16308,16362,16416,16470,16524,16578,16632,16686,16740,16794,16848,16902,16956,17010,17064,17118,17172,17226,17280,17334,17388,17442,17496,17550,17604,17658,17712,17766,17820,17874,17928,17982,18036,18090,18144,18198,18252,18306,18360,18414,18468,18522,18576,18630,18684,18738,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3135,3190,3245,3300,3355,3410,3465,3520,3575,3630,3685,3740,3795,3850,3905,3960,4015,4070,4125,4180,4235,4290,4345,4400,4455,4510,4565,4620,4675,4730,4785,4840,4895,4950,5005,5060,5115,5170,5225,5280,5335,5390,5445,5500,5555,5610,5665,5720,5775,5830,5885,5940,5995,6050,6105,6160,6215,6270,6325,6380,6435,6490,6545,6600,6655,6710,6765,6820,6875,6930,6985,7040,7095,7150,7205,7260,7315,7370,7425,7480,7535,7590,7645,7700,7755,7810,7865,7920,7975,8030,8085,8140,8195,8250,8305,8360,8415,8470,8525,8580,8635,8690,8745,8800,8855,8910,8965,9020,9075,9130,9185,9240,9295,9350,9405,9460,9515,9570,9625,9680,9735,9790,9845,9900,9955,10010,10065,10120,10175,10230,10285,10340,10395,10450,10505,10560,10615,10670,10725,10780,10835,10890,10945,11000,11055,11110,11165,11220,11275,11330,11385,11440,11495,11550,11605,11660,11715,11770,11825,11880,11935,11990,12045,12100,12155,12210,12265,12320,12375,12430,12485,12540,12595,12650,12705,12760,12815,12870,12925,12980,13035,13090,13145,13200,13255,13310,13365,13420,13475,13530,13585,13640,13695,13750,13805,13860,13915,13970,14025,14080,14135,14190,14245,14300,14355,14410,14465,14520,14575,14630,14685,14740,14795,14850,14905,14960,15015,15070,15125,15180,15235,15290,15345,15400,15455,15510,15565,15620,15675,15730,15785,15840,15895,15950,16005,16060,16115,16170,16225,16280,16335,16390,16445,16500,16555,16610,16665,16720,16775,16830,16885,16940,16995,17050,17105,17160,17215,17270,17325,17380,17435,17490,17545,17600,17655,17710,17765,17820,17875,17930,17985,18040,18095,18150,18205,18260,18315,18370,18425,18480,18535,18590,18645,18700,18755,18810,18865,18920,18975,19030,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3135,3190,3245,3300,3355,3410,3465,3520,3575,3630,3685,3740,3795,3850,3905,3960,4015,4070,4125,4180,4235,4290,4345,4400,4455,4510,4565,4620,4675,4730,4785,4840,4895,4950,5005,5060,5115,5170,5225,5280,5335,5390,5445,5500,5555,5610,5665,5720,5775,5830,5885,5940,5995,6050,6105,6160,6215,6270,6325,6380,6435,6490,6545,6600,6655,6710,6765,6820,6875,6930,6985,7040,7095,7150,7205,7260,7315,7370,7425,7480,7535,7590,7645,7700,7755,7810,7865,7920,7975,8030,8085,8140,8195,8250,8305,8360,8415,8470,8525,8580,8635,8690,8745,8800,8855,8910,8965,9020,9075,9130,9185,9240,9295,9350,9405,9460,9515,9570,9625,9680,9735,9790,9845,9900,9955,10010,10065,10120,10175,10230,10285,10340,10395,10450,10505,10560,10615,10670,10725,10780,10835,10890,10945,11000,11055,11110,11165,11220,11275,11330,11385,11440,11495,11550,11605,11660,11715,11770,11825,11880,11935,11990,12045,12100,12155,12210,12265,12320,12375,12430,12485,12540,12595,12650,12705,12760,12815,12870,12925,12980,13035,13090,13145,13200,13255,13310,13365,13420,13475,13530,13585,13640,13695,13750,13805,13860,13915,13970,14025,14080,14135,14190,14245,14300,14355,14410,14465,14520,14575,14630,14685,14740,14795,14850,14905,14960,15015,15070,15125,15180,15235,15290,15345,15400,15455,15510,15565,15620,15675,15730,15785,15840,15895,15950,16005,16060,16115,16170,16225,16280,16335,16390,16445,16500,16555,16610,16665,16720,16775,16830,16885,16940,16995,17050,17105,17160,17215,17270,17325,17380,17435,17490,17545,17600,17655,17710,17765,17820,17875,17930,17985,18040,18095,18150,18205,18260,18315,18370,18425,18480,18535,18590,18645,18700,18755,18810,18865,18920,18975,19030,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3248,3304,3360,3416,3472,3528,3584,3640,3696,3752,3808,3864,3920,3976,4032,4088,4144,4200,4256,4312,4368,4424,4480,4536,4592,4648,4704,4760,4816,4872,4928,4984,5040,5096,5152,5208,5264,5320,5376,5432,5488,5544,5600,5656,5712,5768,5824,5880,5936,5992,6048,6104,6160,6216,6272,6328,6384,6440,6496,6552,6608,6664,6720,6776,6832,6888,6944,7000,7056,7112,7168,7224,7280,7336,7392,7448,7504,7560,7616,7672,7728,7784,7840,7896,7952,8008,8064,8120,8176,8232,8288,8344,8400,8456,8512,8568,8624,8680,8736,8792,8848,8904,8960,9016,9072,9128,9184,9240,9296,9352,9408,9464,9520,9576,9632,9688,9744,9800,9856,9912,9968,10024,10080,10136,10192,10248,10304,10360,10416,10472,10528,10584,10640,10696,10752,10808,10864,10920,10976,11032,11088,11144,11200,11256,11312,11368,11424,11480,11536,11592,11648,11704,11760,11816,11872,11928,11984,12040,12096,12152,12208,12264,12320,12376,12432,12488,12544,12600,12656,12712,12768,12824,12880,12936,12992,13048,13104,13160,13216,13272,13328,13384,13440,13496,13552,13608,13664,13720,13776,13832,13888,13944,14000,14056,14112,14168,14224,14280,14336,14392,14448,14504,14560,14616,14672,14728,14784,14840,14896,14952,15008,15064,15120,15176,15232,15288,15344,15400,15456,15512,15568,15624,15680,15736,15792,15848,15904,15960,16016,16072,16128,16184,16240,16296,16352,16408,16464,16520,16576,16632,16688,16744,16800,16856,16912,16968,17024,17080,17136,17192,17248,17304,17360,17416,17472,17528,17584,17640,17696,17752,17808,17864,17920,17976,18032,18088,18144,18200,18256,18312,18368,18424,18480,18536,18592,18648,18704,18760,18816,18872,18928,18984,19040,19096,19152,19208,19264,19320,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3248,3304,3360,3416,3472,3528,3584,3640,3696,3752,3808,3864,3920,3976,4032,4088,4144,4200,4256,4312,4368,4424,4480,4536,4592,4648,4704,4760,4816,4872,4928,4984,5040,5096,5152,5208,5264,5320,5376,5432,5488,5544,5600,5656,5712,5768,5824,5880,5936,5992,6048,6104,6160,6216,6272,6328,6384,6440,6496,6552,6608,6664,6720,6776,6832,6888,6944,7000,7056,7112,7168,7224,7280,7336,7392,7448,7504,7560,7616,7672,7728,7784,7840,7896,7952,8008,8064,8120,8176,8232,8288,8344,8400,8456,8512,8568,8624,8680,8736,8792,8848,8904,8960,9016,9072,9128,9184,9240,9296,9352,9408,9464,9520,9576,9632,9688,9744,9800,9856,9912,9968,10024,10080,10136,10192,10248,10304,10360,10416,10472,10528,10584,10640,10696,10752,10808,10864,10920,10976,11032,11088,11144,11200,11256,11312,11368,11424,11480,11536,11592,11648,11704,11760,11816,11872,11928,11984,12040,12096,12152,12208,12264,12320,12376,12432,12488,12544,12600,12656,12712,12768,12824,12880,12936,12992,13048,13104,13160,13216,13272,13328,13384,13440,13496,13552,13608,13664,13720,13776,13832,13888,13944,14000,14056,14112,14168,14224,14280,14336,14392,14448,14504,14560,14616,14672,14728,14784,14840,14896,14952,15008,15064,15120,15176,15232,15288,15344,15400,15456,15512,15568,15624,15680,15736,15792,15848,15904,15960,16016,16072,16128,16184,16240,16296,16352,16408,16464,16520,16576,16632,16688,16744,16800,16856,16912,16968,17024,17080,17136,17192,17248,17304,17360,17416,17472,17528,17584,17640,17696,17752,17808,17864,17920,17976,18032,18088,18144,18200,18256,18312,18368,18424,18480,18536,18592,18648,18704,18760,18816,18872,18928,18984,19040,19096,19152,19208,19264,19320,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3363,3420,3477,3534,3591,3648,3705,3762,3819,3876,3933,3990,4047,4104,4161,4218,4275,4332,4389,4446,4503,4560,4617,4674,4731,4788,4845,4902,4959,5016,5073,5130,5187,5244,5301,5358,5415,5472,5529,5586,5643,5700,5757,5814,5871,5928,5985,6042,6099,6156,6213,6270,6327,6384,6441,6498,6555,6612,6669,6726,6783,6840,6897,6954,7011,7068,7125,7182,7239,7296,7353,7410,7467,7524,7581,7638,7695,7752,7809,7866,7923,7980,8037,8094,8151,8208,8265,8322,8379,8436,8493,8550,8607,8664,8721,8778,8835,8892,8949,9006,9063,9120,9177,9234,9291,9348,9405,9462,9519,9576,9633,9690,9747,9804,9861,9918,9975,10032,10089,10146,10203,10260,10317,10374,10431,10488,10545,10602,10659,10716,10773,10830,10887,10944,11001,11058,11115,11172,11229,11286,11343,11400,11457,11514,11571,11628,11685,11742,11799,11856,11913,11970,12027,12084,12141,12198,12255,12312,12369,12426,12483,12540,12597,12654,12711,12768,12825,12882,12939,12996,13053,13110,13167,13224,13281,13338,13395,13452,13509,13566,13623,13680,13737,13794,13851,13908,13965,14022,14079,14136,14193,14250,14307,14364,14421,14478,14535,14592,14649,14706,14763,14820,14877,14934,14991,15048,15105,15162,15219,15276,15333,15390,15447,15504,15561,15618,15675,15732,15789,15846,15903,15960,16017,16074,16131,16188,16245,16302,16359,16416,16473,16530,16587,16644,16701,16758,16815,16872,16929,16986,17043,17100,17157,17214,17271,17328,17385,17442,17499,17556,17613,17670,17727,17784,17841,17898,17955,18012,18069,18126,18183,18240,18297,18354,18411,18468,18525,18582,18639,18696,18753,18810,18867,18924,18981,19038,19095,19152,19209,19266,19323,19380,19437,19494,19551,19608,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3363,3420,3477,3534,3591,3648,3705,3762,3819,3876,3933,3990,4047,4104,4161,4218,4275,4332,4389,4446,4503,4560,4617,4674,4731,4788,4845,4902,4959,5016,5073,5130,5187,5244,5301,5358,5415,5472,5529,5586,5643,5700,5757,5814,5871,5928,5985,6042,6099,6156,6213,6270,6327,6384,6441,6498,6555,6612,6669,6726,6783,6840,6897,6954,7011,7068,7125,7182,7239,7296,7353,7410,7467,7524,7581,7638,7695,7752,7809,7866,7923,7980,8037,8094,8151,8208,8265,8322,8379,8436,8493,8550,8607,8664,8721,8778,8835,8892,8949,9006,9063,9120,9177,9234,9291,9348,9405,9462,9519,9576,9633,9690,9747,9804,9861,9918,9975,10032,10089,10146,10203,10260,10317,10374,10431,10488,10545,10602,10659,10716,10773,10830,10887,10944,11001,11058,11115,11172,11229,11286,11343,11400,11457,11514,11571,11628,11685,11742,11799,11856,11913,11970,12027,12084,12141,12198,12255,12312,12369,12426,12483,12540,12597,12654,12711,12768,12825,12882,12939,12996,13053,13110,13167,13224,13281,13338,13395,13452,13509,13566,13623,13680,13737,13794,13851,13908,13965,14022,14079,14136,14193,14250,14307,14364,14421,14478,14535,14592,14649,14706,14763,14820,14877,14934,14991,15048,15105,15162,15219,15276,15333,15390,15447,15504,15561,15618,15675,15732,15789,15846,15903,15960,16017,16074,16131,16188,16245,16302,16359,16416,16473,16530,16587,16644,16701,16758,16815,16872,16929,16986,17043,17100,17157,17214,17271,17328,17385,17442,17499,17556,17613,17670,17727,17784,17841,17898,17955,18012,18069,18126,18183,18240,18297,18354,18411,18468,18525,18582,18639,18696,18753,18810,18867,18924,18981,19038,19095,19152,19209,19266,19323,19380,19437,19494,19551,19608,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3480,3538,3596,3654,3712,3770,3828,3886,3944,4002,4060,4118,4176,4234,4292,4350,4408,4466,4524,4582,4640,4698,4756,4814,4872,4930,4988,5046,5104,5162,5220,5278,5336,5394,5452,5510,5568,5626,5684,5742,5800,5858,5916,5974,6032,6090,6148,6206,6264,6322,6380,6438,6496,6554,6612,6670,6728,6786,6844,6902,6960,7018,7076,7134,7192,7250,7308,7366,7424,7482,7540,7598,7656,7714,7772,7830,7888,7946,8004,8062,8120,8178,8236,8294,8352,8410,8468,8526,8584,8642,8700,8758,8816,8874,8932,8990,9048,9106,9164,9222,9280,9338,9396,9454,9512,9570,9628,9686,9744,9802,9860,9918,9976,10034,10092,10150,10208,10266,10324,10382,10440,10498,10556,10614,10672,10730,10788,10846,10904,10962,11020,11078,11136,11194,11252,11310,11368,11426,11484,11542,11600,11658,11716,11774,11832,11890,11948,12006,12064,12122,12180,12238,12296,12354,12412,12470,12528,12586,12644,12702,12760,12818,12876,12934,12992,13050,13108,13166,13224,13282,13340,13398,13456,13514,13572,13630,13688,13746,13804,13862,13920,13978,14036,14094,14152,14210,14268,14326,14384,14442,14500,14558,14616,14674,14732,14790,14848,14906,14964,15022,15080,15138,15196,15254,15312,15370,15428,15486,15544,15602,15660,15718,15776,15834,15892,15950,16008,16066,16124,16182,16240,16298,16356,16414,16472,16530,16588,16646,16704,16762,16820,16878,16936,16994,17052,17110,17168,17226,17284,17342,17400,17458,17516,17574,17632,17690,17748,17806,17864,17922,17980,18038,18096,18154,18212,18270,18328,18386,18444,18502,18560,18618,18676,18734,18792,18850,18908,18966,19024,19082,19140,19198,19256,19314,19372,19430,19488,19546,19604,19662,19720,19778,19836,19894,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3480,3538,3596,3654,3712,3770,3828,3886,3944,4002,4060,4118,4176,4234,4292,4350,4408,4466,4524,4582,4640,4698,4756,4814,4872,4930,4988,5046,5104,5162,5220,5278,5336,5394,5452,5510,5568,5626,5684,5742,5800,5858,5916,5974,6032,6090,6148,6206,6264,6322,6380,6438,6496,6554,6612,6670,6728,6786,6844,6902,6960,7018,7076,7134,7192,7250,7308,7366,7424,7482,7540,7598,7656,7714,7772,7830,7888,7946,8004,8062,8120,8178,8236,8294,8352,8410,8468,8526,8584,8642,8700,8758,8816,8874,8932,8990,9048,9106,9164,9222,9280,9338,9396,9454,9512,9570,9628,9686,9744,9802,9860,9918,9976,10034,10092,10150,10208,10266,10324,10382,10440,10498,10556,10614,10672,10730,10788,10846,10904,10962,11020,11078,11136,11194,11252,11310,11368,11426,11484,11542,11600,11658,11716,11774,11832,11890,11948,12006,12064,12122,12180,12238,12296,12354,12412,12470,12528,12586,12644,12702,12760,12818,12876,12934,12992,13050,13108,13166,13224,13282,13340,13398,13456,13514,13572,13630,13688,13746,13804,13862,13920,13978,14036,14094,14152,14210,14268,14326,14384,14442,14500,14558,14616,14674,14732,14790,14848,14906,14964,15022,15080,15138,15196,15254,15312,15370,15428,15486,15544,15602,15660,15718,15776,15834,15892,15950,16008,16066,16124,16182,16240,16298,16356,16414,16472,16530,16588,16646,16704,16762,16820,16878,16936,16994,17052,17110,17168,17226,17284,17342,17400,17458,17516,17574,17632,17690,17748,17806,17864,17922,17980,18038,18096,18154,18212,18270,18328,18386,18444,18502,18560,18618,18676,18734,18792,18850,18908,18966,19024,19082,19140,19198,19256,19314,19372,19430,19488,19546,19604,19662,19720,19778,19836,19894,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3599,3658,3717,3776,3835,3894,3953,4012,4071,4130,4189,4248,4307,4366,4425,4484,4543,4602,4661,4720,4779,4838,4897,4956,5015,5074,5133,5192,5251,5310,5369,5428,5487,5546,5605,5664,5723,5782,5841,5900,5959,6018,6077,6136,6195,6254,6313,6372,6431,6490,6549,6608,6667,6726,6785,6844,6903,6962,7021,7080,7139,7198,7257,7316,7375,7434,7493,7552,7611,7670,7729,7788,7847,7906,7965,8024,8083,8142,8201,8260,8319,8378,8437,8496,8555,8614,8673,8732,8791,8850,8909,8968,9027,9086,9145,9204,9263,9322,9381,9440,9499,9558,9617,9676,9735,9794,9853,9912,9971,10030,10089,10148,10207,10266,10325,10384,10443,10502,10561,10620,10679,10738,10797,10856,10915,10974,11033,11092,11151,11210,11269,11328,11387,11446,11505,11564,11623,11682,11741,11800,11859,11918,11977,12036,12095,12154,12213,12272,12331,12390,12449,12508,12567,12626,12685,12744,12803,12862,12921,12980,13039,13098,13157,13216,13275,13334,13393,13452,13511,13570,13629,13688,13747,13806,13865,13924,13983,14042,14101,14160,14219,14278,14337,14396,14455,14514,14573,14632,14691,14750,14809,14868,14927,14986,15045,15104,15163,15222,15281,15340,15399,15458,15517,15576,15635,15694,15753,15812,15871,15930,15989,16048,16107,16166,16225,16284,16343,16402,16461,16520,16579,16638,16697,16756,16815,16874,16933,16992,17051,17110,17169,17228,17287,17346,17405,17464,17523,17582,17641,17700,17759,17818,17877,17936,17995,18054,18113,18172,18231,18290,18349,18408,18467,18526,18585,18644,18703,18762,18821,18880,18939,18998,19057,19116,19175,19234,19293,19352,19411,19470,19529,19588,19647,19706,19765,19824,19883,19942,20001,20060,20119,20178,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3599,3658,3717,3776,3835,3894,3953,4012,4071,4130,4189,4248,4307,4366,4425,4484,4543,4602,4661,4720,4779,4838,4897,4956,5015,5074,5133,5192,5251,5310,5369,5428,5487,5546,5605,5664,5723,5782,5841,5900,5959,6018,6077,6136,6195,6254,6313,6372,6431,6490,6549,6608,6667,6726,6785,6844,6903,6962,7021,7080,7139,7198,7257,7316,7375,7434,7493,7552,7611,7670,7729,7788,7847,7906,7965,8024,8083,8142,8201,8260,8319,8378,8437,8496,8555,8614,8673,8732,8791,8850,8909,8968,9027,9086,9145,9204,9263,9322,9381,9440,9499,9558,9617,9676,9735,9794,9853,9912,9971,10030,10089,10148,10207,10266,10325,10384,10443,10502,10561,10620,10679,10738,10797,10856,10915,10974,11033,11092,11151,11210,11269,11328,11387,11446,11505,11564,11623,11682,11741,11800,11859,11918,11977,12036,12095,12154,12213,12272,12331,12390,12449,12508,12567,12626,12685,12744,12803,12862,12921,12980,13039,13098,13157,13216,13275,13334,13393,13452,13511,13570,13629,13688,13747,13806,13865,13924,13983,14042,14101,14160,14219,14278,14337,14396,14455,14514,14573,14632,14691,14750,14809,14868,14927,14986,15045,15104,15163,15222,15281,15340,15399,15458,15517,15576,15635,15694,15753,15812,15871,15930,15989,16048,16107,16166,16225,16284,16343,16402,16461,16520,16579,16638,16697,16756,16815,16874,16933,16992,17051,17110,17169,17228,17287,17346,17405,17464,17523,17582,17641,17700,17759,17818,17877,17936,17995,18054,18113,18172,18231,18290,18349,18408,18467,18526,18585,18644,18703,18762,18821,18880,18939,18998,19057,19116,19175,19234,19293,19352,19411,19470,19529,19588,19647,19706,19765,19824,19883,19942,20001,20060,20119,20178,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3720,3780,3840,3900,3960,4020,4080,4140,4200,4260,4320,4380,4440,4500,4560,4620,4680,4740,4800,4860,4920,4980,5040,5100,5160,5220,5280,5340,5400,5460,5520,5580,5640,5700,5760,5820,5880,5940,6000,6060,6120,6180,6240,6300,6360,6420,6480,6540,6600,6660,6720,6780,6840,6900,6960,7020,7080,7140,7200,7260,7320,7380,7440,7500,7560,7620,7680,7740,7800,7860,7920,7980,8040,8100,8160,8220,8280,8340,8400,8460,8520,8580,8640,8700,8760,8820,8880,8940,9000,9060,9120,9180,9240,9300,9360,9420,9480,9540,9600,9660,9720,9780,9840,9900,9960,10020,10080,10140,10200,10260,10320,10380,10440,10500,10560,10620,10680,10740,10800,10860,10920,10980,11040,11100,11160,11220,11280,11340,11400,11460,11520,11580,11640,11700,11760,11820,11880,11940,12000,12060,12120,12180,12240,12300,12360,12420,12480,12540,12600,12660,12720,12780,12840,12900,12960,13020,13080,13140,13200,13260,13320,13380,13440,13500,13560,13620,13680,13740,13800,13860,13920,13980,14040,14100,14160,14220,14280,14340,14400,14460,14520,14580,14640,14700,14760,14820,14880,14940,15000,15060,15120,15180,15240,15300,15360,15420,15480,15540,15600,15660,15720,15780,15840,15900,15960,16020,16080,16140,16200,16260,16320,16380,16440,16500,16560,16620,16680,16740,16800,16860,16920,16980,17040,17100,17160,17220,17280,17340,17400,17460,17520,17580,17640,17700,17760,17820,17880,17940,18000,18060,18120,18180,18240,18300,18360,18420,18480,18540,18600,18660,18720,18780,18840,18900,18960,19020,19080,19140,19200,19260,19320,19380,19440,19500,19560,19620,19680,19740,19800,19860,19920,19980,20040,20100,20160,20220,20280,20340,20400,20460,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3720,3780,3840,3900,3960,4020,4080,4140,4200,4260,4320,4380,4440,4500,4560,4620,4680,4740,4800,4860,4920,4980,5040,5100,5160,5220,5280,5340,5400,5460,5520,5580,5640,5700,5760,5820,5880,5940,6000,6060,6120,6180,6240,6300,6360,6420,6480,6540,6600,6660,6720,6780,6840,6900,6960,7020,7080,7140,7200,7260,7320,7380,7440,7500,7560,7620,7680,7740,7800,7860,7920,7980,8040,8100,8160,8220,8280,8340,8400,8460,8520,8580,8640,8700,8760,8820,8880,8940,9000,9060,9120,9180,9240,9300,9360,9420,9480,9540,9600,9660,9720,9780,9840,9900,9960,10020,10080,10140,10200,10260,10320,10380,10440,10500,10560,10620,10680,10740,10800,10860,10920,10980,11040,11100,11160,11220,11280,11340,11400,11460,11520,11580,11640,11700,11760,11820,11880,11940,12000,12060,12120,12180,12240,12300,12360,12420,12480,12540,12600,12660,12720,12780,12840,12900,12960,13020,13080,13140,13200,13260,13320,13380,13440,13500,13560,13620,13680,13740,13800,13860,13920,13980,14040,14100,14160,14220,14280,14340,14400,14460,14520,14580,14640,14700,14760,14820,14880,14940,15000,15060,15120,15180,15240,15300,15360,15420,15480,15540,15600,15660,15720,15780,15840,15900,15960,16020,16080,16140,16200,16260,16320,16380,16440,16500,16560,16620,16680,16740,16800,16860,16920,16980,17040,17100,17160,17220,17280,17340,17400,17460,17520,17580,17640,17700,17760,17820,17880,17940,18000,18060,18120,18180,18240,18300,18360,18420,18480,18540,18600,18660,18720,18780,18840,18900,18960,19020,19080,19140,19200,19260,19320,19380,19440,19500,19560,19620,19680,19740,19800,19860,19920,19980,20040,20100,20160,20220,20280,20340,20400,20460,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3843,3904,3965,4026,4087,4148,4209,4270,4331,4392,4453,4514,4575,4636,4697,4758,4819,4880,4941,5002,5063,5124,5185,5246,5307,5368,5429,5490,5551,5612,5673,5734,5795,5856,5917,5978,6039,6100,6161,6222,6283,6344,6405,6466,6527,6588,6649,6710,6771,6832,6893,6954,7015,7076,7137,7198,7259,7320,7381,7442,7503,7564,7625,7686,7747,7808,7869,7930,7991,8052,8113,8174,8235,8296,8357,8418,8479,8540,8601,8662,8723,8784,8845,8906,8967,9028,9089,9150,9211,9272,9333,9394,9455,9516,9577,9638,9699,9760,9821,9882,9943,10004,10065,10126,10187,10248,10309,10370,10431,10492,10553,10614,10675,10736,10797,10858,10919,10980,11041,11102,11163,11224,11285,11346,11407,11468,11529,11590,11651,11712,11773,11834,11895,11956,12017,12078,12139,12200,12261,12322,12383,12444,12505,12566,12627,12688,12749,12810,12871,12932,12993,13054,13115,13176,13237,13298,13359,13420,13481,13542,13603,13664,13725,13786,13847,13908,13969,14030,14091,14152,14213,14274,14335,14396,14457,14518,14579,14640,14701,14762,14823,14884,14945,15006,15067,15128,15189,15250,15311,15372,15433,15494,15555,15616,15677,15738,15799,15860,15921,15982,16043,16104,16165,16226,16287,16348,16409,16470,16531,16592,16653,16714,16775,16836,16897,16958,17019,17080,17141,17202,17263,17324,17385,17446,17507,17568,17629,17690,17751,17812,17873,17934,17995,18056,18117,18178,18239,18300,18361,18422,18483,18544,18605,18666,18727,18788,18849,18910,18971,19032,19093,19154,19215,19276,19337,19398,19459,19520,19581,19642,19703,19764,19825,19886,19947,20008,20069,20130,20191,20252,20313,20374,20435,20496,20557,20618,20679,20740,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3843,3904,3965,4026,4087,4148,4209,4270,4331,4392,4453,4514,4575,4636,4697,4758,4819,4880,4941,5002,5063,5124,5185,5246,5307,5368,5429,5490,5551,5612,5673,5734,5795,5856,5917,5978,6039,6100,6161,6222,6283,6344,6405,6466,6527,6588,6649,6710,6771,6832,6893,6954,7015,7076,7137,7198,7259,7320,7381,7442,7503,7564,7625,7686,7747,7808,7869,7930,7991,8052,8113,8174,8235,8296,8357,8418,8479,8540,8601,8662,8723,8784,8845,8906,8967,9028,9089,9150,9211,9272,9333,9394,9455,9516,9577,9638,9699,9760,9821,9882,9943,10004,10065,10126,10187,10248,10309,10370,10431,10492,10553,10614,10675,10736,10797,10858,10919,10980,11041,11102,11163,11224,11285,11346,11407,11468,11529,11590,11651,11712,11773,11834,11895,11956,12017,12078,12139,12200,12261,12322,12383,12444,12505,12566,12627,12688,12749,12810,12871,12932,12993,13054,13115,13176,13237,13298,13359,13420,13481,13542,13603,13664,13725,13786,13847,13908,13969,14030,14091,14152,14213,14274,14335,14396,14457,14518,14579,14640,14701,14762,14823,14884,14945,15006,15067,15128,15189,15250,15311,15372,15433,15494,15555,15616,15677,15738,15799,15860,15921,15982,16043,16104,16165,16226,16287,16348,16409,16470,16531,16592,16653,16714,16775,16836,16897,16958,17019,17080,17141,17202,17263,17324,17385,17446,17507,17568,17629,17690,17751,17812,17873,17934,17995,18056,18117,18178,18239,18300,18361,18422,18483,18544,18605,18666,18727,18788,18849,18910,18971,19032,19093,19154,19215,19276,19337,19398,19459,19520,19581,19642,19703,19764,19825,19886,19947,20008,20069,20130,20191,20252,20313,20374,20435,20496,20557,20618,20679,20740,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3968,4030,4092,4154,4216,4278,4340,4402,4464,4526,4588,4650,4712,4774,4836,4898,4960,5022,5084,5146,5208,5270,5332,5394,5456,5518,5580,5642,5704,5766,5828,5890,5952,6014,6076,6138,6200,6262,6324,6386,6448,6510,6572,6634,6696,6758,6820,6882,6944,7006,7068,7130,7192,7254,7316,7378,7440,7502,7564,7626,7688,7750,7812,7874,7936,7998,8060,8122,8184,8246,8308,8370,8432,8494,8556,8618,8680,8742,8804,8866,8928,8990,9052,9114,9176,9238,9300,9362,9424,9486,9548,9610,9672,9734,9796,9858,9920,9982,10044,10106,10168,10230,10292,10354,10416,10478,10540,10602,10664,10726,10788,10850,10912,10974,11036,11098,11160,11222,11284,11346,11408,11470,11532,11594,11656,11718,11780,11842,11904,11966,12028,12090,12152,12214,12276,12338,12400,12462,12524,12586,12648,12710,12772,12834,12896,12958,13020,13082,13144,13206,13268,13330,13392,13454,13516,13578,13640,13702,13764,13826,13888,13950,14012,14074,14136,14198,14260,14322,14384,14446,14508,14570,14632,14694,14756,14818,14880,14942,15004,15066,15128,15190,15252,15314,15376,15438,15500,15562,15624,15686,15748,15810,15872,15934,15996,16058,16120,16182,16244,16306,16368,16430,16492,16554,16616,16678,16740,16802,16864,16926,16988,17050,17112,17174,17236,17298,17360,17422,17484,17546,17608,17670,17732,17794,17856,17918,17980,18042,18104,18166,18228,18290,18352,18414,18476,18538,18600,18662,18724,18786,18848,18910,18972,19034,19096,19158,19220,19282,19344,19406,19468,19530,19592,19654,19716,19778,19840,19902,19964,20026,20088,20150,20212,20274,20336,20398,20460,20522,20584,20646,20708,20770,20832,20894,20956,21018,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3968,4030,4092,4154,4216,4278,4340,4402,4464,4526,4588,4650,4712,4774,4836,4898,4960,5022,5084,5146,5208,5270,5332,5394,5456,5518,5580,5642,5704,5766,5828,5890,5952,6014,6076,6138,6200,6262,6324,6386,6448,6510,6572,6634,6696,6758,6820,6882,6944,7006,7068,7130,7192,7254,7316,7378,7440,7502,7564,7626,7688,7750,7812,7874,7936,7998,8060,8122,8184,8246,8308,8370,8432,8494,8556,8618,8680,8742,8804,8866,8928,8990,9052,9114,9176,9238,9300,9362,9424,9486,9548,9610,9672,9734,9796,9858,9920,9982,10044,10106,10168,10230,10292,10354,10416,10478,10540,10602,10664,10726,10788,10850,10912,10974,11036,11098,11160,11222,11284,11346,11408,11470,11532,11594,11656,11718,11780,11842,11904,11966,12028,12090,12152,12214,12276,12338,12400,12462,12524,12586,12648,12710,12772,12834,12896,12958,13020,13082,13144,13206,13268,13330,13392,13454,13516,13578,13640,13702,13764,13826,13888,13950,14012,14074,14136,14198,14260,14322,14384,14446,14508,14570,14632,14694,14756,14818,14880,14942,15004,15066,15128,15190,15252,15314,15376,15438,15500,15562,15624,15686,15748,15810,15872,15934,15996,16058,16120,16182,16244,16306,16368,16430,16492,16554,16616,16678,16740,16802,16864,16926,16988,17050,17112,17174,17236,17298,17360,17422,17484,17546,17608,17670,17732,17794,17856,17918,17980,18042,18104,18166,18228,18290,18352,18414,18476,18538,18600,18662,18724,18786,18848,18910,18972,19034,19096,19158,19220,19282,19344,19406,19468,19530,19592,19654,19716,19778,19840,19902,19964,20026,20088,20150,20212,20274,20336,20398,20460,20522,20584,20646,20708,20770,20832,20894,20956,21018,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4095,4158,4221,4284,4347,4410,4473,4536,4599,4662,4725,4788,4851,4914,4977,5040,5103,5166,5229,5292,5355,5418,5481,5544,5607,5670,5733,5796,5859,5922,5985,6048,6111,6174,6237,6300,6363,6426,6489,6552,6615,6678,6741,6804,6867,6930,6993,7056,7119,7182,7245,7308,7371,7434,7497,7560,7623,7686,7749,7812,7875,7938,8001,8064,8127,8190,8253,8316,8379,8442,8505,8568,8631,8694,8757,8820,8883,8946,9009,9072,9135,9198,9261,9324,9387,9450,9513,9576,9639,9702,9765,9828,9891,9954,10017,10080,10143,10206,10269,10332,10395,10458,10521,10584,10647,10710,10773,10836,10899,10962,11025,11088,11151,11214,11277,11340,11403,11466,11529,11592,11655,11718,11781,11844,11907,11970,12033,12096,12159,12222,12285,12348,12411,12474,12537,12600,12663,12726,12789,12852,12915,12978,13041,13104,13167,13230,13293,13356,13419,13482,13545,13608,13671,13734,13797,13860,13923,13986,14049,14112,14175,14238,14301,14364,14427,14490,14553,14616,14679,14742,14805,14868,14931,14994,15057,15120,15183,15246,15309,15372,15435,15498,15561,15624,15687,15750,15813,15876,15939,16002,16065,16128,16191,16254,16317,16380,16443,16506,16569,16632,16695,16758,16821,16884,16947,17010,17073,17136,17199,17262,17325,17388,17451,17514,17577,17640,17703,17766,17829,17892,17955,18018,18081,18144,18207,18270,18333,18396,18459,18522,18585,18648,18711,18774,18837,18900,18963,19026,19089,19152,19215,19278,19341,19404,19467,19530,19593,19656,19719,19782,19845,19908,19971,20034,20097,20160,20223,20286,20349,20412,20475,20538,20601,20664,20727,20790,20853,20916,20979,21042,21105,21168,21231,21294,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4095,4158,4221,4284,4347,4410,4473,4536,4599,4662,4725,4788,4851,4914,4977,5040,5103,5166,5229,5292,5355,5418,5481,5544,5607,5670,5733,5796,5859,5922,5985,6048,6111,6174,6237,6300,6363,6426,6489,6552,6615,6678,6741,6804,6867,6930,6993,7056,7119,7182,7245,7308,7371,7434,7497,7560,7623,7686,7749,7812,7875,7938,8001,8064,8127,8190,8253,8316,8379,8442,8505,8568,8631,8694,8757,8820,8883,8946,9009,9072,9135,9198,9261,9324,9387,9450,9513,9576,9639,9702,9765,9828,9891,9954,10017,10080,10143,10206,10269,10332,10395,10458,10521,10584,10647,10710,10773,10836,10899,10962,11025,11088,11151,11214,11277,11340,11403,11466,11529,11592,11655,11718,11781,11844,11907,11970,12033,12096,12159,12222,12285,12348,12411,12474,12537,12600,12663,12726,12789,12852,12915,12978,13041,13104,13167,13230,13293,13356,13419,13482,13545,13608,13671,13734,13797,13860,13923,13986,14049,14112,14175,14238,14301,14364,14427,14490,14553,14616,14679,14742,14805,14868,14931,14994,15057,15120,15183,15246,15309,15372,15435,15498,15561,15624,15687,15750,15813,15876,15939,16002,16065,16128,16191,16254,16317,16380,16443,16506,16569,16632,16695,16758,16821,16884,16947,17010,17073,17136,17199,17262,17325,17388,17451,17514,17577,17640,17703,17766,17829,17892,17955,18018,18081,18144,18207,18270,18333,18396,18459,18522,18585,18648,18711,18774,18837,18900,18963,19026,19089,19152,19215,19278,19341,19404,19467,19530,19593,19656,19719,19782,19845,19908,19971,20034,20097,20160,20223,20286,20349,20412,20475,20538,20601,20664,20727,20790,20853,20916,20979,21042,21105,21168,21231,21294,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4224,4288,4352,4416,4480,4544,4608,4672,4736,4800,4864,4928,4992,5056,5120,5184,5248,5312,5376,5440,5504,5568,5632,5696,5760,5824,5888,5952,6016,6080,6144,6208,6272,6336,6400,6464,6528,6592,6656,6720,6784,6848,6912,6976,7040,7104,7168,7232,7296,7360,7424,7488,7552,7616,7680,7744,7808,7872,7936,8000,8064,8128,8192,8256,8320,8384,8448,8512,8576,8640,8704,8768,8832,8896,8960,9024,9088,9152,9216,9280,9344,9408,9472,9536,9600,9664,9728,9792,9856,9920,9984,10048,10112,10176,10240,10304,10368,10432,10496,10560,10624,10688,10752,10816,10880,10944,11008,11072,11136,11200,11264,11328,11392,11456,11520,11584,11648,11712,11776,11840,11904,11968,12032,12096,12160,12224,12288,12352,12416,12480,12544,12608,12672,12736,12800,12864,12928,12992,13056,13120,13184,13248,13312,13376,13440,13504,13568,13632,13696,13760,13824,13888,13952,14016,14080,14144,14208,14272,14336,14400,14464,14528,14592,14656,14720,14784,14848,14912,14976,15040,15104,15168,15232,15296,15360,15424,15488,15552,15616,15680,15744,15808,15872,15936,16000,16064,16128,16192,16256,16320,16384,16448,16512,16576,16640,16704,16768,16832,16896,16960,17024,17088,17152,17216,17280,17344,17408,17472,17536,17600,17664,17728,17792,17856,17920,17984,18048,18112,18176,18240,18304,18368,18432,18496,18560,18624,18688,18752,18816,18880,18944,19008,19072,19136,19200,19264,19328,19392,19456,19520,19584,19648,19712,19776,19840,19904,19968,20032,20096,20160,20224,20288,20352,20416,20480,20544,20608,20672,20736,20800,20864,20928,20992,21056,21120,21184,21248,21312,21376,21440,21504,21568,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4224,4288,4352,4416,4480,4544,4608,4672,4736,4800,4864,4928,4992,5056,5120,5184,5248,5312,5376,5440,5504,5568,5632,5696,5760,5824,5888,5952,6016,6080,6144,6208,6272,6336,6400,6464,6528,6592,6656,6720,6784,6848,6912,6976,7040,7104,7168,7232,7296,7360,7424,7488,7552,7616,7680,7744,7808,7872,7936,8000,8064,8128,8192,8256,8320,8384,8448,8512,8576,8640,8704,8768,8832,8896,8960,9024,9088,9152,9216,9280,9344,9408,9472,9536,9600,9664,9728,9792,9856,9920,9984,10048,10112,10176,10240,10304,10368,10432,10496,10560,10624,10688,10752,10816,10880,10944,11008,11072,11136,11200,11264,11328,11392,11456,11520,11584,11648,11712,11776,11840,11904,11968,12032,12096,12160,12224,12288,12352,12416,12480,12544,12608,12672,12736,12800,12864,12928,12992,13056,13120,13184,13248,13312,13376,13440,13504,13568,13632,13696,13760,13824,13888,13952,14016,14080,14144,14208,14272,14336,14400,14464,14528,14592,14656,14720,14784,14848,14912,14976,15040,15104,15168,15232,15296,15360,15424,15488,15552,15616,15680,15744,15808,15872,15936,16000,16064,16128,16192,16256,16320,16384,16448,16512,16576,16640,16704,16768,16832,16896,16960,17024,17088,17152,17216,17280,17344,17408,17472,17536,17600,17664,17728,17792,17856,17920,17984,18048,18112,18176,18240,18304,18368,18432,18496,18560,18624,18688,18752,18816,18880,18944,19008,19072,19136,19200,19264,19328,19392,19456,19520,19584,19648,19712,19776,19840,19904,19968,20032,20096,20160,20224,20288,20352,20416,20480,20544,20608,20672,20736,20800,20864,20928,20992,21056,21120,21184,21248,21312,21376,21440,21504,21568,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4355,4420,4485,4550,4615,4680,4745,4810,4875,4940,5005,5070,5135,5200,5265,5330,5395,5460,5525,5590,5655,5720,5785,5850,5915,5980,6045,6110,6175,6240,6305,6370,6435,6500,6565,6630,6695,6760,6825,6890,6955,7020,7085,7150,7215,7280,7345,7410,7475,7540,7605,7670,7735,7800,7865,7930,7995,8060,8125,8190,8255,8320,8385,8450,8515,8580,8645,8710,8775,8840,8905,8970,9035,9100,9165,9230,9295,9360,9425,9490,9555,9620,9685,9750,9815,9880,9945,10010,10075,10140,10205,10270,10335,10400,10465,10530,10595,10660,10725,10790,10855,10920,10985,11050,11115,11180,11245,11310,11375,11440,11505,11570,11635,11700,11765,11830,11895,11960,12025,12090,12155,12220,12285,12350,12415,12480,12545,12610,12675,12740,12805,12870,12935,13000,13065,13130,13195,13260,13325,13390,13455,13520,13585,13650,13715,13780,13845,13910,13975,14040,14105,14170,14235,14300,14365,14430,14495,14560,14625,14690,14755,14820,14885,14950,15015,15080,15145,15210,15275,15340,15405,15470,15535,15600,15665,15730,15795,15860,15925,15990,16055,16120,16185,16250,16315,16380,16445,16510,16575,16640,16705,16770,16835,16900,16965,17030,17095,17160,17225,17290,17355,17420,17485,17550,17615,17680,17745,17810,17875,17940,18005,18070,18135,18200,18265,18330,18395,18460,18525,18590,18655,18720,18785,18850,18915,18980,19045,19110,19175,19240,19305,19370,19435,19500,19565,19630,19695,19760,19825,19890,19955,20020,20085,20150,20215,20280,20345,20410,20475,20540,20605,20670,20735,20800,20865,20930,20995,21060,21125,21190,21255,21320,21385,21450,21515,21580,21645,21710,21775,21840,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4355,4420,4485,4550,4615,4680,4745,4810,4875,4940,5005,5070,5135,5200,5265,5330,5395,5460,5525,5590,5655,5720,5785,5850,5915,5980,6045,6110,6175,6240,6305,6370,6435,6500,6565,6630,6695,6760,6825,6890,6955,7020,7085,7150,7215,7280,7345,7410,7475,7540,7605,7670,7735,7800,7865,7930,7995,8060,8125,8190,8255,8320,8385,8450,8515,8580,8645,8710,8775,8840,8905,8970,9035,9100,9165,9230,9295,9360,9425,9490,9555,9620,9685,9750,9815,9880,9945,10010,10075,10140,10205,10270,10335,10400,10465,10530,10595,10660,10725,10790,10855,10920,10985,11050,11115,11180,11245,11310,11375,11440,11505,11570,11635,11700,11765,11830,11895,11960,12025,12090,12155,12220,12285,12350,12415,12480,12545,12610,12675,12740,12805,12870,12935,13000,13065,13130,13195,13260,13325,13390,13455,13520,13585,13650,13715,13780,13845,13910,13975,14040,14105,14170,14235,14300,14365,14430,14495,14560,14625,14690,14755,14820,14885,14950,15015,15080,15145,15210,15275,15340,15405,15470,15535,15600,15665,15730,15795,15860,15925,15990,16055,16120,16185,16250,16315,16380,16445,16510,16575,16640,16705,16770,16835,16900,16965,17030,17095,17160,17225,17290,17355,17420,17485,17550,17615,17680,17745,17810,17875,17940,18005,18070,18135,18200,18265,18330,18395,18460,18525,18590,18655,18720,18785,18850,18915,18980,19045,19110,19175,19240,19305,19370,19435,19500,19565,19630,19695,19760,19825,19890,19955,20020,20085,20150,20215,20280,20345,20410,20475,20540,20605,20670,20735,20800,20865,20930,20995,21060,21125,21190,21255,21320,21385,21450,21515,21580,21645,21710,21775,21840,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4488,4554,4620,4686,4752,4818,4884,4950,5016,5082,5148,5214,5280,5346,5412,5478,5544,5610,5676,5742,5808,5874,5940,6006,6072,6138,6204,6270,6336,6402,6468,6534,6600,6666,6732,6798,6864,6930,6996,7062,7128,7194,7260,7326,7392,7458,7524,7590,7656,7722,7788,7854,7920,7986,8052,8118,8184,8250,8316,8382,8448,8514,8580,8646,8712,8778,8844,8910,8976,9042,9108,9174,9240,9306,9372,9438,9504,9570,9636,9702,9768,9834,9900,9966,10032,10098,10164,10230,10296,10362,10428,10494,10560,10626,10692,10758,10824,10890,10956,11022,11088,11154,11220,11286,11352,11418,11484,11550,11616,11682,11748,11814,11880,11946,12012,12078,12144,12210,12276,12342,12408,12474,12540,12606,12672,12738,12804,12870,12936,13002,13068,13134,13200,13266,13332,13398,13464,13530,13596,13662,13728,13794,13860,13926,13992,14058,14124,14190,14256,14322,14388,14454,14520,14586,14652,14718,14784,14850,14916,14982,15048,15114,15180,15246,15312,15378,15444,15510,15576,15642,15708,15774,15840,15906,15972,16038,16104,16170,16236,16302,16368,16434,16500,16566,16632,16698,16764,16830,16896,16962,17028,17094,17160,17226,17292,17358,17424,17490,17556,17622,17688,17754,17820,17886,17952,18018,18084,18150,18216,18282,18348,18414,18480,18546,18612,18678,18744,18810,18876,18942,19008,19074,19140,19206,19272,19338,19404,19470,19536,19602,19668,19734,19800,19866,19932,19998,20064,20130,20196,20262,20328,20394,20460,20526,20592,20658,20724,20790,20856,20922,20988,21054,21120,21186,21252,21318,21384,21450,21516,21582,21648,21714,21780,21846,21912,21978,22044,22110,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4488,4554,4620,4686,4752,4818,4884,4950,5016,5082,5148,5214,5280,5346,5412,5478,5544,5610,5676,5742,5808,5874,5940,6006,6072,6138,6204,6270,6336,6402,6468,6534,6600,6666,6732,6798,6864,6930,6996,7062,7128,7194,7260,7326,7392,7458,7524,7590,7656,7722,7788,7854,7920,7986,8052,8118,8184,8250,8316,8382,8448,8514,8580,8646,8712,8778,8844,8910,8976,9042,9108,9174,9240,9306,9372,9438,9504,9570,9636,9702,9768,9834,9900,9966,10032,10098,10164,10230,10296,10362,10428,10494,10560,10626,10692,10758,10824,10890,10956,11022,11088,11154,11220,11286,11352,11418,11484,11550,11616,11682,11748,11814,11880,11946,12012,12078,12144,12210,12276,12342,12408,12474,12540,12606,12672,12738,12804,12870,12936,13002,13068,13134,13200,13266,13332,13398,13464,13530,13596,13662,13728,13794,13860,13926,13992,14058,14124,14190,14256,14322,14388,14454,14520,14586,14652,14718,14784,14850,14916,14982,15048,15114,15180,15246,15312,15378,15444,15510,15576,15642,15708,15774,15840,15906,15972,16038,16104,16170,16236,16302,16368,16434,16500,16566,16632,16698,16764,16830,16896,16962,17028,17094,17160,17226,17292,17358,17424,17490,17556,17622,17688,17754,17820,17886,17952,18018,18084,18150,18216,18282,18348,18414,18480,18546,18612,18678,18744,18810,18876,18942,19008,19074,19140,19206,19272,19338,19404,19470,19536,19602,19668,19734,19800,19866,19932,19998,20064,20130,20196,20262,20328,20394,20460,20526,20592,20658,20724,20790,20856,20922,20988,21054,21120,21186,21252,21318,21384,21450,21516,21582,21648,21714,21780,21846,21912,21978,22044,22110,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4623,4690,4757,4824,4891,4958,5025,5092,5159,5226,5293,5360,5427,5494,5561,5628,5695,5762,5829,5896,5963,6030,6097,6164,6231,6298,6365,6432,6499,6566,6633,6700,6767,6834,6901,6968,7035,7102,7169,7236,7303,7370,7437,7504,7571,7638,7705,7772,7839,7906,7973,8040,8107,8174,8241,8308,8375,8442,8509,8576,8643,8710,8777,8844,8911,8978,9045,9112,9179,9246,9313,9380,9447,9514,9581,9648,9715,9782,9849,9916,9983,10050,10117,10184,10251,10318,10385,10452,10519,10586,10653,10720,10787,10854,10921,10988,11055,11122,11189,11256,11323,11390,11457,11524,11591,11658,11725,11792,11859,11926,11993,12060,12127,12194,12261,12328,12395,12462,12529,12596,12663,12730,12797,12864,12931,12998,13065,13132,13199,13266,13333,13400,13467,13534,13601,13668,13735,13802,13869,13936,14003,14070,14137,14204,14271,14338,14405,14472,14539,14606,14673,14740,14807,14874,14941,15008,15075,15142,15209,15276,15343,15410,15477,15544,15611,15678,15745,15812,15879,15946,16013,16080,16147,16214,16281,16348,16415,16482,16549,16616,16683,16750,16817,16884,16951,17018,17085,17152,17219,17286,17353,17420,17487,17554,17621,17688,17755,17822,17889,17956,18023,18090,18157,18224,18291,18358,18425,18492,18559,18626,18693,18760,18827,18894,18961,19028,19095,19162,19229,19296,19363,19430,19497,19564,19631,19698,19765,19832,19899,19966,20033,20100,20167,20234,20301,20368,20435,20502,20569,20636,20703,20770,20837,20904,20971,21038,21105,21172,21239,21306,21373,21440,21507,21574,21641,21708,21775,21842,21909,21976,22043,22110,22177,22244,22311,22378,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4623,4690,4757,4824,4891,4958,5025,5092,5159,5226,5293,5360,5427,5494,5561,5628,5695,5762,5829,5896,5963,6030,6097,6164,6231,6298,6365,6432,6499,6566,6633,6700,6767,6834,6901,6968,7035,7102,7169,7236,7303,7370,7437,7504,7571,7638,7705,7772,7839,7906,7973,8040,8107,8174,8241,8308,8375,8442,8509,8576,8643,8710,8777,8844,8911,8978,9045,9112,9179,9246,9313,9380,9447,9514,9581,9648,9715,9782,9849,9916,9983,10050,10117,10184,10251,10318,10385,10452,10519,10586,10653,10720,10787,10854,10921,10988,11055,11122,11189,11256,11323,11390,11457,11524,11591,11658,11725,11792,11859,11926,11993,12060,12127,12194,12261,12328,12395,12462,12529,12596,12663,12730,12797,12864,12931,12998,13065,13132,13199,13266,13333,13400,13467,13534,13601,13668,13735,13802,13869,13936,14003,14070,14137,14204,14271,14338,14405,14472,14539,14606,14673,14740,14807,14874,14941,15008,15075,15142,15209,15276,15343,15410,15477,15544,15611,15678,15745,15812,15879,15946,16013,16080,16147,16214,16281,16348,16415,16482,16549,16616,16683,16750,16817,16884,16951,17018,17085,17152,17219,17286,17353,17420,17487,17554,17621,17688,17755,17822,17889,17956,18023,18090,18157,18224,18291,18358,18425,18492,18559,18626,18693,18760,18827,18894,18961,19028,19095,19162,19229,19296,19363,19430,19497,19564,19631,19698,19765,19832,19899,19966,20033,20100,20167,20234,20301,20368,20435,20502,20569,20636,20703,20770,20837,20904,20971,21038,21105,21172,21239,21306,21373,21440,21507,21574,21641,21708,21775,21842,21909,21976,22043,22110,22177,22244,22311,22378,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4760,4828,4896,4964,5032,5100,5168,5236,5304,5372,5440,5508,5576,5644,5712,5780,5848,5916,5984,6052,6120,6188,6256,6324,6392,6460,6528,6596,6664,6732,6800,6868,6936,7004,7072,7140,7208,7276,7344,7412,7480,7548,7616,7684,7752,7820,7888,7956,8024,8092,8160,8228,8296,8364,8432,8500,8568,8636,8704,8772,8840,8908,8976,9044,9112,9180,9248,9316,9384,9452,9520,9588,9656,9724,9792,9860,9928,9996,10064,10132,10200,10268,10336,10404,10472,10540,10608,10676,10744,10812,10880,10948,11016,11084,11152,11220,11288,11356,11424,11492,11560,11628,11696,11764,11832,11900,11968,12036,12104,12172,12240,12308,12376,12444,12512,12580,12648,12716,12784,12852,12920,12988,13056,13124,13192,13260,13328,13396,13464,13532,13600,13668,13736,13804,13872,13940,14008,14076,14144,14212,14280,14348,14416,14484,14552,14620,14688,14756,14824,14892,14960,15028,15096,15164,15232,15300,15368,15436,15504,15572,15640,15708,15776,15844,15912,15980,16048,16116,16184,16252,16320,16388,16456,16524,16592,16660,16728,16796,16864,16932,17000,17068,17136,17204,17272,17340,17408,17476,17544,17612,17680,17748,17816,17884,17952,18020,18088,18156,18224,18292,18360,18428,18496,18564,18632,18700,18768,18836,18904,18972,19040,19108,19176,19244,19312,19380,19448,19516,19584,19652,19720,19788,19856,19924,19992,20060,20128,20196,20264,20332,20400,20468,20536,20604,20672,20740,20808,20876,20944,21012,21080,21148,21216,21284,21352,21420,21488,21556,21624,21692,21760,21828,21896,21964,22032,22100,22168,22236,22304,22372,22440,22508,22576,22644,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4760,4828,4896,4964,5032,5100,5168,5236,5304,5372,5440,5508,5576,5644,5712,5780,5848,5916,5984,6052,6120,6188,6256,6324,6392,6460,6528,6596,6664,6732,6800,6868,6936,7004,7072,7140,7208,7276,7344,7412,7480,7548,7616,7684,7752,7820,7888,7956,8024,8092,8160,8228,8296,8364,8432,8500,8568,8636,8704,8772,8840,8908,8976,9044,9112,9180,9248,9316,9384,9452,9520,9588,9656,9724,9792,9860,9928,9996,10064,10132,10200,10268,10336,10404,10472,10540,10608,10676,10744,10812,10880,10948,11016,11084,11152,11220,11288,11356,11424,11492,11560,11628,11696,11764,11832,11900,11968,12036,12104,12172,12240,12308,12376,12444,12512,12580,12648,12716,12784,12852,12920,12988,13056,13124,13192,13260,13328,13396,13464,13532,13600,13668,13736,13804,13872,13940,14008,14076,14144,14212,14280,14348,14416,14484,14552,14620,14688,14756,14824,14892,14960,15028,15096,15164,15232,15300,15368,15436,15504,15572,15640,15708,15776,15844,15912,15980,16048,16116,16184,16252,16320,16388,16456,16524,16592,16660,16728,16796,16864,16932,17000,17068,17136,17204,17272,17340,17408,17476,17544,17612,17680,17748,17816,17884,17952,18020,18088,18156,18224,18292,18360,18428,18496,18564,18632,18700,18768,18836,18904,18972,19040,19108,19176,19244,19312,19380,19448,19516,19584,19652,19720,19788,19856,19924,19992,20060,20128,20196,20264,20332,20400,20468,20536,20604,20672,20740,20808,20876,20944,21012,21080,21148,21216,21284,21352,21420,21488,21556,21624,21692,21760,21828,21896,21964,22032,22100,22168,22236,22304,22372,22440,22508,22576,22644,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4899,4968,5037,5106,5175,5244,5313,5382,5451,5520,5589,5658,5727,5796,5865,5934,6003,6072,6141,6210,6279,6348,6417,6486,6555,6624,6693,6762,6831,6900,6969,7038,7107,7176,7245,7314,7383,7452,7521,7590,7659,7728,7797,7866,7935,8004,8073,8142,8211,8280,8349,8418,8487,8556,8625,8694,8763,8832,8901,8970,9039,9108,9177,9246,9315,9384,9453,9522,9591,9660,9729,9798,9867,9936,10005,10074,10143,10212,10281,10350,10419,10488,10557,10626,10695,10764,10833,10902,10971,11040,11109,11178,11247,11316,11385,11454,11523,11592,11661,11730,11799,11868,11937,12006,12075,12144,12213,12282,12351,12420,12489,12558,12627,12696,12765,12834,12903,12972,13041,13110,13179,13248,13317,13386,13455,13524,13593,13662,13731,13800,13869,13938,14007,14076,14145,14214,14283,14352,14421,14490,14559,14628,14697,14766,14835,14904,14973,15042,15111,15180,15249,15318,15387,15456,15525,15594,15663,15732,15801,15870,15939,16008,16077,16146,16215,16284,16353,16422,16491,16560,16629,16698,16767,16836,16905,16974,17043,17112,17181,17250,17319,17388,17457,17526,17595,17664,17733,17802,17871,17940,18009,18078,18147,18216,18285,18354,18423,18492,18561,18630,18699,18768,18837,18906,18975,19044,19113,19182,19251,19320,19389,19458,19527,19596,19665,19734,19803,19872,19941,20010,20079,20148,20217,20286,20355,20424,20493,20562,20631,20700,20769,20838,20907,20976,21045,21114,21183,21252,21321,21390,21459,21528,21597,21666,21735,21804,21873,21942,22011,22080,22149,22218,22287,22356,22425,22494,22563,22632,22701,22770,22839,22908,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4899,4968,5037,5106,5175,5244,5313,5382,5451,5520,5589,5658,5727,5796,5865,5934,6003,6072,6141,6210,6279,6348,6417,6486,6555,6624,6693,6762,6831,6900,6969,7038,7107,7176,7245,7314,7383,7452,7521,7590,7659,7728,7797,7866,7935,8004,8073,8142,8211,8280,8349,8418,8487,8556,8625,8694,8763,8832,8901,8970,9039,9108,9177,9246,9315,9384,9453,9522,9591,9660,9729,9798,9867,9936,10005,10074,10143,10212,10281,10350,10419,10488,10557,10626,10695,10764,10833,10902,10971,11040,11109,11178,11247,11316,11385,11454,11523,11592,11661,11730,11799,11868,11937,12006,12075,12144,12213,12282,12351,12420,12489,12558,12627,12696,12765,12834,12903,12972,13041,13110,13179,13248,13317,13386,13455,13524,13593,13662,13731,13800,13869,13938,14007,14076,14145,14214,14283,14352,14421,14490,14559,14628,14697,14766,14835,14904,14973,15042,15111,15180,15249,15318,15387,15456,15525,15594,15663,15732,15801,15870,15939,16008,16077,16146,16215,16284,16353,16422,16491,16560,16629,16698,16767,16836,16905,16974,17043,17112,17181,17250,17319,17388,17457,17526,17595,17664,17733,17802,17871,17940,18009,18078,18147,18216,18285,18354,18423,18492,18561,18630,18699,18768,18837,18906,18975,19044,19113,19182,19251,19320,19389,19458,19527,19596,19665,19734,19803,19872,19941,20010,20079,20148,20217,20286,20355,20424,20493,20562,20631,20700,20769,20838,20907,20976,21045,21114,21183,21252,21321,21390,21459,21528,21597,21666,21735,21804,21873,21942,22011,22080,22149,22218,22287,22356,22425,22494,22563,22632,22701,22770,22839,22908,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5040,5110,5180,5250,5320,5390,5460,5530,5600,5670,5740,5810,5880,5950,6020,6090,6160,6230,6300,6370,6440,6510,6580,6650,6720,6790,6860,6930,7000,7070,7140,7210,7280,7350,7420,7490,7560,7630,7700,7770,7840,7910,7980,8050,8120,8190,8260,8330,8400,8470,8540,8610,8680,8750,8820,8890,8960,9030,9100,9170,9240,9310,9380,9450,9520,9590,9660,9730,9800,9870,9940,10010,10080,10150,10220,10290,10360,10430,10500,10570,10640,10710,10780,10850,10920,10990,11060,11130,11200,11270,11340,11410,11480,11550,11620,11690,11760,11830,11900,11970,12040,12110,12180,12250,12320,12390,12460,12530,12600,12670,12740,12810,12880,12950,13020,13090,13160,13230,13300,13370,13440,13510,13580,13650,13720,13790,13860,13930,14000,14070,14140,14210,14280,14350,14420,14490,14560,14630,14700,14770,14840,14910,14980,15050,15120,15190,15260,15330,15400,15470,15540,15610,15680,15750,15820,15890,15960,16030,16100,16170,16240,16310,16380,16450,16520,16590,16660,16730,16800,16870,16940,17010,17080,17150,17220,17290,17360,17430,17500,17570,17640,17710,17780,17850,17920,17990,18060,18130,18200,18270,18340,18410,18480,18550,18620,18690,18760,18830,18900,18970,19040,19110,19180,19250,19320,19390,19460,19530,19600,19670,19740,19810,19880,19950,20020,20090,20160,20230,20300,20370,20440,20510,20580,20650,20720,20790,20860,20930,21000,21070,21140,21210,21280,21350,21420,21490,21560,21630,21700,21770,21840,21910,21980,22050,22120,22190,22260,22330,22400,22470,22540,22610,22680,22750,22820,22890,22960,23030,23100,23170,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5040,5110,5180,5250,5320,5390,5460,5530,5600,5670,5740,5810,5880,5950,6020,6090,6160,6230,6300,6370,6440,6510,6580,6650,6720,6790,6860,6930,7000,7070,7140,7210,7280,7350,7420,7490,7560,7630,7700,7770,7840,7910,7980,8050,8120,8190,8260,8330,8400,8470,8540,8610,8680,8750,8820,8890,8960,9030,9100,9170,9240,9310,9380,9450,9520,9590,9660,9730,9800,9870,9940,10010,10080,10150,10220,10290,10360,10430,10500,10570,10640,10710,10780,10850,10920,10990,11060,11130,11200,11270,11340,11410,11480,11550,11620,11690,11760,11830,11900,11970,12040,12110,12180,12250,12320,12390,12460,12530,12600,12670,12740,12810,12880,12950,13020,13090,13160,13230,13300,13370,13440,13510,13580,13650,13720,13790,13860,13930,14000,14070,14140,14210,14280,14350,14420,14490,14560,14630,14700,14770,14840,14910,14980,15050,15120,15190,15260,15330,15400,15470,15540,15610,15680,15750,15820,15890,15960,16030,16100,16170,16240,16310,16380,16450,16520,16590,16660,16730,16800,16870,16940,17010,17080,17150,17220,17290,17360,17430,17500,17570,17640,17710,17780,17850,17920,17990,18060,18130,18200,18270,18340,18410,18480,18550,18620,18690,18760,18830,18900,18970,19040,19110,19180,19250,19320,19390,19460,19530,19600,19670,19740,19810,19880,19950,20020,20090,20160,20230,20300,20370,20440,20510,20580,20650,20720,20790,20860,20930,21000,21070,21140,21210,21280,21350,21420,21490,21560,21630,21700,21770,21840,21910,21980,22050,22120,22190,22260,22330,22400,22470,22540,22610,22680,22750,22820,22890,22960,23030,23100,23170,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5183,5254,5325,5396,5467,5538,5609,5680,5751,5822,5893,5964,6035,6106,6177,6248,6319,6390,6461,6532,6603,6674,6745,6816,6887,6958,7029,7100,7171,7242,7313,7384,7455,7526,7597,7668,7739,7810,7881,7952,8023,8094,8165,8236,8307,8378,8449,8520,8591,8662,8733,8804,8875,8946,9017,9088,9159,9230,9301,9372,9443,9514,9585,9656,9727,9798,9869,9940,10011,10082,10153,10224,10295,10366,10437,10508,10579,10650,10721,10792,10863,10934,11005,11076,11147,11218,11289,11360,11431,11502,11573,11644,11715,11786,11857,11928,11999,12070,12141,12212,12283,12354,12425,12496,12567,12638,12709,12780,12851,12922,12993,13064,13135,13206,13277,13348,13419,13490,13561,13632,13703,13774,13845,13916,13987,14058,14129,14200,14271,14342,14413,14484,14555,14626,14697,14768,14839,14910,14981,15052,15123,15194,15265,15336,15407,15478,15549,15620,15691,15762,15833,15904,15975,16046,16117,16188,16259,16330,16401,16472,16543,16614,16685,16756,16827,16898,16969,17040,17111,17182,17253,17324,17395,17466,17537,17608,17679,17750,17821,17892,17963,18034,18105,18176,18247,18318,18389,18460,18531,18602,18673,18744,18815,18886,18957,19028,19099,19170,19241,19312,19383,19454,19525,19596,19667,19738,19809,19880,19951,20022,20093,20164,20235,20306,20377,20448,20519,20590,20661,20732,20803,20874,20945,21016,21087,21158,21229,21300,21371,21442,21513,21584,21655,21726,21797,21868,21939,22010,22081,22152,22223,22294,22365,22436,22507,22578,22649,22720,22791,22862,22933,23004,23075,23146,23217,23288,23359,23430,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5183,5254,5325,5396,5467,5538,5609,5680,5751,5822,5893,5964,6035,6106,6177,6248,6319,6390,6461,6532,6603,6674,6745,6816,6887,6958,7029,7100,7171,7242,7313,7384,7455,7526,7597,7668,7739,7810,7881,7952,8023,8094,8165,8236,8307,8378,8449,8520,8591,8662,8733,8804,8875,8946,9017,9088,9159,9230,9301,9372,9443,9514,9585,9656,9727,9798,9869,9940,10011,10082,10153,10224,10295,10366,10437,10508,10579,10650,10721,10792,10863,10934,11005,11076,11147,11218,11289,11360,11431,11502,11573,11644,11715,11786,11857,11928,11999,12070,12141,12212,12283,12354,12425,12496,12567,12638,12709,12780,12851,12922,12993,13064,13135,13206,13277,13348,13419,13490,13561,13632,13703,13774,13845,13916,13987,14058,14129,14200,14271,14342,14413,14484,14555,14626,14697,14768,14839,14910,14981,15052,15123,15194,15265,15336,15407,15478,15549,15620,15691,15762,15833,15904,15975,16046,16117,16188,16259,16330,16401,16472,16543,16614,16685,16756,16827,16898,16969,17040,17111,17182,17253,17324,17395,17466,17537,17608,17679,17750,17821,17892,17963,18034,18105,18176,18247,18318,18389,18460,18531,18602,18673,18744,18815,18886,18957,19028,19099,19170,19241,19312,19383,19454,19525,19596,19667,19738,19809,19880,19951,20022,20093,20164,20235,20306,20377,20448,20519,20590,20661,20732,20803,20874,20945,21016,21087,21158,21229,21300,21371,21442,21513,21584,21655,21726,21797,21868,21939,22010,22081,22152,22223,22294,22365,22436,22507,22578,22649,22720,22791,22862,22933,23004,23075,23146,23217,23288,23359,23430,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5328,5400,5472,5544,5616,5688,5760,5832,5904,5976,6048,6120,6192,6264,6336,6408,6480,6552,6624,6696,6768,6840,6912,6984,7056,7128,7200,7272,7344,7416,7488,7560,7632,7704,7776,7848,7920,7992,8064,8136,8208,8280,8352,8424,8496,8568,8640,8712,8784,8856,8928,9000,9072,9144,9216,9288,9360,9432,9504,9576,9648,9720,9792,9864,9936,10008,10080,10152,10224,10296,10368,10440,10512,10584,10656,10728,10800,10872,10944,11016,11088,11160,11232,11304,11376,11448,11520,11592,11664,11736,11808,11880,11952,12024,12096,12168,12240,12312,12384,12456,12528,12600,12672,12744,12816,12888,12960,13032,13104,13176,13248,13320,13392,13464,13536,13608,13680,13752,13824,13896,13968,14040,14112,14184,14256,14328,14400,14472,14544,14616,14688,14760,14832,14904,14976,15048,15120,15192,15264,15336,15408,15480,15552,15624,15696,15768,15840,15912,15984,16056,16128,16200,16272,16344,16416,16488,16560,16632,16704,16776,16848,16920,16992,17064,17136,17208,17280,17352,17424,17496,17568,17640,17712,17784,17856,17928,18000,18072,18144,18216,18288,18360,18432,18504,18576,18648,18720,18792,18864,18936,19008,19080,19152,19224,19296,19368,19440,19512,19584,19656,19728,19800,19872,19944,20016,20088,20160,20232,20304,20376,20448,20520,20592,20664,20736,20808,20880,20952,21024,21096,21168,21240,21312,21384,21456,21528,21600,21672,21744,21816,21888,21960,22032,22104,22176,22248,22320,22392,22464,22536,22608,22680,22752,22824,22896,22968,23040,23112,23184,23256,23328,23400,23472,23544,23616,23688,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5328,5400,5472,5544,5616,5688,5760,5832,5904,5976,6048,6120,6192,6264,6336,6408,6480,6552,6624,6696,6768,6840,6912,6984,7056,7128,7200,7272,7344,7416,7488,7560,7632,7704,7776,7848,7920,7992,8064,8136,8208,8280,8352,8424,8496,8568,8640,8712,8784,8856,8928,9000,9072,9144,9216,9288,9360,9432,9504,9576,9648,9720,9792,9864,9936,10008,10080,10152,10224,10296,10368,10440,10512,10584,10656,10728,10800,10872,10944,11016,11088,11160,11232,11304,11376,11448,11520,11592,11664,11736,11808,11880,11952,12024,12096,12168,12240,12312,12384,12456,12528,12600,12672,12744,12816,12888,12960,13032,13104,13176,13248,13320,13392,13464,13536,13608,13680,13752,13824,13896,13968,14040,14112,14184,14256,14328,14400,14472,14544,14616,14688,14760,14832,14904,14976,15048,15120,15192,15264,15336,15408,15480,15552,15624,15696,15768,15840,15912,15984,16056,16128,16200,16272,16344,16416,16488,16560,16632,16704,16776,16848,16920,16992,17064,17136,17208,17280,17352,17424,17496,17568,17640,17712,17784,17856,17928,18000,18072,18144,18216,18288,18360,18432,18504,18576,18648,18720,18792,18864,18936,19008,19080,19152,19224,19296,19368,19440,19512,19584,19656,19728,19800,19872,19944,20016,20088,20160,20232,20304,20376,20448,20520,20592,20664,20736,20808,20880,20952,21024,21096,21168,21240,21312,21384,21456,21528,21600,21672,21744,21816,21888,21960,22032,22104,22176,22248,22320,22392,22464,22536,22608,22680,22752,22824,22896,22968,23040,23112,23184,23256,23328,23400,23472,23544,23616,23688,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5475,5548,5621,5694,5767,5840,5913,5986,6059,6132,6205,6278,6351,6424,6497,6570,6643,6716,6789,6862,6935,7008,7081,7154,7227,7300,7373,7446,7519,7592,7665,7738,7811,7884,7957,8030,8103,8176,8249,8322,8395,8468,8541,8614,8687,8760,8833,8906,8979,9052,9125,9198,9271,9344,9417,9490,9563,9636,9709,9782,9855,9928,10001,10074,10147,10220,10293,10366,10439,10512,10585,10658,10731,10804,10877,10950,11023,11096,11169,11242,11315,11388,11461,11534,11607,11680,11753,11826,11899,11972,12045,12118,12191,12264,12337,12410,12483,12556,12629,12702,12775,12848,12921,12994,13067,13140,13213,13286,13359,13432,13505,13578,13651,13724,13797,13870,13943,14016,14089,14162,14235,14308,14381,14454,14527,14600,14673,14746,14819,14892,14965,15038,15111,15184,15257,15330,15403,15476,15549,15622,15695,15768,15841,15914,15987,16060,16133,16206,16279,16352,16425,16498,16571,16644,16717,16790,16863,16936,17009,17082,17155,17228,17301,17374,17447,17520,17593,17666,17739,17812,17885,17958,18031,18104,18177,18250,18323,18396,18469,18542,18615,18688,18761,18834,18907,18980,19053,19126,19199,19272,19345,19418,19491,19564,19637,19710,19783,19856,19929,20002,20075,20148,20221,20294,20367,20440,20513,20586,20659,20732,20805,20878,20951,21024,21097,21170,21243,21316,21389,21462,21535,21608,21681,21754,21827,21900,21973,22046,22119,22192,22265,22338,22411,22484,22557,22630,22703,22776,22849,22922,22995,23068,23141,23214,23287,23360,23433,23506,23579,23652,23725,23798,23871,23944,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5475,5548,5621,5694,5767,5840,5913,5986,6059,6132,6205,6278,6351,6424,6497,6570,6643,6716,6789,6862,6935,7008,7081,7154,7227,7300,7373,7446,7519,7592,7665,7738,7811,7884,7957,8030,8103,8176,8249,8322,8395,8468,8541,8614,8687,8760,8833,8906,8979,9052,9125,9198,9271,9344,9417,9490,9563,9636,9709,9782,9855,9928,10001,10074,10147,10220,10293,10366,10439,10512,10585,10658,10731,10804,10877,10950,11023,11096,11169,11242,11315,11388,11461,11534,11607,11680,11753,11826,11899,11972,12045,12118,12191,12264,12337,12410,12483,12556,12629,12702,12775,12848,12921,12994,13067,13140,13213,13286,13359,13432,13505,13578,13651,13724,13797,13870,13943,14016,14089,14162,14235,14308,14381,14454,14527,14600,14673,14746,14819,14892,14965,15038,15111,15184,15257,15330,15403,15476,15549,15622,15695,15768,15841,15914,15987,16060,16133,16206,16279,16352,16425,16498,16571,16644,16717,16790,16863,16936,17009,17082,17155,17228,17301,17374,17447,17520,17593,17666,17739,17812,17885,17958,18031,18104,18177,18250,18323,18396,18469,18542,18615,18688,18761,18834,18907,18980,19053,19126,19199,19272,19345,19418,19491,19564,19637,19710,19783,19856,19929,20002,20075,20148,20221,20294,20367,20440,20513,20586,20659,20732,20805,20878,20951,21024,21097,21170,21243,21316,21389,21462,21535,21608,21681,21754,21827,21900,21973,22046,22119,22192,22265,22338,22411,22484,22557,22630,22703,22776,22849,22922,22995,23068,23141,23214,23287,23360,23433,23506,23579,23652,23725,23798,23871,23944,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5624,5698,5772,5846,5920,5994,6068,6142,6216,6290,6364,6438,6512,6586,6660,6734,6808,6882,6956,7030,7104,7178,7252,7326,7400,7474,7548,7622,7696,7770,7844,7918,7992,8066,8140,8214,8288,8362,8436,8510,8584,8658,8732,8806,8880,8954,9028,9102,9176,9250,9324,9398,9472,9546,9620,9694,9768,9842,9916,9990,10064,10138,10212,10286,10360,10434,10508,10582,10656,10730,10804,10878,10952,11026,11100,11174,11248,11322,11396,11470,11544,11618,11692,11766,11840,11914,11988,12062,12136,12210,12284,12358,12432,12506,12580,12654,12728,12802,12876,12950,13024,13098,13172,13246,13320,13394,13468,13542,13616,13690,13764,13838,13912,13986,14060,14134,14208,14282,14356,14430,14504,14578,14652,14726,14800,14874,14948,15022,15096,15170,15244,15318,15392,15466,15540,15614,15688,15762,15836,15910,15984,16058,16132,16206,16280,16354,16428,16502,16576,16650,16724,16798,16872,16946,17020,17094,17168,17242,17316,17390,17464,17538,17612,17686,17760,17834,17908,17982,18056,18130,18204,18278,18352,18426,18500,18574,18648,18722,18796,18870,18944,19018,19092,19166,19240,19314,19388,19462,19536,19610,19684,19758,19832,19906,19980,20054,20128,20202,20276,20350,20424,20498,20572,20646,20720,20794,20868,20942,21016,21090,21164,21238,21312,21386,21460,21534,21608,21682,21756,21830,21904,21978,22052,22126,22200,22274,22348,22422,22496,22570,22644,22718,22792,22866,22940,23014,23088,23162,23236,23310,23384,23458,23532,23606,23680,23754,23828,23902,23976,24050,24124,24198,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5624,5698,5772,5846,5920,5994,6068,6142,6216,6290,6364,6438,6512,6586,6660,6734,6808,6882,6956,7030,7104,7178,7252,7326,7400,7474,7548,7622,7696,7770,7844,7918,7992,8066,8140,8214,8288,8362,8436,8510,8584,8658,8732,8806,8880,8954,9028,9102,9176,9250,9324,9398,9472,9546,9620,9694,9768,9842,9916,9990,10064,10138,10212,10286,10360,10434,10508,10582,10656,10730,10804,10878,10952,11026,11100,11174,11248,11322,11396,11470,11544,11618,11692,11766,11840,11914,11988,12062,12136,12210,12284,12358,12432,12506,12580,12654,12728,12802,12876,12950,13024,13098,13172,13246,13320,13394,13468,13542,13616,13690,13764,13838,13912,13986,14060,14134,14208,14282,14356,14430,14504,14578,14652,14726,14800,14874,14948,15022,15096,15170,15244,15318,15392,15466,15540,15614,15688,15762,15836,15910,15984,16058,16132,16206,16280,16354,16428,16502,16576,16650,16724,16798,16872,16946,17020,17094,17168,17242,17316,17390,17464,17538,17612,17686,17760,17834,17908,17982,18056,18130,18204,18278,18352,18426,18500,18574,18648,18722,18796,18870,18944,19018,19092,19166,19240,19314,19388,19462,19536,19610,19684,19758,19832,19906,19980,20054,20128,20202,20276,20350,20424,20498,20572,20646,20720,20794,20868,20942,21016,21090,21164,21238,21312,21386,21460,21534,21608,21682,21756,21830,21904,21978,22052,22126,22200,22274,22348,22422,22496,22570,22644,22718,22792,22866,22940,23014,23088,23162,23236,23310,23384,23458,23532,23606,23680,23754,23828,23902,23976,24050,24124,24198,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5775,5850,5925,6000,6075,6150,6225,6300,6375,6450,6525,6600,6675,6750,6825,6900,6975,7050,7125,7200,7275,7350,7425,7500,7575,7650,7725,7800,7875,7950,8025,8100,8175,8250,8325,8400,8475,8550,8625,8700,8775,8850,8925,9000,9075,9150,9225,9300,9375,9450,9525,9600,9675,9750,9825,9900,9975,10050,10125,10200,10275,10350,10425,10500,10575,10650,10725,10800,10875,10950,11025,11100,11175,11250,11325,11400,11475,11550,11625,11700,11775,11850,11925,12000,12075,12150,12225,12300,12375,12450,12525,12600,12675,12750,12825,12900,12975,13050,13125,13200,13275,13350,13425,13500,13575,13650,13725,13800,13875,13950,14025,14100,14175,14250,14325,14400,14475,14550,14625,14700,14775,14850,14925,15000,15075,15150,15225,15300,15375,15450,15525,15600,15675,15750,15825,15900,15975,16050,16125,16200,16275,16350,16425,16500,16575,16650,16725,16800,16875,16950,17025,17100,17175,17250,17325,17400,17475,17550,17625,17700,17775,17850,17925,18000,18075,18150,18225,18300,18375,18450,18525,18600,18675,18750,18825,18900,18975,19050,19125,19200,19275,19350,19425,19500,19575,19650,19725,19800,19875,19950,20025,20100,20175,20250,20325,20400,20475,20550,20625,20700,20775,20850,20925,21000,21075,21150,21225,21300,21375,21450,21525,21600,21675,21750,21825,21900,21975,22050,22125,22200,22275,22350,22425,22500,22575,22650,22725,22800,22875,22950,23025,23100,23175,23250,23325,23400,23475,23550,23625,23700,23775,23850,23925,24000,24075,24150,24225,24300,24375,24450,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5775,5850,5925,6000,6075,6150,6225,6300,6375,6450,6525,6600,6675,6750,6825,6900,6975,7050,7125,7200,7275,7350,7425,7500,7575,7650,7725,7800,7875,7950,8025,8100,8175,8250,8325,8400,8475,8550,8625,8700,8775,8850,8925,9000,9075,9150,9225,9300,9375,9450,9525,9600,9675,9750,9825,9900,9975,10050,10125,10200,10275,10350,10425,10500,10575,10650,10725,10800,10875,10950,11025,11100,11175,11250,11325,11400,11475,11550,11625,11700,11775,11850,11925,12000,12075,12150,12225,12300,12375,12450,12525,12600,12675,12750,12825,12900,12975,13050,13125,13200,13275,13350,13425,13500,13575,13650,13725,13800,13875,13950,14025,14100,14175,14250,14325,14400,14475,14550,14625,14700,14775,14850,14925,15000,15075,15150,15225,15300,15375,15450,15525,15600,15675,15750,15825,15900,15975,16050,16125,16200,16275,16350,16425,16500,16575,16650,16725,16800,16875,16950,17025,17100,17175,17250,17325,17400,17475,17550,17625,17700,17775,17850,17925,18000,18075,18150,18225,18300,18375,18450,18525,18600,18675,18750,18825,18900,18975,19050,19125,19200,19275,19350,19425,19500,19575,19650,19725,19800,19875,19950,20025,20100,20175,20250,20325,20400,20475,20550,20625,20700,20775,20850,20925,21000,21075,21150,21225,21300,21375,21450,21525,21600,21675,21750,21825,21900,21975,22050,22125,22200,22275,22350,22425,22500,22575,22650,22725,22800,22875,22950,23025,23100,23175,23250,23325,23400,23475,23550,23625,23700,23775,23850,23925,24000,24075,24150,24225,24300,24375,24450,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5928,6004,6080,6156,6232,6308,6384,6460,6536,6612,6688,6764,6840,6916,6992,7068,7144,7220,7296,7372,7448,7524,7600,7676,7752,7828,7904,7980,8056,8132,8208,8284,8360,8436,8512,8588,8664,8740,8816,8892,8968,9044,9120,9196,9272,9348,9424,9500,9576,9652,9728,9804,9880,9956,10032,10108,10184,10260,10336,10412,10488,10564,10640,10716,10792,10868,10944,11020,11096,11172,11248,11324,11400,11476,11552,11628,11704,11780,11856,11932,12008,12084,12160,12236,12312,12388,12464,12540,12616,12692,12768,12844,12920,12996,13072,13148,13224,13300,13376,13452,13528,13604,13680,13756,13832,13908,13984,14060,14136,14212,14288,14364,14440,14516,14592,14668,14744,14820,14896,14972,15048,15124,15200,15276,15352,15428,15504,15580,15656,15732,15808,15884,15960,16036,16112,16188,16264,16340,16416,16492,16568,16644,16720,16796,16872,16948,17024,17100,17176,17252,17328,17404,17480,17556,17632,17708,17784,17860,17936,18012,18088,18164,18240,18316,18392,18468,18544,18620,18696,18772,18848,18924,19000,19076,19152,19228,19304,19380,19456,19532,19608,19684,19760,19836,19912,19988,20064,20140,20216,20292,20368,20444,20520,20596,20672,20748,20824,20900,20976,21052,21128,21204,21280,21356,21432,21508,21584,21660,21736,21812,21888,21964,22040,22116,22192,22268,22344,22420,22496,22572,22648,22724,22800,22876,22952,23028,23104,23180,23256,23332,23408,23484,23560,23636,23712,23788,23864,23940,24016,24092,24168,24244,24320,24396,24472,24548,24624,24700,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5928,6004,6080,6156,6232,6308,6384,6460,6536,6612,6688,6764,6840,6916,6992,7068,7144,7220,7296,7372,7448,7524,7600,7676,7752,7828,7904,7980,8056,8132,8208,8284,8360,8436,8512,8588,8664,8740,8816,8892,8968,9044,9120,9196,9272,9348,9424,9500,9576,9652,9728,9804,9880,9956,10032,10108,10184,10260,10336,10412,10488,10564,10640,10716,10792,10868,10944,11020,11096,11172,11248,11324,11400,11476,11552,11628,11704,11780,11856,11932,12008,12084,12160,12236,12312,12388,12464,12540,12616,12692,12768,12844,12920,12996,13072,13148,13224,13300,13376,13452,13528,13604,13680,13756,13832,13908,13984,14060,14136,14212,14288,14364,14440,14516,14592,14668,14744,14820,14896,14972,15048,15124,15200,15276,15352,15428,15504,15580,15656,15732,15808,15884,15960,16036,16112,16188,16264,16340,16416,16492,16568,16644,16720,16796,16872,16948,17024,17100,17176,17252,17328,17404,17480,17556,17632,17708,17784,17860,17936,18012,18088,18164,18240,18316,18392,18468,18544,18620,18696,18772,18848,18924,19000,19076,19152,19228,19304,19380,19456,19532,19608,19684,19760,19836,19912,19988,20064,20140,20216,20292,20368,20444,20520,20596,20672,20748,20824,20900,20976,21052,21128,21204,21280,21356,21432,21508,21584,21660,21736,21812,21888,21964,22040,22116,22192,22268,22344,22420,22496,22572,22648,22724,22800,22876,22952,23028,23104,23180,23256,23332,23408,23484,23560,23636,23712,23788,23864,23940,24016,24092,24168,24244,24320,24396,24472,24548,24624,24700,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6083,6160,6237,6314,6391,6468,6545,6622,6699,6776,6853,6930,7007,7084,7161,7238,7315,7392,7469,7546,7623,7700,7777,7854,7931,8008,8085,8162,8239,8316,8393,8470,8547,8624,8701,8778,8855,8932,9009,9086,9163,9240,9317,9394,9471,9548,9625,9702,9779,9856,9933,10010,10087,10164,10241,10318,10395,10472,10549,10626,10703,10780,10857,10934,11011,11088,11165,11242,11319,11396,11473,11550,11627,11704,11781,11858,11935,12012,12089,12166,12243,12320,12397,12474,12551,12628,12705,12782,12859,12936,13013,13090,13167,13244,13321,13398,13475,13552,13629,13706,13783,13860,13937,14014,14091,14168,14245,14322,14399,14476,14553,14630,14707,14784,14861,14938,15015,15092,15169,15246,15323,15400,15477,15554,15631,15708,15785,15862,15939,16016,16093,16170,16247,16324,16401,16478,16555,16632,16709,16786,16863,16940,17017,17094,17171,17248,17325,17402,17479,17556,17633,17710,17787,17864,17941,18018,18095,18172,18249,18326,18403,18480,18557,18634,18711,18788,18865,18942,19019,19096,19173,19250,19327,19404,19481,19558,19635,19712,19789,19866,19943,20020,20097,20174,20251,20328,20405,20482,20559,20636,20713,20790,20867,20944,21021,21098,21175,21252,21329,21406,21483,21560,21637,21714,21791,21868,21945,22022,22099,22176,22253,22330,22407,22484,22561,22638,22715,22792,22869,22946,23023,23100,23177,23254,23331,23408,23485,23562,23639,23716,23793,23870,23947,24024,24101,24178,24255,24332,24409,24486,24563,24640,24717,24794,24871,24948,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6083,6160,6237,6314,6391,6468,6545,6622,6699,6776,6853,6930,7007,7084,7161,7238,7315,7392,7469,7546,7623,7700,7777,7854,7931,8008,8085,8162,8239,8316,8393,8470,8547,8624,8701,8778,8855,8932,9009,9086,9163,9240,9317,9394,9471,9548,9625,9702,9779,9856,9933,10010,10087,10164,10241,10318,10395,10472,10549,10626,10703,10780,10857,10934,11011,11088,11165,11242,11319,11396,11473,11550,11627,11704,11781,11858,11935,12012,12089,12166,12243,12320,12397,12474,12551,12628,12705,12782,12859,12936,13013,13090,13167,13244,13321,13398,13475,13552,13629,13706,13783,13860,13937,14014,14091,14168,14245,14322,14399,14476,14553,14630,14707,14784,14861,14938,15015,15092,15169,15246,15323,15400,15477,15554,15631,15708,15785,15862,15939,16016,16093,16170,16247,16324,16401,16478,16555,16632,16709,16786,16863,16940,17017,17094,17171,17248,17325,17402,17479,17556,17633,17710,17787,17864,17941,18018,18095,18172,18249,18326,18403,18480,18557,18634,18711,18788,18865,18942,19019,19096,19173,19250,19327,19404,19481,19558,19635,19712,19789,19866,19943,20020,20097,20174,20251,20328,20405,20482,20559,20636,20713,20790,20867,20944,21021,21098,21175,21252,21329,21406,21483,21560,21637,21714,21791,21868,21945,22022,22099,22176,22253,22330,22407,22484,22561,22638,22715,22792,22869,22946,23023,23100,23177,23254,23331,23408,23485,23562,23639,23716,23793,23870,23947,24024,24101,24178,24255,24332,24409,24486,24563,24640,24717,24794,24871,24948,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6240,6318,6396,6474,6552,6630,6708,6786,6864,6942,7020,7098,7176,7254,7332,7410,7488,7566,7644,7722,7800,7878,7956,8034,8112,8190,8268,8346,8424,8502,8580,8658,8736,8814,8892,8970,9048,9126,9204,9282,9360,9438,9516,9594,9672,9750,9828,9906,9984,10062,10140,10218,10296,10374,10452,10530,10608,10686,10764,10842,10920,10998,11076,11154,11232,11310,11388,11466,11544,11622,11700,11778,11856,11934,12012,12090,12168,12246,12324,12402,12480,12558,12636,12714,12792,12870,12948,13026,13104,13182,13260,13338,13416,13494,13572,13650,13728,13806,13884,13962,14040,14118,14196,14274,14352,14430,14508,14586,14664,14742,14820,14898,14976,15054,15132,15210,15288,15366,15444,15522,15600,15678,15756,15834,15912,15990,16068,16146,16224,16302,16380,16458,16536,16614,16692,16770,16848,16926,17004,17082,17160,17238,17316,17394,17472,17550,17628,17706,17784,17862,17940,18018,18096,18174,18252,18330,18408,18486,18564,18642,18720,18798,18876,18954,19032,19110,19188,19266,19344,19422,19500,19578,19656,19734,19812,19890,19968,20046,20124,20202,20280,20358,20436,20514,20592,20670,20748,20826,20904,20982,21060,21138,21216,21294,21372,21450,21528,21606,21684,21762,21840,21918,21996,22074,22152,22230,22308,22386,22464,22542,22620,22698,22776,22854,22932,23010,23088,23166,23244,23322,23400,23478,23556,23634,23712,23790,23868,23946,24024,24102,24180,24258,24336,24414,24492,24570,24648,24726,24804,24882,24960,25038,25116,25194,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6240,6318,6396,6474,6552,6630,6708,6786,6864,6942,7020,7098,7176,7254,7332,7410,7488,7566,7644,7722,7800,7878,7956,8034,8112,8190,8268,8346,8424,8502,8580,8658,8736,8814,8892,8970,9048,9126,9204,9282,9360,9438,9516,9594,9672,9750,9828,9906,9984,10062,10140,10218,10296,10374,10452,10530,10608,10686,10764,10842,10920,10998,11076,11154,11232,11310,11388,11466,11544,11622,11700,11778,11856,11934,12012,12090,12168,12246,12324,12402,12480,12558,12636,12714,12792,12870,12948,13026,13104,13182,13260,13338,13416,13494,13572,13650,13728,13806,13884,13962,14040,14118,14196,14274,14352,14430,14508,14586,14664,14742,14820,14898,14976,15054,15132,15210,15288,15366,15444,15522,15600,15678,15756,15834,15912,15990,16068,16146,16224,16302,16380,16458,16536,16614,16692,16770,16848,16926,17004,17082,17160,17238,17316,17394,17472,17550,17628,17706,17784,17862,17940,18018,18096,18174,18252,18330,18408,18486,18564,18642,18720,18798,18876,18954,19032,19110,19188,19266,19344,19422,19500,19578,19656,19734,19812,19890,19968,20046,20124,20202,20280,20358,20436,20514,20592,20670,20748,20826,20904,20982,21060,21138,21216,21294,21372,21450,21528,21606,21684,21762,21840,21918,21996,22074,22152,22230,22308,22386,22464,22542,22620,22698,22776,22854,22932,23010,23088,23166,23244,23322,23400,23478,23556,23634,23712,23790,23868,23946,24024,24102,24180,24258,24336,24414,24492,24570,24648,24726,24804,24882,24960,25038,25116,25194,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6399,6478,6557,6636,6715,6794,6873,6952,7031,7110,7189,7268,7347,7426,7505,7584,7663,7742,7821,7900,7979,8058,8137,8216,8295,8374,8453,8532,8611,8690,8769,8848,8927,9006,9085,9164,9243,9322,9401,9480,9559,9638,9717,9796,9875,9954,10033,10112,10191,10270,10349,10428,10507,10586,10665,10744,10823,10902,10981,11060,11139,11218,11297,11376,11455,11534,11613,11692,11771,11850,11929,12008,12087,12166,12245,12324,12403,12482,12561,12640,12719,12798,12877,12956,13035,13114,13193,13272,13351,13430,13509,13588,13667,13746,13825,13904,13983,14062,14141,14220,14299,14378,14457,14536,14615,14694,14773,14852,14931,15010,15089,15168,15247,15326,15405,15484,15563,15642,15721,15800,15879,15958,16037,16116,16195,16274,16353,16432,16511,16590,16669,16748,16827,16906,16985,17064,17143,17222,17301,17380,17459,17538,17617,17696,17775,17854,17933,18012,18091,18170,18249,18328,18407,18486,18565,18644,18723,18802,18881,18960,19039,19118,19197,19276,19355,19434,19513,19592,19671,19750,19829,19908,19987,20066,20145,20224,20303,20382,20461,20540,20619,20698,20777,20856,20935,21014,21093,21172,21251,21330,21409,21488,21567,21646,21725,21804,21883,21962,22041,22120,22199,22278,22357,22436,22515,22594,22673,22752,22831,22910,22989,23068,23147,23226,23305,23384,23463,23542,23621,23700,23779,23858,23937,24016,24095,24174,24253,24332,24411,24490,24569,24648,24727,24806,24885,24964,25043,25122,25201,25280,25359,25438,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6399,6478,6557,6636,6715,6794,6873,6952,7031,7110,7189,7268,7347,7426,7505,7584,7663,7742,7821,7900,7979,8058,8137,8216,8295,8374,8453,8532,8611,8690,8769,8848,8927,9006,9085,9164,9243,9322,9401,9480,9559,9638,9717,9796,9875,9954,10033,10112,10191,10270,10349,10428,10507,10586,10665,10744,10823,10902,10981,11060,11139,11218,11297,11376,11455,11534,11613,11692,11771,11850,11929,12008,12087,12166,12245,12324,12403,12482,12561,12640,12719,12798,12877,12956,13035,13114,13193,13272,13351,13430,13509,13588,13667,13746,13825,13904,13983,14062,14141,14220,14299,14378,14457,14536,14615,14694,14773,14852,14931,15010,15089,15168,15247,15326,15405,15484,15563,15642,15721,15800,15879,15958,16037,16116,16195,16274,16353,16432,16511,16590,16669,16748,16827,16906,16985,17064,17143,17222,17301,17380,17459,17538,17617,17696,17775,17854,17933,18012,18091,18170,18249,18328,18407,18486,18565,18644,18723,18802,18881,18960,19039,19118,19197,19276,19355,19434,19513,19592,19671,19750,19829,19908,19987,20066,20145,20224,20303,20382,20461,20540,20619,20698,20777,20856,20935,21014,21093,21172,21251,21330,21409,21488,21567,21646,21725,21804,21883,21962,22041,22120,22199,22278,22357,22436,22515,22594,22673,22752,22831,22910,22989,23068,23147,23226,23305,23384,23463,23542,23621,23700,23779,23858,23937,24016,24095,24174,24253,24332,24411,24490,24569,24648,24727,24806,24885,24964,25043,25122,25201,25280,25359,25438,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6560,6640,6720,6800,6880,6960,7040,7120,7200,7280,7360,7440,7520,7600,7680,7760,7840,7920,8000,8080,8160,8240,8320,8400,8480,8560,8640,8720,8800,8880,8960,9040,9120,9200,9280,9360,9440,9520,9600,9680,9760,9840,9920,10000,10080,10160,10240,10320,10400,10480,10560,10640,10720,10800,10880,10960,11040,11120,11200,11280,11360,11440,11520,11600,11680,11760,11840,11920,12000,12080,12160,12240,12320,12400,12480,12560,12640,12720,12800,12880,12960,13040,13120,13200,13280,13360,13440,13520,13600,13680,13760,13840,13920,14000,14080,14160,14240,14320,14400,14480,14560,14640,14720,14800,14880,14960,15040,15120,15200,15280,15360,15440,15520,15600,15680,15760,15840,15920,16000,16080,16160,16240,16320,16400,16480,16560,16640,16720,16800,16880,16960,17040,17120,17200,17280,17360,17440,17520,17600,17680,17760,17840,17920,18000,18080,18160,18240,18320,18400,18480,18560,18640,18720,18800,18880,18960,19040,19120,19200,19280,19360,19440,19520,19600,19680,19760,19840,19920,20000,20080,20160,20240,20320,20400,20480,20560,20640,20720,20800,20880,20960,21040,21120,21200,21280,21360,21440,21520,21600,21680,21760,21840,21920,22000,22080,22160,22240,22320,22400,22480,22560,22640,22720,22800,22880,22960,23040,23120,23200,23280,23360,23440,23520,23600,23680,23760,23840,23920,24000,24080,24160,24240,24320,24400,24480,24560,24640,24720,24800,24880,24960,25040,25120,25200,25280,25360,25440,25520,25600,25680,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6560,6640,6720,6800,6880,6960,7040,7120,7200,7280,7360,7440,7520,7600,7680,7760,7840,7920,8000,8080,8160,8240,8320,8400,8480,8560,8640,8720,8800,8880,8960,9040,9120,9200,9280,9360,9440,9520,9600,9680,9760,9840,9920,10000,10080,10160,10240,10320,10400,10480,10560,10640,10720,10800,10880,10960,11040,11120,11200,11280,11360,11440,11520,11600,11680,11760,11840,11920,12000,12080,12160,12240,12320,12400,12480,12560,12640,12720,12800,12880,12960,13040,13120,13200,13280,13360,13440,13520,13600,13680,13760,13840,13920,14000,14080,14160,14240,14320,14400,14480,14560,14640,14720,14800,14880,14960,15040,15120,15200,15280,15360,15440,15520,15600,15680,15760,15840,15920,16000,16080,16160,16240,16320,16400,16480,16560,16640,16720,16800,16880,16960,17040,17120,17200,17280,17360,17440,17520,17600,17680,17760,17840,17920,18000,18080,18160,18240,18320,18400,18480,18560,18640,18720,18800,18880,18960,19040,19120,19200,19280,19360,19440,19520,19600,19680,19760,19840,19920,20000,20080,20160,20240,20320,20400,20480,20560,20640,20720,20800,20880,20960,21040,21120,21200,21280,21360,21440,21520,21600,21680,21760,21840,21920,22000,22080,22160,22240,22320,22400,22480,22560,22640,22720,22800,22880,22960,23040,23120,23200,23280,23360,23440,23520,23600,23680,23760,23840,23920,24000,24080,24160,24240,24320,24400,24480,24560,24640,24720,24800,24880,24960,25040,25120,25200,25280,25360,25440,25520,25600,25680,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6723,6804,6885,6966,7047,7128,7209,7290,7371,7452,7533,7614,7695,7776,7857,7938,8019,8100,8181,8262,8343,8424,8505,8586,8667,8748,8829,8910,8991,9072,9153,9234,9315,9396,9477,9558,9639,9720,9801,9882,9963,10044,10125,10206,10287,10368,10449,10530,10611,10692,10773,10854,10935,11016,11097,11178,11259,11340,11421,11502,11583,11664,11745,11826,11907,11988,12069,12150,12231,12312,12393,12474,12555,12636,12717,12798,12879,12960,13041,13122,13203,13284,13365,13446,13527,13608,13689,13770,13851,13932,14013,14094,14175,14256,14337,14418,14499,14580,14661,14742,14823,14904,14985,15066,15147,15228,15309,15390,15471,15552,15633,15714,15795,15876,15957,16038,16119,16200,16281,16362,16443,16524,16605,16686,16767,16848,16929,17010,17091,17172,17253,17334,17415,17496,17577,17658,17739,17820,17901,17982,18063,18144,18225,18306,18387,18468,18549,18630,18711,18792,18873,18954,19035,19116,19197,19278,19359,19440,19521,19602,19683,19764,19845,19926,20007,20088,20169,20250,20331,20412,20493,20574,20655,20736,20817,20898,20979,21060,21141,21222,21303,21384,21465,21546,21627,21708,21789,21870,21951,22032,22113,22194,22275,22356,22437,22518,22599,22680,22761,22842,22923,23004,23085,23166,23247,23328,23409,23490,23571,23652,23733,23814,23895,23976,24057,24138,24219,24300,24381,24462,24543,24624,24705,24786,24867,24948,25029,25110,25191,25272,25353,25434,25515,25596,25677,25758,25839,25920,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6723,6804,6885,6966,7047,7128,7209,7290,7371,7452,7533,7614,7695,7776,7857,7938,8019,8100,8181,8262,8343,8424,8505,8586,8667,8748,8829,8910,8991,9072,9153,9234,9315,9396,9477,9558,9639,9720,9801,9882,9963,10044,10125,10206,10287,10368,10449,10530,10611,10692,10773,10854,10935,11016,11097,11178,11259,11340,11421,11502,11583,11664,11745,11826,11907,11988,12069,12150,12231,12312,12393,12474,12555,12636,12717,12798,12879,12960,13041,13122,13203,13284,13365,13446,13527,13608,13689,13770,13851,13932,14013,14094,14175,14256,14337,14418,14499,14580,14661,14742,14823,14904,14985,15066,15147,15228,15309,15390,15471,15552,15633,15714,15795,15876,15957,16038,16119,16200,16281,16362,16443,16524,16605,16686,16767,16848,16929,17010,17091,17172,17253,17334,17415,17496,17577,17658,17739,17820,17901,17982,18063,18144,18225,18306,18387,18468,18549,18630,18711,18792,18873,18954,19035,19116,19197,19278,19359,19440,19521,19602,19683,19764,19845,19926,20007,20088,20169,20250,20331,20412,20493,20574,20655,20736,20817,20898,20979,21060,21141,21222,21303,21384,21465,21546,21627,21708,21789,21870,21951,22032,22113,22194,22275,22356,22437,22518,22599,22680,22761,22842,22923,23004,23085,23166,23247,23328,23409,23490,23571,23652,23733,23814,23895,23976,24057,24138,24219,24300,24381,24462,24543,24624,24705,24786,24867,24948,25029,25110,25191,25272,25353,25434,25515,25596,25677,25758,25839,25920,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6888,6970,7052,7134,7216,7298,7380,7462,7544,7626,7708,7790,7872,7954,8036,8118,8200,8282,8364,8446,8528,8610,8692,8774,8856,8938,9020,9102,9184,9266,9348,9430,9512,9594,9676,9758,9840,9922,10004,10086,10168,10250,10332,10414,10496,10578,10660,10742,10824,10906,10988,11070,11152,11234,11316,11398,11480,11562,11644,11726,11808,11890,11972,12054,12136,12218,12300,12382,12464,12546,12628,12710,12792,12874,12956,13038,13120,13202,13284,13366,13448,13530,13612,13694,13776,13858,13940,14022,14104,14186,14268,14350,14432,14514,14596,14678,14760,14842,14924,15006,15088,15170,15252,15334,15416,15498,15580,15662,15744,15826,15908,15990,16072,16154,16236,16318,16400,16482,16564,16646,16728,16810,16892,16974,17056,17138,17220,17302,17384,17466,17548,17630,17712,17794,17876,17958,18040,18122,18204,18286,18368,18450,18532,18614,18696,18778,18860,18942,19024,19106,19188,19270,19352,19434,19516,19598,19680,19762,19844,19926,20008,20090,20172,20254,20336,20418,20500,20582,20664,20746,20828,20910,20992,21074,21156,21238,21320,21402,21484,21566,21648,21730,21812,21894,21976,22058,22140,22222,22304,22386,22468,22550,22632,22714,22796,22878,22960,23042,23124,23206,23288,23370,23452,23534,23616,23698,23780,23862,23944,24026,24108,24190,24272,24354,24436,24518,24600,24682,24764,24846,24928,25010,25092,25174,25256,25338,25420,25502,25584,25666,25748,25830,25912,25994,26076,26158,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6888,6970,7052,7134,7216,7298,7380,7462,7544,7626,7708,7790,7872,7954,8036,8118,8200,8282,8364,8446,8528,8610,8692,8774,8856,8938,9020,9102,9184,9266,9348,9430,9512,9594,9676,9758,9840,9922,10004,10086,10168,10250,10332,10414,10496,10578,10660,10742,10824,10906,10988,11070,11152,11234,11316,11398,11480,11562,11644,11726,11808,11890,11972,12054,12136,12218,12300,12382,12464,12546,12628,12710,12792,12874,12956,13038,13120,13202,13284,13366,13448,13530,13612,13694,13776,13858,13940,14022,14104,14186,14268,14350,14432,14514,14596,14678,14760,14842,14924,15006,15088,15170,15252,15334,15416,15498,15580,15662,15744,15826,15908,15990,16072,16154,16236,16318,16400,16482,16564,16646,16728,16810,16892,16974,17056,17138,17220,17302,17384,17466,17548,17630,17712,17794,17876,17958,18040,18122,18204,18286,18368,18450,18532,18614,18696,18778,18860,18942,19024,19106,19188,19270,19352,19434,19516,19598,19680,19762,19844,19926,20008,20090,20172,20254,20336,20418,20500,20582,20664,20746,20828,20910,20992,21074,21156,21238,21320,21402,21484,21566,21648,21730,21812,21894,21976,22058,22140,22222,22304,22386,22468,22550,22632,22714,22796,22878,22960,23042,23124,23206,23288,23370,23452,23534,23616,23698,23780,23862,23944,24026,24108,24190,24272,24354,24436,24518,24600,24682,24764,24846,24928,25010,25092,25174,25256,25338,25420,25502,25584,25666,25748,25830,25912,25994,26076,26158,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7055,7138,7221,7304,7387,7470,7553,7636,7719,7802,7885,7968,8051,8134,8217,8300,8383,8466,8549,8632,8715,8798,8881,8964,9047,9130,9213,9296,9379,9462,9545,9628,9711,9794,9877,9960,10043,10126,10209,10292,10375,10458,10541,10624,10707,10790,10873,10956,11039,11122,11205,11288,11371,11454,11537,11620,11703,11786,11869,11952,12035,12118,12201,12284,12367,12450,12533,12616,12699,12782,12865,12948,13031,13114,13197,13280,13363,13446,13529,13612,13695,13778,13861,13944,14027,14110,14193,14276,14359,14442,14525,14608,14691,14774,14857,14940,15023,15106,15189,15272,15355,15438,15521,15604,15687,15770,15853,15936,16019,16102,16185,16268,16351,16434,16517,16600,16683,16766,16849,16932,17015,17098,17181,17264,17347,17430,17513,17596,17679,17762,17845,17928,18011,18094,18177,18260,18343,18426,18509,18592,18675,18758,18841,18924,19007,19090,19173,19256,19339,19422,19505,19588,19671,19754,19837,19920,20003,20086,20169,20252,20335,20418,20501,20584,20667,20750,20833,20916,20999,21082,21165,21248,21331,21414,21497,21580,21663,21746,21829,21912,21995,22078,22161,22244,22327,22410,22493,22576,22659,22742,22825,22908,22991,23074,23157,23240,23323,23406,23489,23572,23655,23738,23821,23904,23987,24070,24153,24236,24319,24402,24485,24568,24651,24734,24817,24900,24983,25066,25149,25232,25315,25398,25481,25564,25647,25730,25813,25896,25979,26062,26145,26228,26311,26394,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7055,7138,7221,7304,7387,7470,7553,7636,7719,7802,7885,7968,8051,8134,8217,8300,8383,8466,8549,8632,8715,8798,8881,8964,9047,9130,9213,9296,9379,9462,9545,9628,9711,9794,9877,9960,10043,10126,10209,10292,10375,10458,10541,10624,10707,10790,10873,10956,11039,11122,11205,11288,11371,11454,11537,11620,11703,11786,11869,11952,12035,12118,12201,12284,12367,12450,12533,12616,12699,12782,12865,12948,13031,13114,13197,13280,13363,13446,13529,13612,13695,13778,13861,13944,14027,14110,14193,14276,14359,14442,14525,14608,14691,14774,14857,14940,15023,15106,15189,15272,15355,15438,15521,15604,15687,15770,15853,15936,16019,16102,16185,16268,16351,16434,16517,16600,16683,16766,16849,16932,17015,17098,17181,17264,17347,17430,17513,17596,17679,17762,17845,17928,18011,18094,18177,18260,18343,18426,18509,18592,18675,18758,18841,18924,19007,19090,19173,19256,19339,19422,19505,19588,19671,19754,19837,19920,20003,20086,20169,20252,20335,20418,20501,20584,20667,20750,20833,20916,20999,21082,21165,21248,21331,21414,21497,21580,21663,21746,21829,21912,21995,22078,22161,22244,22327,22410,22493,22576,22659,22742,22825,22908,22991,23074,23157,23240,23323,23406,23489,23572,23655,23738,23821,23904,23987,24070,24153,24236,24319,24402,24485,24568,24651,24734,24817,24900,24983,25066,25149,25232,25315,25398,25481,25564,25647,25730,25813,25896,25979,26062,26145,26228,26311,26394,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7224,7308,7392,7476,7560,7644,7728,7812,7896,7980,8064,8148,8232,8316,8400,8484,8568,8652,8736,8820,8904,8988,9072,9156,9240,9324,9408,9492,9576,9660,9744,9828,9912,9996,10080,10164,10248,10332,10416,10500,10584,10668,10752,10836,10920,11004,11088,11172,11256,11340,11424,11508,11592,11676,11760,11844,11928,12012,12096,12180,12264,12348,12432,12516,12600,12684,12768,12852,12936,13020,13104,13188,13272,13356,13440,13524,13608,13692,13776,13860,13944,14028,14112,14196,14280,14364,14448,14532,14616,14700,14784,14868,14952,15036,15120,15204,15288,15372,15456,15540,15624,15708,15792,15876,15960,16044,16128,16212,16296,16380,16464,16548,16632,16716,16800,16884,16968,17052,17136,17220,17304,17388,17472,17556,17640,17724,17808,17892,17976,18060,18144,18228,18312,18396,18480,18564,18648,18732,18816,18900,18984,19068,19152,19236,19320,19404,19488,19572,19656,19740,19824,19908,19992,20076,20160,20244,20328,20412,20496,20580,20664,20748,20832,20916,21000,21084,21168,21252,21336,21420,21504,21588,21672,21756,21840,21924,22008,22092,22176,22260,22344,22428,22512,22596,22680,22764,22848,22932,23016,23100,23184,23268,23352,23436,23520,23604,23688,23772,23856,23940,24024,24108,24192,24276,24360,24444,24528,24612,24696,24780,24864,24948,25032,25116,25200,25284,25368,25452,25536,25620,25704,25788,25872,25956,26040,26124,26208,26292,26376,26460,26544,26628,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7224,7308,7392,7476,7560,7644,7728,7812,7896,7980,8064,8148,8232,8316,8400,8484,8568,8652,8736,8820,8904,8988,9072,9156,9240,9324,9408,9492,9576,9660,9744,9828,9912,9996,10080,10164,10248,10332,10416,10500,10584,10668,10752,10836,10920,11004,11088,11172,11256,11340,11424,11508,11592,11676,11760,11844,11928,12012,12096,12180,12264,12348,12432,12516,12600,12684,12768,12852,12936,13020,13104,13188,13272,13356,13440,13524,13608,13692,13776,13860,13944,14028,14112,14196,14280,14364,14448,14532,14616,14700,14784,14868,14952,15036,15120,15204,15288,15372,15456,15540,15624,15708,15792,15876,15960,16044,16128,16212,16296,16380,16464,16548,16632,16716,16800,16884,16968,17052,17136,17220,17304,17388,17472,17556,17640,17724,17808,17892,17976,18060,18144,18228,18312,18396,18480,18564,18648,18732,18816,18900,18984,19068,19152,19236,19320,19404,19488,19572,19656,19740,19824,19908,19992,20076,20160,20244,20328,20412,20496,20580,20664,20748,20832,20916,21000,21084,21168,21252,21336,21420,21504,21588,21672,21756,21840,21924,22008,22092,22176,22260,22344,22428,22512,22596,22680,22764,22848,22932,23016,23100,23184,23268,23352,23436,23520,23604,23688,23772,23856,23940,24024,24108,24192,24276,24360,24444,24528,24612,24696,24780,24864,24948,25032,25116,25200,25284,25368,25452,25536,25620,25704,25788,25872,25956,26040,26124,26208,26292,26376,26460,26544,26628,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7395,7480,7565,7650,7735,7820,7905,7990,8075,8160,8245,8330,8415,8500,8585,8670,8755,8840,8925,9010,9095,9180,9265,9350,9435,9520,9605,9690,9775,9860,9945,10030,10115,10200,10285,10370,10455,10540,10625,10710,10795,10880,10965,11050,11135,11220,11305,11390,11475,11560,11645,11730,11815,11900,11985,12070,12155,12240,12325,12410,12495,12580,12665,12750,12835,12920,13005,13090,13175,13260,13345,13430,13515,13600,13685,13770,13855,13940,14025,14110,14195,14280,14365,14450,14535,14620,14705,14790,14875,14960,15045,15130,15215,15300,15385,15470,15555,15640,15725,15810,15895,15980,16065,16150,16235,16320,16405,16490,16575,16660,16745,16830,16915,17000,17085,17170,17255,17340,17425,17510,17595,17680,17765,17850,17935,18020,18105,18190,18275,18360,18445,18530,18615,18700,18785,18870,18955,19040,19125,19210,19295,19380,19465,19550,19635,19720,19805,19890,19975,20060,20145,20230,20315,20400,20485,20570,20655,20740,20825,20910,20995,21080,21165,21250,21335,21420,21505,21590,21675,21760,21845,21930,22015,22100,22185,22270,22355,22440,22525,22610,22695,22780,22865,22950,23035,23120,23205,23290,23375,23460,23545,23630,23715,23800,23885,23970,24055,24140,24225,24310,24395,24480,24565,24650,24735,24820,24905,24990,25075,25160,25245,25330,25415,25500,25585,25670,25755,25840,25925,26010,26095,26180,26265,26350,26435,26520,26605,26690,26775,26860,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7395,7480,7565,7650,7735,7820,7905,7990,8075,8160,8245,8330,8415,8500,8585,8670,8755,8840,8925,9010,9095,9180,9265,9350,9435,9520,9605,9690,9775,9860,9945,10030,10115,10200,10285,10370,10455,10540,10625,10710,10795,10880,10965,11050,11135,11220,11305,11390,11475,11560,11645,11730,11815,11900,11985,12070,12155,12240,12325,12410,12495,12580,12665,12750,12835,12920,13005,13090,13175,13260,13345,13430,13515,13600,13685,13770,13855,13940,14025,14110,14195,14280,14365,14450,14535,14620,14705,14790,14875,14960,15045,15130,15215,15300,15385,15470,15555,15640,15725,15810,15895,15980,16065,16150,16235,16320,16405,16490,16575,16660,16745,16830,16915,17000,17085,17170,17255,17340,17425,17510,17595,17680,17765,17850,17935,18020,18105,18190,18275,18360,18445,18530,18615,18700,18785,18870,18955,19040,19125,19210,19295,19380,19465,19550,19635,19720,19805,19890,19975,20060,20145,20230,20315,20400,20485,20570,20655,20740,20825,20910,20995,21080,21165,21250,21335,21420,21505,21590,21675,21760,21845,21930,22015,22100,22185,22270,22355,22440,22525,22610,22695,22780,22865,22950,23035,23120,23205,23290,23375,23460,23545,23630,23715,23800,23885,23970,24055,24140,24225,24310,24395,24480,24565,24650,24735,24820,24905,24990,25075,25160,25245,25330,25415,25500,25585,25670,25755,25840,25925,26010,26095,26180,26265,26350,26435,26520,26605,26690,26775,26860,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7568,7654,7740,7826,7912,7998,8084,8170,8256,8342,8428,8514,8600,8686,8772,8858,8944,9030,9116,9202,9288,9374,9460,9546,9632,9718,9804,9890,9976,10062,10148,10234,10320,10406,10492,10578,10664,10750,10836,10922,11008,11094,11180,11266,11352,11438,11524,11610,11696,11782,11868,11954,12040,12126,12212,12298,12384,12470,12556,12642,12728,12814,12900,12986,13072,13158,13244,13330,13416,13502,13588,13674,13760,13846,13932,14018,14104,14190,14276,14362,14448,14534,14620,14706,14792,14878,14964,15050,15136,15222,15308,15394,15480,15566,15652,15738,15824,15910,15996,16082,16168,16254,16340,16426,16512,16598,16684,16770,16856,16942,17028,17114,17200,17286,17372,17458,17544,17630,17716,17802,17888,17974,18060,18146,18232,18318,18404,18490,18576,18662,18748,18834,18920,19006,19092,19178,19264,19350,19436,19522,19608,19694,19780,19866,19952,20038,20124,20210,20296,20382,20468,20554,20640,20726,20812,20898,20984,21070,21156,21242,21328,21414,21500,21586,21672,21758,21844,21930,22016,22102,22188,22274,22360,22446,22532,22618,22704,22790,22876,22962,23048,23134,23220,23306,23392,23478,23564,23650,23736,23822,23908,23994,24080,24166,24252,24338,24424,24510,24596,24682,24768,24854,24940,25026,25112,25198,25284,25370,25456,25542,25628,25714,25800,25886,25972,26058,26144,26230,26316,26402,26488,26574,26660,26746,26832,26918,27004,27090,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7568,7654,7740,7826,7912,7998,8084,8170,8256,8342,8428,8514,8600,8686,8772,8858,8944,9030,9116,9202,9288,9374,9460,9546,9632,9718,9804,9890,9976,10062,10148,10234,10320,10406,10492,10578,10664,10750,10836,10922,11008,11094,11180,11266,11352,11438,11524,11610,11696,11782,11868,11954,12040,12126,12212,12298,12384,12470,12556,12642,12728,12814,12900,12986,13072,13158,13244,13330,13416,13502,13588,13674,13760,13846,13932,14018,14104,14190,14276,14362,14448,14534,14620,14706,14792,14878,14964,15050,15136,15222,15308,15394,15480,15566,15652,15738,15824,15910,15996,16082,16168,16254,16340,16426,16512,16598,16684,16770,16856,16942,17028,17114,17200,17286,17372,17458,17544,17630,17716,17802,17888,17974,18060,18146,18232,18318,18404,18490,18576,18662,18748,18834,18920,19006,19092,19178,19264,19350,19436,19522,19608,19694,19780,19866,19952,20038,20124,20210,20296,20382,20468,20554,20640,20726,20812,20898,20984,21070,21156,21242,21328,21414,21500,21586,21672,21758,21844,21930,22016,22102,22188,22274,22360,22446,22532,22618,22704,22790,22876,22962,23048,23134,23220,23306,23392,23478,23564,23650,23736,23822,23908,23994,24080,24166,24252,24338,24424,24510,24596,24682,24768,24854,24940,25026,25112,25198,25284,25370,25456,25542,25628,25714,25800,25886,25972,26058,26144,26230,26316,26402,26488,26574,26660,26746,26832,26918,27004,27090,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7743,7830,7917,8004,8091,8178,8265,8352,8439,8526,8613,8700,8787,8874,8961,9048,9135,9222,9309,9396,9483,9570,9657,9744,9831,9918,10005,10092,10179,10266,10353,10440,10527,10614,10701,10788,10875,10962,11049,11136,11223,11310,11397,11484,11571,11658,11745,11832,11919,12006,12093,12180,12267,12354,12441,12528,12615,12702,12789,12876,12963,13050,13137,13224,13311,13398,13485,13572,13659,13746,13833,13920,14007,14094,14181,14268,14355,14442,14529,14616,14703,14790,14877,14964,15051,15138,15225,15312,15399,15486,15573,15660,15747,15834,15921,16008,16095,16182,16269,16356,16443,16530,16617,16704,16791,16878,16965,17052,17139,17226,17313,17400,17487,17574,17661,17748,17835,17922,18009,18096,18183,18270,18357,18444,18531,18618,18705,18792,18879,18966,19053,19140,19227,19314,19401,19488,19575,19662,19749,19836,19923,20010,20097,20184,20271,20358,20445,20532,20619,20706,20793,20880,20967,21054,21141,21228,21315,21402,21489,21576,21663,21750,21837,21924,22011,22098,22185,22272,22359,22446,22533,22620,22707,22794,22881,22968,23055,23142,23229,23316,23403,23490,23577,23664,23751,23838,23925,24012,24099,24186,24273,24360,24447,24534,24621,24708,24795,24882,24969,25056,25143,25230,25317,25404,25491,25578,25665,25752,25839,25926,26013,26100,26187,26274,26361,26448,26535,26622,26709,26796,26883,26970,27057,27144,27231,27318,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7743,7830,7917,8004,8091,8178,8265,8352,8439,8526,8613,8700,8787,8874,8961,9048,9135,9222,9309,9396,9483,9570,9657,9744,9831,9918,10005,10092,10179,10266,10353,10440,10527,10614,10701,10788,10875,10962,11049,11136,11223,11310,11397,11484,11571,11658,11745,11832,11919,12006,12093,12180,12267,12354,12441,12528,12615,12702,12789,12876,12963,13050,13137,13224,13311,13398,13485,13572,13659,13746,13833,13920,14007,14094,14181,14268,14355,14442,14529,14616,14703,14790,14877,14964,15051,15138,15225,15312,15399,15486,15573,15660,15747,15834,15921,16008,16095,16182,16269,16356,16443,16530,16617,16704,16791,16878,16965,17052,17139,17226,17313,17400,17487,17574,17661,17748,17835,17922,18009,18096,18183,18270,18357,18444,18531,18618,18705,18792,18879,18966,19053,19140,19227,19314,19401,19488,19575,19662,19749,19836,19923,20010,20097,20184,20271,20358,20445,20532,20619,20706,20793,20880,20967,21054,21141,21228,21315,21402,21489,21576,21663,21750,21837,21924,22011,22098,22185,22272,22359,22446,22533,22620,22707,22794,22881,22968,23055,23142,23229,23316,23403,23490,23577,23664,23751,23838,23925,24012,24099,24186,24273,24360,24447,24534,24621,24708,24795,24882,24969,25056,25143,25230,25317,25404,25491,25578,25665,25752,25839,25926,26013,26100,26187,26274,26361,26448,26535,26622,26709,26796,26883,26970,27057,27144,27231,27318,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7920,8008,8096,8184,8272,8360,8448,8536,8624,8712,8800,8888,8976,9064,9152,9240,9328,9416,9504,9592,9680,9768,9856,9944,10032,10120,10208,10296,10384,10472,10560,10648,10736,10824,10912,11000,11088,11176,11264,11352,11440,11528,11616,11704,11792,11880,11968,12056,12144,12232,12320,12408,12496,12584,12672,12760,12848,12936,13024,13112,13200,13288,13376,13464,13552,13640,13728,13816,13904,13992,14080,14168,14256,14344,14432,14520,14608,14696,14784,14872,14960,15048,15136,15224,15312,15400,15488,15576,15664,15752,15840,15928,16016,16104,16192,16280,16368,16456,16544,16632,16720,16808,16896,16984,17072,17160,17248,17336,17424,17512,17600,17688,17776,17864,17952,18040,18128,18216,18304,18392,18480,18568,18656,18744,18832,18920,19008,19096,19184,19272,19360,19448,19536,19624,19712,19800,19888,19976,20064,20152,20240,20328,20416,20504,20592,20680,20768,20856,20944,21032,21120,21208,21296,21384,21472,21560,21648,21736,21824,21912,22000,22088,22176,22264,22352,22440,22528,22616,22704,22792,22880,22968,23056,23144,23232,23320,23408,23496,23584,23672,23760,23848,23936,24024,24112,24200,24288,24376,24464,24552,24640,24728,24816,24904,24992,25080,25168,25256,25344,25432,25520,25608,25696,25784,25872,25960,26048,26136,26224,26312,26400,26488,26576,26664,26752,26840,26928,27016,27104,27192,27280,27368,27456,27544,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7920,8008,8096,8184,8272,8360,8448,8536,8624,8712,8800,8888,8976,9064,9152,9240,9328,9416,9504,9592,9680,9768,9856,9944,10032,10120,10208,10296,10384,10472,10560,10648,10736,10824,10912,11000,11088,11176,11264,11352,11440,11528,11616,11704,11792,11880,11968,12056,12144,12232,12320,12408,12496,12584,12672,12760,12848,12936,13024,13112,13200,13288,13376,13464,13552,13640,13728,13816,13904,13992,14080,14168,14256,14344,14432,14520,14608,14696,14784,14872,14960,15048,15136,15224,15312,15400,15488,15576,15664,15752,15840,15928,16016,16104,16192,16280,16368,16456,16544,16632,16720,16808,16896,16984,17072,17160,17248,17336,17424,17512,17600,17688,17776,17864,17952,18040,18128,18216,18304,18392,18480,18568,18656,18744,18832,18920,19008,19096,19184,19272,19360,19448,19536,19624,19712,19800,19888,19976,20064,20152,20240,20328,20416,20504,20592,20680,20768,20856,20944,21032,21120,21208,21296,21384,21472,21560,21648,21736,21824,21912,22000,22088,22176,22264,22352,22440,22528,22616,22704,22792,22880,22968,23056,23144,23232,23320,23408,23496,23584,23672,23760,23848,23936,24024,24112,24200,24288,24376,24464,24552,24640,24728,24816,24904,24992,25080,25168,25256,25344,25432,25520,25608,25696,25784,25872,25960,26048,26136,26224,26312,26400,26488,26576,26664,26752,26840,26928,27016,27104,27192,27280,27368,27456,27544,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8099,8188,8277,8366,8455,8544,8633,8722,8811,8900,8989,9078,9167,9256,9345,9434,9523,9612,9701,9790,9879,9968,10057,10146,10235,10324,10413,10502,10591,10680,10769,10858,10947,11036,11125,11214,11303,11392,11481,11570,11659,11748,11837,11926,12015,12104,12193,12282,12371,12460,12549,12638,12727,12816,12905,12994,13083,13172,13261,13350,13439,13528,13617,13706,13795,13884,13973,14062,14151,14240,14329,14418,14507,14596,14685,14774,14863,14952,15041,15130,15219,15308,15397,15486,15575,15664,15753,15842,15931,16020,16109,16198,16287,16376,16465,16554,16643,16732,16821,16910,16999,17088,17177,17266,17355,17444,17533,17622,17711,17800,17889,17978,18067,18156,18245,18334,18423,18512,18601,18690,18779,18868,18957,19046,19135,19224,19313,19402,19491,19580,19669,19758,19847,19936,20025,20114,20203,20292,20381,20470,20559,20648,20737,20826,20915,21004,21093,21182,21271,21360,21449,21538,21627,21716,21805,21894,21983,22072,22161,22250,22339,22428,22517,22606,22695,22784,22873,22962,23051,23140,23229,23318,23407,23496,23585,23674,23763,23852,23941,24030,24119,24208,24297,24386,24475,24564,24653,24742,24831,24920,25009,25098,25187,25276,25365,25454,25543,25632,25721,25810,25899,25988,26077,26166,26255,26344,26433,26522,26611,26700,26789,26878,26967,27056,27145,27234,27323,27412,27501,27590,27679,27768,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8099,8188,8277,8366,8455,8544,8633,8722,8811,8900,8989,9078,9167,9256,9345,9434,9523,9612,9701,9790,9879,9968,10057,10146,10235,10324,10413,10502,10591,10680,10769,10858,10947,11036,11125,11214,11303,11392,11481,11570,11659,11748,11837,11926,12015,12104,12193,12282,12371,12460,12549,12638,12727,12816,12905,12994,13083,13172,13261,13350,13439,13528,13617,13706,13795,13884,13973,14062,14151,14240,14329,14418,14507,14596,14685,14774,14863,14952,15041,15130,15219,15308,15397,15486,15575,15664,15753,15842,15931,16020,16109,16198,16287,16376,16465,16554,16643,16732,16821,16910,16999,17088,17177,17266,17355,17444,17533,17622,17711,17800,17889,17978,18067,18156,18245,18334,18423,18512,18601,18690,18779,18868,18957,19046,19135,19224,19313,19402,19491,19580,19669,19758,19847,19936,20025,20114,20203,20292,20381,20470,20559,20648,20737,20826,20915,21004,21093,21182,21271,21360,21449,21538,21627,21716,21805,21894,21983,22072,22161,22250,22339,22428,22517,22606,22695,22784,22873,22962,23051,23140,23229,23318,23407,23496,23585,23674,23763,23852,23941,24030,24119,24208,24297,24386,24475,24564,24653,24742,24831,24920,25009,25098,25187,25276,25365,25454,25543,25632,25721,25810,25899,25988,26077,26166,26255,26344,26433,26522,26611,26700,26789,26878,26967,27056,27145,27234,27323,27412,27501,27590,27679,27768,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8280,8370,8460,8550,8640,8730,8820,8910,9000,9090,9180,9270,9360,9450,9540,9630,9720,9810,9900,9990,10080,10170,10260,10350,10440,10530,10620,10710,10800,10890,10980,11070,11160,11250,11340,11430,11520,11610,11700,11790,11880,11970,12060,12150,12240,12330,12420,12510,12600,12690,12780,12870,12960,13050,13140,13230,13320,13410,13500,13590,13680,13770,13860,13950,14040,14130,14220,14310,14400,14490,14580,14670,14760,14850,14940,15030,15120,15210,15300,15390,15480,15570,15660,15750,15840,15930,16020,16110,16200,16290,16380,16470,16560,16650,16740,16830,16920,17010,17100,17190,17280,17370,17460,17550,17640,17730,17820,17910,18000,18090,18180,18270,18360,18450,18540,18630,18720,18810,18900,18990,19080,19170,19260,19350,19440,19530,19620,19710,19800,19890,19980,20070,20160,20250,20340,20430,20520,20610,20700,20790,20880,20970,21060,21150,21240,21330,21420,21510,21600,21690,21780,21870,21960,22050,22140,22230,22320,22410,22500,22590,22680,22770,22860,22950,23040,23130,23220,23310,23400,23490,23580,23670,23760,23850,23940,24030,24120,24210,24300,24390,24480,24570,24660,24750,24840,24930,25020,25110,25200,25290,25380,25470,25560,25650,25740,25830,25920,26010,26100,26190,26280,26370,26460,26550,26640,26730,26820,26910,27000,27090,27180,27270,27360,27450,27540,27630,27720,27810,27900,27990,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8280,8370,8460,8550,8640,8730,8820,8910,9000,9090,9180,9270,9360,9450,9540,9630,9720,9810,9900,9990,10080,10170,10260,10350,10440,10530,10620,10710,10800,10890,10980,11070,11160,11250,11340,11430,11520,11610,11700,11790,11880,11970,12060,12150,12240,12330,12420,12510,12600,12690,12780,12870,12960,13050,13140,13230,13320,13410,13500,13590,13680,13770,13860,13950,14040,14130,14220,14310,14400,14490,14580,14670,14760,14850,14940,15030,15120,15210,15300,15390,15480,15570,15660,15750,15840,15930,16020,16110,16200,16290,16380,16470,16560,16650,16740,16830,16920,17010,17100,17190,17280,17370,17460,17550,17640,17730,17820,17910,18000,18090,18180,18270,18360,18450,18540,18630,18720,18810,18900,18990,19080,19170,19260,19350,19440,19530,19620,19710,19800,19890,19980,20070,20160,20250,20340,20430,20520,20610,20700,20790,20880,20970,21060,21150,21240,21330,21420,21510,21600,21690,21780,21870,21960,22050,22140,22230,22320,22410,22500,22590,22680,22770,22860,22950,23040,23130,23220,23310,23400,23490,23580,23670,23760,23850,23940,24030,24120,24210,24300,24390,24480,24570,24660,24750,24840,24930,25020,25110,25200,25290,25380,25470,25560,25650,25740,25830,25920,26010,26100,26190,26280,26370,26460,26550,26640,26730,26820,26910,27000,27090,27180,27270,27360,27450,27540,27630,27720,27810,27900,27990,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8463,8554,8645,8736,8827,8918,9009,9100,9191,9282,9373,9464,9555,9646,9737,9828,9919,10010,10101,10192,10283,10374,10465,10556,10647,10738,10829,10920,11011,11102,11193,11284,11375,11466,11557,11648,11739,11830,11921,12012,12103,12194,12285,12376,12467,12558,12649,12740,12831,12922,13013,13104,13195,13286,13377,13468,13559,13650,13741,13832,13923,14014,14105,14196,14287,14378,14469,14560,14651,14742,14833,14924,15015,15106,15197,15288,15379,15470,15561,15652,15743,15834,15925,16016,16107,16198,16289,16380,16471,16562,16653,16744,16835,16926,17017,17108,17199,17290,17381,17472,17563,17654,17745,17836,17927,18018,18109,18200,18291,18382,18473,18564,18655,18746,18837,18928,19019,19110,19201,19292,19383,19474,19565,19656,19747,19838,19929,20020,20111,20202,20293,20384,20475,20566,20657,20748,20839,20930,21021,21112,21203,21294,21385,21476,21567,21658,21749,21840,21931,22022,22113,22204,22295,22386,22477,22568,22659,22750,22841,22932,23023,23114,23205,23296,23387,23478,23569,23660,23751,23842,23933,24024,24115,24206,24297,24388,24479,24570,24661,24752,24843,24934,25025,25116,25207,25298,25389,25480,25571,25662,25753,25844,25935,26026,26117,26208,26299,26390,26481,26572,26663,26754,26845,26936,27027,27118,27209,27300,27391,27482,27573,27664,27755,27846,27937,28028,28119,28210,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8463,8554,8645,8736,8827,8918,9009,9100,9191,9282,9373,9464,9555,9646,9737,9828,9919,10010,10101,10192,10283,10374,10465,10556,10647,10738,10829,10920,11011,11102,11193,11284,11375,11466,11557,11648,11739,11830,11921,12012,12103,12194,12285,12376,12467,12558,12649,12740,12831,12922,13013,13104,13195,13286,13377,13468,13559,13650,13741,13832,13923,14014,14105,14196,14287,14378,14469,14560,14651,14742,14833,14924,15015,15106,15197,15288,15379,15470,15561,15652,15743,15834,15925,16016,16107,16198,16289,16380,16471,16562,16653,16744,16835,16926,17017,17108,17199,17290,17381,17472,17563,17654,17745,17836,17927,18018,18109,18200,18291,18382,18473,18564,18655,18746,18837,18928,19019,19110,19201,19292,19383,19474,19565,19656,19747,19838,19929,20020,20111,20202,20293,20384,20475,20566,20657,20748,20839,20930,21021,21112,21203,21294,21385,21476,21567,21658,21749,21840,21931,22022,22113,22204,22295,22386,22477,22568,22659,22750,22841,22932,23023,23114,23205,23296,23387,23478,23569,23660,23751,23842,23933,24024,24115,24206,24297,24388,24479,24570,24661,24752,24843,24934,25025,25116,25207,25298,25389,25480,25571,25662,25753,25844,25935,26026,26117,26208,26299,26390,26481,26572,26663,26754,26845,26936,27027,27118,27209,27300,27391,27482,27573,27664,27755,27846,27937,28028,28119,28210,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8648,8740,8832,8924,9016,9108,9200,9292,9384,9476,9568,9660,9752,9844,9936,10028,10120,10212,10304,10396,10488,10580,10672,10764,10856,10948,11040,11132,11224,11316,11408,11500,11592,11684,11776,11868,11960,12052,12144,12236,12328,12420,12512,12604,12696,12788,12880,12972,13064,13156,13248,13340,13432,13524,13616,13708,13800,13892,13984,14076,14168,14260,14352,14444,14536,14628,14720,14812,14904,14996,15088,15180,15272,15364,15456,15548,15640,15732,15824,15916,16008,16100,16192,16284,16376,16468,16560,16652,16744,16836,16928,17020,17112,17204,17296,17388,17480,17572,17664,17756,17848,17940,18032,18124,18216,18308,18400,18492,18584,18676,18768,18860,18952,19044,19136,19228,19320,19412,19504,19596,19688,19780,19872,19964,20056,20148,20240,20332,20424,20516,20608,20700,20792,20884,20976,21068,21160,21252,21344,21436,21528,21620,21712,21804,21896,21988,22080,22172,22264,22356,22448,22540,22632,22724,22816,22908,23000,23092,23184,23276,23368,23460,23552,23644,23736,23828,23920,24012,24104,24196,24288,24380,24472,24564,24656,24748,24840,24932,25024,25116,25208,25300,25392,25484,25576,25668,25760,25852,25944,26036,26128,26220,26312,26404,26496,26588,26680,26772,26864,26956,27048,27140,27232,27324,27416,27508,27600,27692,27784,27876,27968,28060,28152,28244,28336,28428,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8648,8740,8832,8924,9016,9108,9200,9292,9384,9476,9568,9660,9752,9844,9936,10028,10120,10212,10304,10396,10488,10580,10672,10764,10856,10948,11040,11132,11224,11316,11408,11500,11592,11684,11776,11868,11960,12052,12144,12236,12328,12420,12512,12604,12696,12788,12880,12972,13064,13156,13248,13340,13432,13524,13616,13708,13800,13892,13984,14076,14168,14260,14352,14444,14536,14628,14720,14812,14904,14996,15088,15180,15272,15364,15456,15548,15640,15732,15824,15916,16008,16100,16192,16284,16376,16468,16560,16652,16744,16836,16928,17020,17112,17204,17296,17388,17480,17572,17664,17756,17848,17940,18032,18124,18216,18308,18400,18492,18584,18676,18768,18860,18952,19044,19136,19228,19320,19412,19504,19596,19688,19780,19872,19964,20056,20148,20240,20332,20424,20516,20608,20700,20792,20884,20976,21068,21160,21252,21344,21436,21528,21620,21712,21804,21896,21988,22080,22172,22264,22356,22448,22540,22632,22724,22816,22908,23000,23092,23184,23276,23368,23460,23552,23644,23736,23828,23920,24012,24104,24196,24288,24380,24472,24564,24656,24748,24840,24932,25024,25116,25208,25300,25392,25484,25576,25668,25760,25852,25944,26036,26128,26220,26312,26404,26496,26588,26680,26772,26864,26956,27048,27140,27232,27324,27416,27508,27600,27692,27784,27876,27968,28060,28152,28244,28336,28428,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8835,8928,9021,9114,9207,9300,9393,9486,9579,9672,9765,9858,9951,10044,10137,10230,10323,10416,10509,10602,10695,10788,10881,10974,11067,11160,11253,11346,11439,11532,11625,11718,11811,11904,11997,12090,12183,12276,12369,12462,12555,12648,12741,12834,12927,13020,13113,13206,13299,13392,13485,13578,13671,13764,13857,13950,14043,14136,14229,14322,14415,14508,14601,14694,14787,14880,14973,15066,15159,15252,15345,15438,15531,15624,15717,15810,15903,15996,16089,16182,16275,16368,16461,16554,16647,16740,16833,16926,17019,17112,17205,17298,17391,17484,17577,17670,17763,17856,17949,18042,18135,18228,18321,18414,18507,18600,18693,18786,18879,18972,19065,19158,19251,19344,19437,19530,19623,19716,19809,19902,19995,20088,20181,20274,20367,20460,20553,20646,20739,20832,20925,21018,21111,21204,21297,21390,21483,21576,21669,21762,21855,21948,22041,22134,22227,22320,22413,22506,22599,22692,22785,22878,22971,23064,23157,23250,23343,23436,23529,23622,23715,23808,23901,23994,24087,24180,24273,24366,24459,24552,24645,24738,24831,24924,25017,25110,25203,25296,25389,25482,25575,25668,25761,25854,25947,26040,26133,26226,26319,26412,26505,26598,26691,26784,26877,26970,27063,27156,27249,27342,27435,27528,27621,27714,27807,27900,27993,28086,28179,28272,28365,28458,28551,28644,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8835,8928,9021,9114,9207,9300,9393,9486,9579,9672,9765,9858,9951,10044,10137,10230,10323,10416,10509,10602,10695,10788,10881,10974,11067,11160,11253,11346,11439,11532,11625,11718,11811,11904,11997,12090,12183,12276,12369,12462,12555,12648,12741,12834,12927,13020,13113,13206,13299,13392,13485,13578,13671,13764,13857,13950,14043,14136,14229,14322,14415,14508,14601,14694,14787,14880,14973,15066,15159,15252,15345,15438,15531,15624,15717,15810,15903,15996,16089,16182,16275,16368,16461,16554,16647,16740,16833,16926,17019,17112,17205,17298,17391,17484,17577,17670,17763,17856,17949,18042,18135,18228,18321,18414,18507,18600,18693,18786,18879,18972,19065,19158,19251,19344,19437,19530,19623,19716,19809,19902,19995,20088,20181,20274,20367,20460,20553,20646,20739,20832,20925,21018,21111,21204,21297,21390,21483,21576,21669,21762,21855,21948,22041,22134,22227,22320,22413,22506,22599,22692,22785,22878,22971,23064,23157,23250,23343,23436,23529,23622,23715,23808,23901,23994,24087,24180,24273,24366,24459,24552,24645,24738,24831,24924,25017,25110,25203,25296,25389,25482,25575,25668,25761,25854,25947,26040,26133,26226,26319,26412,26505,26598,26691,26784,26877,26970,27063,27156,27249,27342,27435,27528,27621,27714,27807,27900,27993,28086,28179,28272,28365,28458,28551,28644,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9024,9118,9212,9306,9400,9494,9588,9682,9776,9870,9964,10058,10152,10246,10340,10434,10528,10622,10716,10810,10904,10998,11092,11186,11280,11374,11468,11562,11656,11750,11844,11938,12032,12126,12220,12314,12408,12502,12596,12690,12784,12878,12972,13066,13160,13254,13348,13442,13536,13630,13724,13818,13912,14006,14100,14194,14288,14382,14476,14570,14664,14758,14852,14946,15040,15134,15228,15322,15416,15510,15604,15698,15792,15886,15980,16074,16168,16262,16356,16450,16544,16638,16732,16826,16920,17014,17108,17202,17296,17390,17484,17578,17672,17766,17860,17954,18048,18142,18236,18330,18424,18518,18612,18706,18800,18894,18988,19082,19176,19270,19364,19458,19552,19646,19740,19834,19928,20022,20116,20210,20304,20398,20492,20586,20680,20774,20868,20962,21056,21150,21244,21338,21432,21526,21620,21714,21808,21902,21996,22090,22184,22278,22372,22466,22560,22654,22748,22842,22936,23030,23124,23218,23312,23406,23500,23594,23688,23782,23876,23970,24064,24158,24252,24346,24440,24534,24628,24722,24816,24910,25004,25098,25192,25286,25380,25474,25568,25662,25756,25850,25944,26038,26132,26226,26320,26414,26508,26602,26696,26790,26884,26978,27072,27166,27260,27354,27448,27542,27636,27730,27824,27918,28012,28106,28200,28294,28388,28482,28576,28670,28764,28858,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9024,9118,9212,9306,9400,9494,9588,9682,9776,9870,9964,10058,10152,10246,10340,10434,10528,10622,10716,10810,10904,10998,11092,11186,11280,11374,11468,11562,11656,11750,11844,11938,12032,12126,12220,12314,12408,12502,12596,12690,12784,12878,12972,13066,13160,13254,13348,13442,13536,13630,13724,13818,13912,14006,14100,14194,14288,14382,14476,14570,14664,14758,14852,14946,15040,15134,15228,15322,15416,15510,15604,15698,15792,15886,15980,16074,16168,16262,16356,16450,16544,16638,16732,16826,16920,17014,17108,17202,17296,17390,17484,17578,17672,17766,17860,17954,18048,18142,18236,18330,18424,18518,18612,18706,18800,18894,18988,19082,19176,19270,19364,19458,19552,19646,19740,19834,19928,20022,20116,20210,20304,20398,20492,20586,20680,20774,20868,20962,21056,21150,21244,21338,21432,21526,21620,21714,21808,21902,21996,22090,22184,22278,22372,22466,22560,22654,22748,22842,22936,23030,23124,23218,23312,23406,23500,23594,23688,23782,23876,23970,24064,24158,24252,24346,24440,24534,24628,24722,24816,24910,25004,25098,25192,25286,25380,25474,25568,25662,25756,25850,25944,26038,26132,26226,26320,26414,26508,26602,26696,26790,26884,26978,27072,27166,27260,27354,27448,27542,27636,27730,27824,27918,28012,28106,28200,28294,28388,28482,28576,28670,28764,28858,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9215,9310,9405,9500,9595,9690,9785,9880,9975,10070,10165,10260,10355,10450,10545,10640,10735,10830,10925,11020,11115,11210,11305,11400,11495,11590,11685,11780,11875,11970,12065,12160,12255,12350,12445,12540,12635,12730,12825,12920,13015,13110,13205,13300,13395,13490,13585,13680,13775,13870,13965,14060,14155,14250,14345,14440,14535,14630,14725,14820,14915,15010,15105,15200,15295,15390,15485,15580,15675,15770,15865,15960,16055,16150,16245,16340,16435,16530,16625,16720,16815,16910,17005,17100,17195,17290,17385,17480,17575,17670,17765,17860,17955,18050,18145,18240,18335,18430,18525,18620,18715,18810,18905,19000,19095,19190,19285,19380,19475,19570,19665,19760,19855,19950,20045,20140,20235,20330,20425,20520,20615,20710,20805,20900,20995,21090,21185,21280,21375,21470,21565,21660,21755,21850,21945,22040,22135,22230,22325,22420,22515,22610,22705,22800,22895,22990,23085,23180,23275,23370,23465,23560,23655,23750,23845,23940,24035,24130,24225,24320,24415,24510,24605,24700,24795,24890,24985,25080,25175,25270,25365,25460,25555,25650,25745,25840,25935,26030,26125,26220,26315,26410,26505,26600,26695,26790,26885,26980,27075,27170,27265,27360,27455,27550,27645,27740,27835,27930,28025,28120,28215,28310,28405,28500,28595,28690,28785,28880,28975,29070,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9215,9310,9405,9500,9595,9690,9785,9880,9975,10070,10165,10260,10355,10450,10545,10640,10735,10830,10925,11020,11115,11210,11305,11400,11495,11590,11685,11780,11875,11970,12065,12160,12255,12350,12445,12540,12635,12730,12825,12920,13015,13110,13205,13300,13395,13490,13585,13680,13775,13870,13965,14060,14155,14250,14345,14440,14535,14630,14725,14820,14915,15010,15105,15200,15295,15390,15485,15580,15675,15770,15865,15960,16055,16150,16245,16340,16435,16530,16625,16720,16815,16910,17005,17100,17195,17290,17385,17480,17575,17670,17765,17860,17955,18050,18145,18240,18335,18430,18525,18620,18715,18810,18905,19000,19095,19190,19285,19380,19475,19570,19665,19760,19855,19950,20045,20140,20235,20330,20425,20520,20615,20710,20805,20900,20995,21090,21185,21280,21375,21470,21565,21660,21755,21850,21945,22040,22135,22230,22325,22420,22515,22610,22705,22800,22895,22990,23085,23180,23275,23370,23465,23560,23655,23750,23845,23940,24035,24130,24225,24320,24415,24510,24605,24700,24795,24890,24985,25080,25175,25270,25365,25460,25555,25650,25745,25840,25935,26030,26125,26220,26315,26410,26505,26600,26695,26790,26885,26980,27075,27170,27265,27360,27455,27550,27645,27740,27835,27930,28025,28120,28215,28310,28405,28500,28595,28690,28785,28880,28975,29070,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9408,9504,9600,9696,9792,9888,9984,10080,10176,10272,10368,10464,10560,10656,10752,10848,10944,11040,11136,11232,11328,11424,11520,11616,11712,11808,11904,12000,12096,12192,12288,12384,12480,12576,12672,12768,12864,12960,13056,13152,13248,13344,13440,13536,13632,13728,13824,13920,14016,14112,14208,14304,14400,14496,14592,14688,14784,14880,14976,15072,15168,15264,15360,15456,15552,15648,15744,15840,15936,16032,16128,16224,16320,16416,16512,16608,16704,16800,16896,16992,17088,17184,17280,17376,17472,17568,17664,17760,17856,17952,18048,18144,18240,18336,18432,18528,18624,18720,18816,18912,19008,19104,19200,19296,19392,19488,19584,19680,19776,19872,19968,20064,20160,20256,20352,20448,20544,20640,20736,20832,20928,21024,21120,21216,21312,21408,21504,21600,21696,21792,21888,21984,22080,22176,22272,22368,22464,22560,22656,22752,22848,22944,23040,23136,23232,23328,23424,23520,23616,23712,23808,23904,24000,24096,24192,24288,24384,24480,24576,24672,24768,24864,24960,25056,25152,25248,25344,25440,25536,25632,25728,25824,25920,26016,26112,26208,26304,26400,26496,26592,26688,26784,26880,26976,27072,27168,27264,27360,27456,27552,27648,27744,27840,27936,28032,28128,28224,28320,28416,28512,28608,28704,28800,28896,28992,29088,29184,29280,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9408,9504,9600,9696,9792,9888,9984,10080,10176,10272,10368,10464,10560,10656,10752,10848,10944,11040,11136,11232,11328,11424,11520,11616,11712,11808,11904,12000,12096,12192,12288,12384,12480,12576,12672,12768,12864,12960,13056,13152,13248,13344,13440,13536,13632,13728,13824,13920,14016,14112,14208,14304,14400,14496,14592,14688,14784,14880,14976,15072,15168,15264,15360,15456,15552,15648,15744,15840,15936,16032,16128,16224,16320,16416,16512,16608,16704,16800,16896,16992,17088,17184,17280,17376,17472,17568,17664,17760,17856,17952,18048,18144,18240,18336,18432,18528,18624,18720,18816,18912,19008,19104,19200,19296,19392,19488,19584,19680,19776,19872,19968,20064,20160,20256,20352,20448,20544,20640,20736,20832,20928,21024,21120,21216,21312,21408,21504,21600,21696,21792,21888,21984,22080,22176,22272,22368,22464,22560,22656,22752,22848,22944,23040,23136,23232,23328,23424,23520,23616,23712,23808,23904,24000,24096,24192,24288,24384,24480,24576,24672,24768,24864,24960,25056,25152,25248,25344,25440,25536,25632,25728,25824,25920,26016,26112,26208,26304,26400,26496,26592,26688,26784,26880,26976,27072,27168,27264,27360,27456,27552,27648,27744,27840,27936,28032,28128,28224,28320,28416,28512,28608,28704,28800,28896,28992,29088,29184,29280,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9603,9700,9797,9894,9991,10088,10185,10282,10379,10476,10573,10670,10767,10864,10961,11058,11155,11252,11349,11446,11543,11640,11737,11834,11931,12028,12125,12222,12319,12416,12513,12610,12707,12804,12901,12998,13095,13192,13289,13386,13483,13580,13677,13774,13871,13968,14065,14162,14259,14356,14453,14550,14647,14744,14841,14938,15035,15132,15229,15326,15423,15520,15617,15714,15811,15908,16005,16102,16199,16296,16393,16490,16587,16684,16781,16878,16975,17072,17169,17266,17363,17460,17557,17654,17751,17848,17945,18042,18139,18236,18333,18430,18527,18624,18721,18818,18915,19012,19109,19206,19303,19400,19497,19594,19691,19788,19885,19982,20079,20176,20273,20370,20467,20564,20661,20758,20855,20952,21049,21146,21243,21340,21437,21534,21631,21728,21825,21922,22019,22116,22213,22310,22407,22504,22601,22698,22795,22892,22989,23086,23183,23280,23377,23474,23571,23668,23765,23862,23959,24056,24153,24250,24347,24444,24541,24638,24735,24832,24929,25026,25123,25220,25317,25414,25511,25608,25705,25802,25899,25996,26093,26190,26287,26384,26481,26578,26675,26772,26869,26966,27063,27160,27257,27354,27451,27548,27645,27742,27839,27936,28033,28130,28227,28324,28421,28518,28615,28712,28809,28906,29003,29100,29197,29294,29391,29488,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9603,9700,9797,9894,9991,10088,10185,10282,10379,10476,10573,10670,10767,10864,10961,11058,11155,11252,11349,11446,11543,11640,11737,11834,11931,12028,12125,12222,12319,12416,12513,12610,12707,12804,12901,12998,13095,13192,13289,13386,13483,13580,13677,13774,13871,13968,14065,14162,14259,14356,14453,14550,14647,14744,14841,14938,15035,15132,15229,15326,15423,15520,15617,15714,15811,15908,16005,16102,16199,16296,16393,16490,16587,16684,16781,16878,16975,17072,17169,17266,17363,17460,17557,17654,17751,17848,17945,18042,18139,18236,18333,18430,18527,18624,18721,18818,18915,19012,19109,19206,19303,19400,19497,19594,19691,19788,19885,19982,20079,20176,20273,20370,20467,20564,20661,20758,20855,20952,21049,21146,21243,21340,21437,21534,21631,21728,21825,21922,22019,22116,22213,22310,22407,22504,22601,22698,22795,22892,22989,23086,23183,23280,23377,23474,23571,23668,23765,23862,23959,24056,24153,24250,24347,24444,24541,24638,24735,24832,24929,25026,25123,25220,25317,25414,25511,25608,25705,25802,25899,25996,26093,26190,26287,26384,26481,26578,26675,26772,26869,26966,27063,27160,27257,27354,27451,27548,27645,27742,27839,27936,28033,28130,28227,28324,28421,28518,28615,28712,28809,28906,29003,29100,29197,29294,29391,29488,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9800,9898,9996,10094,10192,10290,10388,10486,10584,10682,10780,10878,10976,11074,11172,11270,11368,11466,11564,11662,11760,11858,11956,12054,12152,12250,12348,12446,12544,12642,12740,12838,12936,13034,13132,13230,13328,13426,13524,13622,13720,13818,13916,14014,14112,14210,14308,14406,14504,14602,14700,14798,14896,14994,15092,15190,15288,15386,15484,15582,15680,15778,15876,15974,16072,16170,16268,16366,16464,16562,16660,16758,16856,16954,17052,17150,17248,17346,17444,17542,17640,17738,17836,17934,18032,18130,18228,18326,18424,18522,18620,18718,18816,18914,19012,19110,19208,19306,19404,19502,19600,19698,19796,19894,19992,20090,20188,20286,20384,20482,20580,20678,20776,20874,20972,21070,21168,21266,21364,21462,21560,21658,21756,21854,21952,22050,22148,22246,22344,22442,22540,22638,22736,22834,22932,23030,23128,23226,23324,23422,23520,23618,23716,23814,23912,24010,24108,24206,24304,24402,24500,24598,24696,24794,24892,24990,25088,25186,25284,25382,25480,25578,25676,25774,25872,25970,26068,26166,26264,26362,26460,26558,26656,26754,26852,26950,27048,27146,27244,27342,27440,27538,27636,27734,27832,27930,28028,28126,28224,28322,28420,28518,28616,28714,28812,28910,29008,29106,29204,29302,29400,29498,29596,29694,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9800,9898,9996,10094,10192,10290,10388,10486,10584,10682,10780,10878,10976,11074,11172,11270,11368,11466,11564,11662,11760,11858,11956,12054,12152,12250,12348,12446,12544,12642,12740,12838,12936,13034,13132,13230,13328,13426,13524,13622,13720,13818,13916,14014,14112,14210,14308,14406,14504,14602,14700,14798,14896,14994,15092,15190,15288,15386,15484,15582,15680,15778,15876,15974,16072,16170,16268,16366,16464,16562,16660,16758,16856,16954,17052,17150,17248,17346,17444,17542,17640,17738,17836,17934,18032,18130,18228,18326,18424,18522,18620,18718,18816,18914,19012,19110,19208,19306,19404,19502,19600,19698,19796,19894,19992,20090,20188,20286,20384,20482,20580,20678,20776,20874,20972,21070,21168,21266,21364,21462,21560,21658,21756,21854,21952,22050,22148,22246,22344,22442,22540,22638,22736,22834,22932,23030,23128,23226,23324,23422,23520,23618,23716,23814,23912,24010,24108,24206,24304,24402,24500,24598,24696,24794,24892,24990,25088,25186,25284,25382,25480,25578,25676,25774,25872,25970,26068,26166,26264,26362,26460,26558,26656,26754,26852,26950,27048,27146,27244,27342,27440,27538,27636,27734,27832,27930,28028,28126,28224,28322,28420,28518,28616,28714,28812,28910,29008,29106,29204,29302,29400,29498,29596,29694,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,9999,10098,10197,10296,10395,10494,10593,10692,10791,10890,10989,11088,11187,11286,11385,11484,11583,11682,11781,11880,11979,12078,12177,12276,12375,12474,12573,12672,12771,12870,12969,13068,13167,13266,13365,13464,13563,13662,13761,13860,13959,14058,14157,14256,14355,14454,14553,14652,14751,14850,14949,15048,15147,15246,15345,15444,15543,15642,15741,15840,15939,16038,16137,16236,16335,16434,16533,16632,16731,16830,16929,17028,17127,17226,17325,17424,17523,17622,17721,17820,17919,18018,18117,18216,18315,18414,18513,18612,18711,18810,18909,19008,19107,19206,19305,19404,19503,19602,19701,19800,19899,19998,20097,20196,20295,20394,20493,20592,20691,20790,20889,20988,21087,21186,21285,21384,21483,21582,21681,21780,21879,21978,22077,22176,22275,22374,22473,22572,22671,22770,22869,22968,23067,23166,23265,23364,23463,23562,23661,23760,23859,23958,24057,24156,24255,24354,24453,24552,24651,24750,24849,24948,25047,25146,25245,25344,25443,25542,25641,25740,25839,25938,26037,26136,26235,26334,26433,26532,26631,26730,26829,26928,27027,27126,27225,27324,27423,27522,27621,27720,27819,27918,28017,28116,28215,28314,28413,28512,28611,28710,28809,28908,29007,29106,29205,29304,29403,29502,29601,29700,29799,29898,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,9999,10098,10197,10296,10395,10494,10593,10692,10791,10890,10989,11088,11187,11286,11385,11484,11583,11682,11781,11880,11979,12078,12177,12276,12375,12474,12573,12672,12771,12870,12969,13068,13167,13266,13365,13464,13563,13662,13761,13860,13959,14058,14157,14256,14355,14454,14553,14652,14751,14850,14949,15048,15147,15246,15345,15444,15543,15642,15741,15840,15939,16038,16137,16236,16335,16434,16533,16632,16731,16830,16929,17028,17127,17226,17325,17424,17523,17622,17721,17820,17919,18018,18117,18216,18315,18414,18513,18612,18711,18810,18909,19008,19107,19206,19305,19404,19503,19602,19701,19800,19899,19998,20097,20196,20295,20394,20493,20592,20691,20790,20889,20988,21087,21186,21285,21384,21483,21582,21681,21780,21879,21978,22077,22176,22275,22374,22473,22572,22671,22770,22869,22968,23067,23166,23265,23364,23463,23562,23661,23760,23859,23958,24057,24156,24255,24354,24453,24552,24651,24750,24849,24948,25047,25146,25245,25344,25443,25542,25641,25740,25839,25938,26037,26136,26235,26334,26433,26532,26631,26730,26829,26928,27027,27126,27225,27324,27423,27522,27621,27720,27819,27918,28017,28116,28215,28314,28413,28512,28611,28710,28809,28908,29007,29106,29205,29304,29403,29502,29601,29700,29799,29898,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10200,10300,10400,10500,10600,10700,10800,10900,11000,11100,11200,11300,11400,11500,11600,11700,11800,11900,12000,12100,12200,12300,12400,12500,12600,12700,12800,12900,13000,13100,13200,13300,13400,13500,13600,13700,13800,13900,14000,14100,14200,14300,14400,14500,14600,14700,14800,14900,15000,15100,15200,15300,15400,15500,15600,15700,15800,15900,16000,16100,16200,16300,16400,16500,16600,16700,16800,16900,17000,17100,17200,17300,17400,17500,17600,17700,17800,17900,18000,18100,18200,18300,18400,18500,18600,18700,18800,18900,19000,19100,19200,19300,19400,19500,19600,19700,19800,19900,20000,20100,20200,20300,20400,20500,20600,20700,20800,20900,21000,21100,21200,21300,21400,21500,21600,21700,21800,21900,22000,22100,22200,22300,22400,22500,22600,22700,22800,22900,23000,23100,23200,23300,23400,23500,23600,23700,23800,23900,24000,24100,24200,24300,24400,24500,24600,24700,24800,24900,25000,25100,25200,25300,25400,25500,25600,25700,25800,25900,26000,26100,26200,26300,26400,26500,26600,26700,26800,26900,27000,27100,27200,27300,27400,27500,27600,27700,27800,27900,28000,28100,28200,28300,28400,28500,28600,28700,28800,28900,29000,29100,29200,29300,29400,29500,29600,29700,29800,29900,30000,30100,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10200,10300,10400,10500,10600,10700,10800,10900,11000,11100,11200,11300,11400,11500,11600,11700,11800,11900,12000,12100,12200,12300,12400,12500,12600,12700,12800,12900,13000,13100,13200,13300,13400,13500,13600,13700,13800,13900,14000,14100,14200,14300,14400,14500,14600,14700,14800,14900,15000,15100,15200,15300,15400,15500,15600,15700,15800,15900,16000,16100,16200,16300,16400,16500,16600,16700,16800,16900,17000,17100,17200,17300,17400,17500,17600,17700,17800,17900,18000,18100,18200,18300,18400,18500,18600,18700,18800,18900,19000,19100,19200,19300,19400,19500,19600,19700,19800,19900,20000,20100,20200,20300,20400,20500,20600,20700,20800,20900,21000,21100,21200,21300,21400,21500,21600,21700,21800,21900,22000,22100,22200,22300,22400,22500,22600,22700,22800,22900,23000,23100,23200,23300,23400,23500,23600,23700,23800,23900,24000,24100,24200,24300,24400,24500,24600,24700,24800,24900,25000,25100,25200,25300,25400,25500,25600,25700,25800,25900,26000,26100,26200,26300,26400,26500,26600,26700,26800,26900,27000,27100,27200,27300,27400,27500,27600,27700,27800,27900,28000,28100,28200,28300,28400,28500,28600,28700,28800,28900,29000,29100,29200,29300,29400,29500,29600,29700,29800,29900,30000,30100,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10403,10504,10605,10706,10807,10908,11009,11110,11211,11312,11413,11514,11615,11716,11817,11918,12019,12120,12221,12322,12423,12524,12625,12726,12827,12928,13029,13130,13231,13332,13433,13534,13635,13736,13837,13938,14039,14140,14241,14342,14443,14544,14645,14746,14847,14948,15049,15150,15251,15352,15453,15554,15655,15756,15857,15958,16059,16160,16261,16362,16463,16564,16665,16766,16867,16968,17069,17170,17271,17372,17473,17574,17675,17776,17877,17978,18079,18180,18281,18382,18483,18584,18685,18786,18887,18988,19089,19190,19291,19392,19493,19594,19695,19796,19897,19998,20099,20200,20301,20402,20503,20604,20705,20806,20907,21008,21109,21210,21311,21412,21513,21614,21715,21816,21917,22018,22119,22220,22321,22422,22523,22624,22725,22826,22927,23028,23129,23230,23331,23432,23533,23634,23735,23836,23937,24038,24139,24240,24341,24442,24543,24644,24745,24846,24947,25048,25149,25250,25351,25452,25553,25654,25755,25856,25957,26058,26159,26260,26361,26462,26563,26664,26765,26866,26967,27068,27169,27270,27371,27472,27573,27674,27775,27876,27977,28078,28179,28280,28381,28482,28583,28684,28785,28886,28987,29088,29189,29290,29391,29492,29593,29694,29795,29896,29997,30098,30199,30300,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10403,10504,10605,10706,10807,10908,11009,11110,11211,11312,11413,11514,11615,11716,11817,11918,12019,12120,12221,12322,12423,12524,12625,12726,12827,12928,13029,13130,13231,13332,13433,13534,13635,13736,13837,13938,14039,14140,14241,14342,14443,14544,14645,14746,14847,14948,15049,15150,15251,15352,15453,15554,15655,15756,15857,15958,16059,16160,16261,16362,16463,16564,16665,16766,16867,16968,17069,17170,17271,17372,17473,17574,17675,17776,17877,17978,18079,18180,18281,18382,18483,18584,18685,18786,18887,18988,19089,19190,19291,19392,19493,19594,19695,19796,19897,19998,20099,20200,20301,20402,20503,20604,20705,20806,20907,21008,21109,21210,21311,21412,21513,21614,21715,21816,21917,22018,22119,22220,22321,22422,22523,22624,22725,22826,22927,23028,23129,23230,23331,23432,23533,23634,23735,23836,23937,24038,24139,24240,24341,24442,24543,24644,24745,24846,24947,25048,25149,25250,25351,25452,25553,25654,25755,25856,25957,26058,26159,26260,26361,26462,26563,26664,26765,26866,26967,27068,27169,27270,27371,27472,27573,27674,27775,27876,27977,28078,28179,28280,28381,28482,28583,28684,28785,28886,28987,29088,29189,29290,29391,29492,29593,29694,29795,29896,29997,30098,30199,30300,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10608,10710,10812,10914,11016,11118,11220,11322,11424,11526,11628,11730,11832,11934,12036,12138,12240,12342,12444,12546,12648,12750,12852,12954,13056,13158,13260,13362,13464,13566,13668,13770,13872,13974,14076,14178,14280,14382,14484,14586,14688,14790,14892,14994,15096,15198,15300,15402,15504,15606,15708,15810,15912,16014,16116,16218,16320,16422,16524,16626,16728,16830,16932,17034,17136,17238,17340,17442,17544,17646,17748,17850,17952,18054,18156,18258,18360,18462,18564,18666,18768,18870,18972,19074,19176,19278,19380,19482,19584,19686,19788,19890,19992,20094,20196,20298,20400,20502,20604,20706,20808,20910,21012,21114,21216,21318,21420,21522,21624,21726,21828,21930,22032,22134,22236,22338,22440,22542,22644,22746,22848,22950,23052,23154,23256,23358,23460,23562,23664,23766,23868,23970,24072,24174,24276,24378,24480,24582,24684,24786,24888,24990,25092,25194,25296,25398,25500,25602,25704,25806,25908,26010,26112,26214,26316,26418,26520,26622,26724,26826,26928,27030,27132,27234,27336,27438,27540,27642,27744,27846,27948,28050,28152,28254,28356,28458,28560,28662,28764,28866,28968,29070,29172,29274,29376,29478,29580,29682,29784,29886,29988,30090,30192,30294,30396,30498,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10608,10710,10812,10914,11016,11118,11220,11322,11424,11526,11628,11730,11832,11934,12036,12138,12240,12342,12444,12546,12648,12750,12852,12954,13056,13158,13260,13362,13464,13566,13668,13770,13872,13974,14076,14178,14280,14382,14484,14586,14688,14790,14892,14994,15096,15198,15300,15402,15504,15606,15708,15810,15912,16014,16116,16218,16320,16422,16524,16626,16728,16830,16932,17034,17136,17238,17340,17442,17544,17646,17748,17850,17952,18054,18156,18258,18360,18462,18564,18666,18768,18870,18972,19074,19176,19278,19380,19482,19584,19686,19788,19890,19992,20094,20196,20298,20400,20502,20604,20706,20808,20910,21012,21114,21216,21318,21420,21522,21624,21726,21828,21930,22032,22134,22236,22338,22440,22542,22644,22746,22848,22950,23052,23154,23256,23358,23460,23562,23664,23766,23868,23970,24072,24174,24276,24378,24480,24582,24684,24786,24888,24990,25092,25194,25296,25398,25500,25602,25704,25806,25908,26010,26112,26214,26316,26418,26520,26622,26724,26826,26928,27030,27132,27234,27336,27438,27540,27642,27744,27846,27948,28050,28152,28254,28356,28458,28560,28662,28764,28866,28968,29070,29172,29274,29376,29478,29580,29682,29784,29886,29988,30090,30192,30294,30396,30498,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10815,10918,11021,11124,11227,11330,11433,11536,11639,11742,11845,11948,12051,12154,12257,12360,12463,12566,12669,12772,12875,12978,13081,13184,13287,13390,13493,13596,13699,13802,13905,14008,14111,14214,14317,14420,14523,14626,14729,14832,14935,15038,15141,15244,15347,15450,15553,15656,15759,15862,15965,16068,16171,16274,16377,16480,16583,16686,16789,16892,16995,17098,17201,17304,17407,17510,17613,17716,17819,17922,18025,18128,18231,18334,18437,18540,18643,18746,18849,18952,19055,19158,19261,19364,19467,19570,19673,19776,19879,19982,20085,20188,20291,20394,20497,20600,20703,20806,20909,21012,21115,21218,21321,21424,21527,21630,21733,21836,21939,22042,22145,22248,22351,22454,22557,22660,22763,22866,22969,23072,23175,23278,23381,23484,23587,23690,23793,23896,23999,24102,24205,24308,24411,24514,24617,24720,24823,24926,25029,25132,25235,25338,25441,25544,25647,25750,25853,25956,26059,26162,26265,26368,26471,26574,26677,26780,26883,26986,27089,27192,27295,27398,27501,27604,27707,27810,27913,28016,28119,28222,28325,28428,28531,28634,28737,28840,28943,29046,29149,29252,29355,29458,29561,29664,29767,29870,29973,30076,30179,30282,30385,30488,30591,30694,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10815,10918,11021,11124,11227,11330,11433,11536,11639,11742,11845,11948,12051,12154,12257,12360,12463,12566,12669,12772,12875,12978,13081,13184,13287,13390,13493,13596,13699,13802,13905,14008,14111,14214,14317,14420,14523,14626,14729,14832,14935,15038,15141,15244,15347,15450,15553,15656,15759,15862,15965,16068,16171,16274,16377,16480,16583,16686,16789,16892,16995,17098,17201,17304,17407,17510,17613,17716,17819,17922,18025,18128,18231,18334,18437,18540,18643,18746,18849,18952,19055,19158,19261,19364,19467,19570,19673,19776,19879,19982,20085,20188,20291,20394,20497,20600,20703,20806,20909,21012,21115,21218,21321,21424,21527,21630,21733,21836,21939,22042,22145,22248,22351,22454,22557,22660,22763,22866,22969,23072,23175,23278,23381,23484,23587,23690,23793,23896,23999,24102,24205,24308,24411,24514,24617,24720,24823,24926,25029,25132,25235,25338,25441,25544,25647,25750,25853,25956,26059,26162,26265,26368,26471,26574,26677,26780,26883,26986,27089,27192,27295,27398,27501,27604,27707,27810,27913,28016,28119,28222,28325,28428,28531,28634,28737,28840,28943,29046,29149,29252,29355,29458,29561,29664,29767,29870,29973,30076,30179,30282,30385,30488,30591,30694,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11024,11128,11232,11336,11440,11544,11648,11752,11856,11960,12064,12168,12272,12376,12480,12584,12688,12792,12896,13000,13104,13208,13312,13416,13520,13624,13728,13832,13936,14040,14144,14248,14352,14456,14560,14664,14768,14872,14976,15080,15184,15288,15392,15496,15600,15704,15808,15912,16016,16120,16224,16328,16432,16536,16640,16744,16848,16952,17056,17160,17264,17368,17472,17576,17680,17784,17888,17992,18096,18200,18304,18408,18512,18616,18720,18824,18928,19032,19136,19240,19344,19448,19552,19656,19760,19864,19968,20072,20176,20280,20384,20488,20592,20696,20800,20904,21008,21112,21216,21320,21424,21528,21632,21736,21840,21944,22048,22152,22256,22360,22464,22568,22672,22776,22880,22984,23088,23192,23296,23400,23504,23608,23712,23816,23920,24024,24128,24232,24336,24440,24544,24648,24752,24856,24960,25064,25168,25272,25376,25480,25584,25688,25792,25896,26000,26104,26208,26312,26416,26520,26624,26728,26832,26936,27040,27144,27248,27352,27456,27560,27664,27768,27872,27976,28080,28184,28288,28392,28496,28600,28704,28808,28912,29016,29120,29224,29328,29432,29536,29640,29744,29848,29952,30056,30160,30264,30368,30472,30576,30680,30784,30888,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11024,11128,11232,11336,11440,11544,11648,11752,11856,11960,12064,12168,12272,12376,12480,12584,12688,12792,12896,13000,13104,13208,13312,13416,13520,13624,13728,13832,13936,14040,14144,14248,14352,14456,14560,14664,14768,14872,14976,15080,15184,15288,15392,15496,15600,15704,15808,15912,16016,16120,16224,16328,16432,16536,16640,16744,16848,16952,17056,17160,17264,17368,17472,17576,17680,17784,17888,17992,18096,18200,18304,18408,18512,18616,18720,18824,18928,19032,19136,19240,19344,19448,19552,19656,19760,19864,19968,20072,20176,20280,20384,20488,20592,20696,20800,20904,21008,21112,21216,21320,21424,21528,21632,21736,21840,21944,22048,22152,22256,22360,22464,22568,22672,22776,22880,22984,23088,23192,23296,23400,23504,23608,23712,23816,23920,24024,24128,24232,24336,24440,24544,24648,24752,24856,24960,25064,25168,25272,25376,25480,25584,25688,25792,25896,26000,26104,26208,26312,26416,26520,26624,26728,26832,26936,27040,27144,27248,27352,27456,27560,27664,27768,27872,27976,28080,28184,28288,28392,28496,28600,28704,28808,28912,29016,29120,29224,29328,29432,29536,29640,29744,29848,29952,30056,30160,30264,30368,30472,30576,30680,30784,30888,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11235,11340,11445,11550,11655,11760,11865,11970,12075,12180,12285,12390,12495,12600,12705,12810,12915,13020,13125,13230,13335,13440,13545,13650,13755,13860,13965,14070,14175,14280,14385,14490,14595,14700,14805,14910,15015,15120,15225,15330,15435,15540,15645,15750,15855,15960,16065,16170,16275,16380,16485,16590,16695,16800,16905,17010,17115,17220,17325,17430,17535,17640,17745,17850,17955,18060,18165,18270,18375,18480,18585,18690,18795,18900,19005,19110,19215,19320,19425,19530,19635,19740,19845,19950,20055,20160,20265,20370,20475,20580,20685,20790,20895,21000,21105,21210,21315,21420,21525,21630,21735,21840,21945,22050,22155,22260,22365,22470,22575,22680,22785,22890,22995,23100,23205,23310,23415,23520,23625,23730,23835,23940,24045,24150,24255,24360,24465,24570,24675,24780,24885,24990,25095,25200,25305,25410,25515,25620,25725,25830,25935,26040,26145,26250,26355,26460,26565,26670,26775,26880,26985,27090,27195,27300,27405,27510,27615,27720,27825,27930,28035,28140,28245,28350,28455,28560,28665,28770,28875,28980,29085,29190,29295,29400,29505,29610,29715,29820,29925,30030,30135,30240,30345,30450,30555,30660,30765,30870,30975,31080,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11235,11340,11445,11550,11655,11760,11865,11970,12075,12180,12285,12390,12495,12600,12705,12810,12915,13020,13125,13230,13335,13440,13545,13650,13755,13860,13965,14070,14175,14280,14385,14490,14595,14700,14805,14910,15015,15120,15225,15330,15435,15540,15645,15750,15855,15960,16065,16170,16275,16380,16485,16590,16695,16800,16905,17010,17115,17220,17325,17430,17535,17640,17745,17850,17955,18060,18165,18270,18375,18480,18585,18690,18795,18900,19005,19110,19215,19320,19425,19530,19635,19740,19845,19950,20055,20160,20265,20370,20475,20580,20685,20790,20895,21000,21105,21210,21315,21420,21525,21630,21735,21840,21945,22050,22155,22260,22365,22470,22575,22680,22785,22890,22995,23100,23205,23310,23415,23520,23625,23730,23835,23940,24045,24150,24255,24360,24465,24570,24675,24780,24885,24990,25095,25200,25305,25410,25515,25620,25725,25830,25935,26040,26145,26250,26355,26460,26565,26670,26775,26880,26985,27090,27195,27300,27405,27510,27615,27720,27825,27930,28035,28140,28245,28350,28455,28560,28665,28770,28875,28980,29085,29190,29295,29400,29505,29610,29715,29820,29925,30030,30135,30240,30345,30450,30555,30660,30765,30870,30975,31080,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11448,11554,11660,11766,11872,11978,12084,12190,12296,12402,12508,12614,12720,12826,12932,13038,13144,13250,13356,13462,13568,13674,13780,13886,13992,14098,14204,14310,14416,14522,14628,14734,14840,14946,15052,15158,15264,15370,15476,15582,15688,15794,15900,16006,16112,16218,16324,16430,16536,16642,16748,16854,16960,17066,17172,17278,17384,17490,17596,17702,17808,17914,18020,18126,18232,18338,18444,18550,18656,18762,18868,18974,19080,19186,19292,19398,19504,19610,19716,19822,19928,20034,20140,20246,20352,20458,20564,20670,20776,20882,20988,21094,21200,21306,21412,21518,21624,21730,21836,21942,22048,22154,22260,22366,22472,22578,22684,22790,22896,23002,23108,23214,23320,23426,23532,23638,23744,23850,23956,24062,24168,24274,24380,24486,24592,24698,24804,24910,25016,25122,25228,25334,25440,25546,25652,25758,25864,25970,26076,26182,26288,26394,26500,26606,26712,26818,26924,27030,27136,27242,27348,27454,27560,27666,27772,27878,27984,28090,28196,28302,28408,28514,28620,28726,28832,28938,29044,29150,29256,29362,29468,29574,29680,29786,29892,29998,30104,30210,30316,30422,30528,30634,30740,30846,30952,31058,31164,31270,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11448,11554,11660,11766,11872,11978,12084,12190,12296,12402,12508,12614,12720,12826,12932,13038,13144,13250,13356,13462,13568,13674,13780,13886,13992,14098,14204,14310,14416,14522,14628,14734,14840,14946,15052,15158,15264,15370,15476,15582,15688,15794,15900,16006,16112,16218,16324,16430,16536,16642,16748,16854,16960,17066,17172,17278,17384,17490,17596,17702,17808,17914,18020,18126,18232,18338,18444,18550,18656,18762,18868,18974,19080,19186,19292,19398,19504,19610,19716,19822,19928,20034,20140,20246,20352,20458,20564,20670,20776,20882,20988,21094,21200,21306,21412,21518,21624,21730,21836,21942,22048,22154,22260,22366,22472,22578,22684,22790,22896,23002,23108,23214,23320,23426,23532,23638,23744,23850,23956,24062,24168,24274,24380,24486,24592,24698,24804,24910,25016,25122,25228,25334,25440,25546,25652,25758,25864,25970,26076,26182,26288,26394,26500,26606,26712,26818,26924,27030,27136,27242,27348,27454,27560,27666,27772,27878,27984,28090,28196,28302,28408,28514,28620,28726,28832,28938,29044,29150,29256,29362,29468,29574,29680,29786,29892,29998,30104,30210,30316,30422,30528,30634,30740,30846,30952,31058,31164,31270,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11663,11770,11877,11984,12091,12198,12305,12412,12519,12626,12733,12840,12947,13054,13161,13268,13375,13482,13589,13696,13803,13910,14017,14124,14231,14338,14445,14552,14659,14766,14873,14980,15087,15194,15301,15408,15515,15622,15729,15836,15943,16050,16157,16264,16371,16478,16585,16692,16799,16906,17013,17120,17227,17334,17441,17548,17655,17762,17869,17976,18083,18190,18297,18404,18511,18618,18725,18832,18939,19046,19153,19260,19367,19474,19581,19688,19795,19902,20009,20116,20223,20330,20437,20544,20651,20758,20865,20972,21079,21186,21293,21400,21507,21614,21721,21828,21935,22042,22149,22256,22363,22470,22577,22684,22791,22898,23005,23112,23219,23326,23433,23540,23647,23754,23861,23968,24075,24182,24289,24396,24503,24610,24717,24824,24931,25038,25145,25252,25359,25466,25573,25680,25787,25894,26001,26108,26215,26322,26429,26536,26643,26750,26857,26964,27071,27178,27285,27392,27499,27606,27713,27820,27927,28034,28141,28248,28355,28462,28569,28676,28783,28890,28997,29104,29211,29318,29425,29532,29639,29746,29853,29960,30067,30174,30281,30388,30495,30602,30709,30816,30923,31030,31137,31244,31351,31458,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11663,11770,11877,11984,12091,12198,12305,12412,12519,12626,12733,12840,12947,13054,13161,13268,13375,13482,13589,13696,13803,13910,14017,14124,14231,14338,14445,14552,14659,14766,14873,14980,15087,15194,15301,15408,15515,15622,15729,15836,15943,16050,16157,16264,16371,16478,16585,16692,16799,16906,17013,17120,17227,17334,17441,17548,17655,17762,17869,17976,18083,18190,18297,18404,18511,18618,18725,18832,18939,19046,19153,19260,19367,19474,19581,19688,19795,19902,20009,20116,20223,20330,20437,20544,20651,20758,20865,20972,21079,21186,21293,21400,21507,21614,21721,21828,21935,22042,22149,22256,22363,22470,22577,22684,22791,22898,23005,23112,23219,23326,23433,23540,23647,23754,23861,23968,24075,24182,24289,24396,24503,24610,24717,24824,24931,25038,25145,25252,25359,25466,25573,25680,25787,25894,26001,26108,26215,26322,26429,26536,26643,26750,26857,26964,27071,27178,27285,27392,27499,27606,27713,27820,27927,28034,28141,28248,28355,28462,28569,28676,28783,28890,28997,29104,29211,29318,29425,29532,29639,29746,29853,29960,30067,30174,30281,30388,30495,30602,30709,30816,30923,31030,31137,31244,31351,31458,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11880,11988,12096,12204,12312,12420,12528,12636,12744,12852,12960,13068,13176,13284,13392,13500,13608,13716,13824,13932,14040,14148,14256,14364,14472,14580,14688,14796,14904,15012,15120,15228,15336,15444,15552,15660,15768,15876,15984,16092,16200,16308,16416,16524,16632,16740,16848,16956,17064,17172,17280,17388,17496,17604,17712,17820,17928,18036,18144,18252,18360,18468,18576,18684,18792,18900,19008,19116,19224,19332,19440,19548,19656,19764,19872,19980,20088,20196,20304,20412,20520,20628,20736,20844,20952,21060,21168,21276,21384,21492,21600,21708,21816,21924,22032,22140,22248,22356,22464,22572,22680,22788,22896,23004,23112,23220,23328,23436,23544,23652,23760,23868,23976,24084,24192,24300,24408,24516,24624,24732,24840,24948,25056,25164,25272,25380,25488,25596,25704,25812,25920,26028,26136,26244,26352,26460,26568,26676,26784,26892,27000,27108,27216,27324,27432,27540,27648,27756,27864,27972,28080,28188,28296,28404,28512,28620,28728,28836,28944,29052,29160,29268,29376,29484,29592,29700,29808,29916,30024,30132,30240,30348,30456,30564,30672,30780,30888,30996,31104,31212,31320,31428,31536,31644,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11880,11988,12096,12204,12312,12420,12528,12636,12744,12852,12960,13068,13176,13284,13392,13500,13608,13716,13824,13932,14040,14148,14256,14364,14472,14580,14688,14796,14904,15012,15120,15228,15336,15444,15552,15660,15768,15876,15984,16092,16200,16308,16416,16524,16632,16740,16848,16956,17064,17172,17280,17388,17496,17604,17712,17820,17928,18036,18144,18252,18360,18468,18576,18684,18792,18900,19008,19116,19224,19332,19440,19548,19656,19764,19872,19980,20088,20196,20304,20412,20520,20628,20736,20844,20952,21060,21168,21276,21384,21492,21600,21708,21816,21924,22032,22140,22248,22356,22464,22572,22680,22788,22896,23004,23112,23220,23328,23436,23544,23652,23760,23868,23976,24084,24192,24300,24408,24516,24624,24732,24840,24948,25056,25164,25272,25380,25488,25596,25704,25812,25920,26028,26136,26244,26352,26460,26568,26676,26784,26892,27000,27108,27216,27324,27432,27540,27648,27756,27864,27972,28080,28188,28296,28404,28512,28620,28728,28836,28944,29052,29160,29268,29376,29484,29592,29700,29808,29916,30024,30132,30240,30348,30456,30564,30672,30780,30888,30996,31104,31212,31320,31428,31536,31644,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12099,12208,12317,12426,12535,12644,12753,12862,12971,13080,13189,13298,13407,13516,13625,13734,13843,13952,14061,14170,14279,14388,14497,14606,14715,14824,14933,15042,15151,15260,15369,15478,15587,15696,15805,15914,16023,16132,16241,16350,16459,16568,16677,16786,16895,17004,17113,17222,17331,17440,17549,17658,17767,17876,17985,18094,18203,18312,18421,18530,18639,18748,18857,18966,19075,19184,19293,19402,19511,19620,19729,19838,19947,20056,20165,20274,20383,20492,20601,20710,20819,20928,21037,21146,21255,21364,21473,21582,21691,21800,21909,22018,22127,22236,22345,22454,22563,22672,22781,22890,22999,23108,23217,23326,23435,23544,23653,23762,23871,23980,24089,24198,24307,24416,24525,24634,24743,24852,24961,25070,25179,25288,25397,25506,25615,25724,25833,25942,26051,26160,26269,26378,26487,26596,26705,26814,26923,27032,27141,27250,27359,27468,27577,27686,27795,27904,28013,28122,28231,28340,28449,28558,28667,28776,28885,28994,29103,29212,29321,29430,29539,29648,29757,29866,29975,30084,30193,30302,30411,30520,30629,30738,30847,30956,31065,31174,31283,31392,31501,31610,31719,31828,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12099,12208,12317,12426,12535,12644,12753,12862,12971,13080,13189,13298,13407,13516,13625,13734,13843,13952,14061,14170,14279,14388,14497,14606,14715,14824,14933,15042,15151,15260,15369,15478,15587,15696,15805,15914,16023,16132,16241,16350,16459,16568,16677,16786,16895,17004,17113,17222,17331,17440,17549,17658,17767,17876,17985,18094,18203,18312,18421,18530,18639,18748,18857,18966,19075,19184,19293,19402,19511,19620,19729,19838,19947,20056,20165,20274,20383,20492,20601,20710,20819,20928,21037,21146,21255,21364,21473,21582,21691,21800,21909,22018,22127,22236,22345,22454,22563,22672,22781,22890,22999,23108,23217,23326,23435,23544,23653,23762,23871,23980,24089,24198,24307,24416,24525,24634,24743,24852,24961,25070,25179,25288,25397,25506,25615,25724,25833,25942,26051,26160,26269,26378,26487,26596,26705,26814,26923,27032,27141,27250,27359,27468,27577,27686,27795,27904,28013,28122,28231,28340,28449,28558,28667,28776,28885,28994,29103,29212,29321,29430,29539,29648,29757,29866,29975,30084,30193,30302,30411,30520,30629,30738,30847,30956,31065,31174,31283,31392,31501,31610,31719,31828,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12320,12430,12540,12650,12760,12870,12980,13090,13200,13310,13420,13530,13640,13750,13860,13970,14080,14190,14300,14410,14520,14630,14740,14850,14960,15070,15180,15290,15400,15510,15620,15730,15840,15950,16060,16170,16280,16390,16500,16610,16720,16830,16940,17050,17160,17270,17380,17490,17600,17710,17820,17930,18040,18150,18260,18370,18480,18590,18700,18810,18920,19030,19140,19250,19360,19470,19580,19690,19800,19910,20020,20130,20240,20350,20460,20570,20680,20790,20900,21010,21120,21230,21340,21450,21560,21670,21780,21890,22000,22110,22220,22330,22440,22550,22660,22770,22880,22990,23100,23210,23320,23430,23540,23650,23760,23870,23980,24090,24200,24310,24420,24530,24640,24750,24860,24970,25080,25190,25300,25410,25520,25630,25740,25850,25960,26070,26180,26290,26400,26510,26620,26730,26840,26950,27060,27170,27280,27390,27500,27610,27720,27830,27940,28050,28160,28270,28380,28490,28600,28710,28820,28930,29040,29150,29260,29370,29480,29590,29700,29810,29920,30030,30140,30250,30360,30470,30580,30690,30800,30910,31020,31130,31240,31350,31460,31570,31680,31790,31900,32010,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12320,12430,12540,12650,12760,12870,12980,13090,13200,13310,13420,13530,13640,13750,13860,13970,14080,14190,14300,14410,14520,14630,14740,14850,14960,15070,15180,15290,15400,15510,15620,15730,15840,15950,16060,16170,16280,16390,16500,16610,16720,16830,16940,17050,17160,17270,17380,17490,17600,17710,17820,17930,18040,18150,18260,18370,18480,18590,18700,18810,18920,19030,19140,19250,19360,19470,19580,19690,19800,19910,20020,20130,20240,20350,20460,20570,20680,20790,20900,21010,21120,21230,21340,21450,21560,21670,21780,21890,22000,22110,22220,22330,22440,22550,22660,22770,22880,22990,23100,23210,23320,23430,23540,23650,23760,23870,23980,24090,24200,24310,24420,24530,24640,24750,24860,24970,25080,25190,25300,25410,25520,25630,25740,25850,25960,26070,26180,26290,26400,26510,26620,26730,26840,26950,27060,27170,27280,27390,27500,27610,27720,27830,27940,28050,28160,28270,28380,28490,28600,28710,28820,28930,29040,29150,29260,29370,29480,29590,29700,29810,29920,30030,30140,30250,30360,30470,30580,30690,30800,30910,31020,31130,31240,31350,31460,31570,31680,31790,31900,32010,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12543,12654,12765,12876,12987,13098,13209,13320,13431,13542,13653,13764,13875,13986,14097,14208,14319,14430,14541,14652,14763,14874,14985,15096,15207,15318,15429,15540,15651,15762,15873,15984,16095,16206,16317,16428,16539,16650,16761,16872,16983,17094,17205,17316,17427,17538,17649,17760,17871,17982,18093,18204,18315,18426,18537,18648,18759,18870,18981,19092,19203,19314,19425,19536,19647,19758,19869,19980,20091,20202,20313,20424,20535,20646,20757,20868,20979,21090,21201,21312,21423,21534,21645,21756,21867,21978,22089,22200,22311,22422,22533,22644,22755,22866,22977,23088,23199,23310,23421,23532,23643,23754,23865,23976,24087,24198,24309,24420,24531,24642,24753,24864,24975,25086,25197,25308,25419,25530,25641,25752,25863,25974,26085,26196,26307,26418,26529,26640,26751,26862,26973,27084,27195,27306,27417,27528,27639,27750,27861,27972,28083,28194,28305,28416,28527,28638,28749,28860,28971,29082,29193,29304,29415,29526,29637,29748,29859,29970,30081,30192,30303,30414,30525,30636,30747,30858,30969,31080,31191,31302,31413,31524,31635,31746,31857,31968,32079,32190,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12543,12654,12765,12876,12987,13098,13209,13320,13431,13542,13653,13764,13875,13986,14097,14208,14319,14430,14541,14652,14763,14874,14985,15096,15207,15318,15429,15540,15651,15762,15873,15984,16095,16206,16317,16428,16539,16650,16761,16872,16983,17094,17205,17316,17427,17538,17649,17760,17871,17982,18093,18204,18315,18426,18537,18648,18759,18870,18981,19092,19203,19314,19425,19536,19647,19758,19869,19980,20091,20202,20313,20424,20535,20646,20757,20868,20979,21090,21201,21312,21423,21534,21645,21756,21867,21978,22089,22200,22311,22422,22533,22644,22755,22866,22977,23088,23199,23310,23421,23532,23643,23754,23865,23976,24087,24198,24309,24420,24531,24642,24753,24864,24975,25086,25197,25308,25419,25530,25641,25752,25863,25974,26085,26196,26307,26418,26529,26640,26751,26862,26973,27084,27195,27306,27417,27528,27639,27750,27861,27972,28083,28194,28305,28416,28527,28638,28749,28860,28971,29082,29193,29304,29415,29526,29637,29748,29859,29970,30081,30192,30303,30414,30525,30636,30747,30858,30969,31080,31191,31302,31413,31524,31635,31746,31857,31968,32079,32190,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12768,12880,12992,13104,13216,13328,13440,13552,13664,13776,13888,14000,14112,14224,14336,14448,14560,14672,14784,14896,15008,15120,15232,15344,15456,15568,15680,15792,15904,16016,16128,16240,16352,16464,16576,16688,16800,16912,17024,17136,17248,17360,17472,17584,17696,17808,17920,18032,18144,18256,18368,18480,18592,18704,18816,18928,19040,19152,19264,19376,19488,19600,19712,19824,19936,20048,20160,20272,20384,20496,20608,20720,20832,20944,21056,21168,21280,21392,21504,21616,21728,21840,21952,22064,22176,22288,22400,22512,22624,22736,22848,22960,23072,23184,23296,23408,23520,23632,23744,23856,23968,24080,24192,24304,24416,24528,24640,24752,24864,24976,25088,25200,25312,25424,25536,25648,25760,25872,25984,26096,26208,26320,26432,26544,26656,26768,26880,26992,27104,27216,27328,27440,27552,27664,27776,27888,28000,28112,28224,28336,28448,28560,28672,28784,28896,29008,29120,29232,29344,29456,29568,29680,29792,29904,30016,30128,30240,30352,30464,30576,30688,30800,30912,31024,31136,31248,31360,31472,31584,31696,31808,31920,32032,32144,32256,32368,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12768,12880,12992,13104,13216,13328,13440,13552,13664,13776,13888,14000,14112,14224,14336,14448,14560,14672,14784,14896,15008,15120,15232,15344,15456,15568,15680,15792,15904,16016,16128,16240,16352,16464,16576,16688,16800,16912,17024,17136,17248,17360,17472,17584,17696,17808,17920,18032,18144,18256,18368,18480,18592,18704,18816,18928,19040,19152,19264,19376,19488,19600,19712,19824,19936,20048,20160,20272,20384,20496,20608,20720,20832,20944,21056,21168,21280,21392,21504,21616,21728,21840,21952,22064,22176,22288,22400,22512,22624,22736,22848,22960,23072,23184,23296,23408,23520,23632,23744,23856,23968,24080,24192,24304,24416,24528,24640,24752,24864,24976,25088,25200,25312,25424,25536,25648,25760,25872,25984,26096,26208,26320,26432,26544,26656,26768,26880,26992,27104,27216,27328,27440,27552,27664,27776,27888,28000,28112,28224,28336,28448,28560,28672,28784,28896,29008,29120,29232,29344,29456,29568,29680,29792,29904,30016,30128,30240,30352,30464,30576,30688,30800,30912,31024,31136,31248,31360,31472,31584,31696,31808,31920,32032,32144,32256,32368,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12995,13108,13221,13334,13447,13560,13673,13786,13899,14012,14125,14238,14351,14464,14577,14690,14803,14916,15029,15142,15255,15368,15481,15594,15707,15820,15933,16046,16159,16272,16385,16498,16611,16724,16837,16950,17063,17176,17289,17402,17515,17628,17741,17854,17967,18080,18193,18306,18419,18532,18645,18758,18871,18984,19097,19210,19323,19436,19549,19662,19775,19888,20001,20114,20227,20340,20453,20566,20679,20792,20905,21018,21131,21244,21357,21470,21583,21696,21809,21922,22035,22148,22261,22374,22487,22600,22713,22826,22939,23052,23165,23278,23391,23504,23617,23730,23843,23956,24069,24182,24295,24408,24521,24634,24747,24860,24973,25086,25199,25312,25425,25538,25651,25764,25877,25990,26103,26216,26329,26442,26555,26668,26781,26894,27007,27120,27233,27346,27459,27572,27685,27798,27911,28024,28137,28250,28363,28476,28589,28702,28815,28928,29041,29154,29267,29380,29493,29606,29719,29832,29945,30058,30171,30284,30397,30510,30623,30736,30849,30962,31075,31188,31301,31414,31527,31640,31753,31866,31979,32092,32205,32318,32431,32544,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12995,13108,13221,13334,13447,13560,13673,13786,13899,14012,14125,14238,14351,14464,14577,14690,14803,14916,15029,15142,15255,15368,15481,15594,15707,15820,15933,16046,16159,16272,16385,16498,16611,16724,16837,16950,17063,17176,17289,17402,17515,17628,17741,17854,17967,18080,18193,18306,18419,18532,18645,18758,18871,18984,19097,19210,19323,19436,19549,19662,19775,19888,20001,20114,20227,20340,20453,20566,20679,20792,20905,21018,21131,21244,21357,21470,21583,21696,21809,21922,22035,22148,22261,22374,22487,22600,22713,22826,22939,23052,23165,23278,23391,23504,23617,23730,23843,23956,24069,24182,24295,24408,24521,24634,24747,24860,24973,25086,25199,25312,25425,25538,25651,25764,25877,25990,26103,26216,26329,26442,26555,26668,26781,26894,27007,27120,27233,27346,27459,27572,27685,27798,27911,28024,28137,28250,28363,28476,28589,28702,28815,28928,29041,29154,29267,29380,29493,29606,29719,29832,29945,30058,30171,30284,30397,30510,30623,30736,30849,30962,31075,31188,31301,31414,31527,31640,31753,31866,31979,32092,32205,32318,32431,32544,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13224,13338,13452,13566,13680,13794,13908,14022,14136,14250,14364,14478,14592,14706,14820,14934,15048,15162,15276,15390,15504,15618,15732,15846,15960,16074,16188,16302,16416,16530,16644,16758,16872,16986,17100,17214,17328,17442,17556,17670,17784,17898,18012,18126,18240,18354,18468,18582,18696,18810,18924,19038,19152,19266,19380,19494,19608,19722,19836,19950,20064,20178,20292,20406,20520,20634,20748,20862,20976,21090,21204,21318,21432,21546,21660,21774,21888,22002,22116,22230,22344,22458,22572,22686,22800,22914,23028,23142,23256,23370,23484,23598,23712,23826,23940,24054,24168,24282,24396,24510,24624,24738,24852,24966,25080,25194,25308,25422,25536,25650,25764,25878,25992,26106,26220,26334,26448,26562,26676,26790,26904,27018,27132,27246,27360,27474,27588,27702,27816,27930,28044,28158,28272,28386,28500,28614,28728,28842,28956,29070,29184,29298,29412,29526,29640,29754,29868,29982,30096,30210,30324,30438,30552,30666,30780,30894,31008,31122,31236,31350,31464,31578,31692,31806,31920,32034,32148,32262,32376,32490,32604,32718,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13224,13338,13452,13566,13680,13794,13908,14022,14136,14250,14364,14478,14592,14706,14820,14934,15048,15162,15276,15390,15504,15618,15732,15846,15960,16074,16188,16302,16416,16530,16644,16758,16872,16986,17100,17214,17328,17442,17556,17670,17784,17898,18012,18126,18240,18354,18468,18582,18696,18810,18924,19038,19152,19266,19380,19494,19608,19722,19836,19950,20064,20178,20292,20406,20520,20634,20748,20862,20976,21090,21204,21318,21432,21546,21660,21774,21888,22002,22116,22230,22344,22458,22572,22686,22800,22914,23028,23142,23256,23370,23484,23598,23712,23826,23940,24054,24168,24282,24396,24510,24624,24738,24852,24966,25080,25194,25308,25422,25536,25650,25764,25878,25992,26106,26220,26334,26448,26562,26676,26790,26904,27018,27132,27246,27360,27474,27588,27702,27816,27930,28044,28158,28272,28386,28500,28614,28728,28842,28956,29070,29184,29298,29412,29526,29640,29754,29868,29982,30096,30210,30324,30438,30552,30666,30780,30894,31008,31122,31236,31350,31464,31578,31692,31806,31920,32034,32148,32262,32376,32490,32604,32718,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13455,13570,13685,13800,13915,14030,14145,14260,14375,14490,14605,14720,14835,14950,15065,15180,15295,15410,15525,15640,15755,15870,15985,16100,16215,16330,16445,16560,16675,16790,16905,17020,17135,17250,17365,17480,17595,17710,17825,17940,18055,18170,18285,18400,18515,18630,18745,18860,18975,19090,19205,19320,19435,19550,19665,19780,19895,20010,20125,20240,20355,20470,20585,20700,20815,20930,21045,21160,21275,21390,21505,21620,21735,21850,21965,22080,22195,22310,22425,22540,22655,22770,22885,23000,23115,23230,23345,23460,23575,23690,23805,23920,24035,24150,24265,24380,24495,24610,24725,24840,24955,25070,25185,25300,25415,25530,25645,25760,25875,25990,26105,26220,26335,26450,26565,26680,26795,26910,27025,27140,27255,27370,27485,27600,27715,27830,27945,28060,28175,28290,28405,28520,28635,28750,28865,28980,29095,29210,29325,29440,29555,29670,29785,29900,30015,30130,30245,30360,30475,30590,30705,30820,30935,31050,31165,31280,31395,31510,31625,31740,31855,31970,32085,32200,32315,32430,32545,32660,32775,32890,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13455,13570,13685,13800,13915,14030,14145,14260,14375,14490,14605,14720,14835,14950,15065,15180,15295,15410,15525,15640,15755,15870,15985,16100,16215,16330,16445,16560,16675,16790,16905,17020,17135,17250,17365,17480,17595,17710,17825,17940,18055,18170,18285,18400,18515,18630,18745,18860,18975,19090,19205,19320,19435,19550,19665,19780,19895,20010,20125,20240,20355,20470,20585,20700,20815,20930,21045,21160,21275,21390,21505,21620,21735,21850,21965,22080,22195,22310,22425,22540,22655,22770,22885,23000,23115,23230,23345,23460,23575,23690,23805,23920,24035,24150,24265,24380,24495,24610,24725,24840,24955,25070,25185,25300,25415,25530,25645,25760,25875,25990,26105,26220,26335,26450,26565,26680,26795,26910,27025,27140,27255,27370,27485,27600,27715,27830,27945,28060,28175,28290,28405,28520,28635,28750,28865,28980,29095,29210,29325,29440,29555,29670,29785,29900,30015,30130,30245,30360,30475,30590,30705,30820,30935,31050,31165,31280,31395,31510,31625,31740,31855,31970,32085,32200,32315,32430,32545,32660,32775,32890,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13688,13804,13920,14036,14152,14268,14384,14500,14616,14732,14848,14964,15080,15196,15312,15428,15544,15660,15776,15892,16008,16124,16240,16356,16472,16588,16704,16820,16936,17052,17168,17284,17400,17516,17632,17748,17864,17980,18096,18212,18328,18444,18560,18676,18792,18908,19024,19140,19256,19372,19488,19604,19720,19836,19952,20068,20184,20300,20416,20532,20648,20764,20880,20996,21112,21228,21344,21460,21576,21692,21808,21924,22040,22156,22272,22388,22504,22620,22736,22852,22968,23084,23200,23316,23432,23548,23664,23780,23896,24012,24128,24244,24360,24476,24592,24708,24824,24940,25056,25172,25288,25404,25520,25636,25752,25868,25984,26100,26216,26332,26448,26564,26680,26796,26912,27028,27144,27260,27376,27492,27608,27724,27840,27956,28072,28188,28304,28420,28536,28652,28768,28884,29000,29116,29232,29348,29464,29580,29696,29812,29928,30044,30160,30276,30392,30508,30624,30740,30856,30972,31088,31204,31320,31436,31552,31668,31784,31900,32016,32132,32248,32364,32480,32596,32712,32828,32944,33060,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13688,13804,13920,14036,14152,14268,14384,14500,14616,14732,14848,14964,15080,15196,15312,15428,15544,15660,15776,15892,16008,16124,16240,16356,16472,16588,16704,16820,16936,17052,17168,17284,17400,17516,17632,17748,17864,17980,18096,18212,18328,18444,18560,18676,18792,18908,19024,19140,19256,19372,19488,19604,19720,19836,19952,20068,20184,20300,20416,20532,20648,20764,20880,20996,21112,21228,21344,21460,21576,21692,21808,21924,22040,22156,22272,22388,22504,22620,22736,22852,22968,23084,23200,23316,23432,23548,23664,23780,23896,24012,24128,24244,24360,24476,24592,24708,24824,24940,25056,25172,25288,25404,25520,25636,25752,25868,25984,26100,26216,26332,26448,26564,26680,26796,26912,27028,27144,27260,27376,27492,27608,27724,27840,27956,28072,28188,28304,28420,28536,28652,28768,28884,29000,29116,29232,29348,29464,29580,29696,29812,29928,30044,30160,30276,30392,30508,30624,30740,30856,30972,31088,31204,31320,31436,31552,31668,31784,31900,32016,32132,32248,32364,32480,32596,32712,32828,32944,33060,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13923,14040,14157,14274,14391,14508,14625,14742,14859,14976,15093,15210,15327,15444,15561,15678,15795,15912,16029,16146,16263,16380,16497,16614,16731,16848,16965,17082,17199,17316,17433,17550,17667,17784,17901,18018,18135,18252,18369,18486,18603,18720,18837,18954,19071,19188,19305,19422,19539,19656,19773,19890,20007,20124,20241,20358,20475,20592,20709,20826,20943,21060,21177,21294,21411,21528,21645,21762,21879,21996,22113,22230,22347,22464,22581,22698,22815,22932,23049,23166,23283,23400,23517,23634,23751,23868,23985,24102,24219,24336,24453,24570,24687,24804,24921,25038,25155,25272,25389,25506,25623,25740,25857,25974,26091,26208,26325,26442,26559,26676,26793,26910,27027,27144,27261,27378,27495,27612,27729,27846,27963,28080,28197,28314,28431,28548,28665,28782,28899,29016,29133,29250,29367,29484,29601,29718,29835,29952,30069,30186,30303,30420,30537,30654,30771,30888,31005,31122,31239,31356,31473,31590,31707,31824,31941,32058,32175,32292,32409,32526,32643,32760,32877,32994,33111,33228,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13923,14040,14157,14274,14391,14508,14625,14742,14859,14976,15093,15210,15327,15444,15561,15678,15795,15912,16029,16146,16263,16380,16497,16614,16731,16848,16965,17082,17199,17316,17433,17550,17667,17784,17901,18018,18135,18252,18369,18486,18603,18720,18837,18954,19071,19188,19305,19422,19539,19656,19773,19890,20007,20124,20241,20358,20475,20592,20709,20826,20943,21060,21177,21294,21411,21528,21645,21762,21879,21996,22113,22230,22347,22464,22581,22698,22815,22932,23049,23166,23283,23400,23517,23634,23751,23868,23985,24102,24219,24336,24453,24570,24687,24804,24921,25038,25155,25272,25389,25506,25623,25740,25857,25974,26091,26208,26325,26442,26559,26676,26793,26910,27027,27144,27261,27378,27495,27612,27729,27846,27963,28080,28197,28314,28431,28548,28665,28782,28899,29016,29133,29250,29367,29484,29601,29718,29835,29952,30069,30186,30303,30420,30537,30654,30771,30888,31005,31122,31239,31356,31473,31590,31707,31824,31941,32058,32175,32292,32409,32526,32643,32760,32877,32994,33111,33228,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14160,14278,14396,14514,14632,14750,14868,14986,15104,15222,15340,15458,15576,15694,15812,15930,16048,16166,16284,16402,16520,16638,16756,16874,16992,17110,17228,17346,17464,17582,17700,17818,17936,18054,18172,18290,18408,18526,18644,18762,18880,18998,19116,19234,19352,19470,19588,19706,19824,19942,20060,20178,20296,20414,20532,20650,20768,20886,21004,21122,21240,21358,21476,21594,21712,21830,21948,22066,22184,22302,22420,22538,22656,22774,22892,23010,23128,23246,23364,23482,23600,23718,23836,23954,24072,24190,24308,24426,24544,24662,24780,24898,25016,25134,25252,25370,25488,25606,25724,25842,25960,26078,26196,26314,26432,26550,26668,26786,26904,27022,27140,27258,27376,27494,27612,27730,27848,27966,28084,28202,28320,28438,28556,28674,28792,28910,29028,29146,29264,29382,29500,29618,29736,29854,29972,30090,30208,30326,30444,30562,30680,30798,30916,31034,31152,31270,31388,31506,31624,31742,31860,31978,32096,32214,32332,32450,32568,32686,32804,32922,33040,33158,33276,33394,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14160,14278,14396,14514,14632,14750,14868,14986,15104,15222,15340,15458,15576,15694,15812,15930,16048,16166,16284,16402,16520,16638,16756,16874,16992,17110,17228,17346,17464,17582,17700,17818,17936,18054,18172,18290,18408,18526,18644,18762,18880,18998,19116,19234,19352,19470,19588,19706,19824,19942,20060,20178,20296,20414,20532,20650,20768,20886,21004,21122,21240,21358,21476,21594,21712,21830,21948,22066,22184,22302,22420,22538,22656,22774,22892,23010,23128,23246,23364,23482,23600,23718,23836,23954,24072,24190,24308,24426,24544,24662,24780,24898,25016,25134,25252,25370,25488,25606,25724,25842,25960,26078,26196,26314,26432,26550,26668,26786,26904,27022,27140,27258,27376,27494,27612,27730,27848,27966,28084,28202,28320,28438,28556,28674,28792,28910,29028,29146,29264,29382,29500,29618,29736,29854,29972,30090,30208,30326,30444,30562,30680,30798,30916,31034,31152,31270,31388,31506,31624,31742,31860,31978,32096,32214,32332,32450,32568,32686,32804,32922,33040,33158,33276,33394,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14399,14518,14637,14756,14875,14994,15113,15232,15351,15470,15589,15708,15827,15946,16065,16184,16303,16422,16541,16660,16779,16898,17017,17136,17255,17374,17493,17612,17731,17850,17969,18088,18207,18326,18445,18564,18683,18802,18921,19040,19159,19278,19397,19516,19635,19754,19873,19992,20111,20230,20349,20468,20587,20706,20825,20944,21063,21182,21301,21420,21539,21658,21777,21896,22015,22134,22253,22372,22491,22610,22729,22848,22967,23086,23205,23324,23443,23562,23681,23800,23919,24038,24157,24276,24395,24514,24633,24752,24871,24990,25109,25228,25347,25466,25585,25704,25823,25942,26061,26180,26299,26418,26537,26656,26775,26894,27013,27132,27251,27370,27489,27608,27727,27846,27965,28084,28203,28322,28441,28560,28679,28798,28917,29036,29155,29274,29393,29512,29631,29750,29869,29988,30107,30226,30345,30464,30583,30702,30821,30940,31059,31178,31297,31416,31535,31654,31773,31892,32011,32130,32249,32368,32487,32606,32725,32844,32963,33082,33201,33320,33439,33558,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14399,14518,14637,14756,14875,14994,15113,15232,15351,15470,15589,15708,15827,15946,16065,16184,16303,16422,16541,16660,16779,16898,17017,17136,17255,17374,17493,17612,17731,17850,17969,18088,18207,18326,18445,18564,18683,18802,18921,19040,19159,19278,19397,19516,19635,19754,19873,19992,20111,20230,20349,20468,20587,20706,20825,20944,21063,21182,21301,21420,21539,21658,21777,21896,22015,22134,22253,22372,22491,22610,22729,22848,22967,23086,23205,23324,23443,23562,23681,23800,23919,24038,24157,24276,24395,24514,24633,24752,24871,24990,25109,25228,25347,25466,25585,25704,25823,25942,26061,26180,26299,26418,26537,26656,26775,26894,27013,27132,27251,27370,27489,27608,27727,27846,27965,28084,28203,28322,28441,28560,28679,28798,28917,29036,29155,29274,29393,29512,29631,29750,29869,29988,30107,30226,30345,30464,30583,30702,30821,30940,31059,31178,31297,31416,31535,31654,31773,31892,32011,32130,32249,32368,32487,32606,32725,32844,32963,33082,33201,33320,33439,33558,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14640,14760,14880,15000,15120,15240,15360,15480,15600,15720,15840,15960,16080,16200,16320,16440,16560,16680,16800,16920,17040,17160,17280,17400,17520,17640,17760,17880,18000,18120,18240,18360,18480,18600,18720,18840,18960,19080,19200,19320,19440,19560,19680,19800,19920,20040,20160,20280,20400,20520,20640,20760,20880,21000,21120,21240,21360,21480,21600,21720,21840,21960,22080,22200,22320,22440,22560,22680,22800,22920,23040,23160,23280,23400,23520,23640,23760,23880,24000,24120,24240,24360,24480,24600,24720,24840,24960,25080,25200,25320,25440,25560,25680,25800,25920,26040,26160,26280,26400,26520,26640,26760,26880,27000,27120,27240,27360,27480,27600,27720,27840,27960,28080,28200,28320,28440,28560,28680,28800,28920,29040,29160,29280,29400,29520,29640,29760,29880,30000,30120,30240,30360,30480,30600,30720,30840,30960,31080,31200,31320,31440,31560,31680,31800,31920,32040,32160,32280,32400,32520,32640,32760,32880,33000,33120,33240,33360,33480,33600,33720,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14640,14760,14880,15000,15120,15240,15360,15480,15600,15720,15840,15960,16080,16200,16320,16440,16560,16680,16800,16920,17040,17160,17280,17400,17520,17640,17760,17880,18000,18120,18240,18360,18480,18600,18720,18840,18960,19080,19200,19320,19440,19560,19680,19800,19920,20040,20160,20280,20400,20520,20640,20760,20880,21000,21120,21240,21360,21480,21600,21720,21840,21960,22080,22200,22320,22440,22560,22680,22800,22920,23040,23160,23280,23400,23520,23640,23760,23880,24000,24120,24240,24360,24480,24600,24720,24840,24960,25080,25200,25320,25440,25560,25680,25800,25920,26040,26160,26280,26400,26520,26640,26760,26880,27000,27120,27240,27360,27480,27600,27720,27840,27960,28080,28200,28320,28440,28560,28680,28800,28920,29040,29160,29280,29400,29520,29640,29760,29880,30000,30120,30240,30360,30480,30600,30720,30840,30960,31080,31200,31320,31440,31560,31680,31800,31920,32040,32160,32280,32400,32520,32640,32760,32880,33000,33120,33240,33360,33480,33600,33720,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14883,15004,15125,15246,15367,15488,15609,15730,15851,15972,16093,16214,16335,16456,16577,16698,16819,16940,17061,17182,17303,17424,17545,17666,17787,17908,18029,18150,18271,18392,18513,18634,18755,18876,18997,19118,19239,19360,19481,19602,19723,19844,19965,20086,20207,20328,20449,20570,20691,20812,20933,21054,21175,21296,21417,21538,21659,21780,21901,22022,22143,22264,22385,22506,22627,22748,22869,22990,23111,23232,23353,23474,23595,23716,23837,23958,24079,24200,24321,24442,24563,24684,24805,24926,25047,25168,25289,25410,25531,25652,25773,25894,26015,26136,26257,26378,26499,26620,26741,26862,26983,27104,27225,27346,27467,27588,27709,27830,27951,28072,28193,28314,28435,28556,28677,28798,28919,29040,29161,29282,29403,29524,29645,29766,29887,30008,30129,30250,30371,30492,30613,30734,30855,30976,31097,31218,31339,31460,31581,31702,31823,31944,32065,32186,32307,32428,32549,32670,32791,32912,33033,33154,33275,33396,33517,33638,33759,33880,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14883,15004,15125,15246,15367,15488,15609,15730,15851,15972,16093,16214,16335,16456,16577,16698,16819,16940,17061,17182,17303,17424,17545,17666,17787,17908,18029,18150,18271,18392,18513,18634,18755,18876,18997,19118,19239,19360,19481,19602,19723,19844,19965,20086,20207,20328,20449,20570,20691,20812,20933,21054,21175,21296,21417,21538,21659,21780,21901,22022,22143,22264,22385,22506,22627,22748,22869,22990,23111,23232,23353,23474,23595,23716,23837,23958,24079,24200,24321,24442,24563,24684,24805,24926,25047,25168,25289,25410,25531,25652,25773,25894,26015,26136,26257,26378,26499,26620,26741,26862,26983,27104,27225,27346,27467,27588,27709,27830,27951,28072,28193,28314,28435,28556,28677,28798,28919,29040,29161,29282,29403,29524,29645,29766,29887,30008,30129,30250,30371,30492,30613,30734,30855,30976,31097,31218,31339,31460,31581,31702,31823,31944,32065,32186,32307,32428,32549,32670,32791,32912,33033,33154,33275,33396,33517,33638,33759,33880,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15128,15250,15372,15494,15616,15738,15860,15982,16104,16226,16348,16470,16592,16714,16836,16958,17080,17202,17324,17446,17568,17690,17812,17934,18056,18178,18300,18422,18544,18666,18788,18910,19032,19154,19276,19398,19520,19642,19764,19886,20008,20130,20252,20374,20496,20618,20740,20862,20984,21106,21228,21350,21472,21594,21716,21838,21960,22082,22204,22326,22448,22570,22692,22814,22936,23058,23180,23302,23424,23546,23668,23790,23912,24034,24156,24278,24400,24522,24644,24766,24888,25010,25132,25254,25376,25498,25620,25742,25864,25986,26108,26230,26352,26474,26596,26718,26840,26962,27084,27206,27328,27450,27572,27694,27816,27938,28060,28182,28304,28426,28548,28670,28792,28914,29036,29158,29280,29402,29524,29646,29768,29890,30012,30134,30256,30378,30500,30622,30744,30866,30988,31110,31232,31354,31476,31598,31720,31842,31964,32086,32208,32330,32452,32574,32696,32818,32940,33062,33184,33306,33428,33550,33672,33794,33916,34038,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15128,15250,15372,15494,15616,15738,15860,15982,16104,16226,16348,16470,16592,16714,16836,16958,17080,17202,17324,17446,17568,17690,17812,17934,18056,18178,18300,18422,18544,18666,18788,18910,19032,19154,19276,19398,19520,19642,19764,19886,20008,20130,20252,20374,20496,20618,20740,20862,20984,21106,21228,21350,21472,21594,21716,21838,21960,22082,22204,22326,22448,22570,22692,22814,22936,23058,23180,23302,23424,23546,23668,23790,23912,24034,24156,24278,24400,24522,24644,24766,24888,25010,25132,25254,25376,25498,25620,25742,25864,25986,26108,26230,26352,26474,26596,26718,26840,26962,27084,27206,27328,27450,27572,27694,27816,27938,28060,28182,28304,28426,28548,28670,28792,28914,29036,29158,29280,29402,29524,29646,29768,29890,30012,30134,30256,30378,30500,30622,30744,30866,30988,31110,31232,31354,31476,31598,31720,31842,31964,32086,32208,32330,32452,32574,32696,32818,32940,33062,33184,33306,33428,33550,33672,33794,33916,34038,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15375,15498,15621,15744,15867,15990,16113,16236,16359,16482,16605,16728,16851,16974,17097,17220,17343,17466,17589,17712,17835,17958,18081,18204,18327,18450,18573,18696,18819,18942,19065,19188,19311,19434,19557,19680,19803,19926,20049,20172,20295,20418,20541,20664,20787,20910,21033,21156,21279,21402,21525,21648,21771,21894,22017,22140,22263,22386,22509,22632,22755,22878,23001,23124,23247,23370,23493,23616,23739,23862,23985,24108,24231,24354,24477,24600,24723,24846,24969,25092,25215,25338,25461,25584,25707,25830,25953,26076,26199,26322,26445,26568,26691,26814,26937,27060,27183,27306,27429,27552,27675,27798,27921,28044,28167,28290,28413,28536,28659,28782,28905,29028,29151,29274,29397,29520,29643,29766,29889,30012,30135,30258,30381,30504,30627,30750,30873,30996,31119,31242,31365,31488,31611,31734,31857,31980,32103,32226,32349,32472,32595,32718,32841,32964,33087,33210,33333,33456,33579,33702,33825,33948,34071,34194,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15375,15498,15621,15744,15867,15990,16113,16236,16359,16482,16605,16728,16851,16974,17097,17220,17343,17466,17589,17712,17835,17958,18081,18204,18327,18450,18573,18696,18819,18942,19065,19188,19311,19434,19557,19680,19803,19926,20049,20172,20295,20418,20541,20664,20787,20910,21033,21156,21279,21402,21525,21648,21771,21894,22017,22140,22263,22386,22509,22632,22755,22878,23001,23124,23247,23370,23493,23616,23739,23862,23985,24108,24231,24354,24477,24600,24723,24846,24969,25092,25215,25338,25461,25584,25707,25830,25953,26076,26199,26322,26445,26568,26691,26814,26937,27060,27183,27306,27429,27552,27675,27798,27921,28044,28167,28290,28413,28536,28659,28782,28905,29028,29151,29274,29397,29520,29643,29766,29889,30012,30135,30258,30381,30504,30627,30750,30873,30996,31119,31242,31365,31488,31611,31734,31857,31980,32103,32226,32349,32472,32595,32718,32841,32964,33087,33210,33333,33456,33579,33702,33825,33948,34071,34194,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15624,15748,15872,15996,16120,16244,16368,16492,16616,16740,16864,16988,17112,17236,17360,17484,17608,17732,17856,17980,18104,18228,18352,18476,18600,18724,18848,18972,19096,19220,19344,19468,19592,19716,19840,19964,20088,20212,20336,20460,20584,20708,20832,20956,21080,21204,21328,21452,21576,21700,21824,21948,22072,22196,22320,22444,22568,22692,22816,22940,23064,23188,23312,23436,23560,23684,23808,23932,24056,24180,24304,24428,24552,24676,24800,24924,25048,25172,25296,25420,25544,25668,25792,25916,26040,26164,26288,26412,26536,26660,26784,26908,27032,27156,27280,27404,27528,27652,27776,27900,28024,28148,28272,28396,28520,28644,28768,28892,29016,29140,29264,29388,29512,29636,29760,29884,30008,30132,30256,30380,30504,30628,30752,30876,31000,31124,31248,31372,31496,31620,31744,31868,31992,32116,32240,32364,32488,32612,32736,32860,32984,33108,33232,33356,33480,33604,33728,33852,33976,34100,34224,34348,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15624,15748,15872,15996,16120,16244,16368,16492,16616,16740,16864,16988,17112,17236,17360,17484,17608,17732,17856,17980,18104,18228,18352,18476,18600,18724,18848,18972,19096,19220,19344,19468,19592,19716,19840,19964,20088,20212,20336,20460,20584,20708,20832,20956,21080,21204,21328,21452,21576,21700,21824,21948,22072,22196,22320,22444,22568,22692,22816,22940,23064,23188,23312,23436,23560,23684,23808,23932,24056,24180,24304,24428,24552,24676,24800,24924,25048,25172,25296,25420,25544,25668,25792,25916,26040,26164,26288,26412,26536,26660,26784,26908,27032,27156,27280,27404,27528,27652,27776,27900,28024,28148,28272,28396,28520,28644,28768,28892,29016,29140,29264,29388,29512,29636,29760,29884,30008,30132,30256,30380,30504,30628,30752,30876,31000,31124,31248,31372,31496,31620,31744,31868,31992,32116,32240,32364,32488,32612,32736,32860,32984,33108,33232,33356,33480,33604,33728,33852,33976,34100,34224,34348,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15875,16000,16125,16250,16375,16500,16625,16750,16875,17000,17125,17250,17375,17500,17625,17750,17875,18000,18125,18250,18375,18500,18625,18750,18875,19000,19125,19250,19375,19500,19625,19750,19875,20000,20125,20250,20375,20500,20625,20750,20875,21000,21125,21250,21375,21500,21625,21750,21875,22000,22125,22250,22375,22500,22625,22750,22875,23000,23125,23250,23375,23500,23625,23750,23875,24000,24125,24250,24375,24500,24625,24750,24875,25000,25125,25250,25375,25500,25625,25750,25875,26000,26125,26250,26375,26500,26625,26750,26875,27000,27125,27250,27375,27500,27625,27750,27875,28000,28125,28250,28375,28500,28625,28750,28875,29000,29125,29250,29375,29500,29625,29750,29875,30000,30125,30250,30375,30500,30625,30750,30875,31000,31125,31250,31375,31500,31625,31750,31875,32000,32125,32250,32375,32500,32625,32750,32875,33000,33125,33250,33375,33500,33625,33750,33875,34000,34125,34250,34375,34500,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15875,16000,16125,16250,16375,16500,16625,16750,16875,17000,17125,17250,17375,17500,17625,17750,17875,18000,18125,18250,18375,18500,18625,18750,18875,19000,19125,19250,19375,19500,19625,19750,19875,20000,20125,20250,20375,20500,20625,20750,20875,21000,21125,21250,21375,21500,21625,21750,21875,22000,22125,22250,22375,22500,22625,22750,22875,23000,23125,23250,23375,23500,23625,23750,23875,24000,24125,24250,24375,24500,24625,24750,24875,25000,25125,25250,25375,25500,25625,25750,25875,26000,26125,26250,26375,26500,26625,26750,26875,27000,27125,27250,27375,27500,27625,27750,27875,28000,28125,28250,28375,28500,28625,28750,28875,29000,29125,29250,29375,29500,29625,29750,29875,30000,30125,30250,30375,30500,30625,30750,30875,31000,31125,31250,31375,31500,31625,31750,31875,32000,32125,32250,32375,32500,32625,32750,32875,33000,33125,33250,33375,33500,33625,33750,33875,34000,34125,34250,34375,34500,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16128,16254,16380,16506,16632,16758,16884,17010,17136,17262,17388,17514,17640,17766,17892,18018,18144,18270,18396,18522,18648,18774,18900,19026,19152,19278,19404,19530,19656,19782,19908,20034,20160,20286,20412,20538,20664,20790,20916,21042,21168,21294,21420,21546,21672,21798,21924,22050,22176,22302,22428,22554,22680,22806,22932,23058,23184,23310,23436,23562,23688,23814,23940,24066,24192,24318,24444,24570,24696,24822,24948,25074,25200,25326,25452,25578,25704,25830,25956,26082,26208,26334,26460,26586,26712,26838,26964,27090,27216,27342,27468,27594,27720,27846,27972,28098,28224,28350,28476,28602,28728,28854,28980,29106,29232,29358,29484,29610,29736,29862,29988,30114,30240,30366,30492,30618,30744,30870,30996,31122,31248,31374,31500,31626,31752,31878,32004,32130,32256,32382,32508,32634,32760,32886,33012,33138,33264,33390,33516,33642,33768,33894,34020,34146,34272,34398,34524,34650,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16128,16254,16380,16506,16632,16758,16884,17010,17136,17262,17388,17514,17640,17766,17892,18018,18144,18270,18396,18522,18648,18774,18900,19026,19152,19278,19404,19530,19656,19782,19908,20034,20160,20286,20412,20538,20664,20790,20916,21042,21168,21294,21420,21546,21672,21798,21924,22050,22176,22302,22428,22554,22680,22806,22932,23058,23184,23310,23436,23562,23688,23814,23940,24066,24192,24318,24444,24570,24696,24822,24948,25074,25200,25326,25452,25578,25704,25830,25956,26082,26208,26334,26460,26586,26712,26838,26964,27090,27216,27342,27468,27594,27720,27846,27972,28098,28224,28350,28476,28602,28728,28854,28980,29106,29232,29358,29484,29610,29736,29862,29988,30114,30240,30366,30492,30618,30744,30870,30996,31122,31248,31374,31500,31626,31752,31878,32004,32130,32256,32382,32508,32634,32760,32886,33012,33138,33264,33390,33516,33642,33768,33894,34020,34146,34272,34398,34524,34650,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16383,16510,16637,16764,16891,17018,17145,17272,17399,17526,17653,17780,17907,18034,18161,18288,18415,18542,18669,18796,18923,19050,19177,19304,19431,19558,19685,19812,19939,20066,20193,20320,20447,20574,20701,20828,20955,21082,21209,21336,21463,21590,21717,21844,21971,22098,22225,22352,22479,22606,22733,22860,22987,23114,23241,23368,23495,23622,23749,23876,24003,24130,24257,24384,24511,24638,24765,24892,25019,25146,25273,25400,25527,25654,25781,25908,26035,26162,26289,26416,26543,26670,26797,26924,27051,27178,27305,27432,27559,27686,27813,27940,28067,28194,28321,28448,28575,28702,28829,28956,29083,29210,29337,29464,29591,29718,29845,29972,30099,30226,30353,30480,30607,30734,30861,30988,31115,31242,31369,31496,31623,31750,31877,32004,32131,32258,32385,32512,32639,32766,32893,33020,33147,33274,33401,33528,33655,33782,33909,34036,34163,34290,34417,34544,34671,34798,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16383,16510,16637,16764,16891,17018,17145,17272,17399,17526,17653,17780,17907,18034,18161,18288,18415,18542,18669,18796,18923,19050,19177,19304,19431,19558,19685,19812,19939,20066,20193,20320,20447,20574,20701,20828,20955,21082,21209,21336,21463,21590,21717,21844,21971,22098,22225,22352,22479,22606,22733,22860,22987,23114,23241,23368,23495,23622,23749,23876,24003,24130,24257,24384,24511,24638,24765,24892,25019,25146,25273,25400,25527,25654,25781,25908,26035,26162,26289,26416,26543,26670,26797,26924,27051,27178,27305,27432,27559,27686,27813,27940,28067,28194,28321,28448,28575,28702,28829,28956,29083,29210,29337,29464,29591,29718,29845,29972,30099,30226,30353,30480,30607,30734,30861,30988,31115,31242,31369,31496,31623,31750,31877,32004,32131,32258,32385,32512,32639,32766,32893,33020,33147,33274,33401,33528,33655,33782,33909,34036,34163,34290,34417,34544,34671,34798,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16640,16768,16896,17024,17152,17280,17408,17536,17664,17792,17920,18048,18176,18304,18432,18560,18688,18816,18944,19072,19200,19328,19456,19584,19712,19840,19968,20096,20224,20352,20480,20608,20736,20864,20992,21120,21248,21376,21504,21632,21760,21888,22016,22144,22272,22400,22528,22656,22784,22912,23040,23168,23296,23424,23552,23680,23808,23936,24064,24192,24320,24448,24576,24704,24832,24960,25088,25216,25344,25472,25600,25728,25856,25984,26112,26240,26368,26496,26624,26752,26880,27008,27136,27264,27392,27520,27648,27776,27904,28032,28160,28288,28416,28544,28672,28800,28928,29056,29184,29312,29440,29568,29696,29824,29952,30080,30208,30336,30464,30592,30720,30848,30976,31104,31232,31360,31488,31616,31744,31872,32000,32128,32256,32384,32512,32640,32768,32896,33024,33152,33280,33408,33536,33664,33792,33920,34048,34176,34304,34432,34560,34688,34816,34944,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16640,16768,16896,17024,17152,17280,17408,17536,17664,17792,17920,18048,18176,18304,18432,18560,18688,18816,18944,19072,19200,19328,19456,19584,19712,19840,19968,20096,20224,20352,20480,20608,20736,20864,20992,21120,21248,21376,21504,21632,21760,21888,22016,22144,22272,22400,22528,22656,22784,22912,23040,23168,23296,23424,23552,23680,23808,23936,24064,24192,24320,24448,24576,24704,24832,24960,25088,25216,25344,25472,25600,25728,25856,25984,26112,26240,26368,26496,26624,26752,26880,27008,27136,27264,27392,27520,27648,27776,27904,28032,28160,28288,28416,28544,28672,28800,28928,29056,29184,29312,29440,29568,29696,29824,29952,30080,30208,30336,30464,30592,30720,30848,30976,31104,31232,31360,31488,31616,31744,31872,32000,32128,32256,32384,32512,32640,32768,32896,33024,33152,33280,33408,33536,33664,33792,33920,34048,34176,34304,34432,34560,34688,34816,34944,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16899,17028,17157,17286,17415,17544,17673,17802,17931,18060,18189,18318,18447,18576,18705,18834,18963,19092,19221,19350,19479,19608,19737,19866,19995,20124,20253,20382,20511,20640,20769,20898,21027,21156,21285,21414,21543,21672,21801,21930,22059,22188,22317,22446,22575,22704,22833,22962,23091,23220,23349,23478,23607,23736,23865,23994,24123,24252,24381,24510,24639,24768,24897,25026,25155,25284,25413,25542,25671,25800,25929,26058,26187,26316,26445,26574,26703,26832,26961,27090,27219,27348,27477,27606,27735,27864,27993,28122,28251,28380,28509,28638,28767,28896,29025,29154,29283,29412,29541,29670,29799,29928,30057,30186,30315,30444,30573,30702,30831,30960,31089,31218,31347,31476,31605,31734,31863,31992,32121,32250,32379,32508,32637,32766,32895,33024,33153,33282,33411,33540,33669,33798,33927,34056,34185,34314,34443,34572,34701,34830,34959,35088,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16899,17028,17157,17286,17415,17544,17673,17802,17931,18060,18189,18318,18447,18576,18705,18834,18963,19092,19221,19350,19479,19608,19737,19866,19995,20124,20253,20382,20511,20640,20769,20898,21027,21156,21285,21414,21543,21672,21801,21930,22059,22188,22317,22446,22575,22704,22833,22962,23091,23220,23349,23478,23607,23736,23865,23994,24123,24252,24381,24510,24639,24768,24897,25026,25155,25284,25413,25542,25671,25800,25929,26058,26187,26316,26445,26574,26703,26832,26961,27090,27219,27348,27477,27606,27735,27864,27993,28122,28251,28380,28509,28638,28767,28896,29025,29154,29283,29412,29541,29670,29799,29928,30057,30186,30315,30444,30573,30702,30831,30960,31089,31218,31347,31476,31605,31734,31863,31992,32121,32250,32379,32508,32637,32766,32895,33024,33153,33282,33411,33540,33669,33798,33927,34056,34185,34314,34443,34572,34701,34830,34959,35088,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17160,17290,17420,17550,17680,17810,17940,18070,18200,18330,18460,18590,18720,18850,18980,19110,19240,19370,19500,19630,19760,19890,20020,20150,20280,20410,20540,20670,20800,20930,21060,21190,21320,21450,21580,21710,21840,21970,22100,22230,22360,22490,22620,22750,22880,23010,23140,23270,23400,23530,23660,23790,23920,24050,24180,24310,24440,24570,24700,24830,24960,25090,25220,25350,25480,25610,25740,25870,26000,26130,26260,26390,26520,26650,26780,26910,27040,27170,27300,27430,27560,27690,27820,27950,28080,28210,28340,28470,28600,28730,28860,28990,29120,29250,29380,29510,29640,29770,29900,30030,30160,30290,30420,30550,30680,30810,30940,31070,31200,31330,31460,31590,31720,31850,31980,32110,32240,32370,32500,32630,32760,32890,33020,33150,33280,33410,33540,33670,33800,33930,34060,34190,34320,34450,34580,34710,34840,34970,35100,35230,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17160,17290,17420,17550,17680,17810,17940,18070,18200,18330,18460,18590,18720,18850,18980,19110,19240,19370,19500,19630,19760,19890,20020,20150,20280,20410,20540,20670,20800,20930,21060,21190,21320,21450,21580,21710,21840,21970,22100,22230,22360,22490,22620,22750,22880,23010,23140,23270,23400,23530,23660,23790,23920,24050,24180,24310,24440,24570,24700,24830,24960,25090,25220,25350,25480,25610,25740,25870,26000,26130,26260,26390,26520,26650,26780,26910,27040,27170,27300,27430,27560,27690,27820,27950,28080,28210,28340,28470,28600,28730,28860,28990,29120,29250,29380,29510,29640,29770,29900,30030,30160,30290,30420,30550,30680,30810,30940,31070,31200,31330,31460,31590,31720,31850,31980,32110,32240,32370,32500,32630,32760,32890,33020,33150,33280,33410,33540,33670,33800,33930,34060,34190,34320,34450,34580,34710,34840,34970,35100,35230,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17423,17554,17685,17816,17947,18078,18209,18340,18471,18602,18733,18864,18995,19126,19257,19388,19519,19650,19781,19912,20043,20174,20305,20436,20567,20698,20829,20960,21091,21222,21353,21484,21615,21746,21877,22008,22139,22270,22401,22532,22663,22794,22925,23056,23187,23318,23449,23580,23711,23842,23973,24104,24235,24366,24497,24628,24759,24890,25021,25152,25283,25414,25545,25676,25807,25938,26069,26200,26331,26462,26593,26724,26855,26986,27117,27248,27379,27510,27641,27772,27903,28034,28165,28296,28427,28558,28689,28820,28951,29082,29213,29344,29475,29606,29737,29868,29999,30130,30261,30392,30523,30654,30785,30916,31047,31178,31309,31440,31571,31702,31833,31964,32095,32226,32357,32488,32619,32750,32881,33012,33143,33274,33405,33536,33667,33798,33929,34060,34191,34322,34453,34584,34715,34846,34977,35108,35239,35370,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17423,17554,17685,17816,17947,18078,18209,18340,18471,18602,18733,18864,18995,19126,19257,19388,19519,19650,19781,19912,20043,20174,20305,20436,20567,20698,20829,20960,21091,21222,21353,21484,21615,21746,21877,22008,22139,22270,22401,22532,22663,22794,22925,23056,23187,23318,23449,23580,23711,23842,23973,24104,24235,24366,24497,24628,24759,24890,25021,25152,25283,25414,25545,25676,25807,25938,26069,26200,26331,26462,26593,26724,26855,26986,27117,27248,27379,27510,27641,27772,27903,28034,28165,28296,28427,28558,28689,28820,28951,29082,29213,29344,29475,29606,29737,29868,29999,30130,30261,30392,30523,30654,30785,30916,31047,31178,31309,31440,31571,31702,31833,31964,32095,32226,32357,32488,32619,32750,32881,33012,33143,33274,33405,33536,33667,33798,33929,34060,34191,34322,34453,34584,34715,34846,34977,35108,35239,35370,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17688,17820,17952,18084,18216,18348,18480,18612,18744,18876,19008,19140,19272,19404,19536,19668,19800,19932,20064,20196,20328,20460,20592,20724,20856,20988,21120,21252,21384,21516,21648,21780,21912,22044,22176,22308,22440,22572,22704,22836,22968,23100,23232,23364,23496,23628,23760,23892,24024,24156,24288,24420,24552,24684,24816,24948,25080,25212,25344,25476,25608,25740,25872,26004,26136,26268,26400,26532,26664,26796,26928,27060,27192,27324,27456,27588,27720,27852,27984,28116,28248,28380,28512,28644,28776,28908,29040,29172,29304,29436,29568,29700,29832,29964,30096,30228,30360,30492,30624,30756,30888,31020,31152,31284,31416,31548,31680,31812,31944,32076,32208,32340,32472,32604,32736,32868,33000,33132,33264,33396,33528,33660,33792,33924,34056,34188,34320,34452,34584,34716,34848,34980,35112,35244,35376,35508,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17688,17820,17952,18084,18216,18348,18480,18612,18744,18876,19008,19140,19272,19404,19536,19668,19800,19932,20064,20196,20328,20460,20592,20724,20856,20988,21120,21252,21384,21516,21648,21780,21912,22044,22176,22308,22440,22572,22704,22836,22968,23100,23232,23364,23496,23628,23760,23892,24024,24156,24288,24420,24552,24684,24816,24948,25080,25212,25344,25476,25608,25740,25872,26004,26136,26268,26400,26532,26664,26796,26928,27060,27192,27324,27456,27588,27720,27852,27984,28116,28248,28380,28512,28644,28776,28908,29040,29172,29304,29436,29568,29700,29832,29964,30096,30228,30360,30492,30624,30756,30888,31020,31152,31284,31416,31548,31680,31812,31944,32076,32208,32340,32472,32604,32736,32868,33000,33132,33264,33396,33528,33660,33792,33924,34056,34188,34320,34452,34584,34716,34848,34980,35112,35244,35376,35508,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17955,18088,18221,18354,18487,18620,18753,18886,19019,19152,19285,19418,19551,19684,19817,19950,20083,20216,20349,20482,20615,20748,20881,21014,21147,21280,21413,21546,21679,21812,21945,22078,22211,22344,22477,22610,22743,22876,23009,23142,23275,23408,23541,23674,23807,23940,24073,24206,24339,24472,24605,24738,24871,25004,25137,25270,25403,25536,25669,25802,25935,26068,26201,26334,26467,26600,26733,26866,26999,27132,27265,27398,27531,27664,27797,27930,28063,28196,28329,28462,28595,28728,28861,28994,29127,29260,29393,29526,29659,29792,29925,30058,30191,30324,30457,30590,30723,30856,30989,31122,31255,31388,31521,31654,31787,31920,32053,32186,32319,32452,32585,32718,32851,32984,33117,33250,33383,33516,33649,33782,33915,34048,34181,34314,34447,34580,34713,34846,34979,35112,35245,35378,35511,35644,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17955,18088,18221,18354,18487,18620,18753,18886,19019,19152,19285,19418,19551,19684,19817,19950,20083,20216,20349,20482,20615,20748,20881,21014,21147,21280,21413,21546,21679,21812,21945,22078,22211,22344,22477,22610,22743,22876,23009,23142,23275,23408,23541,23674,23807,23940,24073,24206,24339,24472,24605,24738,24871,25004,25137,25270,25403,25536,25669,25802,25935,26068,26201,26334,26467,26600,26733,26866,26999,27132,27265,27398,27531,27664,27797,27930,28063,28196,28329,28462,28595,28728,28861,28994,29127,29260,29393,29526,29659,29792,29925,30058,30191,30324,30457,30590,30723,30856,30989,31122,31255,31388,31521,31654,31787,31920,32053,32186,32319,32452,32585,32718,32851,32984,33117,33250,33383,33516,33649,33782,33915,34048,34181,34314,34447,34580,34713,34846,34979,35112,35245,35378,35511,35644,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18224,18358,18492,18626,18760,18894,19028,19162,19296,19430,19564,19698,19832,19966,20100,20234,20368,20502,20636,20770,20904,21038,21172,21306,21440,21574,21708,21842,21976,22110,22244,22378,22512,22646,22780,22914,23048,23182,23316,23450,23584,23718,23852,23986,24120,24254,24388,24522,24656,24790,24924,25058,25192,25326,25460,25594,25728,25862,25996,26130,26264,26398,26532,26666,26800,26934,27068,27202,27336,27470,27604,27738,27872,28006,28140,28274,28408,28542,28676,28810,28944,29078,29212,29346,29480,29614,29748,29882,30016,30150,30284,30418,30552,30686,30820,30954,31088,31222,31356,31490,31624,31758,31892,32026,32160,32294,32428,32562,32696,32830,32964,33098,33232,33366,33500,33634,33768,33902,34036,34170,34304,34438,34572,34706,34840,34974,35108,35242,35376,35510,35644,35778,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18224,18358,18492,18626,18760,18894,19028,19162,19296,19430,19564,19698,19832,19966,20100,20234,20368,20502,20636,20770,20904,21038,21172,21306,21440,21574,21708,21842,21976,22110,22244,22378,22512,22646,22780,22914,23048,23182,23316,23450,23584,23718,23852,23986,24120,24254,24388,24522,24656,24790,24924,25058,25192,25326,25460,25594,25728,25862,25996,26130,26264,26398,26532,26666,26800,26934,27068,27202,27336,27470,27604,27738,27872,28006,28140,28274,28408,28542,28676,28810,28944,29078,29212,29346,29480,29614,29748,29882,30016,30150,30284,30418,30552,30686,30820,30954,31088,31222,31356,31490,31624,31758,31892,32026,32160,32294,32428,32562,32696,32830,32964,33098,33232,33366,33500,33634,33768,33902,34036,34170,34304,34438,34572,34706,34840,34974,35108,35242,35376,35510,35644,35778,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18495,18630,18765,18900,19035,19170,19305,19440,19575,19710,19845,19980,20115,20250,20385,20520,20655,20790,20925,21060,21195,21330,21465,21600,21735,21870,22005,22140,22275,22410,22545,22680,22815,22950,23085,23220,23355,23490,23625,23760,23895,24030,24165,24300,24435,24570,24705,24840,24975,25110,25245,25380,25515,25650,25785,25920,26055,26190,26325,26460,26595,26730,26865,27000,27135,27270,27405,27540,27675,27810,27945,28080,28215,28350,28485,28620,28755,28890,29025,29160,29295,29430,29565,29700,29835,29970,30105,30240,30375,30510,30645,30780,30915,31050,31185,31320,31455,31590,31725,31860,31995,32130,32265,32400,32535,32670,32805,32940,33075,33210,33345,33480,33615,33750,33885,34020,34155,34290,34425,34560,34695,34830,34965,35100,35235,35370,35505,35640,35775,35910,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18495,18630,18765,18900,19035,19170,19305,19440,19575,19710,19845,19980,20115,20250,20385,20520,20655,20790,20925,21060,21195,21330,21465,21600,21735,21870,22005,22140,22275,22410,22545,22680,22815,22950,23085,23220,23355,23490,23625,23760,23895,24030,24165,24300,24435,24570,24705,24840,24975,25110,25245,25380,25515,25650,25785,25920,26055,26190,26325,26460,26595,26730,26865,27000,27135,27270,27405,27540,27675,27810,27945,28080,28215,28350,28485,28620,28755,28890,29025,29160,29295,29430,29565,29700,29835,29970,30105,30240,30375,30510,30645,30780,30915,31050,31185,31320,31455,31590,31725,31860,31995,32130,32265,32400,32535,32670,32805,32940,33075,33210,33345,33480,33615,33750,33885,34020,34155,34290,34425,34560,34695,34830,34965,35100,35235,35370,35505,35640,35775,35910,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18768,18904,19040,19176,19312,19448,19584,19720,19856,19992,20128,20264,20400,20536,20672,20808,20944,21080,21216,21352,21488,21624,21760,21896,22032,22168,22304,22440,22576,22712,22848,22984,23120,23256,23392,23528,23664,23800,23936,24072,24208,24344,24480,24616,24752,24888,25024,25160,25296,25432,25568,25704,25840,25976,26112,26248,26384,26520,26656,26792,26928,27064,27200,27336,27472,27608,27744,27880,28016,28152,28288,28424,28560,28696,28832,28968,29104,29240,29376,29512,29648,29784,29920,30056,30192,30328,30464,30600,30736,30872,31008,31144,31280,31416,31552,31688,31824,31960,32096,32232,32368,32504,32640,32776,32912,33048,33184,33320,33456,33592,33728,33864,34000,34136,34272,34408,34544,34680,34816,34952,35088,35224,35360,35496,35632,35768,35904,36040,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18768,18904,19040,19176,19312,19448,19584,19720,19856,19992,20128,20264,20400,20536,20672,20808,20944,21080,21216,21352,21488,21624,21760,21896,22032,22168,22304,22440,22576,22712,22848,22984,23120,23256,23392,23528,23664,23800,23936,24072,24208,24344,24480,24616,24752,24888,25024,25160,25296,25432,25568,25704,25840,25976,26112,26248,26384,26520,26656,26792,26928,27064,27200,27336,27472,27608,27744,27880,28016,28152,28288,28424,28560,28696,28832,28968,29104,29240,29376,29512,29648,29784,29920,30056,30192,30328,30464,30600,30736,30872,31008,31144,31280,31416,31552,31688,31824,31960,32096,32232,32368,32504,32640,32776,32912,33048,33184,33320,33456,33592,33728,33864,34000,34136,34272,34408,34544,34680,34816,34952,35088,35224,35360,35496,35632,35768,35904,36040,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19043,19180,19317,19454,19591,19728,19865,20002,20139,20276,20413,20550,20687,20824,20961,21098,21235,21372,21509,21646,21783,21920,22057,22194,22331,22468,22605,22742,22879,23016,23153,23290,23427,23564,23701,23838,23975,24112,24249,24386,24523,24660,24797,24934,25071,25208,25345,25482,25619,25756,25893,26030,26167,26304,26441,26578,26715,26852,26989,27126,27263,27400,27537,27674,27811,27948,28085,28222,28359,28496,28633,28770,28907,29044,29181,29318,29455,29592,29729,29866,30003,30140,30277,30414,30551,30688,30825,30962,31099,31236,31373,31510,31647,31784,31921,32058,32195,32332,32469,32606,32743,32880,33017,33154,33291,33428,33565,33702,33839,33976,34113,34250,34387,34524,34661,34798,34935,35072,35209,35346,35483,35620,35757,35894,36031,36168,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19043,19180,19317,19454,19591,19728,19865,20002,20139,20276,20413,20550,20687,20824,20961,21098,21235,21372,21509,21646,21783,21920,22057,22194,22331,22468,22605,22742,22879,23016,23153,23290,23427,23564,23701,23838,23975,24112,24249,24386,24523,24660,24797,24934,25071,25208,25345,25482,25619,25756,25893,26030,26167,26304,26441,26578,26715,26852,26989,27126,27263,27400,27537,27674,27811,27948,28085,28222,28359,28496,28633,28770,28907,29044,29181,29318,29455,29592,29729,29866,30003,30140,30277,30414,30551,30688,30825,30962,31099,31236,31373,31510,31647,31784,31921,32058,32195,32332,32469,32606,32743,32880,33017,33154,33291,33428,33565,33702,33839,33976,34113,34250,34387,34524,34661,34798,34935,35072,35209,35346,35483,35620,35757,35894,36031,36168,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19320,19458,19596,19734,19872,20010,20148,20286,20424,20562,20700,20838,20976,21114,21252,21390,21528,21666,21804,21942,22080,22218,22356,22494,22632,22770,22908,23046,23184,23322,23460,23598,23736,23874,24012,24150,24288,24426,24564,24702,24840,24978,25116,25254,25392,25530,25668,25806,25944,26082,26220,26358,26496,26634,26772,26910,27048,27186,27324,27462,27600,27738,27876,28014,28152,28290,28428,28566,28704,28842,28980,29118,29256,29394,29532,29670,29808,29946,30084,30222,30360,30498,30636,30774,30912,31050,31188,31326,31464,31602,31740,31878,32016,32154,32292,32430,32568,32706,32844,32982,33120,33258,33396,33534,33672,33810,33948,34086,34224,34362,34500,34638,34776,34914,35052,35190,35328,35466,35604,35742,35880,36018,36156,36294,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19320,19458,19596,19734,19872,20010,20148,20286,20424,20562,20700,20838,20976,21114,21252,21390,21528,21666,21804,21942,22080,22218,22356,22494,22632,22770,22908,23046,23184,23322,23460,23598,23736,23874,24012,24150,24288,24426,24564,24702,24840,24978,25116,25254,25392,25530,25668,25806,25944,26082,26220,26358,26496,26634,26772,26910,27048,27186,27324,27462,27600,27738,27876,28014,28152,28290,28428,28566,28704,28842,28980,29118,29256,29394,29532,29670,29808,29946,30084,30222,30360,30498,30636,30774,30912,31050,31188,31326,31464,31602,31740,31878,32016,32154,32292,32430,32568,32706,32844,32982,33120,33258,33396,33534,33672,33810,33948,34086,34224,34362,34500,34638,34776,34914,35052,35190,35328,35466,35604,35742,35880,36018,36156,36294,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19599,19738,19877,20016,20155,20294,20433,20572,20711,20850,20989,21128,21267,21406,21545,21684,21823,21962,22101,22240,22379,22518,22657,22796,22935,23074,23213,23352,23491,23630,23769,23908,24047,24186,24325,24464,24603,24742,24881,25020,25159,25298,25437,25576,25715,25854,25993,26132,26271,26410,26549,26688,26827,26966,27105,27244,27383,27522,27661,27800,27939,28078,28217,28356,28495,28634,28773,28912,29051,29190,29329,29468,29607,29746,29885,30024,30163,30302,30441,30580,30719,30858,30997,31136,31275,31414,31553,31692,31831,31970,32109,32248,32387,32526,32665,32804,32943,33082,33221,33360,33499,33638,33777,33916,34055,34194,34333,34472,34611,34750,34889,35028,35167,35306,35445,35584,35723,35862,36001,36140,36279,36418,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19599,19738,19877,20016,20155,20294,20433,20572,20711,20850,20989,21128,21267,21406,21545,21684,21823,21962,22101,22240,22379,22518,22657,22796,22935,23074,23213,23352,23491,23630,23769,23908,24047,24186,24325,24464,24603,24742,24881,25020,25159,25298,25437,25576,25715,25854,25993,26132,26271,26410,26549,26688,26827,26966,27105,27244,27383,27522,27661,27800,27939,28078,28217,28356,28495,28634,28773,28912,29051,29190,29329,29468,29607,29746,29885,30024,30163,30302,30441,30580,30719,30858,30997,31136,31275,31414,31553,31692,31831,31970,32109,32248,32387,32526,32665,32804,32943,33082,33221,33360,33499,33638,33777,33916,34055,34194,34333,34472,34611,34750,34889,35028,35167,35306,35445,35584,35723,35862,36001,36140,36279,36418,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19880,20020,20160,20300,20440,20580,20720,20860,21000,21140,21280,21420,21560,21700,21840,21980,22120,22260,22400,22540,22680,22820,22960,23100,23240,23380,23520,23660,23800,23940,24080,24220,24360,24500,24640,24780,24920,25060,25200,25340,25480,25620,25760,25900,26040,26180,26320,26460,26600,26740,26880,27020,27160,27300,27440,27580,27720,27860,28000,28140,28280,28420,28560,28700,28840,28980,29120,29260,29400,29540,29680,29820,29960,30100,30240,30380,30520,30660,30800,30940,31080,31220,31360,31500,31640,31780,31920,32060,32200,32340,32480,32620,32760,32900,33040,33180,33320,33460,33600,33740,33880,34020,34160,34300,34440,34580,34720,34860,35000,35140,35280,35420,35560,35700,35840,35980,36120,36260,36400,36540,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19880,20020,20160,20300,20440,20580,20720,20860,21000,21140,21280,21420,21560,21700,21840,21980,22120,22260,22400,22540,22680,22820,22960,23100,23240,23380,23520,23660,23800,23940,24080,24220,24360,24500,24640,24780,24920,25060,25200,25340,25480,25620,25760,25900,26040,26180,26320,26460,26600,26740,26880,27020,27160,27300,27440,27580,27720,27860,28000,28140,28280,28420,28560,28700,28840,28980,29120,29260,29400,29540,29680,29820,29960,30100,30240,30380,30520,30660,30800,30940,31080,31220,31360,31500,31640,31780,31920,32060,32200,32340,32480,32620,32760,32900,33040,33180,33320,33460,33600,33740,33880,34020,34160,34300,34440,34580,34720,34860,35000,35140,35280,35420,35560,35700,35840,35980,36120,36260,36400,36540,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20163,20304,20445,20586,20727,20868,21009,21150,21291,21432,21573,21714,21855,21996,22137,22278,22419,22560,22701,22842,22983,23124,23265,23406,23547,23688,23829,23970,24111,24252,24393,24534,24675,24816,24957,25098,25239,25380,25521,25662,25803,25944,26085,26226,26367,26508,26649,26790,26931,27072,27213,27354,27495,27636,27777,27918,28059,28200,28341,28482,28623,28764,28905,29046,29187,29328,29469,29610,29751,29892,30033,30174,30315,30456,30597,30738,30879,31020,31161,31302,31443,31584,31725,31866,32007,32148,32289,32430,32571,32712,32853,32994,33135,33276,33417,33558,33699,33840,33981,34122,34263,34404,34545,34686,34827,34968,35109,35250,35391,35532,35673,35814,35955,36096,36237,36378,36519,36660,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20163,20304,20445,20586,20727,20868,21009,21150,21291,21432,21573,21714,21855,21996,22137,22278,22419,22560,22701,22842,22983,23124,23265,23406,23547,23688,23829,23970,24111,24252,24393,24534,24675,24816,24957,25098,25239,25380,25521,25662,25803,25944,26085,26226,26367,26508,26649,26790,26931,27072,27213,27354,27495,27636,27777,27918,28059,28200,28341,28482,28623,28764,28905,29046,29187,29328,29469,29610,29751,29892,30033,30174,30315,30456,30597,30738,30879,31020,31161,31302,31443,31584,31725,31866,32007,32148,32289,32430,32571,32712,32853,32994,33135,33276,33417,33558,33699,33840,33981,34122,34263,34404,34545,34686,34827,34968,35109,35250,35391,35532,35673,35814,35955,36096,36237,36378,36519,36660,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20448,20590,20732,20874,21016,21158,21300,21442,21584,21726,21868,22010,22152,22294,22436,22578,22720,22862,23004,23146,23288,23430,23572,23714,23856,23998,24140,24282,24424,24566,24708,24850,24992,25134,25276,25418,25560,25702,25844,25986,26128,26270,26412,26554,26696,26838,26980,27122,27264,27406,27548,27690,27832,27974,28116,28258,28400,28542,28684,28826,28968,29110,29252,29394,29536,29678,29820,29962,30104,30246,30388,30530,30672,30814,30956,31098,31240,31382,31524,31666,31808,31950,32092,32234,32376,32518,32660,32802,32944,33086,33228,33370,33512,33654,33796,33938,34080,34222,34364,34506,34648,34790,34932,35074,35216,35358,35500,35642,35784,35926,36068,36210,36352,36494,36636,36778,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20448,20590,20732,20874,21016,21158,21300,21442,21584,21726,21868,22010,22152,22294,22436,22578,22720,22862,23004,23146,23288,23430,23572,23714,23856,23998,24140,24282,24424,24566,24708,24850,24992,25134,25276,25418,25560,25702,25844,25986,26128,26270,26412,26554,26696,26838,26980,27122,27264,27406,27548,27690,27832,27974,28116,28258,28400,28542,28684,28826,28968,29110,29252,29394,29536,29678,29820,29962,30104,30246,30388,30530,30672,30814,30956,31098,31240,31382,31524,31666,31808,31950,32092,32234,32376,32518,32660,32802,32944,33086,33228,33370,33512,33654,33796,33938,34080,34222,34364,34506,34648,34790,34932,35074,35216,35358,35500,35642,35784,35926,36068,36210,36352,36494,36636,36778,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20735,20878,21021,21164,21307,21450,21593,21736,21879,22022,22165,22308,22451,22594,22737,22880,23023,23166,23309,23452,23595,23738,23881,24024,24167,24310,24453,24596,24739,24882,25025,25168,25311,25454,25597,25740,25883,26026,26169,26312,26455,26598,26741,26884,27027,27170,27313,27456,27599,27742,27885,28028,28171,28314,28457,28600,28743,28886,29029,29172,29315,29458,29601,29744,29887,30030,30173,30316,30459,30602,30745,30888,31031,31174,31317,31460,31603,31746,31889,32032,32175,32318,32461,32604,32747,32890,33033,33176,33319,33462,33605,33748,33891,34034,34177,34320,34463,34606,34749,34892,35035,35178,35321,35464,35607,35750,35893,36036,36179,36322,36465,36608,36751,36894,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20735,20878,21021,21164,21307,21450,21593,21736,21879,22022,22165,22308,22451,22594,22737,22880,23023,23166,23309,23452,23595,23738,23881,24024,24167,24310,24453,24596,24739,24882,25025,25168,25311,25454,25597,25740,25883,26026,26169,26312,26455,26598,26741,26884,27027,27170,27313,27456,27599,27742,27885,28028,28171,28314,28457,28600,28743,28886,29029,29172,29315,29458,29601,29744,29887,30030,30173,30316,30459,30602,30745,30888,31031,31174,31317,31460,31603,31746,31889,32032,32175,32318,32461,32604,32747,32890,33033,33176,33319,33462,33605,33748,33891,34034,34177,34320,34463,34606,34749,34892,35035,35178,35321,35464,35607,35750,35893,36036,36179,36322,36465,36608,36751,36894,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21024,21168,21312,21456,21600,21744,21888,22032,22176,22320,22464,22608,22752,22896,23040,23184,23328,23472,23616,23760,23904,24048,24192,24336,24480,24624,24768,24912,25056,25200,25344,25488,25632,25776,25920,26064,26208,26352,26496,26640,26784,26928,27072,27216,27360,27504,27648,27792,27936,28080,28224,28368,28512,28656,28800,28944,29088,29232,29376,29520,29664,29808,29952,30096,30240,30384,30528,30672,30816,30960,31104,31248,31392,31536,31680,31824,31968,32112,32256,32400,32544,32688,32832,32976,33120,33264,33408,33552,33696,33840,33984,34128,34272,34416,34560,34704,34848,34992,35136,35280,35424,35568,35712,35856,36000,36144,36288,36432,36576,36720,36864,37008,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21024,21168,21312,21456,21600,21744,21888,22032,22176,22320,22464,22608,22752,22896,23040,23184,23328,23472,23616,23760,23904,24048,24192,24336,24480,24624,24768,24912,25056,25200,25344,25488,25632,25776,25920,26064,26208,26352,26496,26640,26784,26928,27072,27216,27360,27504,27648,27792,27936,28080,28224,28368,28512,28656,28800,28944,29088,29232,29376,29520,29664,29808,29952,30096,30240,30384,30528,30672,30816,30960,31104,31248,31392,31536,31680,31824,31968,32112,32256,32400,32544,32688,32832,32976,33120,33264,33408,33552,33696,33840,33984,34128,34272,34416,34560,34704,34848,34992,35136,35280,35424,35568,35712,35856,36000,36144,36288,36432,36576,36720,36864,37008,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21315,21460,21605,21750,21895,22040,22185,22330,22475,22620,22765,22910,23055,23200,23345,23490,23635,23780,23925,24070,24215,24360,24505,24650,24795,24940,25085,25230,25375,25520,25665,25810,25955,26100,26245,26390,26535,26680,26825,26970,27115,27260,27405,27550,27695,27840,27985,28130,28275,28420,28565,28710,28855,29000,29145,29290,29435,29580,29725,29870,30015,30160,30305,30450,30595,30740,30885,31030,31175,31320,31465,31610,31755,31900,32045,32190,32335,32480,32625,32770,32915,33060,33205,33350,33495,33640,33785,33930,34075,34220,34365,34510,34655,34800,34945,35090,35235,35380,35525,35670,35815,35960,36105,36250,36395,36540,36685,36830,36975,37120,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21315,21460,21605,21750,21895,22040,22185,22330,22475,22620,22765,22910,23055,23200,23345,23490,23635,23780,23925,24070,24215,24360,24505,24650,24795,24940,25085,25230,25375,25520,25665,25810,25955,26100,26245,26390,26535,26680,26825,26970,27115,27260,27405,27550,27695,27840,27985,28130,28275,28420,28565,28710,28855,29000,29145,29290,29435,29580,29725,29870,30015,30160,30305,30450,30595,30740,30885,31030,31175,31320,31465,31610,31755,31900,32045,32190,32335,32480,32625,32770,32915,33060,33205,33350,33495,33640,33785,33930,34075,34220,34365,34510,34655,34800,34945,35090,35235,35380,35525,35670,35815,35960,36105,36250,36395,36540,36685,36830,36975,37120,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21608,21754,21900,22046,22192,22338,22484,22630,22776,22922,23068,23214,23360,23506,23652,23798,23944,24090,24236,24382,24528,24674,24820,24966,25112,25258,25404,25550,25696,25842,25988,26134,26280,26426,26572,26718,26864,27010,27156,27302,27448,27594,27740,27886,28032,28178,28324,28470,28616,28762,28908,29054,29200,29346,29492,29638,29784,29930,30076,30222,30368,30514,30660,30806,30952,31098,31244,31390,31536,31682,31828,31974,32120,32266,32412,32558,32704,32850,32996,33142,33288,33434,33580,33726,33872,34018,34164,34310,34456,34602,34748,34894,35040,35186,35332,35478,35624,35770,35916,36062,36208,36354,36500,36646,36792,36938,37084,37230,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21608,21754,21900,22046,22192,22338,22484,22630,22776,22922,23068,23214,23360,23506,23652,23798,23944,24090,24236,24382,24528,24674,24820,24966,25112,25258,25404,25550,25696,25842,25988,26134,26280,26426,26572,26718,26864,27010,27156,27302,27448,27594,27740,27886,28032,28178,28324,28470,28616,28762,28908,29054,29200,29346,29492,29638,29784,29930,30076,30222,30368,30514,30660,30806,30952,31098,31244,31390,31536,31682,31828,31974,32120,32266,32412,32558,32704,32850,32996,33142,33288,33434,33580,33726,33872,34018,34164,34310,34456,34602,34748,34894,35040,35186,35332,35478,35624,35770,35916,36062,36208,36354,36500,36646,36792,36938,37084,37230,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21903,22050,22197,22344,22491,22638,22785,22932,23079,23226,23373,23520,23667,23814,23961,24108,24255,24402,24549,24696,24843,24990,25137,25284,25431,25578,25725,25872,26019,26166,26313,26460,26607,26754,26901,27048,27195,27342,27489,27636,27783,27930,28077,28224,28371,28518,28665,28812,28959,29106,29253,29400,29547,29694,29841,29988,30135,30282,30429,30576,30723,30870,31017,31164,31311,31458,31605,31752,31899,32046,32193,32340,32487,32634,32781,32928,33075,33222,33369,33516,33663,33810,33957,34104,34251,34398,34545,34692,34839,34986,35133,35280,35427,35574,35721,35868,36015,36162,36309,36456,36603,36750,36897,37044,37191,37338,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21903,22050,22197,22344,22491,22638,22785,22932,23079,23226,23373,23520,23667,23814,23961,24108,24255,24402,24549,24696,24843,24990,25137,25284,25431,25578,25725,25872,26019,26166,26313,26460,26607,26754,26901,27048,27195,27342,27489,27636,27783,27930,28077,28224,28371,28518,28665,28812,28959,29106,29253,29400,29547,29694,29841,29988,30135,30282,30429,30576,30723,30870,31017,31164,31311,31458,31605,31752,31899,32046,32193,32340,32487,32634,32781,32928,33075,33222,33369,33516,33663,33810,33957,34104,34251,34398,34545,34692,34839,34986,35133,35280,35427,35574,35721,35868,36015,36162,36309,36456,36603,36750,36897,37044,37191,37338,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22200,22348,22496,22644,22792,22940,23088,23236,23384,23532,23680,23828,23976,24124,24272,24420,24568,24716,24864,25012,25160,25308,25456,25604,25752,25900,26048,26196,26344,26492,26640,26788,26936,27084,27232,27380,27528,27676,27824,27972,28120,28268,28416,28564,28712,28860,29008,29156,29304,29452,29600,29748,29896,30044,30192,30340,30488,30636,30784,30932,31080,31228,31376,31524,31672,31820,31968,32116,32264,32412,32560,32708,32856,33004,33152,33300,33448,33596,33744,33892,34040,34188,34336,34484,34632,34780,34928,35076,35224,35372,35520,35668,35816,35964,36112,36260,36408,36556,36704,36852,37000,37148,37296,37444,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22200,22348,22496,22644,22792,22940,23088,23236,23384,23532,23680,23828,23976,24124,24272,24420,24568,24716,24864,25012,25160,25308,25456,25604,25752,25900,26048,26196,26344,26492,26640,26788,26936,27084,27232,27380,27528,27676,27824,27972,28120,28268,28416,28564,28712,28860,29008,29156,29304,29452,29600,29748,29896,30044,30192,30340,30488,30636,30784,30932,31080,31228,31376,31524,31672,31820,31968,32116,32264,32412,32560,32708,32856,33004,33152,33300,33448,33596,33744,33892,34040,34188,34336,34484,34632,34780,34928,35076,35224,35372,35520,35668,35816,35964,36112,36260,36408,36556,36704,36852,37000,37148,37296,37444,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22499,22648,22797,22946,23095,23244,23393,23542,23691,23840,23989,24138,24287,24436,24585,24734,24883,25032,25181,25330,25479,25628,25777,25926,26075,26224,26373,26522,26671,26820,26969,27118,27267,27416,27565,27714,27863,28012,28161,28310,28459,28608,28757,28906,29055,29204,29353,29502,29651,29800,29949,30098,30247,30396,30545,30694,30843,30992,31141,31290,31439,31588,31737,31886,32035,32184,32333,32482,32631,32780,32929,33078,33227,33376,33525,33674,33823,33972,34121,34270,34419,34568,34717,34866,35015,35164,35313,35462,35611,35760,35909,36058,36207,36356,36505,36654,36803,36952,37101,37250,37399,37548,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22499,22648,22797,22946,23095,23244,23393,23542,23691,23840,23989,24138,24287,24436,24585,24734,24883,25032,25181,25330,25479,25628,25777,25926,26075,26224,26373,26522,26671,26820,26969,27118,27267,27416,27565,27714,27863,28012,28161,28310,28459,28608,28757,28906,29055,29204,29353,29502,29651,29800,29949,30098,30247,30396,30545,30694,30843,30992,31141,31290,31439,31588,31737,31886,32035,32184,32333,32482,32631,32780,32929,33078,33227,33376,33525,33674,33823,33972,34121,34270,34419,34568,34717,34866,35015,35164,35313,35462,35611,35760,35909,36058,36207,36356,36505,36654,36803,36952,37101,37250,37399,37548,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22800,22950,23100,23250,23400,23550,23700,23850,24000,24150,24300,24450,24600,24750,24900,25050,25200,25350,25500,25650,25800,25950,26100,26250,26400,26550,26700,26850,27000,27150,27300,27450,27600,27750,27900,28050,28200,28350,28500,28650,28800,28950,29100,29250,29400,29550,29700,29850,30000,30150,30300,30450,30600,30750,30900,31050,31200,31350,31500,31650,31800,31950,32100,32250,32400,32550,32700,32850,33000,33150,33300,33450,33600,33750,33900,34050,34200,34350,34500,34650,34800,34950,35100,35250,35400,35550,35700,35850,36000,36150,36300,36450,36600,36750,36900,37050,37200,37350,37500,37650,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22800,22950,23100,23250,23400,23550,23700,23850,24000,24150,24300,24450,24600,24750,24900,25050,25200,25350,25500,25650,25800,25950,26100,26250,26400,26550,26700,26850,27000,27150,27300,27450,27600,27750,27900,28050,28200,28350,28500,28650,28800,28950,29100,29250,29400,29550,29700,29850,30000,30150,30300,30450,30600,30750,30900,31050,31200,31350,31500,31650,31800,31950,32100,32250,32400,32550,32700,32850,33000,33150,33300,33450,33600,33750,33900,34050,34200,34350,34500,34650,34800,34950,35100,35250,35400,35550,35700,35850,36000,36150,36300,36450,36600,36750,36900,37050,37200,37350,37500,37650,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23103,23254,23405,23556,23707,23858,24009,24160,24311,24462,24613,24764,24915,25066,25217,25368,25519,25670,25821,25972,26123,26274,26425,26576,26727,26878,27029,27180,27331,27482,27633,27784,27935,28086,28237,28388,28539,28690,28841,28992,29143,29294,29445,29596,29747,29898,30049,30200,30351,30502,30653,30804,30955,31106,31257,31408,31559,31710,31861,32012,32163,32314,32465,32616,32767,32918,33069,33220,33371,33522,33673,33824,33975,34126,34277,34428,34579,34730,34881,35032,35183,35334,35485,35636,35787,35938,36089,36240,36391,36542,36693,36844,36995,37146,37297,37448,37599,37750,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23103,23254,23405,23556,23707,23858,24009,24160,24311,24462,24613,24764,24915,25066,25217,25368,25519,25670,25821,25972,26123,26274,26425,26576,26727,26878,27029,27180,27331,27482,27633,27784,27935,28086,28237,28388,28539,28690,28841,28992,29143,29294,29445,29596,29747,29898,30049,30200,30351,30502,30653,30804,30955,31106,31257,31408,31559,31710,31861,32012,32163,32314,32465,32616,32767,32918,33069,33220,33371,33522,33673,33824,33975,34126,34277,34428,34579,34730,34881,35032,35183,35334,35485,35636,35787,35938,36089,36240,36391,36542,36693,36844,36995,37146,37297,37448,37599,37750,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23408,23560,23712,23864,24016,24168,24320,24472,24624,24776,24928,25080,25232,25384,25536,25688,25840,25992,26144,26296,26448,26600,26752,26904,27056,27208,27360,27512,27664,27816,27968,28120,28272,28424,28576,28728,28880,29032,29184,29336,29488,29640,29792,29944,30096,30248,30400,30552,30704,30856,31008,31160,31312,31464,31616,31768,31920,32072,32224,32376,32528,32680,32832,32984,33136,33288,33440,33592,33744,33896,34048,34200,34352,34504,34656,34808,34960,35112,35264,35416,35568,35720,35872,36024,36176,36328,36480,36632,36784,36936,37088,37240,37392,37544,37696,37848,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23408,23560,23712,23864,24016,24168,24320,24472,24624,24776,24928,25080,25232,25384,25536,25688,25840,25992,26144,26296,26448,26600,26752,26904,27056,27208,27360,27512,27664,27816,27968,28120,28272,28424,28576,28728,28880,29032,29184,29336,29488,29640,29792,29944,30096,30248,30400,30552,30704,30856,31008,31160,31312,31464,31616,31768,31920,32072,32224,32376,32528,32680,32832,32984,33136,33288,33440,33592,33744,33896,34048,34200,34352,34504,34656,34808,34960,35112,35264,35416,35568,35720,35872,36024,36176,36328,36480,36632,36784,36936,37088,37240,37392,37544,37696,37848,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23715,23868,24021,24174,24327,24480,24633,24786,24939,25092,25245,25398,25551,25704,25857,26010,26163,26316,26469,26622,26775,26928,27081,27234,27387,27540,27693,27846,27999,28152,28305,28458,28611,28764,28917,29070,29223,29376,29529,29682,29835,29988,30141,30294,30447,30600,30753,30906,31059,31212,31365,31518,31671,31824,31977,32130,32283,32436,32589,32742,32895,33048,33201,33354,33507,33660,33813,33966,34119,34272,34425,34578,34731,34884,35037,35190,35343,35496,35649,35802,35955,36108,36261,36414,36567,36720,36873,37026,37179,37332,37485,37638,37791,37944,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23715,23868,24021,24174,24327,24480,24633,24786,24939,25092,25245,25398,25551,25704,25857,26010,26163,26316,26469,26622,26775,26928,27081,27234,27387,27540,27693,27846,27999,28152,28305,28458,28611,28764,28917,29070,29223,29376,29529,29682,29835,29988,30141,30294,30447,30600,30753,30906,31059,31212,31365,31518,31671,31824,31977,32130,32283,32436,32589,32742,32895,33048,33201,33354,33507,33660,33813,33966,34119,34272,34425,34578,34731,34884,35037,35190,35343,35496,35649,35802,35955,36108,36261,36414,36567,36720,36873,37026,37179,37332,37485,37638,37791,37944,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24024,24178,24332,24486,24640,24794,24948,25102,25256,25410,25564,25718,25872,26026,26180,26334,26488,26642,26796,26950,27104,27258,27412,27566,27720,27874,28028,28182,28336,28490,28644,28798,28952,29106,29260,29414,29568,29722,29876,30030,30184,30338,30492,30646,30800,30954,31108,31262,31416,31570,31724,31878,32032,32186,32340,32494,32648,32802,32956,33110,33264,33418,33572,33726,33880,34034,34188,34342,34496,34650,34804,34958,35112,35266,35420,35574,35728,35882,36036,36190,36344,36498,36652,36806,36960,37114,37268,37422,37576,37730,37884,38038,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24024,24178,24332,24486,24640,24794,24948,25102,25256,25410,25564,25718,25872,26026,26180,26334,26488,26642,26796,26950,27104,27258,27412,27566,27720,27874,28028,28182,28336,28490,28644,28798,28952,29106,29260,29414,29568,29722,29876,30030,30184,30338,30492,30646,30800,30954,31108,31262,31416,31570,31724,31878,32032,32186,32340,32494,32648,32802,32956,33110,33264,33418,33572,33726,33880,34034,34188,34342,34496,34650,34804,34958,35112,35266,35420,35574,35728,35882,36036,36190,36344,36498,36652,36806,36960,37114,37268,37422,37576,37730,37884,38038,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24335,24490,24645,24800,24955,25110,25265,25420,25575,25730,25885,26040,26195,26350,26505,26660,26815,26970,27125,27280,27435,27590,27745,27900,28055,28210,28365,28520,28675,28830,28985,29140,29295,29450,29605,29760,29915,30070,30225,30380,30535,30690,30845,31000,31155,31310,31465,31620,31775,31930,32085,32240,32395,32550,32705,32860,33015,33170,33325,33480,33635,33790,33945,34100,34255,34410,34565,34720,34875,35030,35185,35340,35495,35650,35805,35960,36115,36270,36425,36580,36735,36890,37045,37200,37355,37510,37665,37820,37975,38130,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24335,24490,24645,24800,24955,25110,25265,25420,25575,25730,25885,26040,26195,26350,26505,26660,26815,26970,27125,27280,27435,27590,27745,27900,28055,28210,28365,28520,28675,28830,28985,29140,29295,29450,29605,29760,29915,30070,30225,30380,30535,30690,30845,31000,31155,31310,31465,31620,31775,31930,32085,32240,32395,32550,32705,32860,33015,33170,33325,33480,33635,33790,33945,34100,34255,34410,34565,34720,34875,35030,35185,35340,35495,35650,35805,35960,36115,36270,36425,36580,36735,36890,37045,37200,37355,37510,37665,37820,37975,38130,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24648,24804,24960,25116,25272,25428,25584,25740,25896,26052,26208,26364,26520,26676,26832,26988,27144,27300,27456,27612,27768,27924,28080,28236,28392,28548,28704,28860,29016,29172,29328,29484,29640,29796,29952,30108,30264,30420,30576,30732,30888,31044,31200,31356,31512,31668,31824,31980,32136,32292,32448,32604,32760,32916,33072,33228,33384,33540,33696,33852,34008,34164,34320,34476,34632,34788,34944,35100,35256,35412,35568,35724,35880,36036,36192,36348,36504,36660,36816,36972,37128,37284,37440,37596,37752,37908,38064,38220,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24648,24804,24960,25116,25272,25428,25584,25740,25896,26052,26208,26364,26520,26676,26832,26988,27144,27300,27456,27612,27768,27924,28080,28236,28392,28548,28704,28860,29016,29172,29328,29484,29640,29796,29952,30108,30264,30420,30576,30732,30888,31044,31200,31356,31512,31668,31824,31980,32136,32292,32448,32604,32760,32916,33072,33228,33384,33540,33696,33852,34008,34164,34320,34476,34632,34788,34944,35100,35256,35412,35568,35724,35880,36036,36192,36348,36504,36660,36816,36972,37128,37284,37440,37596,37752,37908,38064,38220,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24963,25120,25277,25434,25591,25748,25905,26062,26219,26376,26533,26690,26847,27004,27161,27318,27475,27632,27789,27946,28103,28260,28417,28574,28731,28888,29045,29202,29359,29516,29673,29830,29987,30144,30301,30458,30615,30772,30929,31086,31243,31400,31557,31714,31871,32028,32185,32342,32499,32656,32813,32970,33127,33284,33441,33598,33755,33912,34069,34226,34383,34540,34697,34854,35011,35168,35325,35482,35639,35796,35953,36110,36267,36424,36581,36738,36895,37052,37209,37366,37523,37680,37837,37994,38151,38308,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24963,25120,25277,25434,25591,25748,25905,26062,26219,26376,26533,26690,26847,27004,27161,27318,27475,27632,27789,27946,28103,28260,28417,28574,28731,28888,29045,29202,29359,29516,29673,29830,29987,30144,30301,30458,30615,30772,30929,31086,31243,31400,31557,31714,31871,32028,32185,32342,32499,32656,32813,32970,33127,33284,33441,33598,33755,33912,34069,34226,34383,34540,34697,34854,35011,35168,35325,35482,35639,35796,35953,36110,36267,36424,36581,36738,36895,37052,37209,37366,37523,37680,37837,37994,38151,38308,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25280,25438,25596,25754,25912,26070,26228,26386,26544,26702,26860,27018,27176,27334,27492,27650,27808,27966,28124,28282,28440,28598,28756,28914,29072,29230,29388,29546,29704,29862,30020,30178,30336,30494,30652,30810,30968,31126,31284,31442,31600,31758,31916,32074,32232,32390,32548,32706,32864,33022,33180,33338,33496,33654,33812,33970,34128,34286,34444,34602,34760,34918,35076,35234,35392,35550,35708,35866,36024,36182,36340,36498,36656,36814,36972,37130,37288,37446,37604,37762,37920,38078,38236,38394,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25280,25438,25596,25754,25912,26070,26228,26386,26544,26702,26860,27018,27176,27334,27492,27650,27808,27966,28124,28282,28440,28598,28756,28914,29072,29230,29388,29546,29704,29862,30020,30178,30336,30494,30652,30810,30968,31126,31284,31442,31600,31758,31916,32074,32232,32390,32548,32706,32864,33022,33180,33338,33496,33654,33812,33970,34128,34286,34444,34602,34760,34918,35076,35234,35392,35550,35708,35866,36024,36182,36340,36498,36656,36814,36972,37130,37288,37446,37604,37762,37920,38078,38236,38394,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25599,25758,25917,26076,26235,26394,26553,26712,26871,27030,27189,27348,27507,27666,27825,27984,28143,28302,28461,28620,28779,28938,29097,29256,29415,29574,29733,29892,30051,30210,30369,30528,30687,30846,31005,31164,31323,31482,31641,31800,31959,32118,32277,32436,32595,32754,32913,33072,33231,33390,33549,33708,33867,34026,34185,34344,34503,34662,34821,34980,35139,35298,35457,35616,35775,35934,36093,36252,36411,36570,36729,36888,37047,37206,37365,37524,37683,37842,38001,38160,38319,38478,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25599,25758,25917,26076,26235,26394,26553,26712,26871,27030,27189,27348,27507,27666,27825,27984,28143,28302,28461,28620,28779,28938,29097,29256,29415,29574,29733,29892,30051,30210,30369,30528,30687,30846,31005,31164,31323,31482,31641,31800,31959,32118,32277,32436,32595,32754,32913,33072,33231,33390,33549,33708,33867,34026,34185,34344,34503,34662,34821,34980,35139,35298,35457,35616,35775,35934,36093,36252,36411,36570,36729,36888,37047,37206,37365,37524,37683,37842,38001,38160,38319,38478,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25920,26080,26240,26400,26560,26720,26880,27040,27200,27360,27520,27680,27840,28000,28160,28320,28480,28640,28800,28960,29120,29280,29440,29600,29760,29920,30080,30240,30400,30560,30720,30880,31040,31200,31360,31520,31680,31840,32000,32160,32320,32480,32640,32800,32960,33120,33280,33440,33600,33760,33920,34080,34240,34400,34560,34720,34880,35040,35200,35360,35520,35680,35840,36000,36160,36320,36480,36640,36800,36960,37120,37280,37440,37600,37760,37920,38080,38240,38400,38560,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25920,26080,26240,26400,26560,26720,26880,27040,27200,27360,27520,27680,27840,28000,28160,28320,28480,28640,28800,28960,29120,29280,29440,29600,29760,29920,30080,30240,30400,30560,30720,30880,31040,31200,31360,31520,31680,31840,32000,32160,32320,32480,32640,32800,32960,33120,33280,33440,33600,33760,33920,34080,34240,34400,34560,34720,34880,35040,35200,35360,35520,35680,35840,36000,36160,36320,36480,36640,36800,36960,37120,37280,37440,37600,37760,37920,38080,38240,38400,38560,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26243,26404,26565,26726,26887,27048,27209,27370,27531,27692,27853,28014,28175,28336,28497,28658,28819,28980,29141,29302,29463,29624,29785,29946,30107,30268,30429,30590,30751,30912,31073,31234,31395,31556,31717,31878,32039,32200,32361,32522,32683,32844,33005,33166,33327,33488,33649,33810,33971,34132,34293,34454,34615,34776,34937,35098,35259,35420,35581,35742,35903,36064,36225,36386,36547,36708,36869,37030,37191,37352,37513,37674,37835,37996,38157,38318,38479,38640,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26243,26404,26565,26726,26887,27048,27209,27370,27531,27692,27853,28014,28175,28336,28497,28658,28819,28980,29141,29302,29463,29624,29785,29946,30107,30268,30429,30590,30751,30912,31073,31234,31395,31556,31717,31878,32039,32200,32361,32522,32683,32844,33005,33166,33327,33488,33649,33810,33971,34132,34293,34454,34615,34776,34937,35098,35259,35420,35581,35742,35903,36064,36225,36386,36547,36708,36869,37030,37191,37352,37513,37674,37835,37996,38157,38318,38479,38640,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26568,26730,26892,27054,27216,27378,27540,27702,27864,28026,28188,28350,28512,28674,28836,28998,29160,29322,29484,29646,29808,29970,30132,30294,30456,30618,30780,30942,31104,31266,31428,31590,31752,31914,32076,32238,32400,32562,32724,32886,33048,33210,33372,33534,33696,33858,34020,34182,34344,34506,34668,34830,34992,35154,35316,35478,35640,35802,35964,36126,36288,36450,36612,36774,36936,37098,37260,37422,37584,37746,37908,38070,38232,38394,38556,38718,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26568,26730,26892,27054,27216,27378,27540,27702,27864,28026,28188,28350,28512,28674,28836,28998,29160,29322,29484,29646,29808,29970,30132,30294,30456,30618,30780,30942,31104,31266,31428,31590,31752,31914,32076,32238,32400,32562,32724,32886,33048,33210,33372,33534,33696,33858,34020,34182,34344,34506,34668,34830,34992,35154,35316,35478,35640,35802,35964,36126,36288,36450,36612,36774,36936,37098,37260,37422,37584,37746,37908,38070,38232,38394,38556,38718,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26895,27058,27221,27384,27547,27710,27873,28036,28199,28362,28525,28688,28851,29014,29177,29340,29503,29666,29829,29992,30155,30318,30481,30644,30807,30970,31133,31296,31459,31622,31785,31948,32111,32274,32437,32600,32763,32926,33089,33252,33415,33578,33741,33904,34067,34230,34393,34556,34719,34882,35045,35208,35371,35534,35697,35860,36023,36186,36349,36512,36675,36838,37001,37164,37327,37490,37653,37816,37979,38142,38305,38468,38631,38794,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26895,27058,27221,27384,27547,27710,27873,28036,28199,28362,28525,28688,28851,29014,29177,29340,29503,29666,29829,29992,30155,30318,30481,30644,30807,30970,31133,31296,31459,31622,31785,31948,32111,32274,32437,32600,32763,32926,33089,33252,33415,33578,33741,33904,34067,34230,34393,34556,34719,34882,35045,35208,35371,35534,35697,35860,36023,36186,36349,36512,36675,36838,37001,37164,37327,37490,37653,37816,37979,38142,38305,38468,38631,38794,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27224,27388,27552,27716,27880,28044,28208,28372,28536,28700,28864,29028,29192,29356,29520,29684,29848,30012,30176,30340,30504,30668,30832,30996,31160,31324,31488,31652,31816,31980,32144,32308,32472,32636,32800,32964,33128,33292,33456,33620,33784,33948,34112,34276,34440,34604,34768,34932,35096,35260,35424,35588,35752,35916,36080,36244,36408,36572,36736,36900,37064,37228,37392,37556,37720,37884,38048,38212,38376,38540,38704,38868,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27224,27388,27552,27716,27880,28044,28208,28372,28536,28700,28864,29028,29192,29356,29520,29684,29848,30012,30176,30340,30504,30668,30832,30996,31160,31324,31488,31652,31816,31980,32144,32308,32472,32636,32800,32964,33128,33292,33456,33620,33784,33948,34112,34276,34440,34604,34768,34932,35096,35260,35424,35588,35752,35916,36080,36244,36408,36572,36736,36900,37064,37228,37392,37556,37720,37884,38048,38212,38376,38540,38704,38868,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27555,27720,27885,28050,28215,28380,28545,28710,28875,29040,29205,29370,29535,29700,29865,30030,30195,30360,30525,30690,30855,31020,31185,31350,31515,31680,31845,32010,32175,32340,32505,32670,32835,33000,33165,33330,33495,33660,33825,33990,34155,34320,34485,34650,34815,34980,35145,35310,35475,35640,35805,35970,36135,36300,36465,36630,36795,36960,37125,37290,37455,37620,37785,37950,38115,38280,38445,38610,38775,38940,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27555,27720,27885,28050,28215,28380,28545,28710,28875,29040,29205,29370,29535,29700,29865,30030,30195,30360,30525,30690,30855,31020,31185,31350,31515,31680,31845,32010,32175,32340,32505,32670,32835,33000,33165,33330,33495,33660,33825,33990,34155,34320,34485,34650,34815,34980,35145,35310,35475,35640,35805,35970,36135,36300,36465,36630,36795,36960,37125,37290,37455,37620,37785,37950,38115,38280,38445,38610,38775,38940,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27888,28054,28220,28386,28552,28718,28884,29050,29216,29382,29548,29714,29880,30046,30212,30378,30544,30710,30876,31042,31208,31374,31540,31706,31872,32038,32204,32370,32536,32702,32868,33034,33200,33366,33532,33698,33864,34030,34196,34362,34528,34694,34860,35026,35192,35358,35524,35690,35856,36022,36188,36354,36520,36686,36852,37018,37184,37350,37516,37682,37848,38014,38180,38346,38512,38678,38844,39010,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27888,28054,28220,28386,28552,28718,28884,29050,29216,29382,29548,29714,29880,30046,30212,30378,30544,30710,30876,31042,31208,31374,31540,31706,31872,32038,32204,32370,32536,32702,32868,33034,33200,33366,33532,33698,33864,34030,34196,34362,34528,34694,34860,35026,35192,35358,35524,35690,35856,36022,36188,36354,36520,36686,36852,37018,37184,37350,37516,37682,37848,38014,38180,38346,38512,38678,38844,39010,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28223,28390,28557,28724,28891,29058,29225,29392,29559,29726,29893,30060,30227,30394,30561,30728,30895,31062,31229,31396,31563,31730,31897,32064,32231,32398,32565,32732,32899,33066,33233,33400,33567,33734,33901,34068,34235,34402,34569,34736,34903,35070,35237,35404,35571,35738,35905,36072,36239,36406,36573,36740,36907,37074,37241,37408,37575,37742,37909,38076,38243,38410,38577,38744,38911,39078,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28223,28390,28557,28724,28891,29058,29225,29392,29559,29726,29893,30060,30227,30394,30561,30728,30895,31062,31229,31396,31563,31730,31897,32064,32231,32398,32565,32732,32899,33066,33233,33400,33567,33734,33901,34068,34235,34402,34569,34736,34903,35070,35237,35404,35571,35738,35905,36072,36239,36406,36573,36740,36907,37074,37241,37408,37575,37742,37909,38076,38243,38410,38577,38744,38911,39078,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28560,28728,28896,29064,29232,29400,29568,29736,29904,30072,30240,30408,30576,30744,30912,31080,31248,31416,31584,31752,31920,32088,32256,32424,32592,32760,32928,33096,33264,33432,33600,33768,33936,34104,34272,34440,34608,34776,34944,35112,35280,35448,35616,35784,35952,36120,36288,36456,36624,36792,36960,37128,37296,37464,37632,37800,37968,38136,38304,38472,38640,38808,38976,39144,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28560,28728,28896,29064,29232,29400,29568,29736,29904,30072,30240,30408,30576,30744,30912,31080,31248,31416,31584,31752,31920,32088,32256,32424,32592,32760,32928,33096,33264,33432,33600,33768,33936,34104,34272,34440,34608,34776,34944,35112,35280,35448,35616,35784,35952,36120,36288,36456,36624,36792,36960,37128,37296,37464,37632,37800,37968,38136,38304,38472,38640,38808,38976,39144,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28899,29068,29237,29406,29575,29744,29913,30082,30251,30420,30589,30758,30927,31096,31265,31434,31603,31772,31941,32110,32279,32448,32617,32786,32955,33124,33293,33462,33631,33800,33969,34138,34307,34476,34645,34814,34983,35152,35321,35490,35659,35828,35997,36166,36335,36504,36673,36842,37011,37180,37349,37518,37687,37856,38025,38194,38363,38532,38701,38870,39039,39208,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28899,29068,29237,29406,29575,29744,29913,30082,30251,30420,30589,30758,30927,31096,31265,31434,31603,31772,31941,32110,32279,32448,32617,32786,32955,33124,33293,33462,33631,33800,33969,34138,34307,34476,34645,34814,34983,35152,35321,35490,35659,35828,35997,36166,36335,36504,36673,36842,37011,37180,37349,37518,37687,37856,38025,38194,38363,38532,38701,38870,39039,39208,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29240,29410,29580,29750,29920,30090,30260,30430,30600,30770,30940,31110,31280,31450,31620,31790,31960,32130,32300,32470,32640,32810,32980,33150,33320,33490,33660,33830,34000,34170,34340,34510,34680,34850,35020,35190,35360,35530,35700,35870,36040,36210,36380,36550,36720,36890,37060,37230,37400,37570,37740,37910,38080,38250,38420,38590,38760,38930,39100,39270,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29240,29410,29580,29750,29920,30090,30260,30430,30600,30770,30940,31110,31280,31450,31620,31790,31960,32130,32300,32470,32640,32810,32980,33150,33320,33490,33660,33830,34000,34170,34340,34510,34680,34850,35020,35190,35360,35530,35700,35870,36040,36210,36380,36550,36720,36890,37060,37230,37400,37570,37740,37910,38080,38250,38420,38590,38760,38930,39100,39270,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29583,29754,29925,30096,30267,30438,30609,30780,30951,31122,31293,31464,31635,31806,31977,32148,32319,32490,32661,32832,33003,33174,33345,33516,33687,33858,34029,34200,34371,34542,34713,34884,35055,35226,35397,35568,35739,35910,36081,36252,36423,36594,36765,36936,37107,37278,37449,37620,37791,37962,38133,38304,38475,38646,38817,38988,39159,39330,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29583,29754,29925,30096,30267,30438,30609,30780,30951,31122,31293,31464,31635,31806,31977,32148,32319,32490,32661,32832,33003,33174,33345,33516,33687,33858,34029,34200,34371,34542,34713,34884,35055,35226,35397,35568,35739,35910,36081,36252,36423,36594,36765,36936,37107,37278,37449,37620,37791,37962,38133,38304,38475,38646,38817,38988,39159,39330,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29928,30100,30272,30444,30616,30788,30960,31132,31304,31476,31648,31820,31992,32164,32336,32508,32680,32852,33024,33196,33368,33540,33712,33884,34056,34228,34400,34572,34744,34916,35088,35260,35432,35604,35776,35948,36120,36292,36464,36636,36808,36980,37152,37324,37496,37668,37840,38012,38184,38356,38528,38700,38872,39044,39216,39388,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29928,30100,30272,30444,30616,30788,30960,31132,31304,31476,31648,31820,31992,32164,32336,32508,32680,32852,33024,33196,33368,33540,33712,33884,34056,34228,34400,34572,34744,34916,35088,35260,35432,35604,35776,35948,36120,36292,36464,36636,36808,36980,37152,37324,37496,37668,37840,38012,38184,38356,38528,38700,38872,39044,39216,39388,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30275,30448,30621,30794,30967,31140,31313,31486,31659,31832,32005,32178,32351,32524,32697,32870,33043,33216,33389,33562,33735,33908,34081,34254,34427,34600,34773,34946,35119,35292,35465,35638,35811,35984,36157,36330,36503,36676,36849,37022,37195,37368,37541,37714,37887,38060,38233,38406,38579,38752,38925,39098,39271,39444,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30275,30448,30621,30794,30967,31140,31313,31486,31659,31832,32005,32178,32351,32524,32697,32870,33043,33216,33389,33562,33735,33908,34081,34254,34427,34600,34773,34946,35119,35292,35465,35638,35811,35984,36157,36330,36503,36676,36849,37022,37195,37368,37541,37714,37887,38060,38233,38406,38579,38752,38925,39098,39271,39444,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30624,30798,30972,31146,31320,31494,31668,31842,32016,32190,32364,32538,32712,32886,33060,33234,33408,33582,33756,33930,34104,34278,34452,34626,34800,34974,35148,35322,35496,35670,35844,36018,36192,36366,36540,36714,36888,37062,37236,37410,37584,37758,37932,38106,38280,38454,38628,38802,38976,39150,39324,39498,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30624,30798,30972,31146,31320,31494,31668,31842,32016,32190,32364,32538,32712,32886,33060,33234,33408,33582,33756,33930,34104,34278,34452,34626,34800,34974,35148,35322,35496,35670,35844,36018,36192,36366,36540,36714,36888,37062,37236,37410,37584,37758,37932,38106,38280,38454,38628,38802,38976,39150,39324,39498,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30975,31150,31325,31500,31675,31850,32025,32200,32375,32550,32725,32900,33075,33250,33425,33600,33775,33950,34125,34300,34475,34650,34825,35000,35175,35350,35525,35700,35875,36050,36225,36400,36575,36750,36925,37100,37275,37450,37625,37800,37975,38150,38325,38500,38675,38850,39025,39200,39375,39550,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30975,31150,31325,31500,31675,31850,32025,32200,32375,32550,32725,32900,33075,33250,33425,33600,33775,33950,34125,34300,34475,34650,34825,35000,35175,35350,35525,35700,35875,36050,36225,36400,36575,36750,36925,37100,37275,37450,37625,37800,37975,38150,38325,38500,38675,38850,39025,39200,39375,39550,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31328,31504,31680,31856,32032,32208,32384,32560,32736,32912,33088,33264,33440,33616,33792,33968,34144,34320,34496,34672,34848,35024,35200,35376,35552,35728,35904,36080,36256,36432,36608,36784,36960,37136,37312,37488,37664,37840,38016,38192,38368,38544,38720,38896,39072,39248,39424,39600,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31328,31504,31680,31856,32032,32208,32384,32560,32736,32912,33088,33264,33440,33616,33792,33968,34144,34320,34496,34672,34848,35024,35200,35376,35552,35728,35904,36080,36256,36432,36608,36784,36960,37136,37312,37488,37664,37840,38016,38192,38368,38544,38720,38896,39072,39248,39424,39600,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31683,31860,32037,32214,32391,32568,32745,32922,33099,33276,33453,33630,33807,33984,34161,34338,34515,34692,34869,35046,35223,35400,35577,35754,35931,36108,36285,36462,36639,36816,36993,37170,37347,37524,37701,37878,38055,38232,38409,38586,38763,38940,39117,39294,39471,39648,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31683,31860,32037,32214,32391,32568,32745,32922,33099,33276,33453,33630,33807,33984,34161,34338,34515,34692,34869,35046,35223,35400,35577,35754,35931,36108,36285,36462,36639,36816,36993,37170,37347,37524,37701,37878,38055,38232,38409,38586,38763,38940,39117,39294,39471,39648,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32040,32218,32396,32574,32752,32930,33108,33286,33464,33642,33820,33998,34176,34354,34532,34710,34888,35066,35244,35422,35600,35778,35956,36134,36312,36490,36668,36846,37024,37202,37380,37558,37736,37914,38092,38270,38448,38626,38804,38982,39160,39338,39516,39694,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32040,32218,32396,32574,32752,32930,33108,33286,33464,33642,33820,33998,34176,34354,34532,34710,34888,35066,35244,35422,35600,35778,35956,36134,36312,36490,36668,36846,37024,37202,37380,37558,37736,37914,38092,38270,38448,38626,38804,38982,39160,39338,39516,39694,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32399,32578,32757,32936,33115,33294,33473,33652,33831,34010,34189,34368,34547,34726,34905,35084,35263,35442,35621,35800,35979,36158,36337,36516,36695,36874,37053,37232,37411,37590,37769,37948,38127,38306,38485,38664,38843,39022,39201,39380,39559,39738,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32399,32578,32757,32936,33115,33294,33473,33652,33831,34010,34189,34368,34547,34726,34905,35084,35263,35442,35621,35800,35979,36158,36337,36516,36695,36874,37053,37232,37411,37590,37769,37948,38127,38306,38485,38664,38843,39022,39201,39380,39559,39738,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32760,32940,33120,33300,33480,33660,33840,34020,34200,34380,34560,34740,34920,35100,35280,35460,35640,35820,36000,36180,36360,36540,36720,36900,37080,37260,37440,37620,37800,37980,38160,38340,38520,38700,38880,39060,39240,39420,39600,39780,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32760,32940,33120,33300,33480,33660,33840,34020,34200,34380,34560,34740,34920,35100,35280,35460,35640,35820,36000,36180,36360,36540,36720,36900,37080,37260,37440,37620,37800,37980,38160,38340,38520,38700,38880,39060,39240,39420,39600,39780,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33123,33304,33485,33666,33847,34028,34209,34390,34571,34752,34933,35114,35295,35476,35657,35838,36019,36200,36381,36562,36743,36924,37105,37286,37467,37648,37829,38010,38191,38372,38553,38734,38915,39096,39277,39458,39639,39820,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33123,33304,33485,33666,33847,34028,34209,34390,34571,34752,34933,35114,35295,35476,35657,35838,36019,36200,36381,36562,36743,36924,37105,37286,37467,37648,37829,38010,38191,38372,38553,38734,38915,39096,39277,39458,39639,39820,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33488,33670,33852,34034,34216,34398,34580,34762,34944,35126,35308,35490,35672,35854,36036,36218,36400,36582,36764,36946,37128,37310,37492,37674,37856,38038,38220,38402,38584,38766,38948,39130,39312,39494,39676,39858,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33488,33670,33852,34034,34216,34398,34580,34762,34944,35126,35308,35490,35672,35854,36036,36218,36400,36582,36764,36946,37128,37310,37492,37674,37856,38038,38220,38402,38584,38766,38948,39130,39312,39494,39676,39858,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33855,34038,34221,34404,34587,34770,34953,35136,35319,35502,35685,35868,36051,36234,36417,36600,36783,36966,37149,37332,37515,37698,37881,38064,38247,38430,38613,38796,38979,39162,39345,39528,39711,39894,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33855,34038,34221,34404,34587,34770,34953,35136,35319,35502,35685,35868,36051,36234,36417,36600,36783,36966,37149,37332,37515,37698,37881,38064,38247,38430,38613,38796,38979,39162,39345,39528,39711,39894,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34224,34408,34592,34776,34960,35144,35328,35512,35696,35880,36064,36248,36432,36616,36800,36984,37168,37352,37536,37720,37904,38088,38272,38456,38640,38824,39008,39192,39376,39560,39744,39928,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34224,34408,34592,34776,34960,35144,35328,35512,35696,35880,36064,36248,36432,36616,36800,36984,37168,37352,37536,37720,37904,38088,38272,38456,38640,38824,39008,39192,39376,39560,39744,39928,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34595,34780,34965,35150,35335,35520,35705,35890,36075,36260,36445,36630,36815,37000,37185,37370,37555,37740,37925,38110,38295,38480,38665,38850,39035,39220,39405,39590,39775,39960,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34595,34780,34965,35150,35335,35520,35705,35890,36075,36260,36445,36630,36815,37000,37185,37370,37555,37740,37925,38110,38295,38480,38665,38850,39035,39220,39405,39590,39775,39960,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34968,35154,35340,35526,35712,35898,36084,36270,36456,36642,36828,37014,37200,37386,37572,37758,37944,38130,38316,38502,38688,38874,39060,39246,39432,39618,39804,39990,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34968,35154,35340,35526,35712,35898,36084,36270,36456,36642,36828,37014,37200,37386,37572,37758,37944,38130,38316,38502,38688,38874,39060,39246,39432,39618,39804,39990,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35343,35530,35717,35904,36091,36278,36465,36652,36839,37026,37213,37400,37587,37774,37961,38148,38335,38522,38709,38896,39083,39270,39457,39644,39831,40018,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35343,35530,35717,35904,36091,36278,36465,36652,36839,37026,37213,37400,37587,37774,37961,38148,38335,38522,38709,38896,39083,39270,39457,39644,39831,40018,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35720,35908,36096,36284,36472,36660,36848,37036,37224,37412,37600,37788,37976,38164,38352,38540,38728,38916,39104,39292,39480,39668,39856,40044,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35720,35908,36096,36284,36472,36660,36848,37036,37224,37412,37600,37788,37976,38164,38352,38540,38728,38916,39104,39292,39480,39668,39856,40044,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36099,36288,36477,36666,36855,37044,37233,37422,37611,37800,37989,38178,38367,38556,38745,38934,39123,39312,39501,39690,39879,40068,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36099,36288,36477,36666,36855,37044,37233,37422,37611,37800,37989,38178,38367,38556,38745,38934,39123,39312,39501,39690,39879,40068,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36480,36670,36860,37050,37240,37430,37620,37810,38000,38190,38380,38570,38760,38950,39140,39330,39520,39710,39900,40090,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36480,36670,36860,37050,37240,37430,37620,37810,38000,38190,38380,38570,38760,38950,39140,39330,39520,39710,39900,40090,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36863,37054,37245,37436,37627,37818,38009,38200,38391,38582,38773,38964,39155,39346,39537,39728,39919,40110,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36863,37054,37245,37436,37627,37818,38009,38200,38391,38582,38773,38964,39155,39346,39537,39728,39919,40110,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37248,37440,37632,37824,38016,38208,38400,38592,38784,38976,39168,39360,39552,39744,39936,40128,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37248,37440,37632,37824,38016,38208,38400,38592,38784,38976,39168,39360,39552,39744,39936,40128,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37635,37828,38021,38214,38407,38600,38793,38986,39179,39372,39565,39758,39951,40144,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37635,37828,38021,38214,38407,38600,38793,38986,39179,39372,39565,39758,39951,40144,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38024,38218,38412,38606,38800,38994,39188,39382,39576,39770,39964,40158,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38024,38218,38412,38606,38800,38994,39188,39382,39576,39770,39964,40158,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38415,38610,38805,39000,39195,39390,39585,39780,39975,40170,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38415,38610,38805,39000,39195,39390,39585,39780,39975,40170,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38808,39004,39200,39396,39592,39788,39984,40180,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38808,39004,39200,39396,39592,39788,39984,40180,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38809,39006,39203,39400,39597,39794,39991,40188,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38809,39006,39203,39400,39597,39794,39991,40188,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38809,39006,39204,39402,39600,39798,39996,40194,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38809,39006,39204,39402,39600,39798,39996,40194,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38809,39006,39204,39402,39601,39800,39999,40198,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38809,39006,39204,39402,39601,39800,39999,40198,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38809,39006,39204,39402,39601,39800,40000,40200,
    1,2,4,6,9,12,16,20,25,30,36,42,49,56,64,72,81,90,100,110,121,132,144,156,169,182,196,210,225,240,256,272,289,306,324,342,361,380,400,420,441,462,484,506,529,552,576,600,625,650,676,702,729,756,784,812,841,870,900,930,961,992,1024,1056,1089,1122,1156,1190,1225,1260,1296,1332,1369,1406,1444,1482,1521,1560,1600,1640,1681,1722,1764,1806,1849,1892,1936,1980,2025,2070,2116,2162,2209,2256,2304,2352,2401,2450,2500,2550,2601,2652,2704,2756,2809,2862,2916,2970,3025,3080,3136,3192,3249,3306,3364,3422,3481,3540,3600,3660,3721,3782,3844,3906,3969,4032,4096,4160,4225,4290,4356,4422,4489,4556,4624,4692,4761,4830,4900,4970,5041,5112,5184,5256,5329,5402,5476,5550,5625,5700,5776,5852,5929,6006,6084,6162,6241,6320,6400,6480,6561,6642,6724,6806,6889,6972,7056,7140,7225,7310,7396,7482,7569,7656,7744,7832,7921,8010,8100,8190,8281,8372,8464,8556,8649,8742,8836,8930,9025,9120,9216,9312,9409,9506,9604,9702,9801,9900,10000,10100,10201,10302,10404,10506,10609,10712,10816,10920,11025,11130,11236,11342,11449,11556,11664,11772,11881,11990,12100,12210,12321,12432,12544,12656,12769,12882,12996,13110,13225,13340,13456,13572,13689,13806,13924,14042,14161,14280,14400,14520,14641,14762,14884,15006,15129,15252,15376,15500,15625,15750,15876,16002,16129,16256,16384,16512,16641,16770,16900,17030,17161,17292,17424,17556,17689,17822,17956,18090,18225,18360,18496,18632,18769,18906,19044,19182,19321,19460,19600,19740,19881,20022,20164,20306,20449,20592,20736,20880,21025,21170,21316,21462,21609,21756,21904,22052,22201,22350,22500,22650,22801,22952,23104,23256,23409,23562,23716,23870,24025,24180,24336,24492,24649,24806,24964,25122,25281,25440,25600,25760,25921,26082,26244,26406,26569,26732,26896,27060,27225,27390,27556,27722,27889,28056,28224,28392,28561,28730,28900,29070,29241,29412,29584,29756,29929,30102,30276,30450,30625,30800,30976,31152,31329,31506,31684,31862,32041,32220,32400,32580,32761,32942,33124,33306,33489,33672,33856,34040,34225,34410,34596,34782,34969,35156,35344,35532,35721,35910,36100,36290,36481,36672,36864,37056,37249,37442,37636,37830,38025,38220,38416,38612,38809,39006,39204,39402,39601,39800,40000,40200,
  };

}}}

#endif

// STATISTICS: int-prop

