/*
 * $Id: chip_intel_82443LX.c,v 1.66 2012-02-22 09:27:19 siflkres Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * This is an implementation of the host controller of the
 * Intel 440LX chipset.
 */

#define DEBUG_SMRAM	0
#define DEBUG_CHIPSET	0

#include "config.h"

#include <assert.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue.h"
#include "pci.h"	/* Should go away - FIXME VOSSI */

#include "chip_intel_82443LX.h"

#define CHIP_(x)	chip_intel_82443LX_ ## x

enum CHIP_(orig) {
	ORIG_HOST_BUS,
	ORIG_PCI_BUS,
};

struct cpssp {
	/*
	 * Config
	 */

	/*
	 * Signals
	 */
	struct sig_host_bus_main *port_host_bus;
	struct sig_pci_bus_main *port_pci_bus;
	struct sig_agp_bus_main *port_agp_bus;
	struct sig_cs *sig_mem_cs[8];

	/*
	 * State
	 */
	uint32_t confsp[64];

	int selected_smm;
	uint32_t selected_type;
	uint32_t selected_addr;

	uint32_t confadd; /* Currently addressed device/register */
};

/* ----------------------------------------------------------------- */
/* Built-In Functions                                                */
/* ----------------------------------------------------------------- */

/*
 * FIXME:
 * This (optional, for ACPI) register is used to disable both
 * the PCI and AGP arbiters in the 82443LX to prevent any
 * external bus masters from acquiring the PCI or AGP bus.
 */
static int
CHIP_(pm2_ctl_read)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int *bsp,
	uint32_t *datap
)
{
	if (((pci_getconfigb(cpssp->confsp, 0x7a) >> 6) & 1)
	 && addr == 0x0020
	 && (*bsp >> 2) & 1) {
		fprintf(stderr, "WARNING: 82443LX (Power Management): "
			"reading from unimplemented PM2_CTL register\n");

		*datap &= ~(0xff << 16);
		*datap |= 0x00 << 16;
		*bsp &= ~(1 << 2);
		return 0;
	}
	return 1;
}

static int
CHIP_(pm2_ctl_write)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int *bsp,
	uint32_t data
)
{
	if (((pci_getconfigb(cpssp->confsp, 0x7a) >> 6) & 1)
	 && addr == 0x0020
	 && (*bsp >> 2) & 1) {
		data >>= 16;
		data &= 0xff;

		fprintf(stderr, "WARNING: 82443LX (Power Management): "
			"writing 0x%02x%s to unimplemented PM2_CTL "
			"register\n", data, (data & ~0x01) ? " (reserved bits!)" : "");

		*bsp &= ~(1 << 2);
		return 0;
	}
	return 1;
}

static int
CHIP_(confadd_read)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int *bsp,
	uint32_t *datap
)
{
	if (addr == 0x0cf8
	 && *bsp == 0xf) {
		*datap = cpssp->confadd;
		*bsp &= ~0xf;
		return 0;
	}
	return 1;
}

static int
CHIP_(confadd_write)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int *bsp,
	uint32_t data
)
{
	if (addr == 0x0cf8
	 && *bsp == 0xf) {
		uint32_t odata;

		data &= ~(1 << 30); /* Clear reserved bits. */
		data &= ~(1 << 29);
		data &= ~(1 << 28);
		data &= ~(1 << 27);
		data &= ~(1 << 26);
		data &= ~(1 << 25);
		data &= ~(1 << 24);
		data &= ~(1 <<  1);
		data &= ~(1 <<  0);

		odata = cpssp->confadd;
		cpssp->confadd = data;

		if (((odata >> 31) & 1) != ((data >> 31) & 1)) {
			sig_host_bus_ior_info_flush(cpssp->port_host_bus, cpssp,
					0x0cfc, 0);
			sig_host_bus_iow_info_flush(cpssp->port_host_bus, cpssp,
					0x0cfc, 0);
		}
		*bsp &= ~0xf;
		return 0;
	}
	return 1;
}

static void
CHIP_(confdata_type_addr)(
	struct cpssp *cpssp,
	unsigned int *typep,
	uint32_t *addrp
)
{
	if ((*typep == SIG_HOST_BUS_IOR
	  || *typep == SIG_HOST_BUS_IOW)
	 && *addrp == 0x0cfc) {
		/*
		 * CONFDATA Register
		 */
		if (((cpssp->confadd >> 16) & 0xff) == 0x00) {
			/* Config Space Type 0 */
			uint32_t dev;
			uint32_t reg;

			dev = (cpssp->confadd >> 11) & 0x1f;
			reg = (cpssp->confadd >>  0) & 0x7fc;

			if (dev < 32 - 11) {
				dev = 1 << (11 + dev);
			} else {
				dev = 0;
			}

			*addrp = dev | reg;
			*typep = (*typep == SIG_HOST_BUS_IOR)
				? SIG_PCI_BUS_C0R : SIG_PCI_BUS_C0W;

		} else {
			/* Config Space Type 1 */
			*addrp = cpssp->confadd & ~3;
			*typep = (*typep == SIG_HOST_BUS_IOR)
				? SIG_PCI_BUS_C1R : SIG_PCI_BUS_C1W;
		}
	}
}

/*
 * This are helper functions - all config spaces writes, no matter if they
 * are byte, word or long accesses, get mapped to this.
 */
static void
CHIP_(_pci_bus_cwriteb)(
	struct cpssp *cpssp,
	unsigned char addr,
	unsigned char val
)
{
	unsigned char oldval;

	switch (addr) {
	case 0x00 ... 0x3f:
		/*
		 * Writing to reserved config space registers *below* 0x40
		 * is allowed. Value written is discarded.
		 */
		/* Nothing to do... */
		break;

		/* NBX Configuration Register */
		/* 3-16 */
	case 0x50: /* Bit 7-0 */
		val &= 0xec; /* Mask "reserved" bits */
		pci_setconfigb(cpssp->confsp, addr, val);
		break;
	case 0x51: /* Bit 15-8 */
		val &= 0xbf; /* Mask "reserved" bits */
		pci_setconfigb(cpssp->confsp, addr, val);
		break;
	case 0x52: /* Bit 23-16 */
		val &= 0x07; /* Mask "reserved" bits */
		pci_setconfigb(cpssp->confsp, addr, val);
		break;
	case 0x53: /* Bit 31-24 */
		pci_setconfigb(cpssp->confsp, addr, val);
		break;

		/* DRAM Control Register */
		/* 3-19 */
	case 0x57:
		/* Just remember settings. Not used by simulation. */
		val &= 0x3f;
		pci_setconfigb(cpssp->confsp, addr, val);
		break;

		/* DRAM Timing Register */
		/* 3-20 */
	case 0x58:
		/* Just remember settings. Not used by simulation. */
		val &= 0x3; /* Mask "reserved" bits */
		pci_setconfigb(cpssp->confsp, addr, val);
		break;

		/* Programmable Attribute Map Register */
		/* 3-20 */
	case 0x59:
		val &= 0x30; /* Mask "reserved" bits */

		oldval = pci_getconfigb(cpssp->confsp, addr);
		if (val != oldval) {
			pci_setconfigb(cpssp->confsp, addr, val);
			sig_host_bus_unmap(cpssp->port_host_bus, cpssp,
					0x000f0000, 0x00010000);
			sig_pci_bus_unmap(cpssp->port_pci_bus, cpssp,
					0x000f0000, 0x00010000);

			if (DEBUG_CHIPSET) {
				faum_log(FAUM_LOG_DEBUG, "82443LX", "", \
					 "Changed PAM register 0x%x to 0x%x\n",
					addr, val);
			}
		}
		break;

		/* Programmable Attribute Map Registers */
		/* 3-20 */
	case 0x5a ... 0x5f:
		val &= 0x33; /* Mask "reserved" bits */

		oldval = pci_getconfigb(cpssp->confsp, addr);
		if (val != oldval) {
			pci_setconfigb(cpssp->confsp, addr, val);
			sig_host_bus_unmap(cpssp->port_host_bus, cpssp,
					0x000c0000 + (addr - 0x5a) * 0x8000, 0x8000);
			sig_pci_bus_unmap(cpssp->port_pci_bus, cpssp,
					0x000c0000 + (addr - 0x5a) * 0x8000, 0x8000);
			if (DEBUG_CHIPSET) {
				faum_log(FAUM_LOG_DEBUG, "82443LX", "", \
					 "Changed PAM register 0x%x to 0x%x\n",
					addr, val);
			}
		}
		break;

		/* DRAM Row Boundary Register */
		/* 3-22 */
	case 0x60 ... 0x67:
		oldval = pci_getconfigb(cpssp->confsp, addr);
		pci_setconfigb(cpssp->confsp, addr, val);
		if (val != oldval) {
			sig_host_bus_unmap(cpssp->port_host_bus, cpssp,
					0x00000000, 0x40000000);
			sig_pci_bus_unmap(cpssp->port_pci_bus, cpssp,
					0x00000000, 0x40000000);
		}
		break;

		/* Fixed DRAM Hole Control Register */
		/* 3-24 */
	case 0x68:
		val &= 0xc0; /* Mask "reserved" bits. */
		if (val != 0) {
			/* FIXME */
			fixme();
		}
		pci_setconfigb(cpssp->confsp, addr, val);
		break;

		/* Memory Buffer Strength Control Register */
		/* 3-25 */
	case 0x69 ... 0x6e:
		/* Just remember settings. Not used by simulation. */
		pci_setconfigb(cpssp->confsp, addr, val);
		break;

		/* System Management RAM Control Register */
		/* 3-28 */
	case 0x72:
		val &= 0x78; /* mask "reserved" bit 7 and readonly bits 2:0 */
		val |= 0x02; /* bits 2:0 hardwired to '010' */
		oldval = pci_getconfigb(cpssp->confsp, addr);
		if (oldval & (1 << 4) /* D_LCK */) {
			/* if D_LCK is set, D_OPEN can't be set any more;
			 * D_LCK can only be cleared by a power-on reset */
			val &= ~(1 << 6); /* D_OPEN */
			val |= (1 << 4); /* D_LCK */
		}
		if (val != oldval) {
			if (DEBUG_SMRAM) {
				faum_log(FAUM_LOG_DEBUG, "82443LX", "",
					"Writing 0x%02x to SMRAM "
					"Control (was 0x%02x)\n", val, oldval);
			}

			pci_setconfigb(cpssp->confsp, addr, val);
			/* SMRAM mapping may have changed */
			sig_host_bus_unmap(cpssp->port_host_bus, cpssp,
					0x000a0000, 0x00020000);
			sig_pci_bus_unmap(cpssp->port_pci_bus, cpssp,
					0x000a0000, 0x00020000);
		}
		break;
		
		/* Extended System Management RAM Control Register */
		/* 3-29 */
	case 0x73:
		goto unimplemented;

		/* SDRAM Row Page Size Register */
		/* 3-30 */
	case 0x74 ... 0x75:
		goto unimplemented;

		/* SDRAM Control Register */
		/* 3-30 */
	case 0x76 ... 0x77:
		goto unimplemented;

		/* Paging Policy Register */
		/* 3-32 */
	case 0x78 ... 0x79:
		goto unimplemented;

		/* Power Management Control Register */
		/* 3-33 */
	case 0x7A:
		val &= ~0x02; /* Bit 1 is read only */
		pci_setconfigb(cpssp->confsp, addr, val);

		sig_host_bus_ior_info_flush(cpssp->port_host_bus, cpssp,
				0x0020, 1 << 2);
		sig_host_bus_iow_info_flush(cpssp->port_host_bus, cpssp,
				0x0020, 1 << 2);
		break;

		/* Suspend CBR Refresh Rate Register */
		/* 3-34 */
	case 0x7b ... 0x7c:
		goto unimplemented;

		/* Error Address Pointer Register */
		/* 3-35 */
	case 0x80 ... 0x83:
		goto unimplemented;

		/* Error Command Register */
		/* 3-36 */
	case 0x90:
		/* Just remember settings. Not used by simulation. */
		pci_setconfigb(cpssp->confsp, addr, val);
		break;

		/* Error Status Register */
		/* 3-37 */
	case 0x91 ... 0x92:
		/* As we don't set any error bits we don't need to
		 * reset them... */
		break;

		/* AGP Capability Identifier Register */
		/* 3-38 */
	case 0xa0 ... 0xa3:
		goto unimplemented;

		/* AGP Status Register */
		/* 3-38 */
	case 0xa4 ... 0xa7:
		goto unimplemented;

		/* AGP Command Register */
		/* 3-39 */
	case 0xa8 ... 0xab:
		goto unimplemented;

		/* AGP Control Register */
		/* 3-40 */
	case 0xb0 ... 0xb3:
		goto unimplemented;

		/* Aperture Size Register */
		/* 3-41 */
	case 0xb4:
		val &= 0x3f; /* Masking "reserved" bits. */
		pci_setconfigb(cpssp->confsp, addr, val);
		break;

		/* Aperture Translation Table Base Register */
		/* 3-41 */
	case 0xb8 ... 0xbb:
		goto unimplemented;

		/* Memory Buffer Frequency Select Register */
		/* 3-42 */
	case 0xca ... 0xcc:
		goto unimplemented;

		/* BIOS Scratch Pad Register */
		/* 3-44 */
	case 0xd0 ... 0xd7:
		goto unimplemented;

		/* DRAM Write Thermal Throttling Control Register */
		/* 3-44 */
		/* Just remember settings. Not used by simulation. */
	case 0xe0: /* Bit 7-0 */
	case 0xe1: /* Bit 15-8 */
	case 0xe2: /* Bit 23-16 */
	case 0xe3: /* Bit 31-24 */
	case 0xe4: /* Bit 39-32 */
		if (! ((pci_getconfigb(cpssp->confsp, 0xe7) >> 7) & 1)) {
			pci_setconfigb(cpssp->confsp, addr, val);
		}
		break;
	case 0xe5: /* Bit 47-40 */
		if (! ((pci_getconfigb(cpssp->confsp, 0xe7) >> 7) & 1)) {
			val &= 0x3f; /* Masking "reserved" bits. */
			pci_setconfigb(cpssp->confsp, addr, val);
		}
		break;
	case 0xe6: /* Bit 55-48 */
		/* All bits "reserved". */
		break;
	case 0xe7: /* Bit 63-56 */
		if (! ((pci_getconfigb(cpssp->confsp, 0xe7) >> 7) & 1)) {
			val &= 0x80; /* Masking "reserved" bits. */
			pci_setconfigb(cpssp->confsp, addr, val);
		}
		break;

		/* DRAM Read Thermal Throttling Control Register */
		/* 3-46 */
		/* Just remember settings. Not used by simulation. */
	case 0xe8: /* Bit 7-0 */
	case 0xe9: /* Bit 15-8 */
	case 0xea: /* Bit 23-16 */
	case 0xeb: /* Bit 31-24 */
	case 0xec: /* Bit 39-32 */
		if (! ((pci_getconfigb(cpssp->confsp, 0xe7) >> 7) & 1)) {
			pci_setconfigb(cpssp->confsp, addr, val);
		}
		break;
	case 0xed: /* Bit 47-40 */
		if (! ((pci_getconfigb(cpssp->confsp, 0xe7) >> 7) & 1)) {
			val &= 0x3f; /* Masking "reserved" bits. */
			pci_setconfigb(cpssp->confsp, addr, val);
		}
		break;
	case 0xee: /* Bit 55-48 */
	case 0xef: /* Bit 63-56 */
		/* All bits "reserved". */
		break;

		/* Buffer Control Register */
		/* 3-47 */
	case 0xf0 ... 0xf1:
		goto unimplemented;

	unimplemented:;
		faum_log(FAUM_LOG_WARNING, "82443LX (Host Controller)", "",
				"Writing 0x%02x to unimplemented register 0x%02x.\n",
				val, addr);
		break;

	default:
		faum_log(FAUM_LOG_WARNING, "82443LX (Host Controller)", "",
				"Writing 0x%02x to reserved register 0x%02x.\n",
				val, addr);
		break;
	}
}

static int
CHIP_(config_space_read)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int *bsp,
	uint32_t *valp
)
{
	if (((addr >> 11) & 1)		/* Device 0 */
	 && ((addr >> 8) & 7) == 0) {	/* Function 0 */
		/* Host/PCI Bridge */
		if ((*bsp >> 0) & 1) {
			*valp &= ~(0xff << 0);
			*valp |= pci_getconfigb(cpssp->confsp,
					(addr + 0) & 0xff) << 0;
		}
		if ((*bsp >> 1) & 1) {
			*valp &= ~(0xff << 8);
			*valp |= pci_getconfigb(cpssp->confsp,
					(addr + 1) & 0xff) << 8;
		}
		if ((*bsp >> 2) & 1) {
			*valp &= ~(0xff << 16);
			*valp |= pci_getconfigb(cpssp->confsp,
					(addr + 2) & 0xff) << 16;
		}
		if ((*bsp >> 3) & 1) {
			*valp &= ~(0xff << 24);
			*valp |= pci_getconfigb(cpssp->confsp,
					(addr + 3) & 0xff) << 24;
		}
		*bsp &= ~0xf;
		return 0;

	} else if (((addr >> 12) & 1)		/* Device 1 */
		&& ((addr >> 8) & 7) == 0) {	/* Function 0 */
		/* AGP - FIXME VOSSI */
		return -1;

	} else {
		return -1;
	}
}

static int
CHIP_(config_space_write)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int *bsp,
	uint32_t val
)
{
	if (((addr >> 11) & 1)		/* Device 0 */
	 && ((addr >> 8) & 7) == 0) {	/* Function 0 */
		/* Host/PCI Bridge */
		if ((*bsp >> 0) & 1) {
			CHIP_(_pci_bus_cwriteb)(cpssp,
				(addr + 0) & 0xff, (val >>  0) & 0xff);
		}
		if ((*bsp >> 1) & 1) {
			CHIP_(_pci_bus_cwriteb)(cpssp,
				(addr + 1) & 0xff, (val >>  8) & 0xff);
		}
		if ((*bsp >> 2) & 1) {
			CHIP_(_pci_bus_cwriteb)(cpssp,
				(addr + 2) & 0xff, (val >>  16) & 0xff);
		}
		if ((*bsp >> 3) & 1) {
			CHIP_(_pci_bus_cwriteb)(cpssp,
				(addr + 3) & 0xff, (val >>  24) & 0xff);
		}
		*bsp &= ~0xf;
		return 0;

	} else if (((addr >> 12) & 1)		/* Device 1 */
		&& ((addr >> 8) & 7) == 0) {	/* Function 0 */
		/* AGP - FIXME VOSSI */
		return -1;

	} else {
		return -1;
	}
}

/* --------------------------------------------------------------------- */
/* Memory Interface                                                      */
/* --------------------------------------------------------------------- */

static int
CHIP_(region)(
	struct cpssp *cpssp,
	int wflag,
	unsigned int smm,
	uint32_t addr
)
{
	int region;
	unsigned int rw;

	if ((addr >> 23) < pci_getconfigb(cpssp->confsp, 0x60)) {
		region = 0;
	} else if ((addr >> 23) < pci_getconfigb(cpssp->confsp, 0x61)) {
		region = 1;
	} else if ((addr >> 23) < pci_getconfigb(cpssp->confsp, 0x62)) {
		region = 2;
	} else if ((addr >> 23) < pci_getconfigb(cpssp->confsp, 0x63)) {
		region = 3;
	} else if ((addr >> 23) < pci_getconfigb(cpssp->confsp, 0x64)) {
		region = 4;
	} else if ((addr >> 23) < pci_getconfigb(cpssp->confsp, 0x65)) {
		region = 5;
	} else if ((addr >> 23) < pci_getconfigb(cpssp->confsp, 0x66)) {
		region = 6;
	} else if ((addr >> 23) < pci_getconfigb(cpssp->confsp, 0x67)) {
		region = 7;
	} else {
		/* Shortcut */
		return -1;
	}

	if (/* 0x00000000 <= addr && */ addr < 0x000a0000) {
		/*
		 * Low memory.
		 */
		rw = 3;

	} else if (0x000a0000 <= addr && addr < 0x000c0000) {
		/*
		 * VGA hole / SMRAM
		 */
		unsigned int n;

		n = pci_getconfigb(cpssp->confsp, 0x72); /*SMRAM Control*/
		if ((n & (1 << 3) /* G_SMRAME */)
		 && (smm || ((n & (1 << 6) /* D_OPEN */)))) {
			/* SMRAM */
			rw = 3;
		} else {
			/* VGA Hole */
			/* Shortcut */
			return -1;
		}

	} else if (0x000c0000 <= addr && addr < 0x000c4000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5a) & 0xf;

	} else if (0x000c4000 <= addr && addr < 0x000c8000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5a) >> 4;

	} else if (0x000c8000 <= addr && addr < 0x000cc000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5b) & 0xf;

	} else if (0x000cc000 <= addr && addr < 0x000d0000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5b) >> 4;

	} else if (0x000d0000 <= addr && addr < 0x000d4000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5c) & 0xf;

	} else if (0x000d4000 <= addr && addr < 0x000d8000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5c) >> 4;

	} else if (0x000d8000 <= addr && addr < 0x000dc000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5d) & 0xf;

	} else if (0x000dc000 <= addr && addr < 0x000e0000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5d) >> 4;

	} else if (0x000e0000 <= addr && addr < 0x000e4000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5e) & 0xf;

	} else if (0x000e4000 <= addr && addr < 0x000e8000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5e) >> 4;

	} else if (0x000e8000 <= addr && addr < 0x000ec000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5f) & 0xf;

	} else if (0x000ec000 <= addr && addr < 0x000f0000) {
		rw = pci_getconfigb(cpssp->confsp, 0x5f) >> 4;

	} else if (0x000f0000 <= addr && addr < 0x00100000) {
		/*
		 * BIOS Area configured by PAM register 0x59.
		 */
		rw = pci_getconfigb(cpssp->confsp, 0x59) >> 4;

	} else {
		/*
		 * High memory.
		 */
		rw = 3;
	}

	if (! (rw & (1 << wflag))) {
		return -1;
	}

	return region;
}

static int
CHIP_(mem_mr)(
	struct cpssp *cpssp,
	int smm,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	int region;

	region = CHIP_(region)(cpssp, 0, smm, addr);
	if (0 <= region
	 && region <= 7) {
		return sig_cs_read(cpssp->sig_mem_cs[region], cpssp,
				addr, bs, valp);
	}
	return 1;
}

static int
CHIP_(mem_mw)(
	struct cpssp *cpssp,
	int smm,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	int region;

	region = CHIP_(region)(cpssp, 1, smm, addr);
	if (0 <= region
	 && region <= 7) {
		return sig_cs_write(cpssp->sig_mem_cs[region], cpssp,
				addr, bs, val);
	}
	return 1;
}

static int
CHIP_(mem_map_r)(
	struct cpssp *cpssp,
	enum CHIP_(orig) orig,
	unsigned int smm,
	uint32_t addr,
	char **haddr_p
)
{
	int region;
	char *haddr_dummy;

	region = CHIP_(region)(cpssp, 0, smm, addr);
	if (0 <= region
	 && region <= 7) {
		return sig_cs_map(cpssp->sig_mem_cs[region], cpssp,
				addr, haddr_p, &haddr_dummy);
	}
	return 1;
}

static int
CHIP_(mem_map_w)(
	struct cpssp *cpssp,
	enum CHIP_(orig) orig,
	unsigned int smm,
	uint32_t addr,
	char **haddr_p
)
{
	int region;
	char *haddr_dummy;

	region = CHIP_(region)(cpssp, 1, smm, addr);
	if (0 <= region
	 && region <= 7) {
		return sig_cs_map(cpssp->sig_mem_cs[region], cpssp,
				addr, &haddr_dummy, haddr_p);
	}
	return 1;
}

static int
CHIP_(memX_unmap)(void *_cpssp, unsigned long pa, unsigned long len)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	/* Should unmap interval [pa, pa + len) only - FIXME VOSSI */
	sig_host_bus_unmap(cpssp->port_host_bus, cpssp, 0, 1024*1024*1024);
	sig_pci_bus_unmap(cpssp->port_pci_bus, cpssp, 0, 1024*1024*1024);

	return 0;
}

/* --------------------------------------------------------------------- */
/* DMI Interface                                                         */
/* --------------------------------------------------------------------- */

static int
CHIP_(pci_read)(
	struct cpssp *cpssp,
	unsigned int type,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	int ret;

	switch (type) {
	case SIG_PCI_BUS_IOR:
		ret = sig_pci_bus_ior(cpssp->port_pci_bus, cpssp,
				addr, bs, valp);
		break;
	case SIG_PCI_BUS_C0R:
		ret = sig_pci_bus_main_c0r(cpssp->port_pci_bus, cpssp,
				addr, bs, valp);
		break;
	case SIG_PCI_BUS_C1R:
		ret = sig_pci_bus_c1r(cpssp->port_pci_bus, cpssp,
				addr, bs, valp);
		break;
	case SIG_PCI_BUS_MR:
		ret = sig_pci_bus_mr(cpssp->port_pci_bus, cpssp,
				addr, bs, valp);
		break;
	default:
		assert(0);
	}
	if (ret) {
		sig_pci_bus_main_type_addr(cpssp->port_pci_bus, cpssp,
				type, addr);
		ret = sig_pci_bus_main_read_data(cpssp->port_pci_bus, cpssp,
				bs, valp);
	}
	return ret;
}

static int
CHIP_(pci_write)(
	struct cpssp *cpssp,
	unsigned int type,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	int ret;

	switch (type) {
	case SIG_PCI_BUS_IOW:
		ret = sig_pci_bus_iow(cpssp->port_pci_bus, cpssp,
				addr, bs, val);
		break;
	case SIG_PCI_BUS_C0W:
		ret = sig_pci_bus_main_c0w(cpssp->port_pci_bus, cpssp,
				addr, bs, val);
		break;
	case SIG_PCI_BUS_C1W:
		ret = sig_pci_bus_c1w(cpssp->port_pci_bus, cpssp,
				addr, bs, val);
		break;
	case SIG_PCI_BUS_MW:
		ret = sig_pci_bus_mw(cpssp->port_pci_bus, cpssp,
				addr, bs, val);
		break;
	default:
		assert(0);
	}
	if (ret) {
		sig_pci_bus_main_type_addr(cpssp->port_pci_bus, cpssp,
				type, addr);
		ret = sig_pci_bus_main_write_data(cpssp->port_pci_bus, cpssp,
				bs, val);
	}
	return ret;
}

static int
CHIP_(pci_map_r)(
	struct cpssp *cpssp,
	uint32_t addr,
	char **haddr_p
)
{
	return sig_pci_bus_map_r(cpssp->port_pci_bus, cpssp, addr, haddr_p);
}

static int
CHIP_(pci_map_w)(
	struct cpssp *cpssp,
	uint32_t addr,
	char **haddr_p
)
{
	return sig_pci_bus_map_w(cpssp->port_pci_bus, cpssp, addr, haddr_p);
}

/* --------------------------------------------------------------------- */
/* Internal Bus                                                          */
/* --------------------------------------------------------------------- */

static int
CHIP_(ior)(
	void *_css,
	uint32_t port,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;
	unsigned int type;
	int ret;

	type = SIG_HOST_BUS_IOR;
	ret = 1;
	ret &= CHIP_(pm2_ctl_read)(cpssp, port, &bs, valp);
	ret &= CHIP_(confadd_read)(cpssp, port, &bs, valp);
	CHIP_(confdata_type_addr)(cpssp, &type, &port);
	if (type == SIG_PCI_BUS_C0R) {
		ret &= CHIP_(config_space_read)(cpssp, port, &bs, valp);
	}
	if (bs) {
		ret &= CHIP_(pci_read)(cpssp, type, port, bs, valp);
	}

	return ret;
}

static int
CHIP_(iow)(
	void *_css,
	uint32_t port,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;
	unsigned int type;
	int ret;

	type = SIG_HOST_BUS_IOW;
	ret = 1;
	ret &= CHIP_(pm2_ctl_write)(cpssp, port, &bs, val);
	ret &= CHIP_(confadd_write)(cpssp, port, &bs, val);
	CHIP_(confdata_type_addr)(cpssp, &type, &port);
	if (type == SIG_PCI_BUS_C0W) {
		ret &= CHIP_(config_space_write)(cpssp, port, &bs, val);
	}
	if (bs) {
		ret &= CHIP_(pci_write)(cpssp, type, port, bs, val);
	}

	return ret;
}

static int
CHIP_(ior_info)(
	struct cpssp *cpssp,
	enum CHIP_(orig) orig,
	uint32_t port,
	unsigned int bs,
	int (**cf)(void *, uint32_t, unsigned int, uint32_t *),
	void **cs
)
{
	if (port == 0x0020
	 || port == 0x0cf8
	 || port == 0x0cfc) {
		*cf = CHIP_(ior);
		*cs = cpssp;
		return 0;
	} else {
		return sig_pci_bus_ior_info(cpssp->port_pci_bus, cpssp,
				port, bs, cf, cs);
	}
}

static int
CHIP_(iow_info)(
	struct cpssp *cpssp,
	enum CHIP_(orig) orig,
	uint32_t port,
	unsigned int bs,
	int (**cf)(void *, uint32_t, unsigned int, uint32_t),
	void **cs
)
{
	if (port == 0x0020
	 || port == 0x0cf8
	 || port == 0x0cfc) {
		*cf = CHIP_(iow);
		*cs = cpssp;
		return 0;
	} else {
		return sig_pci_bus_iow_info(cpssp->port_pci_bus, cpssp,
				port, bs, cf, cs);
	}
}

static int
CHIP_(mem_read)(
	struct cpssp *cpssp,
	enum CHIP_(orig) orig,
	unsigned int smm,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	if (CHIP_(mem_mr)(cpssp, smm, addr, bs, valp) == 0
	 || (orig != ORIG_PCI_BUS
	  && CHIP_(pci_read)(cpssp, SIG_PCI_BUS_MR, addr, bs, valp) == 0)) {
		return 0;
	}
	return 1;
}

static int
CHIP_(mem_write)(
	struct cpssp *cpssp,
	enum CHIP_(orig) orig,
	unsigned int smm,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	if (CHIP_(mem_mw)(cpssp, smm, addr, bs, val) == 0
	 || (orig != ORIG_PCI_BUS
	  && CHIP_(pci_write)(cpssp, SIG_PCI_BUS_MW, addr, bs, val) == 0)) {
		return 0;
	}
	return 1;
}

static int
CHIP_(map_r)(
	void *_cpssp,
	enum CHIP_(orig) orig,
	unsigned int smm,
	uint32_t addr,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (CHIP_(mem_map_r)(cpssp, ORIG_HOST_BUS, smm, addr, haddr_p) == 0
	 || CHIP_(pci_map_r)(cpssp, addr, haddr_p) == 0) {
		return 0;
	}
	return 1;
}

static int
CHIP_(map_w)(
	void *_cpssp,
	enum CHIP_(orig) orig,
	unsigned int smm,
	uint32_t addr,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (CHIP_(mem_map_w)(cpssp, ORIG_HOST_BUS, smm, addr, haddr_p) == 0
	 || CHIP_(pci_map_w)(cpssp, addr, haddr_p) == 0) {
		return 0;
	}
	return 1;
}

/* ----------------------------------------------------------------- */
/* Host Bus Interface                                                */
/* ----------------------------------------------------------------- */

static int
CHIP_(host_bus_ior)(
	void *_css,
	uint32_t port,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(ior)(cpssp, port, bs, valp);
}

static int
CHIP_(host_bus_iow)(
	void *_css,
	uint32_t port,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(iow)(cpssp, port, bs, val);
}

static int
CHIP_(host_bus_ior_info)(
	void *_css,
	uint32_t port,
	unsigned int bs,
	int (**cf)(void *, uint32_t, unsigned int, uint32_t *),
	void **cs
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(ior_info)(cpssp, ORIG_HOST_BUS, port, bs, cf, cs);
}

static int
CHIP_(host_bus_iow_info)(
	void *_css,
	uint32_t port,
	unsigned int bs,
	int (**cf)(void *, uint32_t, unsigned int, uint32_t),
	void **cs
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(iow_info)(cpssp, ORIG_HOST_BUS, port, bs, cf, cs);
}

static int
CHIP_(host_bus_inta_addr)(
	void *_css
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return sig_pci_bus_inta_addr(cpssp->port_pci_bus, cpssp);
}

static int
CHIP_(host_bus_inta_data)(
	void *_css,
	uint8_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return sig_pci_bus_inta_data(cpssp->port_pci_bus, cpssp, valp);
}

static int
CHIP_(host_bus_mr)(
	void *_css,
	unsigned int smm,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(mem_read)(cpssp, ORIG_HOST_BUS,
			smm, addr, bs, valp);
}

static int
CHIP_(host_bus_mw)(
	void *_css,
	unsigned int smm,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(mem_write)(cpssp, ORIG_HOST_BUS,
			smm, addr, bs, val);
}

/* Re-used as CHIP_(host_bus_map_x_check). */
static int
CHIP_(host_bus_map_r_check)(
	void *_css,
	unsigned int smm,
	uint32_t pa
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	/* SMM? FIXME */
	return sig_pci_bus_map_r_check(cpssp->port_pci_bus, cpssp, pa);
}

/* Re-used as CHIP_(host_bus_map_x). */
static int
CHIP_(host_bus_map_r)(
	void *_css,
	unsigned int smm,
	uint32_t pa,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(map_r)(cpssp, ORIG_HOST_BUS, smm, pa, haddr_p);
}

static int
CHIP_(host_bus_map_w_check)(
	void *_css,
	unsigned int smm,
	uint32_t pa
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	/* SMM? FIXME */
	return sig_pci_bus_map_w_check(cpssp->port_pci_bus, cpssp, pa);
}

static int
CHIP_(host_bus_map_w)(
	void *_css,
	unsigned int smm,
	uint32_t pa,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(map_w)(cpssp, ORIG_HOST_BUS, smm, pa, haddr_p);
}

static int
CHIP_(host_bus_type_addr)(
	void *_cpssp,
	int smm,
	unsigned int type,
	uint32_t addr
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->selected_smm = smm;
	cpssp->selected_type = type;
	cpssp->selected_addr = addr;

	return 0;
}

static int
CHIP_(host_bus_read_data)(
	void *_cpssp,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	int smm;
	unsigned int type;
	uint32_t addr;

	smm = cpssp->selected_smm;
	type = cpssp->selected_type;
	addr = cpssp->selected_addr;

	cpssp->selected_addr += 4;

	switch (type) {
	case SIG_HOST_BUS_IOR:
		return CHIP_(host_bus_ior)(cpssp, addr, bs, valp);
	case SIG_HOST_BUS_MR:
		return CHIP_(host_bus_mr)(cpssp, smm, addr, bs, valp);
	default:
		assert(0);
	}
}

static int
CHIP_(host_bus_write_data)(
	void *_cpssp,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	int smm;
	unsigned int type;
	uint32_t addr;

	smm = cpssp->selected_smm;
	type = cpssp->selected_type;
	addr = cpssp->selected_addr;

	cpssp->selected_addr += 4;

	switch (type) {
	case SIG_HOST_BUS_IOW:
		return CHIP_(host_bus_iow)(cpssp, addr, bs, val);
	case SIG_HOST_BUS_MW:
		return CHIP_(host_bus_mw)(cpssp, smm, addr, bs, val);
	default:
		assert(0);
	}
}

/* ----------------------------------------------------------------- */
/* PCI Bus Interface                                                 */
/* ----------------------------------------------------------------- */

static int
CHIP_(pci_bus_mr)(
	void *_css,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(mem_read)(cpssp, ORIG_PCI_BUS,
			0, addr, bs, valp);
}

static int
CHIP_(pci_bus_mw)(
	void *_css,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(mem_write)(cpssp, ORIG_PCI_BUS,
			0, addr, bs, val);
}

static int
CHIP_(pci_bus_map_r_check)(
	void *_css,
	uint32_t pa
)
{
	struct cpssp *cpssp = _css;

	/* SMM? FIXME */
	return sig_host_bus_map_r_check(cpssp->port_host_bus, cpssp, 0, pa);
}

static int
CHIP_(pci_bus_map_r)(
	void *_css,
	uint32_t pa,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(map_r)(cpssp, ORIG_PCI_BUS, 0, pa, haddr_p);
}

static int
CHIP_(pci_bus_map_w_check)(
	void *_css,
	uint32_t pa
)
{
	struct cpssp *cpssp = _css;

	/* SMM? FIXME */
	return sig_host_bus_map_w_check(cpssp->port_host_bus, cpssp, 0, pa);
}

static int
CHIP_(pci_bus_map_w)(
	void *_css,
	uint32_t pa,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	return CHIP_(map_w)(cpssp, ORIG_PCI_BUS, 0, pa, haddr_p);
}

static void
CHIP_(pci_bus_unmap)(
	void *_css,
	uint32_t pa,
	uint32_t len
)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	sig_host_bus_unmap(cpssp->port_host_bus, cpssp, pa, len);
}

static void
CHIP_(power_set)(void *_css, unsigned int val)
{
	/* FIXME */
}

static void
CHIP_(n_reset_set)(void *_css, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _css;
	int i;

	/* Set PAM to default */
	for (i = 0x59; i < 0x5f; i++) {
		pci_setconfigb(cpssp->confsp, i, 0x00);
	}
	for (i = 0x60; i < 0x68; i++) {
		pci_setconfigb(cpssp->confsp, i, 0x01);
	}
	/* SMRAM register defaults to 0x02 on reset */
	pci_setconfigb(cpssp->confsp, 0x72, 0x02);
}

/* ----------------------------------------------------------------- */
/* Init Function                                                     */
/* ----------------------------------------------------------------- */

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_reset_hash_,
	struct sig_host_bus_main *port_host_bus,
	struct sig_cs *port_mem_cs0,
	struct sig_cs *port_mem_cs1,
	struct sig_cs *port_mem_cs2,
	struct sig_cs *port_mem_cs3,
	struct sig_cs *port_mem_cs4,
	struct sig_cs *port_mem_cs5,
	struct sig_cs *port_mem_cs6,
	struct sig_cs *port_mem_cs7,
	struct sig_mem_bus_main *port_mem,
	struct sig_pci_bus_main *port_pci_bus,
	struct sig_agp_bus_main *port_agp_bus
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = CHIP_(power_set),
	};
	static const struct sig_boolean_funcs reset_hash__funcs = {
		.set = CHIP_(n_reset_set),
	};
	static const struct sig_host_bus_main_funcs host_bus_funcs = {
		.type_addr =	CHIP_(host_bus_type_addr),
		.read_data =	CHIP_(host_bus_read_data),
		.write_data =	CHIP_(host_bus_write_data),

		.ior =		CHIP_(host_bus_ior),
		.iow =		CHIP_(host_bus_iow),

		.ior_info =	CHIP_(host_bus_ior_info),
		.iow_info =	CHIP_(host_bus_iow_info),

		.mr =		CHIP_(host_bus_mr),
		.mw =		CHIP_(host_bus_mw),
		.map_r_check =	CHIP_(host_bus_map_r_check),
		.map_r =	CHIP_(host_bus_map_r),
		.map_w_check =	CHIP_(host_bus_map_w_check),
		.map_w =	CHIP_(host_bus_map_w),
		.map_x_check =	CHIP_(host_bus_map_r_check), /* ! */
		.map_x =	CHIP_(host_bus_map_r), /* ! */

		.inta_addr =	CHIP_(host_bus_inta_addr),
		.inta_data =	CHIP_(host_bus_inta_data),

#if 0	/* Not used yet. */
		.unmap =	CHIP_(host_bus_unmap),
#endif
	};
	static const struct sig_cs_funcs mem_cs0_funcs = {
		.unmap =	CHIP_(memX_unmap),
	};
	static const struct sig_cs_funcs mem_cs1_funcs = {
		.unmap =	CHIP_(memX_unmap),
	};
	static const struct sig_cs_funcs mem_cs2_funcs = {
		.unmap =	CHIP_(memX_unmap),
	};
	static const struct sig_cs_funcs mem_cs3_funcs = {
		.unmap =	CHIP_(memX_unmap),
	};
	static const struct sig_cs_funcs mem_cs4_funcs = {
		.unmap =	CHIP_(memX_unmap),
	};
	static const struct sig_cs_funcs mem_cs5_funcs = {
		.unmap =	CHIP_(memX_unmap),
	};
	static const struct sig_cs_funcs mem_cs6_funcs = {
		.unmap =	CHIP_(memX_unmap),
	};
	static const struct sig_cs_funcs mem_cs7_funcs = {
		.unmap =	CHIP_(memX_unmap),
	};
	static const struct sig_pci_bus_main_funcs pci_bus_funcs = {
		.mr =		CHIP_(pci_bus_mr),
		.mw =		CHIP_(pci_bus_mw),
		.map_r_check =	CHIP_(pci_bus_map_r_check),
		.map_r =	CHIP_(pci_bus_map_r),
		.map_w_check =	CHIP_(pci_bus_map_w_check),
		.map_w =	CHIP_(pci_bus_map_w),

#if 0	/* Not used yet. */
		.ack =		CHIP_(pci_bus_ack),
#endif

		.unmap =	CHIP_(pci_bus_unmap),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* initialize PCI config space */
	memset(cpssp->confsp, 0, 0x100);
	pci_setconfigw(cpssp->confsp, PCI_VENDOR_ID, PCI_VENDOR_ID_INTEL);
	/* Set device ID for LX chipset with AGP disabled */
	pci_setconfigw(cpssp->confsp, PCI_DEVICE_ID, 0x7180);
	pci_setconfigw(cpssp->confsp, PCI_COMMAND, 0x0006);
	pci_setconfigw(cpssp->confsp, PCI_STATUS, 0x2210);
	pci_setconfigb(cpssp->confsp, PCI_REVISION_ID, 0x03);
	pci_setconfigw(cpssp->confsp, PCI_CLASS_DEVICE, PCI_CLASS_BRIDGE_HOST);
	pci_setconfigb(cpssp->confsp, PCI_LATENCY_TIMER, 0x40);
	pci_setconfigl(cpssp->confsp, PCI_BASE_ADDRESS_0, 0x00000008);

	/* Out */
	/* Call */
	cpssp->port_host_bus = port_host_bus;
	sig_host_bus_main_connect(port_host_bus, cpssp, &host_bus_funcs);

	cpssp->sig_mem_cs[0] = port_mem_cs0;
	sig_cs_connect(port_mem_cs0, cpssp, &mem_cs0_funcs);

	cpssp->sig_mem_cs[1] = port_mem_cs1;
	sig_cs_connect(port_mem_cs1, cpssp, &mem_cs1_funcs);

	cpssp->sig_mem_cs[2] = port_mem_cs2;
	sig_cs_connect(port_mem_cs2, cpssp, &mem_cs2_funcs);

	cpssp->sig_mem_cs[3] = port_mem_cs3;
	sig_cs_connect(port_mem_cs3, cpssp, &mem_cs3_funcs);

	cpssp->sig_mem_cs[4] = port_mem_cs4;
	sig_cs_connect(port_mem_cs4, cpssp, &mem_cs4_funcs);

	cpssp->sig_mem_cs[5] = port_mem_cs5;
	sig_cs_connect(port_mem_cs5, cpssp, &mem_cs5_funcs);

	cpssp->sig_mem_cs[6] = port_mem_cs6;
	sig_cs_connect(port_mem_cs6, cpssp, &mem_cs6_funcs);

	cpssp->sig_mem_cs[7] = port_mem_cs7;
	sig_cs_connect(port_mem_cs7, cpssp, &mem_cs7_funcs);

	cpssp->port_pci_bus = port_pci_bus;
	sig_pci_bus_main_connect(port_pci_bus, cpssp, &pci_bus_funcs);

	cpssp->port_agp_bus = port_agp_bus;
	/* FIXME */

	/* In */
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);

	sig_boolean_connect_in(port_reset_hash_, cpssp, &reset_hash__funcs);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
