//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/StandardSamples/FeNiBilayerBuilder.cpp
//! @brief     Defines various sample builder classes to
//!            test polarized specular computations
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2020
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Sample/StandardSamples/FeNiBilayerBuilder.h"
#include "Base/Const/PhysicalConstants.h"
#include "Base/Const/Units.h"
#include "Sample/Interface/LayerRoughness.h"
#include "Sample/Material/MaterialFactoryFuncs.h"
#include "Sample/Multilayer/Layer.h"
#include "Sample/Multilayer/MultiLayer.h"

using Units::deg;

namespace {

auto constexpr rhoMconst = -PhysConsts::m_n * PhysConsts::g_factor_n * PhysConsts::mu_N
                           / PhysConsts::h_bar / PhysConsts::h_bar * 1e-27;

const complex_t sldFe = complex_t{8.02e-06, 0};
const complex_t sldAu = complex_t{4.6665e-6, 0};
const complex_t sldNi = complex_t{9.4245e-06, 0};

class Options {
public:
    int m_NBilayers = 4;
    double m_angle = 0.;
    double m_magnetizationMagnitude = 1e7;
    double m_thicknessFe = 100. * Units::angstrom;
    double m_thicknessNi = 40. * Units::angstrom;
    double m_sigmaRoughness = 0.;
    int m_effectiveSLD = 0;
    RoughnessModel m_roughnessModel = RoughnessModel::TANH;

    Options() = default;
    Options NBilayers(int n)
    {
        m_NBilayers = n;
        return *this;
    }
    Options angle(double angle)
    {
        m_angle = angle;
        return *this;
    }
    Options magnetizationMagnitude(double M)
    {
        m_magnetizationMagnitude = M;
        return *this;
    }
    Options thicknessFe(double t)
    {
        m_thicknessFe = t;
        return *this;
    }
    Options thicknessNi(double t)
    {
        m_thicknessNi = t;
        return *this;
    }
    Options sigmaRoughness(double r)
    {
        m_sigmaRoughness = r;
        return *this;
    }
    Options effectiveSLD(int i)
    {
        m_effectiveSLD = i;
        return *this;
    }
    Options roughnessModel(RoughnessModel rm)
    {
        m_roughnessModel = rm;
        return *this;
    }
};


//! Creates the sample demonstrating an Fe-Ni Bilayer with and without roughness
class FeNiBilayer {
public:
    explicit FeNiBilayer(Options opt = {})
        : NBilayers(opt.m_NBilayers)
        , angle(opt.m_angle)
        , magnetizationMagnitude(opt.m_magnetizationMagnitude)
        , thicknessFe(opt.m_thicknessFe)
        , thicknessNi(opt.m_thicknessNi)
        , sigmaRoughness(opt.m_sigmaRoughness)
        , effectiveSLD(opt.m_effectiveSLD)
        , roughnessModel(opt.m_roughnessModel)
    {
        if (angle != 0. && effectiveSLD != 0.)
            throw std::runtime_error("Cannot perform scalar computation "
                                     "for non-colinear magnetization");

        magnetizationVector = R3(magnetizationMagnitude * std::sin(angle),
                                 magnetizationMagnitude * std::cos(angle), 0);
        sample = constructSample();
    }

    MultiLayer* release() { return sample.release(); }

private:
    int NBilayers;
    double angle;
    double magnetizationMagnitude;
    double thicknessFe;
    double thicknessNi;
    double sigmaRoughness;
    int effectiveSLD;
    RoughnessModel roughnessModel;

    R3 magnetizationVector;

    std::unique_ptr<MultiLayer> sample;

    std::unique_ptr<MultiLayer> constructSample();
};

std::unique_ptr<MultiLayer> FeNiBilayer::constructSample()
{
    auto sample = std::make_unique<MultiLayer>();

    auto m_ambient = MaterialBySLD("Ambient", 0.0, 0.0);
    auto m_Fe =
        effectiveSLD == 0
            ? MaterialBySLD("Fe", sldFe.real(), sldFe.imag(), magnetizationVector)
            : MaterialBySLD("Fe", sldFe.real() + effectiveSLD * rhoMconst * magnetizationMagnitude,
                            sldFe.imag(), R3());

    auto m_Ni = MaterialBySLD("Ni", sldNi.real(), sldNi.imag());
    auto m_Substrate = MaterialBySLD("Au", sldAu.real(), sldAu.imag());

    Layer l_Fe{m_Fe, thicknessFe};
    Layer l_Ni{m_Ni, thicknessNi};

    LayerRoughness roughness{sigmaRoughness, 0., 0.};
    sample->addLayer(Layer{m_ambient});

    for (auto i = 0; i < NBilayers; ++i) {
        sample->addLayerWithTopRoughness(l_Fe, roughness);
        sample->addLayerWithTopRoughness(l_Ni, roughness);
    }

    sample->addLayerWithTopRoughness(Layer{m_Substrate}, roughness);
    sample->setRoughnessModel(roughnessModel);
    return sample;
}

} // namespace


MultiLayer* ExemplarySamples::createFeNiBilayer()
{
    auto sample = FeNiBilayer{Options()};
    return sample.release();
}

MultiLayer* ExemplarySamples::createFeNiBilayerTanh()
{
    auto sample = FeNiBilayer{
        Options().sigmaRoughness(2. * Units::angstrom).roughnessModel(RoughnessModel::TANH)};
    return sample.release();
}

MultiLayer* ExemplarySamples::createFeNiBilayerNC()
{
    auto sample = FeNiBilayer{
        Options().sigmaRoughness(2. * Units::angstrom).roughnessModel(RoughnessModel::NEVOT_CROCE)};
    return sample.release();
}

MultiLayer* ExemplarySamples::createFeNiBilayerSpinFlip()
{
    auto sample = FeNiBilayer{Options().angle(38. * deg)};
    return sample.release();
}

MultiLayer* ExemplarySamples::createFeNiBilayerSpinFlipTanh()
{
    auto sample = FeNiBilayer{Options()
                                  .angle(38 * deg)
                                  .sigmaRoughness(2. * Units::angstrom)
                                  .roughnessModel(RoughnessModel::TANH)};
    return sample.release();
}

MultiLayer* ExemplarySamples::createFeNiBilayerSpinFlipNC()
{
    auto sample = FeNiBilayer{Options()
                                  .angle(38 * deg)
                                  .sigmaRoughness(2. * Units::angstrom)
                                  .roughnessModel(RoughnessModel::NEVOT_CROCE)};
    return sample.release();
}
