//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/MaterialModel.cpp
//! @brief     Implements class MaterialModel
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/MaterialModel.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Sample/MaterialItem.h"
#include "GUI/Support/XML/UtilXML.h"
#include <QColor>
#include <QUuid>
#include <random>

namespace {
namespace Tag {

const QString Material("Material");

} // namespace Tag

QColor suggestMaterialColor(const QString& name)
{
    if (name.contains(materialMap.key(DefaultMaterials::Vacuum)))
        return QColor(179, 242, 255);
    if (name.contains(materialMap.key(DefaultMaterials::Substrate)))
        return QColor(205, 102, 0);
    if (name.contains(materialMap.key(DefaultMaterials::Default)))
        return QColor(Qt::green);
    if (name.contains(materialMap.key(DefaultMaterials::Core)))
        return QColor(220, 140, 220);
    if (name.contains(materialMap.key(DefaultMaterials::Particle)))
        return QColor(146, 198, 255);

    // return a random color
    static std::random_device r;
    std::default_random_engine re(r());
    std::uniform_int_distribution<int> ru(0, 255);

    return QColor(ru(re), ru(re), ru(re));
}

} // namespace

QMap<QString, DefaultMaterials> materialMap = {{"Default", DefaultMaterials::Default},
                                               {"Vacuum", DefaultMaterials::Vacuum},
                                               {"Particle", DefaultMaterials::Particle},
                                               {"Core", DefaultMaterials::Core},
                                               {"Substrate", DefaultMaterials::Substrate}};

MaterialModel::MaterialModel() = default;

MaterialModel::~MaterialModel()
{
    clear();
}

void MaterialModel::clear()
{
    qDeleteAll(m_materials);
    m_materials.clear();
}

MaterialItem* MaterialModel::addMaterialItem(MaterialItem* materialItem)
{
    return addMaterialItem(materialItem, true);
}

MaterialItem* MaterialModel::addMaterialItem()
{
    return addMaterialItem(new MaterialItem(), false);
}

MaterialItem* MaterialModel::addMaterialItem(MaterialItem* materialItem, bool signalAdding)
{
    ASSERT(materialItem);
    materialItem->disconnect(this);
    m_materials << materialItem;
    connect(materialItem, &MaterialItem::dataChanged, this, [this] { emit materialChanged(); });

    if (signalAdding)
        emit materialAddedOrRemoved();

    return materialItem;
}

MaterialItem* MaterialModel::addRefractiveMaterialItem(const QString& name, double delta,
                                                       double beta)
{
    auto* materialItem = new MaterialItem();
    materialItem->setMatItemName(name);
    materialItem->setColor(suggestMaterialColor(name));
    materialItem->setRefractiveIndex(delta, beta);
    addMaterialItem(materialItem);

    return materialItem;
}

MaterialItem* MaterialModel::addSLDMaterialItem(const QString& name, double sld, double abs_term)
{
    auto* materialItem = new MaterialItem();
    materialItem->setMatItemName(name);
    materialItem->setColor(suggestMaterialColor(name));
    materialItem->setScatteringLengthDensity(complex_t(sld, abs_term));
    addMaterialItem(materialItem);

    return materialItem;
}

MaterialItem* MaterialModel::materialItemFromName(const QString& name) const
{
    for (auto* materialItem : m_materials)
        if (materialItem->matItemName() == name)
            return materialItem;

    return nullptr;
}

MaterialItem* MaterialModel::materialItemFromIdentifier(const QString& identifier) const
{
    for (auto* materialItem : m_materials)
        if (materialItem->identifier() == identifier)
            return materialItem;

    return nullptr;
}

MaterialItem* MaterialModel::copyMaterialItem(const MaterialItem& materialItem)
{
    auto* newMaterial = new MaterialItem(materialItem);
    newMaterial->createNewIdentifier();
    newMaterial->setMatItemName(materialItem.matItemName() + " (copy)");
    addMaterialItem(newMaterial);

    return newMaterial;
}

const QVector<MaterialItem*>& MaterialModel::materialItems() const
{
    return m_materials;
}

MaterialItem* MaterialModel::defaultMaterialItem() const
{
    ASSERT(!materialItems().isEmpty());
    return materialItems().front();
}

MaterialItem* MaterialModel::defaultCoreMaterialItem() const
{
    for (auto* material : materialItems())
        if (material->matItemName() == materialMap.key(DefaultMaterials::Core))
            return material;

    return defaultMaterialItem();
}

MaterialItem* MaterialModel::defaultParticleMaterialItem() const
{
    for (auto* material : materialItems())
        if (material->matItemName() == materialMap.key(DefaultMaterials::Particle))
            return material;

    return defaultMaterialItem();
}

void MaterialModel::removeMaterialItem(MaterialItem* materialItem)
{
    m_materials.removeAll(materialItem);
    delete materialItem;
    emit materialAddedOrRemoved();
}

void MaterialModel::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // materials
    for (const auto* material : m_materials) {
        w->writeStartElement(Tag::Material);
        material->writeTo(w);
        w->writeEndElement();
    }
}

void MaterialModel::readFrom(QXmlStreamReader* r)
{
    clear();

    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // material
        if (tag == Tag::Material) {
            addMaterialItem()->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

void MaterialModel::initFrom(const MaterialModel& from)
{
    // update existing to new contents (do not delete and recreate to keep references valid)
    for (auto* destItem : m_materials)
        if (auto* fromItem = from.materialItemFromIdentifier(destItem->identifier()))
            destItem->updateFrom(*fromItem);

    bool anyAddedOrRemoved = false;

    // remove non-existing
    QVector<MaterialItem*>::iterator iter = m_materials.begin();
    while (iter != m_materials.end())
        if (!from.materialItemFromIdentifier((*iter)->identifier())) {
            delete *iter;
            iter = m_materials.erase(iter);
            anyAddedOrRemoved = true;
        } else
            iter++;

    // copy new ones
    for (const auto* m : from.materialItems())
        if (!materialItemFromIdentifier(m->identifier())) {
            addMaterialItem(new MaterialItem(*m), false);
            anyAddedOrRemoved = true;
        }

    // copy order
    QVector<MaterialItem*> tmp;
    for (const auto* m : from.materialItems())
        tmp << materialItemFromIdentifier(m->identifier());
    m_materials = tmp;

    if (anyAddedOrRemoved)
        emit materialAddedOrRemoved();
}
