/****************************************************************
**
** Attal : Lords of Doom
**
** gameData.cpp
** Manage data of a whole game
**
** Version : $Id: gameData.cpp,v 1.27 2008/02/26 23:34:57 lusum Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo
**
** Date : 05/01/2002
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "gameData.h"

// generic include files
// include files for QT
#include <QTextStream>

// application specific include files
#include "libCommon/artefact.h"
#include "libCommon/artefactManager.h"
#include "libCommon/calendar.h"
#include "libCommon/dataTheme.h"
#include "libCommon/genericBase.h"
#include "libCommon/genericBonus.h"
#include "libCommon/genericBuilding.h"
#include "libCommon/genericCell.h"
#include "libCommon/genericChest.h"
#include "libCommon/genericEvent.h"
#include "libCommon/genericLord.h"
#include "libCommon/genericMap.h"
#include "libCommon/genericMapCreature.h"
#include "libCommon/genericPlayer.h"
#include "libCommon/quest.h"
#include "libCommon/questManager.h"

extern QString VERSION;

GameData::GameData()
	:_nbPlayer( 0 ), _map( 0 )
{
	_quests = new QuestManager();
	_calendar = new Calendar();
	_nbArtefact = 0;
	_nbBonus = 0;
	_nbChest = 0;

	reinit();
}

GameData::~GameData()
{
	TRACE("GameData::~GameData");

	clearLists();

	if( _quests ) {
		delete _quests;
	}
	if( _calendar ) {
		delete _calendar;
	}
	if( _map ) {
		delete _map;
	}
	
	TRACE("End of GameData::~GameData");
	
}

void GameData::clearLists()
{
	TRACE("GameData::clearLists");

	while( ! _bases.isEmpty() ) {
		delete _bases.takeFirst();
	}
	while( ! _buildings.isEmpty() ) {
		delete _buildings.takeFirst();
	}
	while( ! _lords.isEmpty() ) {
		delete _lords.takeFirst();
	}
	while( ! _events.isEmpty() ) {
		delete _events.takeFirst();
	}
	while( ! _creatures.isEmpty() ) {
		delete _creatures.takeFirst();
	}
}

void GameData::save( QTextStream * ts )
{
	*ts << "<?xml version=\"1.0\" encoding=\"UTF-8\"?><!DOCTYPE scenario>" << endl;
	*ts << "<scenario nbPlayer=\"" << getNbPlayer() << "\">" << endl;
	*ts << "\t<attalversion>"  << VERSION << "</attalversion>" << endl;
	*ts << "\t<name>" << getScenarioName() << "</name>" << endl;
	*ts << "\t<theme>" << getScenarioTheme() << "</theme>" << endl;
	*ts << "\t<description>" << endl;
	*ts << "\t\t" << getScenarioDescription() << endl;
	*ts << "\t</description>" << endl;
	
	_calendar->save( ts, 1 );

	_map->save( ts );

	for( int i = 0; i < _events.count(); i++ ) {
		_events.at( i )->save( ts, 1 );
	}

	for( int i = 0; i < _bases.count(); i++ ) {
		_bases.at( i )->save( ts, 1 );
	}

	for( int i = 0; i < _buildings.count(); i++ ) {
		_buildings.at( i )->save( ts, 1 );
	}

	for( int i = 0; i < _lords.count(); i++ ) {
		_lords.at( i )->save( ts, 1 );
	}

	for( int i = 0; i < _creatures.count(); i++ ) {
		_creatures.at( i )->save( ts, 1 );
	}

	for( uint i = 0; i < getNbPlayer(); i++ ) {
		_players.at( i )->save( ts, 1 );
	}

	_quests->save( ts, 1 );

	*ts << "</scenario>" << endl;
}

void GameData::reinit()
{
	TRACE("GameData::reinit");

	if( _map ) {
		_map->clear();
	}
	
	if( _calendar ) {
		_calendar->reinit();
	}
	
	clearLists();
	
	_quests->clear();
	
	_nbPlayer = 0;
	_nbArtefact = 0;
	_nbBonus = 0;
	_nbChest = 0;
	
	initLords();
	
}

void GameData::initLords()
{
	TRACE("GameData::initLords");

	uint nbLords = DataTheme.lords.count();
	for( uint i = 0; i < nbLords; i++ ) {
		GenericLord * lord = new GenericLord();
		lord->setId( i );
		_lords.append( lord );
	}
}

void GameData::resetOwnership( GenericPlayer * player )
{
	for( int i = 0; i < _bases.count(); i++ ) {
		if(_bases.at( i )->getOwner() == player ) {
			_bases.at( i )->setOwner(0);
		}
	}

	for( int i = 0; i < _buildings.count(); i++ ) {
		if( _buildings.at( i )->getOwner() == player ) {
			_buildings.at( i )->setOwner(0);
		}
	}

	for( int i = 0; i < _lords.count(); i++ ) {
		if(_lords.at( i )->getOwner() == player ) {
			_lords.at( i )->setOwner(0);
		}
	}
}

GenericCell * GameData::getCell( uint row, uint col )
{
	if( ( row < _map->getHeight() ) && ( col < _map->getWidth() ) ) {
		return _map->at( row, col );
	} else {
		return NULL;
	}
}

uint GameData::getBaseNumber()
{
	return _bases.count();
}

uint GameData::getBuildingNumber()
{
	return _buildings.count();
}

GenericBase * GameData::getBase( uint num )
{
	if( (int)num < _bases.count() ) {
		return _bases.at( num );
	}	
	
	return NULL;
}

void GameData::loadMap( QTextStream * stream, int width, int height )
{
	if( _map ) {
		delete _map;
	}
	_map = new GenericMap();
	_map->load( stream, width, height );
}

GenericBase * GameData::getNewBase( uchar race ) 
{
	GenericBase * base = new GenericBase( race );
	_bases.append( base );
	base->setOwner( NULL );
	return base;
}

GenericBase * GameData::getNewBase(	uchar race, int row,	int col,	int population,	uchar id,	int nb, const QList<uchar> & forbidlist)
{
	GenericBase * base = _map->at( row, col )->getBase();

	if( base ) {
		removeBase( base );
	}

	base = getNewBase( race );
	base->setPosition( _map->at( row, col ));
	base->setPopulation( population );
	_map->computeStoppable( base );
	base->setId( id );/// XXX: to inv ??
	base->isUnitBought( false );

	for( int i = 0; i < nb; i++ ) {
		base->addForbiddenBuilding( forbidlist.at(i) );
	}

	return base;

}

void GameData::removeBase( GenericBase * base )
{
	if( base ) {
		if(base->getOwner()){
			base->getOwner()->removeBase(base);
		}
		_bases.removeAll( base );
		delete base;
	}
}

void GameData::setBaseId( GenericBase * base ) 
{
	int id = 0;

	for( int i = 0; i < _bases.count(); i++ ) {
		if(_bases.at( i )->getId() == id && _bases.at( i ) != base ) {
			id++;
			i = 0;
		}
	}
	base->setId( id );
}

void GameData::setBuildingId( GenericBuilding * build ) 
{
	int id = 0;

	for( int i = 0; i < _buildings.count(); i++ ) {
		if(_buildings.at( i )->getId() == id && _buildings.at( i ) != build ) {
			id++;
			i = 0;
		}
	}

	build->setId( id );
}

void GameData::setBase2Player( int idBase, int player ) 
{
	TRACE("GameData::setBase2Player idBase %d, player %d", idBase, player );

	if(  _players.count() > player  )  {
		GenericBase * base = getBaseById( idBase );
		
		if( base->getOwner() ) {
			base->getOwner()->removeBase( base );
		}

		if( ( _players.at( player ) ) && ( base )  ) {
			base->setOwner( _players.at( player ) );
			_players.at( player )->addBase( base );
		}
	}
}
	
void GameData::setBuilding2Player( int idBuild, int player )
{
	TRACE("GameData::setBuilding2Player idBuilding %d, player %d", idBuild, player );

	if( ( _players.count() > player  )) {
		GenericBuilding * build = getBuildingById( idBuild );
		
		if( build->getOwner() ) {
			build->getOwner()->removeBuilding( build );
		}

		if( _players.at( player ) ) {
			build->setOwner( _players.at( player ) );
			_players.at( player )->addBuilding( build );		
		}
	}
}

void GameData::setDateByType( int type, int value )
{
	_calendar->setDateByType(type,value);
}

void GameData::setInitialDateByType( int type, int value )
{
	_calendar->setInitialDateByType(type,value);
}

void GameData::setTurn( uint turn )
{
	_calendar->setTurn( turn );
}

uint GameData::getTurn()
{
	return _calendar->getTurn();
}

uint GameData::getDateByType( int type )
{
	return _calendar->getDateByType(type);
}

GenericLord * GameData::getLord( int idLord )
{
	return _lords.at( idLord );
}

uint GameData::getLordNumber()
{
	return _lords.count();
}

GenericPlayer * GameData::getPlayer( int num )
{
	if( ( num >= 0 ) && (num < _players.count() ) ) {
		return _players.at( num );
	} else {
		return NULL;
	}
}

uint GameData::getPlayerNumber()
{
	return _players.count();
}

GenericArtefact * GameData::getArtefactById( int id )
{
	GenericArtefact * ret = NULL;
	for( int i = 0; i < _events.count(); i++ ) {
		if( ( _events.at( i )->getType() == GenericEvent::EventArtefact ) &&
		  _events.at( i )->getArtefact()->getId() == id ) {
			ret = _events.at( i )->getArtefact();
		}
	}
	return ret;
}

GenericBase * GameData::getBaseById( int id )
{
	GenericBase * ret = NULL;
	for( int i = 0; i < _bases.count(); i++ ) {
		 if( _bases.at( i )->getId() == id ) {
			ret = _bases.at( i );
			break;
		}
	}
	
	return ret;
}

GenericBuilding * GameData::getBuildingById( int id )
{	
	GenericBuilding * ret = NULL;

	for( int i = 0; i < _buildings.count(); i++ ) {
		 if( _buildings.at( i )->getId() == id ) {
			ret = _buildings.at( i );
			break;
		}
	}
	return ret;	
}

void GameData::setLord2Player( int idLord, int player )
{
	if( ( _players.count() > player  ) && ( _lords.count() > idLord ) ) {
		if( ( _players.value( player ) ) && ( _lords.at( idLord ) ) ) {
			_lords.at( idLord )->setOwner( _players.at( player ) );
			_players.at( player )->addLord( _lords.at( idLord ) );		
		}
	}
}

GenericEvent * GameData::getNewArtefact( int id )
{
	GenericEvent * ret = new GenericEvent();
	GenericArtefact * artefact = new GenericArtefact();
	artefact->setId( id );
	ret->setArtefact( artefact );
	_events.append( ret );
	return ret;
}

GenericEvent * GameData::getNewBonus()
{
	GenericEvent * ret = new GenericEvent();
	GenericBonus * bonus = new GenericBonus();
	ret->setBonus( bonus );
	_events.append( ret );
	return ret;

}

GenericEvent * GameData::getNewChest()
{
	GenericEvent * ret = new GenericEvent();
	GenericChest * chest = new GenericChest();
	ret->setChest( chest );
	_events.append( ret );
	return ret;

}

GenericBuilding * GameData::getNewBuilding( uchar type )
{
	GenericBuilding * ret = new GenericBuilding( type );
	_buildings.append( ret );
	return ret;
}

GenericBuilding * GameData::getNewBuilding( uchar type, int id, int row, int col)
{

	TRACE("GenericBuilding * GameData::getNewBuilding( uchar type %d, int id %d, int row %d, int col %d )",type, id, row, col );

	GenericBuilding * building = _map->at( row, col )->getBuilding();

	if( building ) {
		removeBuilding( building );
	}

	building = getNewBuilding( type );
	/* keep id, position in that order **/
	building->setId( id );
	building->setPosition( _map->at( row, col ) );
	_map->computeStoppable( building );

	return building;	
}

void GameData::removeBuilding( GenericBuilding * build )
{
	if( build ) {
		if(build->getOwner()){
			build->getOwner()->removeBuilding( build );
		}
		_buildings.removeAll( build );
		delete build;
	}
}

Quest * GameData::getNewQuest()
{
	Quest * ret = new Quest();
	_quests->appendSecondaryQuest( ret );
	return ret;
}

Quest * GameData::getMainQuest()
{
	Quest * ret = _quests->getMainQuest();

	if( ! ret ) {
		ret = new Quest();
		_quests->setMainQuest( ret );
	}

	return ret;
}

GenericMapCreature * GameData::getNewMapCreature()
{
	GenericMapCreature * ret = new GenericMapCreature();
	_creatures.append( ret );
	return ret;
}

GenericMapCreature * GameData::getNewMapCreature( int row, int col, uchar race, uchar level, int nb, bool /* looking */)
{
	GenericMapCreature * creature = getNewMapCreature(); 
	creature->setCreature( race, level );
	creature->setCategoryNumber( nb );
	creature->setCell( _map->at( row, col ) );
	_map->at( row, col )->setCreature( creature );
	return creature;

}

void GameData::removeMapCreature( GenericMapCreature * creature )
{
	if( creature ) {
		_creatures.removeAll( creature );
		delete creature;
	}
}

void GameData::removeMapCreature( int row, int col )
{
	GenericMapCreature * crea = _map->at( row, col )->getCreature();
	if( crea ) {
		removeMapCreature( crea );
		_map->at( row, col )->setCreature( NULL );
	}

}

void GameData::removeEvent( int row, int col )
{
	GenericEvent * event = _map->at( row, col )->getEvent();
	if( event ) {
		removeEvent( event );
		_map->at( row, col )->setEvent( NULL );
	}
}

void GameData::removeEvent( GenericEvent * event )
{
	if( event ) {
		_events.removeAll( event );
		delete event;
	}
}

void GameData::updateLordUnit(uchar id , uchar pos, uchar race, uchar level, int nb, uchar move, int health)
{

	TRACE("GameData::updateLordUnit id  %d, pos %d , race %d, level %d, nb %d, move %d, health %d", id,pos, race, level, nb, move, health );	
	
	if(_lords.at( id )) {
		_lords.at( id )->updateUnit( pos ,race, level,nb, move, health);
	}

}

void GameData::updateBaseBuilding( int row, int col, uchar level,	bool create )
{
	TRACE("GameData::updateBaseBuilding row %d, col %d, level %d, create %d",row,col,level,create);

  GenericBase * base =	_map->at( row, col )->getBase();
	if( base ) {
		base->updateBuilding( level, create );
	} else {
		logEE( "Base not found" );
	}
}
	
void GameData::exchangeArtefactLord( uchar idLord1, uchar idLord2, int item)
{

	GenericLord * lord1 = NULL;
	GenericLord * lord2 = NULL;
	
	if( idLord1 ) {
		lord1 = _lords.at(  idLord1 );
	}
	if( idLord2 ) {
		lord2 = _lords.at(  idLord2 );
	}
	
	if( lord1 && lord2 ) {
		ArtefactManager * manag1 = lord1->getArtefactManager();
		ArtefactManager * manag2 = lord2->getArtefactManager();
		
		GenericLordArtefact * artefact = manag1->getArtefact( item );
		
		manag1->removeArtefact( item );
		manag2->addArtefact( artefact );
		
	}
}
