// ---------------------------------------------------------------------------
// - Rpi.hpp                                                                 -
// - afnix:mth module - real permutation interface definitions               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2015 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_RPI_HPP
#define  AFNIX_RPI_HPP

#ifndef  AFNIX_SERIAL_HPP
#include "Serial.hpp"
#endif
 
namespace afnix {

  /// This Rpi class is the interface class for the permutation object. The
  /// class provides the base implementation for a permutation object which
  /// can be applied to a vector or a matrix.
  /// @author amaury darsch

  class Rpi : public virtual Serial {
  protected:
    /// the permutation size
    t_long  d_size;
    /// the permutation array
    t_long* p_prow;
    /// the reverse permutation
    t_long* p_revp;

  public:
    /// create a null permutation
    Rpi (void);

    /// create a permutation by size
    /// @param size the permutation size
    Rpi (const t_long size);

    /// copy construct this object
    /// @param that the object to copy
    Rpi (const Rpi& that);

    /// destroy this object
    ~Rpi (void);

    
    /// @return the class name
    String repr (void) const;

    /// @return the object serial id
    t_byte serialid (void) const;

    /// serialize this object
    /// @param os the output stream
    void wrstream (OutputStream& os) const;

    /// deserialize this object
    /// @param is the input stream
    void rdstream (InputStream& os);

    /// assign an object to this one
    /// @param that the object to copy
    Rpi& operator = (const Rpi& that);
    
    /// reset this vector
    virtual void reset (void);

    /// clear this vector
    virtual void clear (void);

    /// @return the permutation size
    virtual t_long getsize (void) const;

    /// set the permutation by position
    /// @param pos the permutation index
    /// @param val the permutation value
    virtual void set (const t_long pos, const t_long val);

    /// get the permuation value by position
    /// @param pos the permutation position
    virtual t_long get (const t_long pos) const;

    /// reverse the permuation value by position
    /// @param pos the permutation position
    virtual t_long rget (const t_long pos) const;

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
