;;; mua-bend.el --- MUA back-end for pre-post processing on the message
;;;                draft.
;;;
;;; $Id: mua-bend.el,v 1.8 1997/09/26 10:28:21 ono Exp $

;; Copyright (C) 1997,1998 by Free Software Foundation, Inc.

;; Author: Fumitoyo ONO <ono@sa.osk.sumikin.co.jp>
;; Keywords: comm, internal, mail

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;;
;;  This utility would take the convenience by coupling with the 'x-pgp-sig'
;; package, etc.
;;
;; [ Usage ]
;;    Install this file on to the appropriate directory ( eg. 'site-lisp'
;;   directory ), and add the definitions below to your ~/.emacs file.
;;
;;    (require 'mua-bend)
;;       ...
;;    (add-hook 'mh-before-send-letter-hook    ;;  For MH-E
;;              '(lambda ()
;;                 (mua-bend-fix-message-fields)
;;                 (x-pgp-sig-sign)) t)
;;
;;    (add-hook 'mew-send-hook                 ;;  For Mew
;;              '(lambda ()
;;                 (x-pgp-sig-mew-encode-message)
;;                 (mua-bend-fix-message-fields)
;;                 (x-pgp-sig-sign)
;;                 (mua-bend-normalize-message-draft)) t)
;;
;; [ Notice ]
;;    Call 'mua-bend-fix-message-fields' just after mime-encoding, and just
;;   before PGP-signing. So if you use this utility with the 'x-pgp-sig'
;;   package, you can also set 'mua-bend' functions as the hook values of the
;;   PGP-signing process like the following example:
;;
;;    (add-hook 'x-pgp-sig-prepare-sign-hook 'mua-bend-fix-message-fields)
;;    (add-hook 'x-pgp-sig-post-sign-hook 'mua-bend-normalize-message-draft)
;;

;;; Code:

(require 'sendmail)

(defvar mua-bend-folder-directory (substitute-in-file-name "$HOME/Mail/")
  "The top of mail folder directories.")

(defvar mua-bend-system-directory mua-bend-folder-directory
  "The top of system directories for the MUA back-end process to drop the
mqueue files / system logs.")

(defvar mua-bend-queuing-directory "queue"
  "The queuing folder to save the mqueue files of the pre-post messages.")

(defvar mua-bend-trash-folder ".trash"
  "The trash folder to save the pre-post messages temporally. It's to be
saved as the MH-style if the folder is a directory. Otherwise, it's to be
saved as the 'traditional' mbox style.")

(defvar mua-bend-local-domain
  (or
   (getenv "FROMDOMAIN")
   (getenv "MSGIDDOMAIN")
   (getenv "DOMAINNAME")
   "sa.osk.sumikin.co.jp")
  "The default local domain name.")

(defvar mua-bend-subject-field "No title in original."
  "The default title of the subject field if it's empty.")

(defvar mua-bend-process "/usr/local/lib/im/impost.pl"
  "The MUA back-end process for fixing undefined message fields up.")

(defvar mua-bend-proc-option-list
  (list "-nodraftfolder" "-noverbose" "-nopush"
        "-FccFile" "-Fcc" mua-bend-trash-folder
        "-JustQueuing" "-QueueDir" mua-bend-queuing-directory
        "-FromDomain" mua-bend-local-domain
        "-Subj" mua-bend-subject-field
        "-IgnoreHeader" "-Date" "-MsgId" "-PidMsgId"
        "-noShowRcpts" "-noNScmpl" "-noPartial" "-noHistory")
  "The list of options for MUA back-end process.")

(defvar mua-bend-replacing-fields-alist
  (list
   '("^From .*\n" . "")
   '("^Message-Id:" . "Message-ID:")
   '("^X-Dispatcher:.*\n" . "")
   '("^Lines:.*\n" . mail-header-separator))
  "The list of fields to be replaced just after the end of the MUA back-end
process. Each cell consists of the format '(REGEXP . REPLACEMENT)', and will
be replaced in turn exactly.")

(defvar mua-bend-preserve-bcc t
  "The toggle switch whether to preserve BCC fields by the time the message
draft has been committed to the deliverer.")

(defvar mua-bend-preserve-dcc nil
  "The toggle switch whether to preserve DCC fields by the time the message
draft has been committed to the deliverer.")

(defvar mua-bend-complete-hook nil
  "Hooks called just before completing 'mua-bend-fix-message-fields'.")

(defun mua-bend-find-the-latest (path)
  "Finds the latest file ( i.e. has the newest updating time-stamp ) in the
directory PATH."
  (let ((target (directory-files path t "^[^.,#]"))
        (last (list 0 0))
        this result)
    (car (nreverse
          (mapcar (function
                   (lambda (x)
                     (setq this (nth 5 (file-attributes x)))
                     (and (or (> (car this) (car last))
                              (and (= (car this) (car last))
                                   (>= (car (cdr this)) (car (cdr last)))))
                          (setq last this)
                          (setq result x))
                     result))
                  target)))))

(defun mua-bend-fix-message-fields ()
  "Fixes the message fields up of the current pre-posting draft file."
  (let ((draft buffer-file-name)
        (output "*MUA-bend-output*") bcc rtn)
    (goto-char (point-min))
    (cond ((null (re-search-forward
                  (concat "^" mail-header-separator "$") nil t nil))
           ;; Maybe the message fields are already fixed up
           (read-string
            (concat
             "Unable to modify this draft, so deliver it immediately ... "
             "(HIT RTN): ")))
          (t
           (let ((case-fold-search t)
                 (match
                  (concat "^\\([" (and mua-bend-preserve-bcc "b")
                          (and mua-bend-preserve-dcc "d") "f]cc:.*\\)\n")))
             ;; Preserve hiden-CC fields in a list if you need
             (while (re-search-backward match nil t nil)
               (setq bcc
                     (append (cons (match-string 1) nil) bcc))
               (and (string= "fcc"
                             (downcase (substring (car bcc) 0 3)))
                    (replace-match ""))))
           (save-buffer)
           (save-excursion
             (set-buffer (get-buffer-create output))
             (erase-buffer)
             ;;  Call the 'sendproc' process synchronously
             ;; by the 'just-queuing' mode
             (setq rtn
                   (apply 'call-process mua-bend-process nil t nil
                          (append mua-bend-proc-option-list
                                  (cons draft nil)))))
           (cond ((null (zerop rtn))
                  ;; Maybe internal error
                  (read-string
                   (concat
                    "Failed to commit the process, "
                    "so deliver it immediately ... (HIT RTN): ")))
                 (t
                  ;;  Move the Fcc file onto the draft folder, and modify it
                  ;; a little
                  (delete-file (mua-bend-find-the-latest
                                (concat mua-bend-system-directory
                                        mua-bend-queuing-directory)))
                  (let ((fccpath
                         (concat mua-bend-folder-directory
                                 mua-bend-trash-folder)))
                    (rename-file (cond ((file-directory-p fccpath)
                                        (mua-bend-find-the-latest fccpath))
                                       (t
                                        fccpath))
                                 draft t))
                  (revert-buffer nil t)
                  (let ((case-fold-search nil)
                        (input-coding-system
                         (if (boundp '*noconv*) *noconv* nil))
                        (coding-system-for-read 'no-conversion))
                    (erase-buffer)
                    (insert-file-contents draft t nil nil)
                    (goto-char (point-min))
                    (mapcar (function
                             (lambda (x)
                               (insert (concat x "\n")))) bcc)
                    (mapcar (function
                             (lambda (x)
                               (and (re-search-forward (car x) nil t nil)
                                    (replace-match (eval (cdr x))))))
                            mua-bend-replacing-fields-alist))
                  ;; Call the hook values
                  (run-hooks 'mua-bend-complete-hook)))))))

(defun mua-bend-normalize-message-draft ()
  "Normalizes the pre-posting message draft so as to examine which
content-type matches appropriately to that."
  (save-buffer)
  (let ((input-coding-system
         (if (boundp '*autoconv*) *autoconv* nil))
        (coding-system-for-read 'automatic-conversion))
    (erase-buffer)
    (insert-file-contents buffer-file-name t nil nil)))

(provide 'mua-bend)
;;; mua-bend.el ends here
