#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include "log.h"
#include "common.h"	/*** debuglevel */
#include <syslog.h>

#include <unistd.h>	/*** close, dup2 */
#include <fcntl.h> 	/*** open for log */

#include <time.h>


static char *typenames[] = {
	"error",
	"warning",
	"info",
	"D2",		/* default level for debugged syserror */
	"assertion"
};

static int log_volatile;

void (*logmsg) (logtypes_t type, char *prefix, char *format, ...); 
void (*logdebugadd) (debuglevel_t level, char *format, ...);
void (*logdebug) (debuglevel_t level, char *prefix, char *format, ...);
void (*loginfo) (char *prefix, char *format, ...);
void (*logwarn) (char *prefix, char *format, ...);
void (*logerr) (char *prefix, char *format, ...); 
void (*logsyserror) (logtypes_t type, char *prefix, char *str);
	

static void do_logmsg_std (logtypes_t type, int level, char *prefix, char *format, va_list fargs) 
{
	FILE *fp;

	switch (type) {
	case Error:
	case Warning:
	case Assertion:
		fp = stderr;
		break;

	case Info:
	case Debug:
	default:
		fp = stdout;
		break;
	}
	
	if (log_volatile) {
		fputc ('}', fp);
		fputc ('\n', fp);
	}
	
	//fprintf (fp, PROGPREFIX ": ");
	fprintf (fp, PROGPREFIX);

	/* time stamp */
	if (type == Debug && level >= Session) {
		struct timespec t;
		static long prev_sec = -1, prev_msec = 0;
		static int rep = 0;
		long sec, msec;

		clock_gettime (CLOCK_REALTIME, &t);
		sec = t.tv_sec % 1000;
		msec = t.tv_nsec / 1000000;
		if (t.tv_nsec >= msec * 1000000 + 500000)
			msec++;
		
		if (sec == prev_sec && msec == prev_msec && rep < 5) {
			fprintf (fp, " %%  ---  ");
			rep++;
		}
		else {
			prev_sec 	= sec;
			prev_msec 	= msec;
			rep 		= 0;
			fprintf (fp, " %% %02ld.%03ld", t.tv_sec % 60, msec);
		}
	}
	fprintf (fp, ": ");

	switch (type) {
	case Debug:
		if (level > 0) 
			fprintf (fp, "D%d: ", level);
		else
			fprintf (fp, "D: ");
		break;

	case Error:
	case Warning:
	case Assertion:
	case Info:
		fprintf (fp, "%s: ", typenames[type]);
	}

	if (prefix)
		fprintf (fp, "%s: ", prefix);

	vfprintf (fp, format, fargs);

	log_volatile = 0;
}

void logdebugadd_std (debuglevel_t level, char *format, ...) {
	va_list fargs;

	if (level > config.debuglevel) 
		return;

	if (level >= Event && ! log_volatile) 
		putchar ('{');

	va_start (fargs, format);
	vprintf (format, fargs);

	if (level >= Event)
		log_volatile = 1;
}

void logdebug_std (debuglevel_t level, char *prefix, char *format, ...) {
	va_list fargs;

	if (! Debugged(level)) 
		return;

	va_start (fargs, format);
	do_logmsg_std (Debug, level, prefix, format, fargs);
}

void loginfo_std (char *prefix, char *format, ...) {
	va_list fargs;

	va_start (fargs, format);
	do_logmsg_std (Info, 0, prefix, format, fargs);
}

void logwarn_std (char *prefix, char *format, ...) {
	va_list fargs;

	va_start (fargs, format);
	do_logmsg_std (Warning, 0, prefix, format, fargs);
}

void logerr_std (char *prefix, char *format, ...) {
	va_list fargs;

	va_start (fargs, format);
	do_logmsg_std (Error, 0, prefix, format, fargs);
}

static void logsyserror_std (logtypes_t type, char *prefix, char *str) {
	FILE *fp;

	switch (type) {
	case Error:
	case Warning:
	case Assertion:
		fp = stderr;
		break;

	case Debug:
		/* don't forget this if change case to table */
		if (! Debugged (NonCriticalSysErrors)) 
			return;

	case Info:
	default:
		fp = stdout;
		break;
	}
	
	fprintf (fp, PROGPREFIX ": %s: ", typenames[type]); 
	if (prefix) fprintf (fp, "%s: ", prefix);
	if (str) fprintf (fp, "%s: ", str);
	fprintf (fp, "%s\n", strerror (errno));
}

extern void logmsg_std (logtypes_t type, char *prefix, char *format, ...) 
{
	va_list fargs;

	va_start (fargs, format);
	do_logmsg_std (type, -1, prefix, format, fargs);
}


/*
void logwarn_std (char *prefix, char *format, ...) {
	va_list fargs;
	int size;

	va_start (fargs, format);
	size = 2 + sizeof PROGPREFIX + strlen(format) +
	       sizeof "warning: " + 
	       strlen (prefix) + 2 + 1; 

	fmt = malloc (size);
	if (fmt == 0)
		return;

	snprintf (fmt, size, PROGPREFIX ": warning: %s: %s", prefix, format);
	vfprintf (stderr, fmt, fargs);
	free (fmt);
}
*/
static void do_logmsg_syslog (logtypes_t type, int level, char *prefix, char *format, va_list fargs) 
{
	unsigned priority;

	switch (type) {
	case Error:
		priority = LOG_ERR;
		break;

	case Warning:
	case Assertion:
		priority = LOG_WARNING;
		break;

	case Info:
		priority = LOG_INFO;
		break;

	case Debug:
		priority = LOG_DEBUG;
		break;

	default:
		/* wrong log type */
		break;
	}
	
	int size;
	char *fmt, *ptr;

	size = 0;
	if (type == Debug) 
		size += (sizeof "Dn: " - 1);
	else if (type >= 0) 
		size += (strlen(typenames[type]) + sizeof ": " - 1);

	size += (prefix ? strlen(prefix) + sizeof ": " - 1 : 0);
	size += (sizeof ": " - 1 + strlen (format));

	size += (sizeof "");
	fmt = malloc (size);
	if (fmt == 0)
		return;
	
	ptr = fmt;
	if (type == Debug) {
		if (level >= 0) 
			SPRINTF_ADJUST (ptr, size, "D%d: ", level);
		else 
			SPRINTF_ADJUST (ptr, size, "Dn: ");
	}
	else if (type >= 0)
		SPRINTF_ADJUST (ptr, size, "%s: ", typenames[type]);

	if (prefix)
		SPRINTF_ADJUST (ptr, size, "%s: ", prefix);

	SPRINTF_ADJUST (ptr, size, "%s", format);

	vsyslog (priority, fmt, fargs);
}

static void logdebugadd_syslog (debuglevel_t level, char *format, ...) {
	va_list fargs;

	if (level > config.debuglevel) 
		return;

	va_start (fargs, format);
	vprintf (format, fargs);

	if (level >= Event)
		log_volatile = 1;
}

static void logdebug_syslog (debuglevel_t level, char *prefix, char *format, ...) {
	va_list fargs;

	if (level > config.debuglevel) 
		return;

	va_start (fargs, format);
	do_logmsg_syslog (Debug, level, prefix, format, fargs);
}

static void loginfo_syslog (char *prefix, char *format, ...) {
	va_list fargs;

	va_start (fargs, format);
	do_logmsg_syslog (Info, 0, prefix, format, fargs);
}

static void logwarn_syslog (char *prefix, char *format, ...) {
	va_list fargs;

	va_start (fargs, format);
	do_logmsg_syslog (Warning, 0, prefix, format, fargs);
}

static void logerr_syslog (char *prefix, char *format, ...) {
	va_list fargs;

	va_start (fargs, format);
	do_logmsg_syslog (Error, 0, prefix, format, fargs);
}

static void logsyserror_syslog (logtypes_t type, char *prefix, char *str) {
	unsigned priority;

	switch (type) {
	case Error:
		priority = LOG_ERR;
		break;

	case Warning:
	case Assertion:
		priority = LOG_WARNING;
		break;

	case Info:
		priority = LOG_INFO;
		break;

	case Debug:
		/* don't forget this if change case to table */
		if (! Debugged (NonCriticalSysErrors)) 
			return;

		priority = LOG_DEBUG;
		break;

	default:
		/* wrong log type */
		break;
	}

	syslog (priority, "%s: %s%s%s%s%m", typenames[type],
		prefix ? prefix : "",
		prefix ? ": " : "",
		str ? str : "",
		str ? ": " : ""
		);
}

static void logmsg_syslog (logtypes_t type, char *prefix, char *format, ...) 
{
	va_list fargs;

	va_start (fargs, format);
	do_logmsg_syslog (type, -1, prefix, format, fargs);
}

void logassertion (char *expr, char *file, int line, const char *func)
{
	logmsg (Assertion, NULL, "\"%s\" in function %s(), file <%s>, line %d\n", 
		expr, func, file, line);
}

int log_setup (logdev_t where) {
	log_volatile = 0;

	switch (where) {
	case Std:
		logmsg = logmsg_std;
		logerr = logerr_std;
		logwarn = logwarn_std;
		loginfo = loginfo_std;
		logdebug = logdebug_std;
		logdebugadd = logdebugadd_std;
		logsyserror = logsyserror_std;
		break;

	case Syslog:
		openlog (PROGPREFIX, 0, LOG_DAEMON);
		logmsg = logmsg_syslog;
		logerr = logerr_syslog;
		logwarn = logwarn_syslog;
		loginfo = loginfo_syslog;
		logdebug = logdebug_syslog;
		logdebugadd = logdebugadd_syslog;
		logsyserror = logsyserror_syslog;
		break;
	}

	return 1;
}

/* duping stdout/err to file */
int log_redirect_to_file (char *file)
{
	int fd = open (file, O_WRONLY | O_APPEND | O_CREAT, 0640);
	if (fd == -1) {
		LOGSYSERR (Debug, "log_redirect: open");

		return -2;
	}
	close (1);
	close (2);
	
	if (dup2 (fd, 1) != -1) {
		if (dup2 (fd, 2) == -1) 
			close (1);
		else {
			close (fd);
			LOGDEBUG (2, "log to file %s\n", file);
			/* success */
			return 0; 
		}
	}

	/* failure to dup */
	close (fd);

	/* try to restore link to tty */
	int tty = open ("/dev/tty", O_WRONLY);
	if (tty != -1) {
		if (tty != 1) dup2 (tty, 1);
		if (tty != 2) dup2 (tty, 2);

		if (tty != 1 && tty != 2)
			close (tty);
	}
	else {
		log_setup (Syslog);
	}
	return -1;
}
