// Boost.Geometry (aka GGL, Generic Geometry Library)
// Unit Test

// Copyright (c) 2012-2014 Barend Gehrels, Amsterdam, the Netherlands.

// This file was modified by Oracle on 2021.
// Modifications copyright (c) 2021, Oracle and/or its affiliates.
// Contributed and/or modified by Adam Wulkiewicz, on behalf of Oracle

// Use, modification and distribution is subject to the Boost Software License,
// Version 1.0. (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

#include "test_buffer.hpp"
#include "aimes_cases.hpp"

#include <boost/geometry/algorithms/buffer.hpp>

#include <boost/geometry/geometries/point.hpp>


template <typename P>
void test_aimes()
{
    static double expectations[][4] =
    {
    { 3.41515232094025E-07, 3.42522426421965E-07, 6.83030293657794E-07, 6.87058957282716E-07},
    { 1.02802331980456E-07, 1.03813675877973E-07, 2.05604735015186E-07, 2.09650025340125E-07},
    { 3.56112451527224E-07, 3.57098969061553E-07, 7.12224860421884E-07, 7.16171342673988E-07},
    { 4.25305444196056E-08, 4.35412488286602E-08, 8.50612309477583E-08, 8.91039491079937E-08},
    { 3.69520876120077E-08, 3.79614419898644E-08, 7.39041894348702E-08, 7.79416069462968E-08},
    { 3.12580681338659E-08, 3.22376649819489E-08, 6.23702476332255E-08, 6.62886350255576E-08},
    { 2.64535032101776E-07, 2.65503302898651E-07, 5.29010577565714E-07, 5.32883049686461E-07},
    { 1.26452604831684E-07, 1.27462314480908E-07, 2.52905124398239E-07, 2.56944261423087E-07},
    { 5.62396706982327E-08, 5.7239461170866E-08, 1.12479426661594E-07, 1.16478773293238E-07},
    { 7.03945403301987E-07, 7.04847309407342E-07, 1.40789144609244E-06, 1.41149810417573E-06},
    { 2.05895901217446E-07, 2.06897283305807E-07, 4.11792029808566E-07, 4.15797074992952E-07},
    { 1.01571146160495E-07, 1.02582319527755E-07, 2.031423207427E-07, 2.07186729994646E-07},
    { 2.42559636376427E-07, 2.43568607061206E-07, 4.85119102222598E-07, 4.89155340233083E-07},
    { 3.41515232094025E-07, 3.42522426421965E-07, 6.83030293657794E-07, 6.87058957282716E-07},
    { 9.08697614931953E-08, 9.1881105390712E-08, 1.81739480353826E-07, 1.85784855943893E-07},
    { 6.0466163631645E-08, 6.14741111348849E-08, 1.20932355684999E-07, 1.24964373071634E-07},
    { 4.04659118657946E-07, 4.05667066161186E-07, 8.09318166261619E-07, 8.13350311545946E-07},
    { 4.80904560618001E-07, 4.81905772176106E-07, 9.61808808597198E-07, 9.65813697462181E-07},
    { 3.57066625156222E-08, 3.67177221960446E-08, 7.14132681878255E-08, 7.54576063854984E-08},
    { 2.44433664420285E-08, 2.54509160413363E-08, 4.88867044623476E-08, 5.29170023355618E-08},
    { 3.61392153536144E-07, 3.62384142249539E-07, 7.22784136542032E-07, 7.26752162449884E-07},
    { 7.03945403301987E-07, 7.04847309407342E-07, 1.40789144609244E-06, 1.41149810417573E-06},
    { 7.91276917766481E-07, 7.91866199278957E-07, 1.58255338078561E-06, 1.58491059210064E-06},
    { 2.80934472129957E-07, 2.81945006008755E-07, 5.61869072157606E-07, 5.65911307148781E-07},
    { 3.5339049020422E-07, 3.54394444457284E-07, 7.06780582504507E-07, 7.10797138481212E-07},
    { 2.44327040377357E-07, 2.4533331099974E-07, 4.88653867591893E-07, 4.9267929114194E-07},
    { 2.6145809783884E-07, 2.62465945866097E-07, 5.22915939882296E-07, 5.26947090406793E-07},
    { 2.21305654690696E-07, 2.22233921931547E-07, 4.42615530005241E-07, 4.46329210035401E-07},
    { 7.88202072499189E-08, 7.98287089764926E-08, 1.5764054239753E-07, 1.61674222454167E-07},
    { 4.08582224054044E-07, 4.09593170047629E-07, 8.17164746536037E-07, 8.21208203660717E-07},
    { 2.61692591152496E-07, 2.62703764519756E-07, 5.23385125461573E-07, 5.27429577346084E-07},
    { 2.01361032736713E-07, 2.0236487330294E-07, 4.02722207581974E-07, 4.06737527214318E-07},
    { 6.18118889406105E-08, 6.28143510539303E-08, 1.23623692616093E-07, 1.27633413171679E-07},
    { 5.01939382502314E-07, 5.02932437029813E-07, 1.00387870816121E-06, 1.00785118206659E-06},
    { 6.52623768360172E-07, 6.53615401802199E-07, 1.30525545216642E-06, 1.30922269647726E-06},
    { 3.24414514807359E-07, 3.25418923807774E-07, 6.48829413307794E-07, 6.52846509296978E-07},
    { 1.65468009072356E-07, 1.66476610274913E-07, 3.30936202885823E-07, 3.34970451376648E-07},
    { 5.23240828442795E-09, 6.24356744083343E-09, 1.04648165688559E-08, 1.45333485717657E-08},
    { 3.62318644420156E-07, 3.63320253882193E-07, 7.24637303051168E-07, 7.28643442471366E-07},
    { 5.94340193060816E-07, 5.95350627463631E-07, 1.18868054244103E-06, 1.19272210952204E-06},
    { 7.32884174681203E-07, 7.33866755808776E-07, 1.46576839199497E-06, 1.46969848913159E-06},
    { 3.02247087802243E-07, 3.0325044519941E-07, 6.04493862965683E-07, 6.08507960464522E-07},
    { 3.23513106081919E-07, 3.24517259286949E-07, 6.47026212163837E-07, 6.51042981303362E-07},
    { 2.82636392512359E-07, 2.83643203147221E-07, 5.65272713970444E-07, 5.6930015546186E-07},
    { 5.57109984811177E-08, 5.67222713243609E-08, 1.11422096438218E-07, 1.15467202022046E-07},
    { 1.69497056390355E-07, 1.70405499488879E-07, 3.38994169624129E-07, 3.42627942018225E-07},
    { 4.37937046626757E-08, 4.4801083731727E-08, 8.75873382710779E-08, 9.16166129627527E-08},
    { 3.69329029581422E-08, 3.79310165499192E-08, 7.38645837827789E-08, 7.78570665715961E-08},
    { 5.61418005418091E-08, 5.71532154935994E-08, 1.12283629505328E-07, 1.16328905619412E-07},
    { 2.30063292860905E-07, 2.31069662959271E-07, 4.60126457824117E-07, 4.64152151380404E-07},
    { 1.53257730062251E-07, 1.54239160110592E-07, 3.06515318015954E-07, 3.10441222950431E-07},
    { 7.43983150641725E-08, 7.54085220933121E-08, 1.4879661591749E-07, 1.52837202449518E-07},
    { 3.73331090486317E-08, 3.83436145057203E-08, 7.46662180972635E-08, 7.87081404496348E-08},
    { 1.35055771011139E-07, 1.36057437316595E-07, 2.70111300437748E-07, 2.74118391985212E-07},
    { 1.49921490333327E-07, 1.50926723563316E-07, 2.99842824347252E-07, 3.03863657791226E-07},
    { 5.4672608484907E-07, 5.47715472976051E-07, 1.09345225496327E-06, 1.09740931009128E-06},
    { 6.82862975054377E-08, 6.92914738920081E-08, 1.3657287922797E-07, 1.40593087394336E-07},
    { 2.99415603421949E-08, 3.09528473962928E-08, 5.9883106473535E-08, 6.39284820636021E-08},
    { 1.15991221605327E-06, 1.16091044333189E-06, 2.31982436105227E-06, 2.32381808018545E-06},
    { 5.37620522322868E-07, 5.38598712296334E-07, 1.07524111570001E-06, 1.07915387559387E-06},
    { 2.19138769352867E-08, 2.29165522114272E-08, 4.38277680814281E-08, 4.78386539271014E-08},
    { 2.25320761160219E-07, 2.26330556074572E-07, 4.50641536531293E-07, 4.54680815664688E-07},
    { 1.01003628572016E-06, 1.01100829397183E-06, 2.02007224459066E-06, 2.02395995074767E-06},
    { 4.75155772505786E-07, 4.76152848705169E-07, 9.50311530800718E-07, 9.5429980717654E-07},
    { 1.65787653827465E-07, 1.66797761380622E-07, 3.31575222389802E-07, 3.35615951030377E-07},
    { 2.30222070740638E-07, 2.31232661462855E-07, 4.60444127270421E-07, 4.64487072804332E-07},
    { 7.98883817765272E-07, 7.99890472080733E-07, 1.59776760710884E-06, 1.60179418173811E-06},
    { 1.04933420175257E-07, 1.05940515027214E-07, 2.0986679771795E-07, 2.13895532397146E-07},
    { 1.77967848458138E-06, 1.78066323996973E-06, 3.55935688389764E-06, 3.56329583439674E-06},
    { 4.1623603408425E-07, 4.17224313764564E-07, 8.3247203974679E-07, 8.36425286365738E-07},
    { 9.68960307545785E-08, 9.78908332172068E-08, 1.93792118352576E-07, 1.97771228727106E-07},
    { 2.7254995416115E-07, 2.73529721539489E-07, 5.45099553050932E-07, 5.49018324136341E-07},
    { 4.01288005491551E-07, 4.02259715315267E-07, 8.02576039404812E-07, 8.06462495006599E-07},
    { 1.35055771011139E-07, 1.36057437316595E-07, 2.70111300437748E-07, 2.74118391985212E-07},
    { 4.29554916081543E-08, 4.39544862729235E-08, 8.59111395357104E-08, 8.99071466164969E-08},
    { 1.76189836054164E-06, 1.76286511077706E-06, 3.52380727974833E-06, 3.52767503386531E-06},
    { 1.41602981784672E-07, 1.42607618158763E-07, 2.83206148310455E-07, 2.87224651174256E-07},
    { 1.15380575493873E-06, 1.15479357987169E-06, 0.000002307610870389, 2.31156330698923E-06},
    { 1.41356807148441E-07, 1.42356668675347E-07, 2.82713486399189E-07, 2.86713202513056E-07},
    { 6.45327844495114E-08, 6.55440999253187E-08, 1.29065526266459E-07, 1.33110816591397E-07},
    { 6.86342787048488E-08, 6.96456083915109E-08, 1.37268628463971E-07, 1.41313861945491E-07},
    { 1.83265086661777E-07, 1.84274682624164E-07, 3.66530358064665E-07, 3.70568429275409E-07},
    { 1.43990902756741E-07, 1.44994970696644E-07, 2.87981706037499E-07, 2.91998276225058E-07},
    { 1.02871865692578E-07, 1.03880751112229E-07, 2.05743802439429E-07, 2.09779301485469E-07},
    { 1.36728246502571E-07, 1.37673012545747E-07, 2.7345657827027E-07, 2.77235542966991E-07},
    { 3.10800132297118E-07, 3.11808250330614E-07, 6.21600349859364E-07, 6.25632381456853E-07},
    { 1.32283005882528E-07, 1.33261437440524E-07, 2.64565414909157E-07, 2.68479169562852E-07},
    { 3.33188367562798E-07, 3.34195561890738E-07, 6.66376735125596E-07, 6.70405427172227E-07},
    { 1.90134230138028E-07, 1.91139307048616E-07, 3.80268787125715E-07, 3.8428883897268E-07},
    { 3.02247087802243E-07, 3.0325044519941E-07, 6.04493862965683E-07, 6.08507960464522E-07},
    { 6.45327844495114E-08, 6.55440999253187E-08, 1.29065526266459E-07, 1.33110816591397E-07},
    { 4.02965838475211E-08, 4.13054976888816E-08, 8.05931534841875E-08, 8.46289225364671E-08},
    { 5.11229515609557E-07, 5.12224545445861E-07, 1.02245836330894E-06, 1.02643905108835E-06},
    { 1.18189390718726E-07, 1.19196172931879E-07, 2.36378838280871E-07, 2.40405839235791E-07},
    { 3.81603527443986E-07, 3.82614828708938E-07, 7.63207339105065E-07, 7.67251918887268E-07},
    { 3.34147500780091E-08, 3.43787576184695E-08, 6.67881465687969E-08, 7.06435514530313E-08},
    { 3.4107115709503E-07, 3.42081634130409E-07, 6.82142385244333E-07, 6.86184307596704E-07},
    { 1.58899425173331E-07, 1.59907060037767E-07, 3.17798821924953E-07, 3.21829773497484E-07},
    { 5.36395234007614E-07, 5.37394143407255E-07, 1.07279079486489E-06, 1.07678548033618E-06},
    { 3.9406657492691E-07, 3.95064475355866E-07, 7.88133220908094E-07, 7.92124779991354E-07},
    { 2.77072558674263E-07, 2.78073869708351E-07, 5.54145316300492E-07, 5.58150745177954E-07},
    { 3.93172285839682E-07, 3.94176680629243E-07, 7.8634455746851E-07, 7.90362207681028E-07},
    { 2.85718613213248E-07, 2.86716215214256E-07, 5.71436899576838E-07, 5.75428146021295E-07},
    { 1.26677917933193E-07, 1.27688309703444E-07, 2.5335566533613E-07, 2.57397502423373E-07},
    { 8.70625598281549E-08, 8.80735200325944E-08, 1.74125048602036E-07, 1.7816884678723E-07},
    { 9.37471753559294E-08, 9.47451752608686E-08, 1.87494393344423E-07, 1.91486549283582E-07},
    { 3.30706200202258E-08, 3.40820349720161E-08, 6.61412968838704E-08, 7.01866440522281E-08},
    { 9.30588299752344E-08, 9.40660100923196E-08, 1.86117645739614E-07, 1.90146408840519E-07},
    { 3.83720646368602E-08, 3.9382058503179E-08, 7.67441576954297E-08, 8.07838489436108E-08},
    { 3.12277990133225E-07, 3.13278405883466E-07, 6.2455600868816E-07, 6.28557373261174E-07},
    { 2.62851173715717E-07, 2.63860243876479E-07, 5.25702404274853E-07, 5.29738812815594E-07},
    { 6.38404742403509E-08, 6.48505391609433E-08, 1.27680962691556E-07, 1.3172102342196E-07},
    { 7.73673903609051E-07, 7.74670056102877E-07, 1.54734770774212E-06, 1.55133201928948E-06},
    { 2.23658261688797E-07, 2.24661988568187E-07, 4.47316281793064E-07, 4.51331573003699E-07},
    { 3.18664490350784E-07, 3.19664522407948E-07, 6.37329037544987E-07, 6.41328341544067E-07},
    { 6.28178327133355E-08, 6.38285229115354E-08, 1.25635679637526E-07, 1.29678596749727E-07},
    { 2.14196546721723E-07, 2.15193523445123E-07, 4.28392880280626E-07, 4.32381142445593E-07},
    { 2.53436667208007E-08, 2.63544563949836E-08, 5.06872623873278E-08, 5.47306626685895E-08},
    { 1.79067058070359E-07, 1.80076483502489E-07, 3.58134101929863E-07, 3.62171505230435E-07},
    { 1.14243619009358E-08, 1.24315846505851E-08, 2.28486385367432E-08, 2.69068462005454E-08},
    { 1.62669806513804E-08, 1.72783103380425E-08, 3.25339186701967E-08, 3.65793084711186E-08},
    { 2.55726050113481E-07, 2.56733443393387E-07, 5.11451958118414E-07, 5.15481701768294E-07},
    { 6.74249776011493E-08, 6.84268144368616E-08, 1.34849770461187E-07, 1.38856961484635E-07},
    { 6.02680273686929E-07, 6.03684966904439E-07, 1.20536046210873E-06, 1.20937964709356E-06},
    { 1.45373832083351E-07, 1.46384664390098E-07, 2.90747806275249E-07, 2.94791007604545E-07},
    { 3.46929951433594E-07, 3.4793785630427E-07, 6.93859675493513E-07, 6.9789133760878E-07},
    { 2.86301002461187E-07, 2.87301332946299E-07, 5.72601834392117E-07, 5.76603540025644E-07},
    { 8.92922287221154E-08, 9.02999346408251E-08, 1.78584627974487E-07, 1.82615082167104E-07},
    { 1.61394524411662E-07, 1.62388303692751E-07, 3.22789020401615E-07, 3.26764009628278E-07},
    { 3.15667904260408E-07, 3.16657960297562E-07, 6.31335680623124E-07, 6.35296956374987E-07},
    { 3.65886961617434E-07, 3.66888201597249E-07, 7.31774122186835E-07, 7.35779138949511E-07},
    { 5.1565010039667E-08, 5.25749044300028E-08, 1.03130020079334E-07, 1.07168872887087E-07},
    { 2.49045257305625E-07, 2.50053105332881E-07, 4.98090642508942E-07, 5.02122333045918E-07},
    { 3.89293546732006E-07, 3.90250974646733E-07, 7.78586866090336E-07, 7.82417473033092E-07},
    { 1.28008643685007E-07, 1.29012775573756E-07, 2.56017258948305E-07, 2.60033857557573E-07},
    { 3.59406385541661E-07, 3.60415128852765E-07, 7.18812870559304E-07, 7.2284757379748E-07},
    { 7.04774407722653E-08, 7.14837113946487E-08, 1.40955009442223E-07, 1.44979907190645E-07},
    { 4.69074876718878E-08, 4.79159325550427E-08, 9.38164674835207E-08, 9.78508154503288E-08},
    { 3.46929951433594E-07, 3.4793785630427E-07, 6.93859675493513E-07, 6.9789133760878E-07},
    { 2.52614157147946E-07, 2.5362039934862E-07, 5.05228484826148E-07, 5.09253013092348E-07},
    { 1.752849669856E-07, 1.76295728238074E-07, 3.50569891338637E-07, 3.54612936348531E-07},
    { 9.97295472870974E-08, 1.00740251696152E-07, 1.99459023519921E-07, 2.03501656415028E-07},
    { 1.27083694678731E-08, 1.37173685743619E-08, 2.54168952551481E-08, 2.94820375698493E-08},
    { 2.88675792603499E-07, 2.89683143250841E-07, 5.77351315200758E-07, 5.81380774633544E-07},
    { 2.2199529325917E-07, 2.22990578890858E-07, 4.43990884946288E-07, 4.47971643779965E-07},
    { 3.101136769601E-08, 3.20127782060808E-08, 6.20226217051822E-08, 6.60284200648675E-08},
    { 6.34119317055593E-08, 6.44227355905969E-08, 1.26823891832828E-07, 1.30867206848961E-07},
    { 2.27363642579803E-07, 2.28347417419172E-07, 4.54726489351742E-07, 4.5866131870298E-07},
    { 5.761698673723E-08, 5.86278900982506E-08, 1.15233817155058E-07, 1.19277274279739E-07},
    { 5.59963240220895E-08, 5.70068436900328E-08, 1.11992690676743E-07, 1.16034897246209E-07},
    { 1.00404065506154E-07, 1.01413306197173E-07, 2.00808131012309E-07, 2.04845150619803E-07},
    { 1.51791383018463E-07, 1.52801391095636E-07, 3.03582666560942E-07, 3.07622912032457E-07},
    { 7.73673903609051E-07, 7.74670056102877E-07, 1.54734770774212E-06, 1.55133201928948E-06},
    { 6.7849948948151E-07, 6.79496110933542E-07, 1.35699917791499E-06, 1.36098604741619E-06},
    { 6.46773514745291E-08, 6.56873169191385E-08, 1.29354639000212E-07, 1.33394671308906E-07},
    { 2.14274791687785E-07, 2.15266346970111E-07, 4.28549661535271E-07, 4.32515697923463E-07},
    { 1.75431694060535E-07, 1.76439094445868E-07, 3.50863423648207E-07, 3.54893018084113E-07},
    { 1.54840201105344E-07, 1.55849093630422E-07, 3.09680387999833E-07, 3.13715879940446E-07},
    { 3.03876547036452E-07, 3.04886683011318E-07, 6.07753179338033E-07, 6.11793765870061E-07},
    { 2.87581904956369E-07, 2.88585042085288E-07, 5.7516393781043E-07, 5.79176422377259E-07},
    { 2.4641536811032E-08, 2.56472958426457E-08, 4.92830878329187E-08, 5.33060031671084E-08},
    { 2.38351063330811E-08, 2.48459812723922E-08, 4.76701984553074E-08, 5.17136911071248E-08},
    { 5.06429515212403E-07, 5.07291815665667E-07, 1.01285916542793E-06, 1.0163081114456E-06},
    { 3.16136237188402E-08, 3.26210596313103E-08, 6.32272758593899E-08, 6.72569697712788E-08},
    { 3.48408235595343E-08, 3.58516203391446E-08, 6.96816471190687E-08, 7.37248200266549E-08},
    { 7.48174926457068E-08, 7.58264846467682E-08, 1.49634963975132E-07, 1.5367105987707E-07},
    { 1.93732120123968E-07, 1.94739627090712E-07, 3.87464289985928E-07, 3.9149432495833E-07},
    { 3.07901757423679E-08, 3.17564925467195E-08, 6.17883486597748E-08, 6.55261729320955E-08},
    { 4.99771878992306E-07, 5.00766930144891E-07, 9.99543672719483E-07, 1.00352374943213E-06},
    { 1.17716950853719E-07, 1.18722240927127E-07, 2.35433859074874E-07, 2.39455012263079E-07},
    { 3.58876590667023E-08, 3.6895563937378E-08, 7.17753749768235E-08, 7.58068807726886E-08},
    { 4.9829156978376E-07, 4.99225798478165E-07, 9.96582841139571E-07, 1.0003212125298E-06},
    { 3.75331644875132E-07, 3.76332543794433E-07, 7.50663197379708E-07, 7.54666956481742E-07},
    { 2.08074482088705E-08, 2.18164970533508E-08, 4.16148893123136E-08, 4.56511770607904E-08},
    { 2.65227200202389E-08, 2.75340639177557E-08, 5.30454471459052E-08, 5.70908014196903E-08},
    { 9.48590397342741E-08, 9.57736716600266E-08, 1.89718221577095E-07, 1.93376600066131E-07},
    { 1.26375994113914E-08, 1.36486946189507E-08, 2.52751917173555E-08, 2.93195441258831E-08},
    { 1.50946419807951E-07, 1.51955084959354E-07, 3.01892768561629E-07, 3.05927514432369E-07},
    { 9.22171139450256E-08, 9.32281949417302E-08, 1.84434327366034E-07, 1.88478459506314E-07},
    { 9.69194388744654E-07, 9.70171512904017E-07, 1.93838912565525E-06, 1.94229757255471E-06},
    { 5.55253905076825E-08, 5.65363365012672E-08, 1.11050773909938E-07, 1.15094508146285E-07},
    { 1.13003537194345E-07, 1.13988456007519E-07, 2.26006307002535E-07, 2.29945861462966E-07},
    { 1.2296865747885E-07, 1.23975510746277E-07, 2.45937201270863E-07, 2.49964827503391E-07},
    { 4.48469990033118E-08, 4.58578668371956E-08, 8.96939909011962E-08, 9.37375403964325E-08},
    { 5.0952010610672E-08, 5.19609670845966E-08, 1.01904042537626E-07, 1.05939804484478E-07},
    { 4.86714952785405E-08, 4.96825975915272E-08, 9.73430047679358E-08, 1.01387534812147E-07},
    { 7.39467296284602E-08, 7.49580095771307E-08, 1.47893480573202E-07, 1.51938643000449E-07},
    { 1.0657652893542E-07, 1.07576717311986E-07, 2.13153157346824E-07, 2.17153882431376E-07},
    { 2.98015478961133E-08, 3.08124867842707E-08, 5.96030602650899E-08, 6.36467447634459E-08},
    { 4.8263878227317E-07, 4.83640725690293E-07, 9.65278132980529E-07, 9.69288855401373E-07},
    { 3.14623996189312E-07, 3.15605028333721E-07, 6.29259531592652E-07, 6.33185010201487E-07},
    { 2.29475041635396E-07, 2.30466028483534E-07, 4.5895009037622E-07, 4.62914151455607E-07},
    { 4.36916394619402E-07, 4.37877190506697E-07, 8.7382976232675E-07, 8.77673194565887E-07},
    { 4.38233769273211E-08, 4.4833392109922E-08, 8.76467751709242E-08, 9.16867648470543E-08},
    { 1.28198685445113E-07, 1.29205858456771E-07, 2.56397569842193E-07, 2.60425942144593E-07},
    { 1.72098651773922E-07, 1.73105441092503E-07, 3.44197339074981E-07, 3.48224730828406E-07},
    { 1.79612058559542E-08, 1.79676149514307E-08, 3.59441060869162E-08, 3.59159741947224E-08}
    };

    typedef bg::model::linestring<P> linestring;
    typedef bg::model::polygon<P> polygon;
    typedef typename bg::coordinate_type<P>::type coor_type;

    if (BOOST_GEOMETRY_CONDITION((std::is_same<coor_type, float>::value)))
    {
      std::cout << "This unit test can't be tested with float,"
                << " the coordinate values are too small." << std::endl;
      return;
    }


    int const n = sizeof(testcases_aimes) / sizeof(testcases_aimes[0]);
    int const ne = sizeof(expectations) / sizeof(expectations[0]);
    BOOST_ASSERT(n == ne);

    boost::ignore_unused(n, ne);

    bg::strategy::buffer::join_miter join_miter;
    bg::strategy::buffer::join_round join_round(100);
    bg::strategy::buffer::end_flat end_flat;
    bg::strategy::buffer::end_round end_round(100);

    // Aimes tested originally with 0.000018 degrees (around 2 m)
    ut_settings settings;
    settings.use_ln_area = true;
    settings.tolerance = 0.001;

    int expectation_index = 0;
    for (int width = 18; width <= 36; width += 18, expectation_index += 2)
    {
        double const aimes_width = width / 1000000.0;
        for (int i = 0; i < n; i++)
        {
#if defined(BOOST_GEOMETRY_USE_RESCALING) || defined(BOOST_GEOMETRY_TEST_FAILURES)
            // There are 4 false positives
            bool const possible_invalid = width == 18 && (i == 75 || i == 80 || i == 140);
            settings.set_test_validity(! possible_invalid);
#endif
            std::ostringstream name;
            try
            {
                name << "aimes_" << i << "_" << width;
                test_one<linestring, polygon>
                (
                    name.str(), testcases_aimes[i], join_miter, end_flat,
                    expectations[i][expectation_index],
                    aimes_width, settings
                );
                test_one<linestring, polygon>
                (
                    name.str(), testcases_aimes[i], join_round, end_round,
                    expectations[i][expectation_index + 1],
                    aimes_width, settings
                );
            }
            catch(std::exception const& e)
            {
                std::cout << "Exception: " << e.what() << " in " << name.str() << std::endl;
            }
        }
    }
}


int test_main(int, char* [])
{
    BoostGeometryWriteTestConfiguration();

    test_aimes<bg::model::point<default_test_type, 2, bg::cs::cartesian> >();

#if defined(BOOST_GEOMETRY_TEST_FAILURES)
    // Type float is not supported for these cases.
    // Type double has (judging the svg) 4 false negatives for validity
    BoostGeometryWriteExpectedFailures(0, 0, 0, 0);
#endif

    return 0;
}
